<?php
namespace App\Controllers;

use App\Services\DealService;
use App\Services\VirtualStageService;
use App\Repositories\PersonRepository;
use App\Repositories\DealRepository;
use App\Services\AuthService;
use App\Services\AuditService;
use App\Utils\Sanitizer;
use App\Utils\Logger;

class DealController extends ApiController
{
    private $dealService;
    private $virtualStageService;
    private $personRepo;
    private $dealRepo;
    private $authService;
    private $auditService;
    private $db;

    public function __construct(
        DealService $dealService,
        VirtualStageService $virtualStageService,
        PersonRepository $personRepo,
        DealRepository $dealRepo,
        AuthService $authService,
        AuditService $auditService,
        $db = null
    ) {
        $this->dealService = $dealService;
        $this->virtualStageService = $virtualStageService;
        $this->personRepo = $personRepo;
        $this->dealRepo = $dealRepo;
        $this->authService = $authService;
        $this->auditService = $auditService;
        $this->db = $db;
    }

    public function getDeals()
    {
        $isAdmin = $this->authService->isAdmin();
        $isAdminOrCrm = $this->authService->isAdminOrCrmSpecialist();
        $didarId = $_SESSION['didar_id'] ?? '';
        
        $filters = [];
        if (!$isAdminOrCrm && !empty($didarId)) {
            $filters['owner_didar_id'] = $didarId;
        }

        // Admin/CRM can filter by owner
        $ownerId = Sanitizer::sanitize($_POST['owner_id'] ?? '');
        if ($isAdminOrCrm && $ownerId) {
            $filters['owner_didar_id'] = $ownerId;
        }
        
        $status = Sanitizer::sanitize($_POST['status'] ?? 'all');
        $search = Sanitizer::sanitize($_POST['search'] ?? '');
        $pipelineStage = Sanitizer::sanitize($_POST['pipeline_stage'] ?? 'all');
        $legacy = $_POST['legacy'] ?? '';
        $limit = isset($_POST['limit']) ? (int)$_POST['limit'] : null;
        $offset = isset($_POST['offset']) ? (int)$_POST['offset'] : null;
        $cursorTime = Sanitizer::sanitize($_POST['cursor_time'] ?? '');
        $cursorId = Sanitizer::sanitize($_POST['cursor_id'] ?? '');
        
        if ($status !== 'all') {
            $filters['status'] = $status;
        }
        
        if ($pipelineStage !== 'all') {
            $filters['pipeline_stage'] = $pipelineStage;
        }
        
        if ($search) {
            $filters['search'] = $search;
        }

        if ($legacy !== '') {
            $filters['legacy'] = Sanitizer::sanitize($legacy);
        }
        if ($limit !== null) {
            $filters['limit'] = $limit;
        }
        if ($offset !== null) {
            $filters['offset'] = $offset;
        }
        if ($cursorTime) {
            $filters['cursor_time'] = $cursorTime;
        }
        if ($cursorId) {
            $filters['cursor_id'] = $cursorId;
        }
        
        $deals = $this->dealService->getDeals($filters);
        // Attach products for each deal
        $deals = array_map(function($d) {
            $d['products'] = $this->dealRepo->getProductsByDealDidarId($d['didar_deal_id']);
            return $d;
        }, $deals);
        $this->successResponse(['deals' => $deals]);
    }

    private function logFieldChanges(string $entityType, string $entityId, array $oldData, array $newData): void
    {
        if (!$this->db) {
            \App\Utils\Logger::logError("DealController.logFieldChanges: DB connection not initialized");
            return;
        }
        $actorId = $_SESSION['didar_id'] ?? 'system';
        $actorName = $_SESSION['name'] ?? 'System';
        $pdo = $this->db->getPdo();
        
        $sql = "INSERT INTO field_audit_logs (entity_type, entity_id, field_name, old_value, new_value, actor_didar_id, actor_name) 
                VALUES (?, ?, ?, ?, ?, ?, ?)";
        $stmt = $pdo->prepare($sql);
        
        foreach ($newData as $field => $newValue) {
            $oldValue = $oldData[$field] ?? null;
            
            // Normalize values for comparison
            $normOld = is_array($oldValue) ? json_encode($oldValue) : (string)$oldValue;
            $normNew = is_array($newValue) ? json_encode($newValue) : (string)$newValue;
            
            if ($normOld !== $normNew) {
                $stmt->execute([
                    $entityType,
                    $entityId,
                    $field,
                    $normOld,
                    $normNew,
                    $actorId,
                    $actorName
                ]);
            }
        }
    }

    public function getLegacyDeals()
    {
        $contactId = Sanitizer::sanitize($_POST['contact_id'] ?? '');
        if (empty($contactId)) {
            $this->errorResponse('O\'U+OO3UØ O\'OrOæ OU,OýOU.UO OO3O¦');
        }

        $person = $this->personRepo->findByIdOrDidarId($contactId);
        if (!$person) {
            $this->errorResponse('O\'OrOæ UOOU?O¦ U+O\'O_');
        }

        $legacyDeals = $this->dealRepo->findByContactIdLegacy($person->didar_contact_id, true);
        $legacyArray = array_map(function($deal) {
            return $deal->toArray();
        }, $legacyDeals);

        $this->successResponse(['deals' => $legacyArray]);
    }

    public function save()
    {
        // CRM specialist is view-only, cannot create or edit deals
        if ($this->authService->isCrmSpecialist()) {
            $this->errorResponse('دسترسی محدود - کارشناس CRM فقط مجاز به مشاهده است');
        }
        
        $products = [];
        if (!empty($_POST['product_ids'])) {
            if (is_array($_POST['product_ids'])) {
                $products = $_POST['product_ids'];
            } else {
                # comma separated
                $products = array_filter(array_map('trim', explode(',', $_POST['product_ids'])));
            }
        }

        # Requested services (can be array, JSON string, or comma separated)
        $requestedServices = [];
        if (isset($_POST['requested_services'])) {
            if (is_array($_POST['requested_services'])) {
                $requestedServices = $_POST['requested_services'];
            } else {
                $raw = trim((string)$_POST['requested_services']);
                if ($raw != '') {
                    $decoded = json_decode($raw, true);
                    if (is_array($decoded)) {
                        $requestedServices = $decoded;
                    } else {
                        $requestedServices = array_filter(array_map('trim', explode(',', $raw)));
                    }
                }
            }
        }
        // اگر هنوز خالی است ولی عنوان خدمات قابل ارائه موجود است، آن را هم به عنوان خدمت درخواستی اضافه کن
        if (empty($requestedServices) && !empty($_POST['title'])) {
            $requestedServices = [Sanitizer::sanitize($_POST['title'])];
        }

        $isNewDeal = empty($_POST['deal_id']);
        \App\Utils\Logger::logInfo('DealController.save start', [
            'is_new' => $isNewDeal,
            'incoming_title' => $_POST['title'] ?? '',
            'contact_id' => $_POST['contact_id'] ?? '',
            'deal_id' => $_POST['deal_id'] ?? '',
            'user_id' => $_SESSION['didar_id'] ?? 'unknown',
            'user_name' => $_SESSION['name'] ?? 'unknown',
            'form_data_keys' => array_keys($_POST),
            'server_time' => date('Y-m-d H:i:s'),
            'session_id' => session_id()
        ]);
        $title = Sanitizer::sanitize($_POST['title'] ?? '');
        $rawContactId = Sanitizer::sanitize($_POST['contact_id'] ?? '');
        $resolvedPerson = $this->resolvePersonByContactId($rawContactId);
        $normalizedContactId = $resolvedPerson ? $resolvedPerson->didar_contact_id : $rawContactId;

        // For new deals, generate title automatically if not provided
        if ($isNewDeal && empty($title)) {
            try {
                if (!empty($normalizedContactId)) {
                    $title = $this->dealService->generateDealTitle($normalizedContactId);
                }
            } catch (\Exception $e) {
                \App\Utils\Logger::logError('Failed to generate deal title', $e, ['contact_id' => $_POST['contact_id'] ?? '']);
                // Continue with empty title - will be caught by validation
            }
        }

        if ($isNewDeal && $this->isGeneratedDealTitle($title)) {
            Logger::logWarning('Deal save blocked: generated title used as service', [
                'deal_title' => $title,
                'contact_id' => $_POST['contact_id'] ?? '',
                'deal_id' => $_POST['deal_id'] ?? '',
                'user_id' => $_SESSION['didar_id'] ?? 'unknown',
                'user_name' => $_SESSION['name'] ?? 'unknown',
                'server_time' => date('Y-m-d H:i:s')
            ]);
            $this->errorResponse("\u{0644}\u{0637}\u{0641}\u{0627}\u{064b} \u{062e}\u{062f}\u{0645}\u{0627}\u{062a} \u{0642}\u{0627}\u{0628}\u{0644} \u{0627}\u{0631}\u{0627}\u{0626}\u{0647} \u{0631}\u{0627} \u{0627}\u{0646}\u{062a}\u{062e}\u{0627}\u{0628} \u{06a9}\u{0646}\u{06cc}\u{062f}.");
        }

        // VALIDATION: Ensure deal phone matches contact's primary phone
        if (!empty($title) && !empty($_POST['contact_id'])) {
            Logger::logInfo('DEAL_PHONE_VALIDATION_START', [
                'deal_title' => $title,
                'contact_id' => $_POST['contact_id'],
                'is_new_deal' => $isNewDeal
            ]);

            $dealPhone = $this->extractPhoneFromDealTitle($title);
            $contactPrimaryPhone = $this->getContactPrimaryPhone($_POST['contact_id']);

            Logger::logInfo('DEAL_PHONE_VALIDATION_EXTRACTED', [
                'deal_phone' => $dealPhone,
                'contact_primary_phone' => $contactPrimaryPhone
            ]);

            if (!empty($dealPhone) && !empty($contactPrimaryPhone)) {
                // Normalize both phones for comparison
                $normalizedDealPhone = $this->normalizePhoneForComparison($dealPhone);
                $normalizedContactPhone = $this->normalizePhoneForComparison($contactPrimaryPhone);

                Logger::logInfo('DEAL_PHONE_VALIDATION_NORMALIZED', [
                    'normalized_deal_phone' => $normalizedDealPhone,
                    'normalized_contact_phone' => $normalizedContactPhone,
                    'phones_match' => $normalizedDealPhone === $normalizedContactPhone
                ]);

                if ($normalizedDealPhone !== $normalizedContactPhone) {
                    Logger::logWarning('DEAL_PHONE_VALIDATION_FAILED: Deal phone does not match contact primary phone', [
                        'deal_title' => $title,
                        'extracted_deal_phone' => $dealPhone,
                        'normalized_deal_phone' => $normalizedDealPhone,
                        'contact_primary_phone' => $contactPrimaryPhone,
                        'normalized_contact_phone' => $normalizedContactPhone,
                        'contact_id' => $_POST['contact_id'],
                        'user_id' => $_SESSION['didar_id'] ?? 'unknown',
                        'user_name' => $_SESSION['name'] ?? 'unknown',
                        'server_time' => date('Y-m-d H:i:s')
                    ]);
                    $this->errorResponse('شماره تماس معامله با شماره تماس اصلی لید مطابقت ندارد. لطفاً از شماره صحیح استفاده کنید.');
                }

                Logger::logInfo('DEAL_PHONE_VALIDATION_PASSED', [
                    'deal_title' => $title,
                    'deal_phone' => $dealPhone,
                    'contact_phone' => $contactPrimaryPhone,
                    'contact_id' => $_POST['contact_id'],
                    'validation' => 'PASSED'
                ]);
            } else {
                Logger::logInfo('DEAL_PHONE_VALIDATION_SKIPPED', [
                    'reason' => 'missing_deal_phone_or_contact_phone',
                    'deal_phone' => $dealPhone,
                    'contact_primary_phone' => $contactPrimaryPhone,
                    'contact_id' => $_POST['contact_id'] ?? '',
                    'deal_id' => $_POST['deal_id'] ?? '',
                    'is_new_deal' => $isNewDeal,
                    'incoming_title' => $title,
                    'user_id' => $_SESSION['didar_id'] ?? 'unknown',
                    'user_name' => $_SESSION['name'] ?? 'unknown',
                    'server_time' => date('Y-m-d H:i:s')
                ]);
            }
        }

        // Payments and financial fields (optional for new deals)
        $paymentsRaw = $_POST['payments'] ?? '';
        if (is_array($paymentsRaw)) {
            $paymentsRaw = json_encode($paymentsRaw, JSON_UNESCAPED_UNICODE);
        }
        $paymentsPayload = [];
        if (!empty($paymentsRaw)) {
            $decodedPayments = json_decode($paymentsRaw, true);
            if (is_array($decodedPayments)) {
                // Normalize payment keys to standard format before processing
                $paymentsPayload = array_map(function($payment) {
                    // Normalize card_last4 (support both card_last4 and card_last_4, use card_last4 as standard)
                    if (isset($payment['card_last_4']) && !isset($payment['card_last4'])) {
                        $payment['card_last4'] = $payment['card_last_4'];
                    }
                    // Normalize destination_bank (support both destination_bank and bank_destination, use destination_bank as standard)
                    if (isset($payment['bank_destination']) && !isset($payment['destination_bank'])) {
                        $payment['destination_bank'] = $payment['bank_destination'];
                    }
                    // Normalize method (support both method and payment_method, use method as standard)
                    if (isset($payment['payment_method']) && !isset($payment['method'])) {
                        $payment['method'] = $payment['payment_method'];
                    }
                    return $payment;
                }, $decodedPayments);
                
                // ✅ Validate payment fields based on field configuration
                // This respects field permissions and conditional requirements
                global $app;
                if (isset($app->services->fieldPermission)) {
                    $userRole = $_SESSION['role'] ?? 'agent';
                    $isExisting = !$isNewDeal;
                    
                    // Load existing deal for comparison if updating
                    $existingPayments = [];
                    $existingDealData = null;
                    if (!$isNewDeal) {
                        $existingDealData = $this->dealRepo->findByDidarId(Sanitizer::sanitize($_POST['deal_id']));
                        if ($existingDealData && !empty($existingDealData->payments)) {
                            $existingPaymentsRaw = json_decode($existingDealData->payments, true) ?: [];
                            // Normalize existing payments keys to standard format
                            $existingPayments = array_map(function($payment) {
                                // Normalize card_last4 (support both card_last4 and card_last_4, use card_last4 as standard)
                                if (isset($payment['card_last_4']) && !isset($payment['card_last4'])) {
                                    $payment['card_last4'] = $payment['card_last_4'];
                                }
                                // Normalize destination_bank (support both destination_bank and bank_destination, use destination_bank as standard)
                                if (isset($payment['bank_destination']) && !isset($payment['destination_bank'])) {
                                    $payment['destination_bank'] = $payment['bank_destination'];
                                }
                                // Normalize method (support both method and payment_method, use method as standard)
                                if (isset($payment['payment_method']) && !isset($payment['method'])) {
                                    $payment['method'] = $payment['payment_method'];
                                }
                                return $payment;
                            }, $existingPaymentsRaw);
                        }
                    }

                    // Check each payment record for field-level constraints
                    // Note: paymentsPayload and existingPayments are already normalized at this point
                    foreach ($paymentsPayload as $idx => $payment) {
                        $existingPayment = $existingPayments[$idx] ?? null;
                        $isNewPayment = ($existingPayment === null);

                        // Helper to check if a field has actually changed
                        // Since payments are normalized, we only need to check the standard key
                        $hasChanged = function($fieldName, $newValue) use ($existingPayment) {
                            if (!$existingPayment) return true; // New record
                            
                            // Get old value using standard key (payments are normalized)
                            $oldValue = $existingPayment[$fieldName] ?? null;
                            
                            // Normalize: treat null, empty string, and '0' as empty
                            $oldNormalized = ($oldValue === null || $oldValue === '' || $oldValue === '0') ? '' : trim((string)$oldValue);
                            $newNormalized = ($newValue === null || $newValue === '' || $newValue === '0') ? '' : trim((string)$newValue);
                            
                            return $oldNormalized !== $newNormalized;
                        };

                        // At this point, payment keys are already normalized, so use standard keys directly
                        $methodValue = $payment['method'] ?? '';
                        $cardValue   = $payment['card_last4'] ?? '';
                        $bankValue   = $payment['destination_bank'] ?? '';

                        // 1. Validate payment_method with editable_after_create support
                        if ($hasChanged('method', $methodValue) && !empty($methodValue)) {
                            if (!$app->services->fieldPermission->canUserEditField('payment_method', $userRole, !$isNewPayment)) {
                                \App\Utils\Logger::logWarning('DEAL_PAYMENT_METHOD_EDIT_DENIED', [
                                    'deal_id' => $_POST['deal_id'] ?? '',
                                    'payment_idx' => $idx,
                                    'user_role' => $userRole,
                                    'method_value' => $methodValue,
                                    'is_new_payment' => $isNewPayment,
                                    'old_value' => $existingPayment['method'] ?? null,
                                    'new_value' => $methodValue
                                ]);
                                $this->errorResponse("شما مجاز به تغییر نحوه پرداخت نیستید (ردیف " . ($idx + 1) . ")");
                            }
                        }

                        // 2. Validate conditional requirements using FieldPermissionService
                        // This replaces hardcoded card_last4 logic
                        $paymentDataForLogic = [
                            'payment_method' => $methodValue,
                            'method' => $methodValue,
                            'destination_bank' => $bankValue,
                            'card_last4' => $cardValue,
                            // Add keys that might be used in conditional logic strings
                            'payment_methods' => $methodValue,
                            'payment_destination_banks' => $bankValue
                        ];

                        // Check if card_last4 is required based on conditional logic in settings
                        if ($app->services->fieldPermission->checkConditionalVisibility('payment_card_last4', $paymentDataForLogic)) {
                            if ($app->services->fieldPermission->isFieldRequired('payment_card_last4') && empty($cardValue)) {
                                $this->errorResponse("4 رقم آخر کارت الزامی است (ردیف " . ($idx + 1) . ")");
                            }
                        }

                        // Check if destination_bank is required based on conditional logic in settings
                        if ($app->services->fieldPermission->checkConditionalVisibility('destination_bank', $paymentDataForLogic)) {
                            if ($app->services->fieldPermission->isFieldRequired('destination_bank') && empty($bankValue)) {
                                $this->errorResponse("انتخاب بانک مقصد الزامی است (ردیف " . ($idx + 1) . ")");
                            }
                        }
                        
                        // 3. Validate card_last4 with editable_after_create support
                        if ($hasChanged('card_last4', $cardValue) && !empty($cardValue)) {
                            if (!$app->services->fieldPermission->canUserEditField('payment_card_last4', $userRole, !$isNewPayment)) {
                                \App\Utils\Logger::logWarning('DEAL_PAYMENT_CARD_EDIT_DENIED', [
                                    'deal_id' => $_POST['deal_id'] ?? '',
                                    'payment_idx' => $idx,
                                    'user_role' => $userRole,
                                    'card_value' => $cardValue,
                                    'is_new_payment' => $isNewPayment,
                                    'old_value' => $existingPayment['card_last4'] ?? null,
                                    'new_value' => $cardValue
                                ]);
                                $this->errorResponse("شما مجاز به تغییر 4 رقم کارت نیستید (ردیف " . ($idx + 1) . ")");
                            }
                            
                            // Apply validation pattern from config if set, otherwise use default 4-digit regex
                            $config = $app->services->fieldPermission->getFieldConfig('payment_card_last4');
                            $pattern = $config['validation_pattern'] ?? '^\d{4}$';
                            if (!empty($pattern)) {
                                if (@preg_match('/' . str_replace('/', '\/', $pattern) . '/', $cardValue) === 0) {
                                    $this->errorResponse($config['validation_message'] ?? "قالب 4 رقم آخر کارت صحیح نیست (ردیف " . ($idx + 1) . ")");
                                }
                            }
                        }
                        
                        // 4. Validate destination_bank with editable_after_create support
                        if ($hasChanged('destination_bank', $bankValue) && !empty($bankValue)) {
                            if (!$app->services->fieldPermission->canUserEditField('destination_bank', $userRole, !$isNewPayment)) {
                                \App\Utils\Logger::logWarning('DEAL_PAYMENT_BANK_EDIT_DENIED', [
                                    'deal_id' => $_POST['deal_id'] ?? '',
                                    'payment_idx' => $idx,
                                    'user_role' => $userRole,
                                    'bank_value' => $bankValue,
                                    'is_new_payment' => $isNewPayment,
                                    'old_value' => $existingPayment['destination_bank'] ?? null,
                                    'new_value' => $bankValue,
                                    'config_found' => $app->services->fieldPermission->getFieldConfig('destination_bank')
                                ]);
                                $this->errorResponse("شما مجاز به تغییر بانک مقصد نیستید (ردیف " . ($idx + 1) . ")");
                            }
                        }
                    }

                    // Prevent deletion of payments without permission (use canUserDeleteField)
                    if (count($existingPayments) > count($paymentsPayload)) {
                        if (!$app->services->fieldPermission->canUserDeleteField('payment_delete', $userRole)) {
                            $this->errorResponse('شما مجاز به حذف پرداخت‌ها نیستید');
                        }
                    }
                }
            }
        }

        $statusInput = Sanitizer::sanitize($_POST['deal_status'] ?? '');
        $pipelineStageInput = Sanitizer::sanitize($_POST['pipeline_stage'] ?? '');
        $stageMeansSuccess = in_array($pipelineStageInput, ['deal_success', 'اتمام']);
        $stageMeansFailed = in_array($pipelineStageInput, ['deal_failed', 'معامله ناموفق']);
        if ($stageMeansSuccess) {
            $statusInput = 'won';
            $pipelineStageInput = 'اتمام';
        } elseif ($stageMeansFailed) {
            $statusInput = 'lost';
            $pipelineStageInput = 'معامله ناموفق';
        }
        
        // If status is 'lost' and pipeline_stage is empty, automatically set it to 'معامله ناموفق'
        if ($statusInput === 'lost' && empty($pipelineStageInput)) {
            $pipelineStageInput = 'معامله ناموفق';
        }
        
        $serviceCostInput = $_POST['service_cost'] ?? null;
        $priceListCodeInput = Sanitizer::sanitize($_POST['price_list_code'] ?? '');
        $payableAmountInput = $_POST['payable_amount'] ?? null;
        $paymentConditionsInput = Sanitizer::sanitize($_POST['payment_conditions'] ?? '');

        $normalizeMoney = function ($val): float {
            $s = is_string($val) ? $val : (string)($val ?? '');
            $s = str_replace(',', '', $s);
            $s = preg_replace('/[^\d.]/', '', $s);
            return floatval($s ?: 0);
        };

        $refundAmountInput = $normalizeMoney($_POST['refund_amount'] ?? 0);
        $refundDescriptionInput = Sanitizer::sanitize($_POST['refund_description'] ?? '');

        $failureReasonCodeInput = Sanitizer::sanitize($_POST['failure_reason'] ?? '');
        $failureReasonDescriptionInput = Sanitizer::sanitize($_POST['failure_other_description'] ?? '');
        $failureReasonTextInput = Sanitizer::sanitize($_POST['failure_reason_text'] ?? '');

        $failureReasonTextFromCodeMap = [
            'no_financial_ability' => 'عدم توان مالی',
            'no_purchase_intent' => 'فعلا قصد خرید ندارد',
            'competitor_purchase' => 'خرید از رقبا',
            'wrong_number' => 'شماره اشتباه',
            'irrelevant_lead' => 'لید نامرتبط',
            'other' => 'سایر',
        ];

        $failureReasonTextFromCode = function(string $code) use ($failureReasonTextFromCodeMap): string {
            $map = [
                'no_financial_ability' => 'عدم توان مالی',
                'no_purchase_intent' => 'فعلا قصد خرید ندارد',
                'competitor_purchase' => 'خرید از رقبا',
                'wrong_number' => 'شماره اشتباه',
                'irrelevant_lead' => 'لید نامرتبط',
                'other' => 'سایر',
            ];
            return $map[$code] ?? '';
        };

        $normalizedFailureCode = $failureReasonCodeInput !== '' ? $failureReasonCodeInput : null;
        $normalizedFailureDesc = $failureReasonDescriptionInput !== '' ? $failureReasonDescriptionInput : null;
        $normalizedFailureText = $failureReasonTextInput !== '' ? $failureReasonTextInput : null;

        if ($statusInput === 'lost') {
            // Keep backend in sync with frontend: for lost deals, reason and description are required.
            if (empty($normalizedFailureCode)) {
                if (!empty($normalizedFailureText)) {
                    $reverse = array_flip($failureReasonTextFromCodeMap ?? []);
                    if (isset($reverse[$normalizedFailureText])) {
                        $normalizedFailureCode = $reverse[$normalizedFailureText];
                    }
                }
                if (empty($normalizedFailureCode)) {
                    $this->errorResponse('دلیل ناموفق بودن معامله الزامی است');
                }
            }
            if (empty($normalizedFailureDesc)) {
                $this->errorResponse('توضیحات دلیل ناموفق بودن معامله الزامی است');
            }

            if (empty($normalizedFailureText)) {
                $base = $failureReasonTextFromCode($normalizedFailureCode);
                $normalizedFailureText = $base !== '' ? $base : $normalizedFailureCode;
            }
            if (empty($normalizedFailureText) && !empty($normalizedFailureDesc)) {
                $normalizedFailureText = $normalizedFailureDesc;
            }
        } else {
            // For non-lost deals, clear any sent failure reason data.
            $normalizedFailureCode = null;
            $normalizedFailureDesc = null;
            $normalizedFailureText = null;
        }

        $data = [
            'didar_deal_id' => Sanitizer::sanitize($_POST['deal_id'] ?? ''),
            'contact_didar_id' => $normalizedContactId,
            'title' => $title,
            'status' => $statusInput,
            'pipeline_stage' => $pipelineStageInput,
            'estimated_price' => isset($serviceCostInput) ? $normalizeMoney($serviceCostInput) : null,
            'final_price' => isset($payableAmountInput) ? $normalizeMoney($payableAmountInput) : null,
            'service_cost' => isset($serviceCostInput) ? $normalizeMoney($serviceCostInput) : null,
            'price_list_code' => $priceListCodeInput,
            'has_discount' => intval($_POST['has_discount'] ?? 0),
            'discount_type' => Sanitizer::sanitize($_POST['discount_type'] ?? ''),
            'discount_occasion' => Sanitizer::sanitize($_POST['discount_occasion'] ?? ''),
            'discount_amount' => $normalizeMoney($_POST['discount_amount'] ?? 0),
            'refund_amount' => $refundAmountInput,
            'refund_description' => $refundDescriptionInput,
            'payable_amount' => isset($payableAmountInput) ? $normalizeMoney($payableAmountInput) : null,
            'payment_conditions' => $paymentConditionsInput,
            'payment_amount' => floatval($_POST['payment_amount'] ?? 0),
            'payments' => '',
            'financial_level' => Sanitizer::sanitize($_POST['financial_level'] ?? ''),
            'asset_estimation' => Sanitizer::sanitize($_POST['asset_estimation'] ?? ''),
            'income_estimation' => Sanitizer::sanitize($_POST['income_estimation'] ?? ''),
            'has_previous_purchase' => isset($_POST['has_previous_purchase']) && $_POST['has_previous_purchase'] !== '' ? (int)$_POST['has_previous_purchase'] : null,
            'owner_didar_id' => $_SESSION['didar_id'] ?? '',
            'register_time' => date('Y-m-d H:i:s'),
            'last_sync' => date('Y-m-d H:i:s'),
            'products' => $products,
            'requested_services' => $requestedServices
        ];

        // Use the robust repository method to find the person
        $person = $resolvedPerson ?: $this->resolvePersonByContactId($data['contact_didar_id']);

        if (!$person) {
            $this->errorResponse('لید با این شناسه یافت نشد');
        }

        // Standardize to the real Didar GUID for ALL internal processing
        $data['contact_didar_id'] = $person->didar_contact_id;
        // --- END STRICT LOOKUP ---

        // Failure reason (stored for reporting)
        $data['failure_reason_code'] = $normalizedFailureCode;
        $data['failure_reason_text'] = $normalizedFailureText;
        $data['failure_reason_description'] = $normalizedFailureDesc;

        // ✅ Validation based on field configurations
        global $app;
        if (isset($app->repositories->fieldConfig)) {
            $configs = $app->repositories->fieldConfig->getByEntity('deal');
            
            // Map field_configurations field names to actual POST field names
            $fieldNameMapping = [
                'deal_title' => 'title',  // deal_title in config maps to title in POST
            ];
            
            foreach ($configs as $config) {
                $fieldName = $config['field_name'];
                $fieldLabel = !empty($config['field_label']) ? $config['field_label'] : $fieldName;

                // Skip payment sub-fields; آنها در بخش پرداخت جداگانه اعتبارسنجی می‌شوند
                // Also skip fields that are actually lookup group names or reference payment-related lookup groups
                $paymentRelatedFields = [
                    'payment_card_last4', 'payment_card_last_4', 'card_last4', 'card_last_4',
                    'destination_bank', 'bank_destination', 'destination_banks', 'payment_destination_banks',
                    'payment_method', 'payment_methods', 
                    'payment_delete'
                ];
                $paymentRelatedLookups = ['payment_methods', 'payment_destination_banks'];
                
                if (in_array($fieldName, $paymentRelatedFields) || 
                    in_array($config['lookup_group_code'] ?? '', $paymentRelatedLookups)) {
                    continue;
                }

                // Skip lookup groups - they don't exist as direct fields in $_POST for deals
                if (!empty($config['lookup_group_code']) && $fieldName === $config['lookup_group_code']) {
                    continue;
                }

                // Skip inactive fields
                if (empty($config['is_active'])) {
                    continue;
                }

                // Map field name to actual POST field name
                $postFieldName = $fieldNameMapping[$fieldName] ?? $fieldName;
                $val = $_POST[$postFieldName] ?? null;
                
                // Conditional Visibility Check: Only validate if field should be visible
                // Pass $_POST as formData for condition evaluation
                if (!$app->services->fieldPermission->checkConditionalVisibility($fieldName, $_POST)) {
                    continue;
                }

                // Required check - only validate if field exists in POST or is a new deal
                if ($config['is_required'] && (is_null($val) || $val === '')) {
                    \App\Utils\Logger::logWarning('DEAL_FIELD_REQUIRED_VALIDATION_FAILED', [
                        'deal_id' => $_POST['deal_id'] ?? '',
                        'field_name' => $fieldName,
                        'post_field_name' => $postFieldName,
                        'field_label' => $fieldLabel,
                        'is_required' => $config['is_required'],
                        'value' => $val,
                        'is_new_deal' => $isNewDeal,
                        'all_post_keys' => array_keys($_POST)
                    ]);
                    $this->errorResponse("فیلد {$fieldLabel} الزامی است");
                }
                
                // Regex Pattern check
                if (!empty($config['validation_pattern']) && !empty($val)) {
                    // Safety: use @ to suppress errors if regex is invalid
                    if (@preg_match('/' . str_replace('/', '\/', $config['validation_pattern']) . '/', $val) === 0) {
                        $this->errorResponse("قالب فیلد {$fieldLabel} صحیح نیست");
                    }
                }
            }
        }

        // Refund validation (as a discount type)
        if ((int)($data['has_discount'] ?? 0) === 1 && ($data['discount_type'] ?? '') === 'refund') {
            if (($data['refund_amount'] ?? 0) <= 0) {
                $this->errorResponse('مبلغ مرجوعی الزامی است');
            }
            if (empty(trim((string)($data['refund_description'] ?? '')))) {
                $this->errorResponse('توضیحات مرجوعی الزامی است');
            }
        } else {
            $data['refund_amount'] = 0;
            $data['refund_description'] = '';
        }

        $existingDeal = null;
        if (!$isNewDeal && !empty($data['didar_deal_id'])) {
            $existingDeal = $this->dealRepo->findByDidarId($data['didar_deal_id']);
            \App\Utils\Logger::logInfo('DealController.save existing deal lookup', [
                'deal_id' => $data['didar_deal_id'],
                'contact_id' => $data['contact_didar_id'],
                'existing_deal_found' => (bool) $existingDeal,
                'existing_contact_id' => $existingDeal ? $existingDeal->contact_didar_id : null,
                'existing_title' => $existingDeal ? $existingDeal->title : null,
                'user_id' => $_SESSION['didar_id'] ?? 'unknown',
                'user_name' => $_SESSION['name'] ?? 'unknown',
                'server_time' => date('Y-m-d H:i:s')
            ]);
        }
        if ($existingDeal && !empty($existingDeal->contact_didar_id) && $existingDeal->contact_didar_id !== $data['contact_didar_id']) {
            Logger::logWarning('Deal save blocked: deal_id does not belong to contact', [
                'deal_id' => $data['didar_deal_id'],
                'incoming_contact_id' => $data['contact_didar_id'],
                'existing_contact_id' => $existingDeal->contact_didar_id,
                'incoming_title' => $data['title'] ?? '',
                'requested_services' => $data['requested_services'] ?? [],
                'user_id' => $_SESSION['didar_id'] ?? 'unknown',
                'user_name' => $_SESSION['name'] ?? 'unknown',
                'server_time' => date('Y-m-d H:i:s')
            ]);
            $this->errorResponse('شناسه معامله با لید انتخاب‌شده مطابقت ندارد. لطفا معامله درست را انتخاب کنید.');
        }

        // Preserve پایه عنوان معامله (شماره و ترتیب) و فقط خدمات را به انتهای آن اضافه کن
        $baseTitle = $data['title'];
        if ($isNewDeal) {
            try {
                $baseTitle = $this->dealService->generateDealTitle($data['contact_didar_id']);
            } catch (\Exception $e) {
                \App\Utils\Logger::logWarning('Failed to generate base title for new deal', [
                    'contact_id' => $data['contact_didar_id'],
                    'error' => $e->getMessage()
                ]);
            }
        } elseif ($existingDeal && !empty($existingDeal->title)) {
            $parts = explode('|', $existingDeal->title, 2);
            $baseTitle = trim($parts[0]);
        }
        if (empty($baseTitle) && !empty($data['contact_didar_id'])) {
            try {
                $baseTitle = $this->dealService->generateDealTitle($data['contact_didar_id']);
            } catch (\Exception $e) {
                \App\Utils\Logger::logWarning('Failed to generate base title for existing deal', [
                    'contact_id' => $data['contact_didar_id'],
                    'error' => $e->getMessage()
                ]);
            }
        }
        $servicesText = '';
        $existingRequestedServices = [];
        if ($existingDeal && !empty($existingDeal->requested_services)) {
            $decodedRequested = json_decode($existingDeal->requested_services, true);
            if (is_array($decodedRequested)) {
                $existingRequestedServices = $decodedRequested;
            }
        }
        $normalizeService = function($value): string {
            $s = trim((string)$value);
            $s = preg_replace('/\s+/', ' ', $s);
            return trim($s);
        };
        $baseTitleNormalized = $normalizeService($baseTitle);
        $filterServices = function(array $services) use ($normalizeService, $baseTitleNormalized) {
            $out = [];
            foreach ($services as $service) {
                $s = $normalizeService($service);
                if ($s === '') {
                    continue;
                }
                if ($baseTitleNormalized !== '' && $s === $baseTitleNormalized) {
                    continue;
                }
                if ($this->isGeneratedDealTitle($s)) {
                    continue;
                }
                $out[] = $s;
            }
            return array_values(array_unique($out));
        };

        if (!empty($existingRequestedServices)) {
            $existingRequestedServices = $filterServices((array)$existingRequestedServices);
        }
        if (!empty($requestedServices)) {
            $requestedServices = $filterServices((array)$requestedServices);
        }
        if (empty($requestedServices) && !empty($existingRequestedServices)) {
            $requestedServices = $existingRequestedServices;
        } elseif (!empty($requestedServices) && !empty($existingRequestedServices)) {
            $requestedServices = array_values(array_unique(array_merge($existingRequestedServices, (array)$requestedServices)));
        }
        if (!empty($requestedServices)) {
            $requestedServices = array_values(array_filter(array_map('trim', (array)$requestedServices)));
        }
        if (!empty($requestedServices)) {
            $servicesText = is_array($requestedServices) ? implode('، ', array_unique(array_map('trim', $requestedServices))) : trim((string)$requestedServices);
        } elseif ($existingDeal && !empty($existingDeal->title) && strpos($existingDeal->title, '|') !== false) {
            $servicesText = trim(substr($existingDeal->title, strpos($existingDeal->title, '|') + 1));
        }
        $data['title'] = $servicesText ? trim($baseTitle . ' | ' . $servicesText) : $baseTitle;
        $data['requested_services'] = $requestedServices;

        if (empty($data['contact_didar_id'])) {
            $this->errorResponse('شناسه لید الزامی است');
        }

        // For new deals, title is generated automatically
        if (empty($data['title'])) {
            $this->errorResponse('عنوان معامله الزامی است');
        }

        if (empty($data['status'])) {
            $this->errorResponse('انتخاب وضعیت معامله الزامی است');
        }
        
        // Pipeline stage is only required if status is NOT lost (lost deals automatically get 'معامله ناموفق')
        if (empty($data['pipeline_stage']) && $data['status'] !== 'lost') {
            $this->errorResponse('انتخاب مرحله کاریز معامله الزامی است');
        }

        $requiredFieldMap = [
            'service_cost' => ['value' => $serviceCostInput, 'label' => 'هزینه خدمات'],
            'price_list_code' => ['value' => $priceListCodeInput, 'label' => 'کد لیست قیمت'],
            'payable_amount' => ['value' => $payableAmountInput, 'label' => 'مبلغ قابل پرداخت'],
            'payment_conditions' => ['value' => $paymentConditionsInput, 'label' => 'شرایط پرداخت']
        ];
        foreach ($requiredFieldMap as $key => $item) {
            // Only validate if key exists in POST (manual submission). Automated flows may skip these.
            if (array_key_exists($key, $_POST)) {
                $value = $item['value'];
                if ($value === null || $value === '' || (is_string($value) && trim($value) === '')) {
                    $this->errorResponse($item['label'] . ' الزامی است');
                }
            }
        }

        $existingPaymentsDecoded = [];
        if ($existingDeal && !empty($existingDeal->payments)) {
            $decodedExisting = json_decode($existingDeal->payments, true);
            if (is_array($decodedExisting)) {
                $existingPaymentsDecoded = $decodedExisting;
            }
        }
        $existingPaymentsCount = count($existingPaymentsDecoded);
        if (empty($paymentsPayload) && !empty($existingPaymentsDecoded)) {
            $paymentsPayload = $existingPaymentsDecoded;
        }

        $rulesMap = $this->buildFieldRulesMap();
        $paymentMethodMetaMap = $this->buildPaymentMethodMetaMap();
        $userRole = $_SESSION['role'] ?? '';

        $dateRule = $this->resolveFieldRule($rulesMap, 'deal.payments.date', ['required' => true]);
        $amountRule = $this->resolveFieldRule($rulesMap, 'deal.payments.amount', ['required' => true]);
        $methodRule = $this->resolveFieldRule($rulesMap, 'deal.payments.method', ['required' => true]);
        $descriptionRule = $this->resolveFieldRule($rulesMap, 'deal.payments.description', ['required' => true]);
        $cardRule = $this->resolveFieldRule($rulesMap, 'deal.payments.card_last4', ['required' => true]);
        $bankRule = $this->resolveFieldRule($rulesMap, 'deal.payments.bank_destination', ['required' => false]);

        $normalizedStatus = strtolower((string)$data['status']);
        $paymentsRequired = $normalizedStatus === 'won' || $stageMeansSuccess;
        $validatedPayments = [];
        foreach ($paymentsPayload as $idx => $payment) {
            $isEntryNew = $isNewDeal || $idx >= $existingPaymentsCount;
            $existingPayment = $existingPaymentsDecoded[$idx] ?? [];

            $date = trim((string)($payment['date'] ?? ''));
            $datePersian = trim((string)($payment['date_persian'] ?? ''));
            $amount = isset($payment['amount']) ? floatval($payment['amount']) : 0;
            // Normalize method key
            $method = Sanitizer::sanitize($payment['method'] ?? $payment['payment_method'] ?? '');
            $description = Sanitizer::sanitize($payment['description'] ?? '');
            // Normalize bank key (support both destination_bank and bank_destination, use destination_bank as standard)
            $bankDestination = Sanitizer::sanitize($payment['destination_bank'] ?? $payment['bank_destination'] ?? '');

            $canEditDate = $this->canEditField($dateRule, $isEntryNew, $userRole);
            $canEditAmount = $this->canEditField($amountRule, $isEntryNew, $userRole);
            $canEditMethod = $this->canEditField($methodRule, $isEntryNew, $userRole);
            $canEditDescription = $this->canEditField($descriptionRule, $isEntryNew, $userRole);
            $canEditCard = $this->canEditField($cardRule, $isEntryNew, $userRole);
            $canEditBank = $this->canEditField($bankRule, $isEntryNew, $userRole);

            if (!$canEditDate) {
                if (isset($existingPayment['date'])) {
                    $date = trim((string)$existingPayment['date']);
                }
                if (isset($existingPayment['date_persian'])) {
                    $datePersian = trim((string)$existingPayment['date_persian']);
                }
            }
            if (!$canEditAmount && isset($existingPayment['amount'])) {
                $amount = floatval($existingPayment['amount']);
            }
            if (!$canEditMethod && isset($existingPayment['method'])) {
                $method = Sanitizer::sanitize($existingPayment['method']);
            }
            if (!$canEditDescription && isset($existingPayment['description'])) {
                $description = Sanitizer::sanitize($existingPayment['description']);
            }
            // Normalize existing payment bank key
            if (!$canEditBank) {
                $existingBank = $existingPayment['destination_bank'] ?? $existingPayment['bank_destination'] ?? '';
                if ($existingBank) {
                    $bankDestination = Sanitizer::sanitize($existingBank);
                }
            }

            // Normalize card_last4 key for hasAnyValue check
            $cardLast4ForCheck = $payment['card_last4'] ?? $payment['card_last_4'] ?? '';
            $hasAnyValue = ($date !== '' || $datePersian !== '' || $amount > 0 || $method !== '' || $description !== '' || $bankDestination !== '' || trim((string)$cardLast4ForCheck) !== '');
            if (!$hasAnyValue) {
                continue;
            }

            $methodMeta = $paymentMethodMetaMap[$method] ?? ['requires_card_last4' => false, 'requires_bank_destination' => false];
            $requiresCardLast4 = !empty($methodMeta['requires_card_last4']);
            $requiresBankDestination = !empty($methodMeta['requires_bank_destination']);

            $requireDate = $this->shouldRequireField($dateRule, $isEntryNew, $canEditDate);
            $requireAmount = $this->shouldRequireField($amountRule, $isEntryNew, $canEditAmount);
            $requireMethod = $this->shouldRequireField($methodRule, $isEntryNew, $canEditMethod);
            $requireDescription = $this->shouldRequireField($descriptionRule, $isEntryNew, $canEditDescription);
            $requireCardLast4 = $requiresCardLast4 && $this->shouldRequireField($cardRule, $isEntryNew, $canEditCard);
            $requireBank = $requiresBankDestination && $this->shouldRequireField($bankRule, $isEntryNew, $canEditBank);

            if ($requireDate && $date === '' && $datePersian === '') {
                $this->errorResponse('تاریخ پرداخت شماره ' . ($idx + 1) . ' الزامی است');
            }
            if ($requireAmount && $amount <= 0) {
                $this->errorResponse('مبلغ پرداخت شماره ' . ($idx + 1) . ' الزامی است');
            }
            if ($requireMethod && $method === '') {
                $this->errorResponse('نحوه پرداخت شماره ' . ($idx + 1) . ' الزامی است');
            }
            if ($requireDescription && $description === '') {
                $this->errorResponse('توضیح پرداخت شماره ' . ($idx + 1) . ' الزامی است');
            }

            // Normalize card_last4 key (support both card_last4 and card_last_4)
            $cardLast4Raw = $payment['card_last4'] ?? $payment['card_last_4'] ?? '';
            $cardLast4 = preg_replace('/\D+/', '', (string)$cardLast4Raw);
            if (!$canEditCard) {
                $existingCardLast4 = $existingPayment['card_last4'] ?? $existingPayment['card_last_4'] ?? '';
                if ($existingCardLast4) {
                    $cardLast4 = preg_replace('/\D+/', '', (string)$existingCardLast4);
                }
            }
            if (strlen($cardLast4) > 4) {
                $cardLast4 = substr($cardLast4, -4);
            }
            if ($requireCardLast4 && !preg_match('/^\d{4}$/', $cardLast4)) {
                $this->errorResponse('Card last 4 is required for payment row ' . ($idx + 1));
            }
            if ($requireBank && $bankDestination === '') {
                $this->errorResponse('Bank destination is required for payment row ' . ($idx + 1));
            }

            // Store with standard keys (card_last4 and destination_bank)
            $validatedPayments[] = [
                'date' => $date,
                'date_persian' => $datePersian,
                'amount' => $amount,
                'method' => $method,
                'description' => $description,
                'card_last4' => $cardLast4,  // Standard key (without underscore)
                'destination_bank' => $bankDestination  // Standard key
            ];
        }

        $isFirstPaymentNow = ($existingPaymentsCount === 0 && count($validatedPayments) > 0);

        if ($paymentsRequired && empty($validatedPayments)) {
            $this->errorResponse('ثبت حداقل یک پرداخت کامل برای معامله الزامی است');
        }
        if (!$paymentsRequired && empty($validatedPayments) && !empty($paymentsPayload)) {
            $this->errorResponse('اطلاعات پرداخت ناقص است');
        }

        if (empty($validatedPayments) && $existingDeal && !empty($existingDeal->payments)) {
            $storedPayments = json_decode($existingDeal->payments, true);
            if (is_array($storedPayments) && !empty($storedPayments)) {
                $validatedPayments = $storedPayments;
            }
        }

        $data['payments'] = !empty($validatedPayments)
            ? json_encode($validatedPayments, JSON_UNESCAPED_UNICODE)
            : '';
        $data['payment_amount'] = array_reduce($validatedPayments, function($sum, $p) {
            return $sum + (isset($p['amount']) ? floatval($p['amount']) : 0);
        }, 0);
        $payableAmount = isset($data['payable_amount']) ? floatval($data['payable_amount']) : 0;
        $paymentAmount = floatval($data['payment_amount']);
        if ($paymentAmount > 0 && $payableAmount <= 0) {
            $this->errorResponse('ثبت پرداخت بدون مبلغ قرارداد مجاز نیست');
        }
        if ($payableAmount > 0 && $paymentAmount > $payableAmount) {
            $this->errorResponse('مجموع پرداخت‌ها بیشتر از مبلغ قرارداد است');
        }

        \App\Utils\Logger::logInfo('DealController.save validated payments', [
            'deal_id' => $data['didar_deal_id'],
            'contact_id' => $data['contact_didar_id'],
            'payments_count' => count($validatedPayments),
            'status' => $data['status'],
            'pipeline_stage' => $data['pipeline_stage']
        ]);
        
        # The person has already been loaded and validated earlier
        # and $data['contact_didar_id'] is already standardized to a GUID.
        
        // Enforce owner consistency and permission
        $currentUserId = $_SESSION['didar_id'] ?? '';
        \App\Utils\Logger::logInfo('DealController.save validating permissions', [
            'current_user_id' => $currentUserId,
            'contact_owner_id' => $person->owner_didar_id,
            'contact_id' => $person->didar_contact_id,
            'is_admin_or_crm' => $this->authService->isAdminOrCrmSpecialist(),
            'permission_check_passed' => $this->authService->isAdminOrCrmSpecialist() || $currentUserId === $person->owner_didar_id
        ]);

        if (!$this->authService->isAdminOrCrmSpecialist() && $currentUserId !== $person->owner_didar_id) {
            \App\Utils\Logger::logWarning('Deal save blocked due to insufficient permissions', [
                'user_id' => $currentUserId,
                'contact_owner_id' => $person->owner_didar_id,
                'contact_id' => $person->didar_contact_id,
                'deal_title' => $data['title'],
                'reason' => 'USER_NOT_OWNER_OF_LEAD'
            ]);
            $this->errorResponse('شما مسئول این لید نیستید');
        }
        // Force deal owner to match lead owner
        $data['owner_didar_id'] = $person->owner_didar_id;
        
        // Extra logging for troubleshooting
        \App\Utils\Logger::logInfo('DealController.save payload - DETAILED', [
            'didar_deal_id' => $data['didar_deal_id'],
            'contact_didar_id' => $data['contact_didar_id'],
            'title' => $data['title'],
            'base_title_used' => $baseTitle,
            'services_text' => $servicesText,
            'requested_services_count' => count($requestedServices),
            'pipeline_stage' => $data['pipeline_stage'],
            'status' => $data['status'],
            'payable_amount' => $data['payable_amount'],
            'payment_amount' => $data['payment_amount'],
            'payments_present' => !empty($data['payments']),
            'requested_services' => $data['requested_services'],
            'user_id' => $_SESSION['didar_id'] ?? 'unknown',
            'user_name' => $_SESSION['name'] ?? 'unknown',
            'contact_owner_id' => $person->owner_didar_id,
            'contact_name' => trim(($person->first_name ?? '') . ' ' . ($person->last_name ?? '')),
            'existing_deal' => $existingDeal ? 'YES' : 'NO',
            'existing_deal_id' => $existingDeal ? $existingDeal->didar_deal_id : null,
            'existing_deal_status' => $existingDeal ? $existingDeal->status : null,
            'server_time' => date('Y-m-d H:i:s'),
            'session_data' => [
                'didar_id' => $_SESSION['didar_id'] ?? null,
                'name' => $_SESSION['name'] ?? null,
                'user_id' => $_SESSION['user_id'] ?? null
            ]
        ]);

        try {
            // Log changes before saving if updating
            if ($existingDeal) {
                $this->logFieldChanges('deal', $existingDeal->didar_deal_id, $existingDeal->toArray(), $data);
            }

            $savedDeal = $this->dealService->saveDeal($data);

            if (!empty($validatedPayments)) {
                try {
                    $this->dealService->syncTransactionsFromPayments($data['didar_deal_id'], $validatedPayments);
                } catch (\Exception $e) {
                    \App\Utils\Logger::logWarning('syncTransactionsFromPayments failed', [
                        'deal_id' => $data['didar_deal_id'],
                        'error' => $e->getMessage()
                    ]);
                }
            }

            if (!empty($requestedServices) || !empty($data['title'])) {
                try {
                    $this->dealService->ensureDealProductsFromText(
                        $data['didar_deal_id'],
                        $requestedServices,
                        $data['title']
                    );
                } catch (\Exception $e) {
                    \App\Utils\Logger::logWarning('ensureDealProductsFromText failed', [
                        'deal_id' => $data['didar_deal_id'],
                        'error' => $e->getMessage()
                    ]);
                }
            }

            // AUDIT TRAIL: Log deal creation/update for data integrity tracking
            \App\Utils\Logger::logInfo('AUDIT: Deal saved successfully', [
                'action' => $isNewDeal ? 'CREATE_DEAL' : 'UPDATE_DEAL',
                'deal_id' => $data['didar_deal_id'],
                'contact_id' => $data['contact_didar_id'],
                'contact_owner_id' => $data['owner_didar_id'],
                'user_id' => $_SESSION['didar_id'] ?? 'unknown',
                'user_name' => $_SESSION['name'] ?? 'unknown',
                'deal_title' => $data['title'],
                'deal_status' => $data['status'],
                'pipeline_stage' => $data['pipeline_stage'],
                'payable_amount' => $data['payable_amount'],
                'payment_amount' => $data['payment_amount'],
                'server_time' => date('Y-m-d H:i:s'),
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
            ]);
        } catch (\Exception $e) {
            \App\Utils\Logger::logError('Failed to save deal', $e, [
                'deal_id' => $data['didar_deal_id'],
                'contact_id' => $data['contact_didar_id'],
                'user_id' => $_SESSION['didar_id'] ?? 'unknown'
            ]);
            $this->errorResponse($e->getMessage() ?: 'خطا در ذخیره معامله');
        }

        // Update person financial fields so profile modal shows them
        $finFields = [];
        if (!empty($data['financial_level'])) $finFields['financial_level'] = $data['financial_level'];
        if (!empty($data['asset_estimation'])) $finFields['asset_estimation'] = $data['asset_estimation'];
        if (!empty($data['income_estimation'])) $finFields['income_estimation'] = $data['income_estimation'];
        if ($data['has_previous_purchase'] !== null) $finFields['has_previous_purchase'] = $data['has_previous_purchase'];
        if (!empty($finFields)) {
            try {
                $pdo = $this->personRepo->getDb()->getPdo();
                $setParts = [];
                $params = [];
                foreach ($finFields as $k => $v) {
                    $setParts[] = "$k = ?";
                    $params[] = $v;
                }
                $params[] = $data['contact_didar_id'];
                $stmt = $pdo->prepare("UPDATE persons SET " . implode(', ', $setParts) . " WHERE didar_contact_id = ?");
                $stmt->execute($params);
            } catch (\Exception $e) {
                \App\Utils\Logger::logWarning('Failed to update person financial fields on deal save', [
                    'contact_id' => $data['contact_didar_id'],
                    'error' => $e->getMessage()
                ]);
            }
        }

        $this->successResponse([], 'معامله با موفقیت ثبت شد');
    }

    public function setStatus()
    {
        $dealId = Sanitizer::sanitize($_POST['deal_id'] ?? '');
        $status = Sanitizer::sanitize($_POST['status'] ?? '');
        $contactId = Sanitizer::sanitize($_POST['contact_id'] ?? '');
        $pipelineStage = Sanitizer::sanitize($_POST['pipeline_stage'] ?? '');
        $failureReasonCode = Sanitizer::sanitize($_POST['failure_reason'] ?? '');
        $failureReasonDescription = Sanitizer::sanitize($_POST['failure_other_description'] ?? '');
        $failureReasonText = Sanitizer::sanitize($_POST['failure_reason_text'] ?? '');
        
        if (empty($dealId) || empty($status)) {
            $this->errorResponse('شناسه معامله و وضعیت الزامی است');
        }
        \App\Utils\Logger::logInfo('setStatus called', [
            'deal_id' => $dealId,
            'status' => $status,
            'contact_id' => $contactId,
            'pipeline_stage' => $pipelineStage
        ]);
        
        $statusText = $status === '1' ? 'Won' : 'Lost';
        $timeField = $status === '1' ? 'won_time' : 'lost_time';

        // Ensure payments exist for successful status
        if ($status === '1') {
            $deal = $this->dealRepo->findByDidarId($dealId);
            $paymentsDecoded = [];
            if ($deal && !empty($deal->payments)) {
                $paymentsDecoded = json_decode($deal->payments, true) ?: [];
            }
            $hasValidPayment = false;
            foreach ($paymentsDecoded as $p) {
                $amount = isset($p['amount']) ? floatval($p['amount']) : 0;
                $date = trim((string)($p['date'] ?? ''));
                $datePersian = trim((string)($p['date_persian'] ?? ''));
                $method = trim((string)($p['method'] ?? ''));
                if ($amount > 0 && ($date !== '' || $datePersian !== '') && $method !== '') {
                    $hasValidPayment = true;
                    break;
                }
            }
            if (!$hasValidPayment) {
                \App\Utils\Logger::logWarning('setStatus blocked: no valid payments for Won', ['deal_id' => $dealId, 'contact_id' => $contactId, 'payments_count' => count($paymentsDecoded)]);
                $this->errorResponse('برای ثبت معامله موفق، پرداخت کامل ثبت نشده است');
            }
        }
        
        $this->dealService->updateStatus($dealId, $statusText, $timeField);
        if (!empty($pipelineStage)) {
            $this->dealService->updatePipelineStage($dealId, $pipelineStage);
        }

        // Persist failure reason for reporting (only for Lost)
        if ($statusText === 'Lost') {
            $map = [
                'no_financial_ability' => 'عدم توان مالی',
                'no_purchase_intent' => 'فعلا قصد خرید ندارد',
                'competitor_purchase' => 'خرید از رقبا',
                'wrong_number' => 'شماره اشتباه',
                'irrelevant_lead' => 'لید نامرتبط',
                'other' => 'سایر',
            ];
            $code = $failureReasonCode !== '' ? $failureReasonCode : null;
            $desc = $failureReasonDescription !== '' ? $failureReasonDescription : null;
            $text = $failureReasonText !== '' ? $failureReasonText : null;
            if (!$code && $text) {
                $reverse = array_flip($map);
                if (isset($reverse[$text])) {
                    $code = $reverse[$text];
                }
            }
            if (empty($code)) {
                $this->errorResponse('دلیل ناموفق بودن معامله الزامی است');
            }
            if ($text === null && $code) {
                $text = $map[$code] ?? $code;
            }
            // Note: older clients might send Persian text instead of code in failure_reason
            if ($code && !isset($map[$code]) && $text === null) {
                $text = $code;
                $code = null;
            }
            $this->dealRepo->updateFailureReason($dealId, $code, $text, $desc);
        } else {
            // Clearing reason when marking as Won
            $this->dealRepo->updateFailureReason($dealId, null, null, null);
        }

        // Sync lead sale_status with successful deal
        if (!empty($contactId)) {
            $saleStatus = $status === '1' ? 1 : 0;
            try {
                $person = $this->personRepo->findByDidarId($contactId);
                $currentSaleStatus = $person ? (int) $person->sale_status : null;
                $shouldUpdate = true;
                // اگر یک بار موفق شده، با معامله ناموفق، لید را ناموفق نکن
                if ($saleStatus === 0 && $currentSaleStatus === 1) {
                    $shouldUpdate = false;
                }
                if ($shouldUpdate) {
                    $this->personRepo->updateSaleStatus($contactId, $saleStatus);
                }
                if ($saleStatus === 1) {
                    $deal = $this->dealRepo->findByDidarId($dealId);
                    $products = $this->dealRepo->getProductsByDealDidarId($dealId);
                    $productIds = array_map(fn($p) => $p['product_id'] ?? null, $products);
                    $this->dealService->handleSuccessfulDeal($contactId, $productIds, $deal ? $deal->title : '');
                }
            } catch (\Exception $e) {
                \App\Utils\Logger::logWarning('Failed to update sale_status for contact', [
                    'contact_id' => $contactId,
                    'status' => $saleStatus,
                    'error' => $e->getMessage()
                ]);
            }
        }
        $this->successResponse([], 'وضعیت معامله با موفقیت تغییر یافت');
    }

    public function setPipelineStage()
    {
        $dealId = Sanitizer::sanitize($_POST['deal_id'] ?? '');
        $stage = Sanitizer::sanitize($_POST['pipeline_stage'] ?? '');
        if (empty($dealId) || empty($stage)) {
            $this->errorResponse('شناسه معامله و مرحله الزامی است');
        }
        $stageNormalized = $stage;
        $isSuccessStage = in_array($stage, ['deal_success', 'اتمام', 'موفق', 'معامله موفق', 'پرداخت']);
        $isFailedStage = in_array($stage, ['deal_failed', 'معامله ناموفق', 'ناموفق']);
        if ($isSuccessStage) {
            $stageNormalized = 'اتمام';
            $this->dealService->updateStatus($dealId, 'Won', 'won_time');
        } elseif ($isFailedStage) {
            $stageNormalized = 'معامله ناموفق';
            $this->dealService->updateStatus($dealId, 'Lost', 'lost_time');
        }
        $this->dealService->updatePipelineStage($dealId, $stageNormalized);
        $this->successResponse([], 'مرحله کاریز معامله به‌روزرسانی شد');
    }

    public function setCourseStatus()
    {
        $data = [
            'deal_id' => Sanitizer::sanitize($_POST['deal_id'] ?? ''),
            'contact_id' => Sanitizer::sanitize($_POST['contact_id'] ?? ''),
            'course_virtual' => Sanitizer::sanitize($_POST['course_virtual'] ?? ''),
            'course_online' => Sanitizer::sanitize($_POST['course_online'] ?? ''),
            'access_created' => Sanitizer::sanitize($_POST['access_created'] ?? ''),
            'link_sent' => Sanitizer::sanitize($_POST['link_sent'] ?? ''),
            'course_date' => Sanitizer::sanitize($_POST['course_date'] ?? ''),
            'next_stage' => Sanitizer::sanitize($_POST['next_stage'] ?? 'برگزاری دوره'),
            'refund' => Sanitizer::sanitize($_POST['refund'] ?? ''),
            'refund_reason' => Sanitizer::sanitize($_POST['refund_reason'] ?? '')
        ];

        if (empty($data['deal_id'])) {
            $this->errorResponse('شناسه معامله الزامی است');
        }

        $this->dealService->updateCourseStatus($data);
        $this->successResponse([], 'وضعیت دوره/پشتیبانی معامله به‌روز شد');
    }

    public function delete()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('فقط مدیر سیستم می‌تواند معاملات را حذف کند');
        }
        
        $dealId = Sanitizer::sanitize($_POST['deal_id'] ?? '');
        
        if (empty($dealId)) {
            $this->errorResponse('شناسه معامله الزامی است');
        }

        $deal = $this->dealRepo->findByDidarId($dealId);
        if ($this->auditService) {
            $this->auditService->log('delete', 'deal', $dealId, [
                'deal' => $deal ? $deal->toArray() : null,
                'deal_title' => $deal ? ($deal->title ?? '') : '',
            ]);
        }

        $this->dealService->deleteDeal($dealId);
        $this->successResponse([], 'معامله با موفقیت حذف شد');
    }

    public function getDeal()
    {
        $dealId = Sanitizer::sanitize($_POST['deal_id'] ?? '');
        if (empty($dealId)) {
            $this->errorResponse('شناسه معامله الزامی است');
        }
        
        $deal = $this->dealService->getDeal($dealId);
        if (!$deal) {
            $this->errorResponse('معامله یافت نشد');
        }
        
        // Parse payments JSON
        $payments = [];
        if (!empty($deal->payments)) {
            $payments = json_decode($deal->payments, true) ?: [];
        }
        
        $dealArray = $deal->toArray();
        $dealArray['payments'] = $payments;
        
        // Add explicit contact info to prevent frontend from doing a wrong lookup
        if (isset($deal->contact_name)) {
            $dealArray['contact_name'] = $deal->contact_name;
        }
        if (isset($deal->contact_mobile)) {
            $dealArray['contact_mobile'] = $deal->contact_mobile;
        }
        if (isset($deal->local_person_id)) {
            $dealArray['local_person_id'] = $deal->local_person_id;
        }

        // Load products
        $products = $this->dealRepo->getProductsByDealDidarId($deal->didar_deal_id);
        $dealArray['products'] = $products;
        
        $this->successResponse(['deal' => $dealArray]);
    }

    public function reactivate()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('فقط مدیر سیستم می‌تواند معاملات بسته شده را فعال کند');
        }

        $dealId = Sanitizer::sanitize($_POST['deal_id'] ?? '');
        $newPipelineStage = Sanitizer::sanitize($_POST['new_pipeline_stage'] ?? '');
        $reactivationReason = Sanitizer::sanitize($_POST['reactivation_reason'] ?? '');

        if (empty($dealId)) {
            $this->errorResponse('شناسه معامله الزامی است');
        }

        if (empty($newPipelineStage)) {
            $this->errorResponse('مرحله کاریز جدید الزامی است');
        }

        // بررسی وجود معامله
        $deal = $this->dealRepo->findByDidarId($dealId);
        if (!$deal) {
            $this->errorResponse('معامله یافت نشد');
        }

        // بررسی اینکه معامله بسته شده باشد
        if ($deal->status === 'Pending') {
            $this->errorResponse('این معامله هم اکنون فعال است');
        }

        if ($deal->status !== 'Won' && $deal->status !== 'Lost') {
            $this->errorResponse('فقط معاملات بسته شده (موفق یا ناموفق) قابل فعال‌سازی مجدد هستند');
        }

        \App\Utils\Logger::logInfo('Deal reactivation requested', [
            'deal_id' => $dealId,
            'old_status' => $deal->status,
            'old_stage' => $deal->pipeline_stage,
            'new_stage' => $newPipelineStage,
            'reason' => $reactivationReason
        ]);

        try {
            $this->dealService->reactivateDeal($dealId, $newPipelineStage, $reactivationReason);

            // Audit logging
            if ($this->auditService) {
                $this->auditService->log('reactivate', 'deal', $dealId, [
                    'old_status' => $deal->status,
                    'old_stage' => $deal->pipeline_stage,
                    'new_stage' => $newPipelineStage,
                    'reason' => $reactivationReason,
                    'user_id' => $_SESSION['user_id'] ?? null,
                    'user_name' => $_SESSION['name'] ?? 'Unknown'
                ]);
            }

            $this->successResponse([], 'معامله با موفقیت فعال شد');

        } catch (\Exception $e) {
            \App\Utils\Logger::logError('Failed to reactivate deal', $e, [
                'deal_id' => $dealId,
                'new_stage' => $newPipelineStage
            ]);
            $this->errorResponse('خطا در فعال‌سازی مجدد معامله: ' . $e->getMessage());
        }
    }

    /**
     * Extract phone number from deal title
     * Expected format: "معامله اول 9127103488 | خدمات"
     */
    private function isGeneratedDealTitle(string $title): bool
    {
        $t = trim($title);
        if ($t === '') {
            return false;
        }
        if (strpos($t, '|') !== false) {
            return false;
        }
        return preg_match('/^Ù…Ø¹Ø§Ù…Ù„Ù‡\s+\S+\s+\d{10,12}$/u', $t) === 1;
    }

    private function extractPhoneFromDealTitle(string $title): string
    {
        Logger::logInfo('extractPhoneFromDealTitle called', [
            'input_title' => $title
        ]);

        // Remove any content after " | " separator
        $titlePart = explode(' | ', $title, 2)[0];

        Logger::logInfo('extractPhoneFromDealTitle processing', [
            'title_part' => $titlePart
        ]);

        // Look for phone number pattern in the title part
        // Pattern: "معامله [ordinal] [phone_number]"
        if (preg_match('/معامله\s+\S+\s+(\d{10,12})/u', $titlePart, $matches)) {
            Logger::logInfo('extractPhoneFromDealTitle found match', [
                'matched_phone' => $matches[1],
                'pattern' => 'معامله_ordinal_phone'
            ]);
            return $matches[1];
        }

        // Fallback: try to find any 10-12 digit number
        if (preg_match('/(\d{10,12})/', $titlePart, $matches)) {
            Logger::logInfo('extractPhoneFromDealTitle found fallback match', [
                'matched_phone' => $matches[1],
                'pattern' => 'any_10-12_digits'
            ]);
            return $matches[1];
        }

        Logger::logInfo('extractPhoneFromDealTitle no match found', [
            'title_part' => $titlePart
        ]);
        return '';
    }

    /**
     * Get contact's primary phone number
     */
    private function getContactPrimaryPhone(string $contactId): string
    {
        Logger::logInfo('getContactPrimaryPhone called', [
            'contact_id' => $contactId
        ]);

        // Use the robust repository method to find the person
        $person = $this->resolvePersonByContactId($contactId);
        
        $primaryPhone = $person ? ($person->mobile_phone ?? '') : '';

        Logger::logInfo('getContactPrimaryPhone result', [
            'contact_id' => $contactId,
            'person_found' => (bool) $person,
            'primary_phone' => $primaryPhone
        ]);

        return $primaryPhone;
    }

    /**
     * Normalize phone number for comparison
     * Remove country code, spaces, and standardize format
     */
    private function normalizePhoneForComparison(string $phone): string
    {
        Logger::logInfo('normalizePhoneForComparison called', [
            'input_phone' => $phone
        ]);

        // Remove all non-digit characters
        $cleaned = preg_replace('/\D/', '', $phone);

        Logger::logInfo('normalizePhoneForComparison cleaned', [
            'original' => $phone,
            'cleaned' => $cleaned
        ]);

        // Remove country code if present (98 or +98)
        if (str_starts_with($cleaned, '98') && strlen($cleaned) > 10) {
            $cleaned = substr($cleaned, 2);
            Logger::logInfo('normalizePhoneForComparison removed 98 prefix', ['cleaned' => $cleaned]);
        }
        if (str_starts_with($cleaned, '+98') && strlen($cleaned) > 10) {
            $cleaned = substr($cleaned, 3);
            Logger::logInfo('normalizePhoneForComparison removed +98 prefix', ['cleaned' => $cleaned]);
        }

        // Ensure it's a valid mobile number (10 digits starting with 9)
        if (strlen($cleaned) === 10 && str_starts_with($cleaned, '9')) {
            Logger::logInfo('normalizePhoneForComparison valid 10-digit mobile', ['final' => $cleaned]);
            return $cleaned;
        }

        // If it's 11 digits starting with 09, remove the 0
        if (strlen($cleaned) === 11 && str_starts_with($cleaned, '09')) {
            $final = substr($cleaned, 1);
            Logger::logInfo('normalizePhoneForComparison converted 11-digit to 10-digit', ['original' => $cleaned, 'final' => $final]);
            return $final;
        }

        Logger::logInfo('normalizePhoneForComparison final result', ['final' => $cleaned]);
        return $cleaned;
    }

    private function parseLookupMeta(?string $metaJson): array
    {
        if ($metaJson === null || $metaJson === '') {
            return [];
        }
        $decoded = json_decode($metaJson, true);
        return is_array($decoded) ? $decoded : [];
    }

    private function getLookupItemsSafe(string $groupCode, bool $includeInactive = false): array
    {
        if (!method_exists($this->dealService, 'getLookupItems')) {
            return [];
        }
        try {
            return $this->dealService->getLookupItems($groupCode, $includeInactive);
        } catch (\Throwable $e) {
            return [];
        }
    }

    private function buildFieldRulesMap(): array
    {
        $items = $this->getLookupItemsSafe('field_rules', false);
        $map = [];
        foreach ($items as $item) {
            $key = $item['value'] ?? $item['title'] ?? '';
            if ($key === '') {
                continue;
            }
            $meta = $this->parseLookupMeta($item['meta_json'] ?? null);
            $map[$key] = [
                'required' => array_key_exists('required', $meta) ? (bool)$meta['required'] : null,
                'editable' => array_key_exists('editable', $meta) ? (bool)$meta['editable'] : null,
                'editable_after_create' => array_key_exists('editable_after_create', $meta) ? (bool)$meta['editable_after_create'] : null,
                'editable_roles' => $meta['editable_roles'] ?? [],
            ];
        }
        return $map;
    }

    private function resolveFieldRule(array $rulesMap, string $key, array $defaults = []): array
    {
        $rule = $rulesMap[$key] ?? [];
        $roles = $rule['editable_roles'] ?? ($defaults['editable_roles'] ?? []);
        if (is_string($roles)) {
            $roles = array_values(array_filter(array_map('trim', explode(',', $roles))));
        }
        if (!is_array($roles)) {
            $roles = [];
        }
        return [
            'required' => array_key_exists('required', $rule) ? (bool)$rule['required'] : (bool)($defaults['required'] ?? false),
            'editable' => array_key_exists('editable', $rule) ? (bool)$rule['editable'] : (bool)($defaults['editable'] ?? true),
            'editable_after_create' => array_key_exists('editable_after_create', $rule) ? (bool)$rule['editable_after_create'] : (bool)($defaults['editable_after_create'] ?? true),
            'editable_roles' => $roles,
        ];
    }

    private function canEditField(array $rule, bool $isNewEntry, string $role): bool
    {
        if (array_key_exists('editable', $rule) && !$rule['editable']) {
            return false;
        }
        if (!$isNewEntry && array_key_exists('editable_after_create', $rule) && !$rule['editable_after_create']) {
            return false;
        }
        $roles = $rule['editable_roles'] ?? [];
        if (!empty($roles) && !in_array($role, $roles, true)) {
            return false;
        }
        return true;
    }

    private function shouldRequireField(array $rule, bool $isNewEntry, bool $canEdit): bool
    {
        if (empty($rule['required'])) {
            return false;
        }
        return $isNewEntry ? true : $canEdit;
    }

    private function buildPaymentMethodMetaMap(): array
    {
        $items = $this->getLookupItemsSafe('payment_methods', true);
        $map = [];
        foreach ($items as $item) {
            $key = $item['value'] ?? $item['title'] ?? '';
            if ($key === '') {
                continue;
            }
            $meta = $this->parseLookupMeta($item['meta_json'] ?? null);
            $map[$key] = [
                'requires_card_last4' => !empty($meta['requires_card_last4']),
                'requires_bank_destination' => !empty($meta['requires_bank_destination']),
            ];
        }
        if (empty($map)) {
            $map = [
                'card_reader' => ['requires_card_last4' => true, 'requires_bank_destination' => true],
                'card_to_card' => ['requires_card_last4' => true, 'requires_bank_destination' => true],
                'check' => ['requires_card_last4' => false, 'requires_bank_destination' => false],
            ];
        }
        return $map;
    }

    private function resolvePersonByContactId(string $contactId)
    {
        if ($contactId === '') {
            return null;
        }

        $person = $this->personRepo->findByDidarId($contactId);
        if ($person) {
            return $person;
        }

        return $this->personRepo->findByIdOrDidarId($contactId);
    }
}
