<?php
namespace App\Controllers;

use App\Services\AuthService;
use App\Services\LeadService;
use App\Repositories\PersonRepository;
use App\Utils\Sanitizer;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ImportController extends ApiController
{
    private AuthService $authService;
    private LeadService $leadService;
    private PersonRepository $personRepo;

    public function __construct(
        AuthService $authService,
        LeadService $leadService,
        PersonRepository $personRepo
    ) {
        $this->authService = $authService;
        $this->leadService = $leadService;
        $this->personRepo = $personRepo;
    }

    /**
     * معتبری کردن و نمایش پیش‌نمایش فایل اکسل
     */
    public function importLeadsPreview()
    {
        // Only admin can import leads
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('فقط مدیر می‌تواند لیدها را وارد کند');
            return;
        }

        // بررسی فایل
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            $this->errorResponse('فایلی آپلود نشده است');
            return;
        }

        $file = $_FILES['file'];
        $allowedExtensions = ['xlsx', 'xls', 'csv'];
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        if (!in_array($fileExtension, $allowedExtensions)) {
            $this->errorResponse('فقط فایل‌های Excel یا CSV قابل پذیرش هستند');
            return;
        }

        if ($file['size'] > 5 * 1024 * 1024) { // 5 MB
            $this->errorResponse('حجم فایل نباید بیشتر از 5 مگابایت باشد');
            return;
        }

        try {
            $spreadsheet = IOFactory::load($file['tmp_name']);
            $worksheet = $spreadsheet->getActiveSheet();
            $rows = $worksheet->toArray();

            if (empty($rows)) {
                $this->errorResponse('فایل خالی است');
                return;
            }

            // ستون‌های الزامی
            $requiredColumns = ['شماره لید', 'نام مسئول', 'شیوه آشنایی', 'جزئیات شیوه آشنایی', 'موضوع محتوا'];
            $headerRow = $rows[0] ?? [];

            // بررسی سرستون‌ها
            $columnMapping = [];
            foreach ($requiredColumns as $colName) {
                $index = array_search($colName, $headerRow);
                if ($index === false) {
                    $this->errorResponse("ستون '{$colName}' در فایل یافت نشد");
                    return;
                }
                $columnMapping[$colName] = $index;
            }

            // پردازش ردیف‌ها
            $preview = [];
            $validRows = [];

            for ($i = 1; $i < count($rows); $i++) {
                if (empty($rows[$i]) || (count(array_filter($rows[$i])) === 0)) {
                    continue; // Skip empty rows
                }

                $row = $rows[$i];
                $rowData = [
                    'row_number' => $i + 1,
                    'lead_id' => trim((string)($row[$columnMapping['شماره لید']] ?? '')),
                    'owner_name' => trim((string)($row[$columnMapping['نام مسئول']] ?? '')),
                    'lead_source' => trim((string)($row[$columnMapping['شیوه آشنایی']] ?? '')),
                    'lead_detail' => trim((string)($row[$columnMapping['جزئیات شیوه آشنایی']] ?? '')),
                    'lead_topic' => trim((string)($row[$columnMapping['موضوع محتوا']] ?? '')),
                    'errors' => []
                ];

                // اعتبارسنجی داده‌ها
                if (empty($rowData['lead_id'])) {
                    $rowData['errors'][] = 'شماره لید خالی است';
                } elseif (!preg_match('/^\d{9}$/', $rowData['lead_id'])) {
                    $rowData['errors'][] = 'شماره لید باید 9 رقمی باشد';
                }

                if (empty($rowData['owner_name'])) {
                    $rowData['errors'][] = 'نام مسئول خالی است';
                }

                if (empty($rowData['lead_source'])) {
                    $rowData['errors'][] = 'شیوه آشنایی خالی است';
                }

                if (empty($rowData['lead_detail'])) {
                    $rowData['errors'][] = 'جزئیات شیوه آشنایی خالی است';
                }

                if (empty($rowData['lead_topic'])) {
                    $rowData['errors'][] = 'موضوع محتوا خالی است';
                }

                if (empty($rowData['errors'])) {
                    $validRows[] = $rowData;
                }

                $preview[] = $rowData;
            }

            $this->successResponse([
                'preview' => $preview,
                'valid_count' => count($validRows),
                'error_count' => count($preview) - count($validRows),
                'column_mapping' => $columnMapping
            ]);

        } catch (\Exception $e) {
            $this->errorResponse('خطا در پردازش فایل: ' . $e->getMessage());
        }
    }

    /**
     * انجام واقعی import لیدها
     */
    public function importLeadsExecute()
    {
        // Only admin can import leads
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('فقط مدیر می‌تواند لیدها را وارد کند');
            return;
        }

        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            $this->errorResponse('فایلی آپلود نشده است');
            return;
        }

        try {
            $file = $_FILES['file'];
            $spreadsheet = IOFactory::load($file['tmp_name']);
            $worksheet = $spreadsheet->getActiveSheet();
            $rows = $worksheet->toArray();

            if (empty($rows)) {
                $this->errorResponse('فایل خالی است');
                return;
            }

            $requiredColumns = ['شماره لید', 'نام مسئول', 'شیوه آشنایی', 'جزئیات شیوه آشنایی', 'موضوع محتوا'];
            $headerRow = $rows[0] ?? [];

            $columnMapping = [];
            foreach ($requiredColumns as $colName) {
                $index = array_search($colName, $headerRow);
                if ($index === false) {
                    $this->errorResponse("ستون '{$colName}' یافت نشد");
                    return;
                }
                $columnMapping[$colName] = $index;
            }

            $successCount = 0;
            $failureCount = 0;
            $results = [];
            $currentUserId = $_SESSION['didar_id'] ?? '';

            for ($i = 1; $i < count($rows); $i++) {
                if (empty($rows[$i]) || (count(array_filter($rows[$i])) === 0)) {
                    continue; // Skip empty rows
                }

                $row = $rows[$i];
                $leadId = trim((string)($row[$columnMapping['شماره لید']] ?? ''));
                $ownerName = trim((string)($row[$columnMapping['نام مسئول']] ?? ''));
                $leadSource = trim((string)($row[$columnMapping['شیوه آشنایی']] ?? ''));
                $leadDetail = trim((string)($row[$columnMapping['جزئیات شیوه آشنایی']] ?? ''));
                $leadTopic = trim((string)($row[$columnMapping['موضوع محتوا']] ?? ''));

                // Validation
                if (!preg_match('/^\d{9}$/', $leadId)) {
                    $failureCount++;
                    $results[] = [
                        'row' => $i + 1,
                        'status' => 'fail',
                        'message' => 'شماره لید نامعتبر (باید 9 رقمی باشد)'
                    ];
                    continue;
                }

                if (empty($leadSource) || empty($leadDetail) || empty($leadTopic)) {
                    $failureCount++;
                    $results[] = [
                        'row' => $i + 1,
                        'status' => 'fail',
                        'message' => 'فیلدهای الزامی خالی هستند'
                    ];
                    continue;
                }

                // بدست آوردن didar_id مسئول
                $owner = null;
                try {
                    // جستجو بر اساس نام
                    $owners = $this->personRepo->findAll(['search' => $ownerName]);
                    if (!empty($owners)) {
                        $owner = $owners[0];
                    }
                } catch (\Exception $e) {
                    // silent
                }

                if (!$owner || !$owner->didar_user_id) {
                    $failureCount++;
                    $results[] = [
                        'row' => $i + 1,
                        'status' => 'fail',
                        'message' => "مسئول '$ownerName' یافت نشد"
                    ];
                    continue;
                }

                try {
                    // ایجاد لید
                    $leadData = [
                        'first_name' => $leadId,
                        'last_name' => '',
                        'lead_source' => $leadSource,
                        'lead_content_detail' => $leadDetail,
                        'lead_content_topic' => $leadTopic,
                        'owner_didar_id' => $owner->didar_user_id
                    ];

                    $this->leadService->create(
                        $owner->didar_user_id,
                        $leadData
                    );

                    $successCount++;
                    $results[] = [
                        'row' => $i + 1,
                        'status' => 'success',
                        'message' => 'لید با موفقیت ایجاد شد'
                    ];

                } catch (\Exception $e) {
                    $failureCount++;
                    $results[] = [
                        'row' => $i + 1,
                        'status' => 'fail',
                        'message' => 'خطا: ' . $e->getMessage()
                    ];
                }
            }

            $this->successResponse([
                'success_count' => $successCount,
                'failure_count' => $failureCount,
                'results' => $results
            ]);

        } catch (\Exception $e) {
            $this->errorResponse('خطا: ' . $e->getMessage());
        }
    }
}


