<?php
namespace App\Controllers;

use App\Services\KpiService;
use App\Services\AuthService;
use App\Utils\Logger;

class KpiController extends ApiController
{
    private $kpiService;
    private $authService;

    public function __construct(
        KpiService $kpiService,
        AuthService $authService
    ) {
        $this->kpiService = $kpiService;
        $this->authService = $authService;
    }

    /**
     * دریافت شاخص‌های KPI
     * Admin: می‌تواند همه را ببیند + اهداف را تعریف کند
     * CRM Specialist: **دسترسی ندارد** (403)
     * Agent: فقط KPI خودش را می‌بیند
     */
    public function getKPI()
    {
        $role = $_SESSION['role'] ?? '';
        $didarId = $_SESSION['didar_id'] ?? '';

        // CRM Specialist کاملاً محروم است
        if ($role === 'crm_specialist') {
            $this->errorResponse('دسترسی محدود - کارشناسان CRM به داشبورد KPI دسترسی ندارند', 403);
            return;
        }

        // دریافت فیلترها
        $dateFrom = $_POST['date_from'] ?? date('Y-m-01'); // اول ماه جاری
        $dateTo = $_POST['date_to'] ?? date('Y-m-t'); // آخر ماه جاری
        $ownerDidarId = $_POST['owner_didar_id'] ?? '';
        $productId = $_POST['product_id'] ?? '';

        // اگر Agent بود، فورس فقط خودش را ببیند
        $isAdmin = $this->authService->isAdmin();
        if (!$isAdmin && !empty($ownerDidarId) && $ownerDidarId !== $didarId) {
            $this->errorResponse('شما فقط می‌توانید KPI خود را ببینید', 403);
            return;
        }

        // ساخت فیلترها
        $filters = [
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
            'owner_didar_id' => $ownerDidarId,
            'product_id' => $productId,
            'person_date_from' => $dateFrom,
            'person_date_to' => $dateTo
        ];

        // محاسبه KPI
        try {
            $kpi = $this->kpiService->calculateKPI($filters);

            $this->successResponse([
                'kpi' => $kpi,
                'filters' => [
                    'date_from' => $dateFrom,
                    'date_to' => $dateTo,
                    'owner_didar_id' => $ownerDidarId,
                    'product_id' => $productId
                ],
                'can_manage_targets' => $isAdmin // فقط Admin می‌تواند اهداف را تعریف کند
            ]);

        } catch (\Exception $e) {
            Logger::logError('Failed to calculate KPI', $e, ['filters' => $filters]);
            $this->errorResponse('خطا در محاسبه KPI: ' . $e->getMessage(), 500);
        }
    }

    /**
     * ثبت هدف KPI (فقط Admin)
     */
    public function setKpiTarget()
    {
        $role = $_SESSION['role'] ?? '';
        $didarId = $_SESSION['didar_id'] ?? '';

        if ($role === 'crm_specialist') {
            $this->errorResponse('دسترسی محدود - کارشناسان CRM به داشبورد KPI دسترسی ندارند', 403);
            return;
        }

        $userId = intval($_POST['user_id'] ?? 0);
        $year = intval($_POST['year'] ?? date('Y'));
        $month = intval($_POST['month'] ?? date('m'));
        $targetAmount = floatval($_POST['target_amount'] ?? 0);
        $targetDealCount = intval($_POST['target_deal_count'] ?? 0);

        if ($userId <= 0 || $targetAmount < 0) {
            $this->errorResponse('پارامترهای نامعتبر');
            return;
        }

        try {
            $result = $this->kpiService->setKpiTarget($userId, $year, $month, $targetAmount, $targetDealCount);

            $this->successResponse([
                'message' => 'هدف KPI با موفقیت ثبت شد',
                'user_id' => $userId,
                'year' => $year,
                'month' => $month,
                'target_amount' => $targetAmount,
                'target_deal_count' => $targetDealCount
            ]);
        } catch (\Exception $e) {
            Logger::logError('Failed to set KPI target', $e, [
                'user_id' => $userId,
                'year' => $year,
                'month' => $month
            ]);
            $this->errorResponse('خطا در ثبت هدف KPI: ' . $e->getMessage(), 500);
        }
    }

    /**
     * دریافت لیست اهداف KPI (فقط Admin)
     */
    public function getKpiTargets()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('دسترسی محدود - فقط مدیر سیستم', 403);
            return;
        }

        try {
            $targets = $this->kpiService->getAllKpiTargets();

            $this->successResponse(['targets' => $targets]);
        } catch (\Exception $e) {
            Logger::logError('Failed to get KPI targets', $e);
            $this->errorResponse('خطا در دریافت لیست اهداف KPI: ' . $e->getMessage(), 500);
        }
    }

    /**
     * دریافت جزئیات معاملات KPI (فقط Admin)
     */
    public function getKpiDealDetails()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('دسترسی محدود - فقط مدیر سیستم', 403);
            return;
        }

        // دریافت فیلترها
        $dateFrom = $_POST['date_from'] ?? date('Y-m-01');
        $dateTo = $_POST['date_to'] ?? date('Y-m-t');
        $ownerDidarId = $_POST['owner_didar_id'] ?? '';
        $productId = $_POST['product_id'] ?? '';

        // Agent فقط KPI خودش را می‌بیند
        if (!$this->authService->isAdmin()) {
            $ownerDidarId = $didarId;
        }

        // ساخت فیلترها
        $filters = [
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
            'owner_didar_id' => $ownerDidarId,
            'product_id' => $productId,
            'person_date_from' => $dateFrom,
            'person_date_to' => $dateTo
        ];

        try {
            $deals = $this->kpiService->getKpiDealDetails($filters);

            $this->successResponse([
                'deals' => $deals,
                'filters' => [
                    'date_from' => $dateFrom,
                    'date_to' => $dateTo,
                    'owner_didar_id' => $ownerDidarId,
                    'product_id' => $productId
                ]
            ]);

        } catch (\Exception $e) {
            Logger::logError('Failed to get KPI deal details', $e, ['filters' => $filters]);
            $this->errorResponse('خطا در دریافت جزئیات معاملات: ' . $e->getMessage(), 500);
        }
    }
}
