<?php
namespace App\Controllers;

use App\Services\NotificationService;
use App\Services\AuthService;
use App\Utils\Sanitizer;

class NotificationController extends ApiController
{
    private $notificationService;
    private $authService;

    public function __construct(
        NotificationService $notificationService,
        AuthService $authService
    ) {
        $this->notificationService = $notificationService;
        $this->authService = $authService;
    }

    /**
     * Get notifications for current user
     */
    public function getNotifications()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        if (empty($userId)) {
            $this->errorResponse('شناسه کاربر یافت نشد');
        }

        $limit = isset($_POST['limit']) ? (int)$_POST['limit'] : 50;
        $offset = isset($_POST['offset']) ? (int)$_POST['offset'] : 0;
        $unreadOnly = isset($_POST['unread_only']) && $_POST['unread_only'] == '1';

        if ($unreadOnly) {
            $notifications = $this->notificationService->getUnread($userId, $limit);
        } else {
            $notifications = $this->notificationService->getAll($userId, $limit, $offset);
        }

        $unreadCount = $this->notificationService->getUnreadCount($userId);

        $notificationsArray = array_map(function($notif) {
            return $notif->toArray();
        }, $notifications);

        $this->successResponse([
            'notifications' => $notificationsArray,
            'unread_count' => $unreadCount
        ]);
    }

    /**
     * Get unread count
     */
    public function getUnreadCount()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        if (empty($userId)) {
            $this->errorResponse('شناسه کاربر یافت نشد');
        }

        $count = $this->notificationService->getUnreadCount($userId);
        $this->successResponse(['count' => $count]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        $notificationId = Sanitizer::sanitize($_POST['notification_id'] ?? '');

        if (empty($notificationId)) {
            $this->errorResponse('شناسه اعلان الزامی است');
        }

        $result = $this->notificationService->markAsRead($notificationId, $userId);
        
        if ($result) {
            $this->successResponse([], 'اعلان به عنوان خوانده شده علامت‌گذاری شد');
        } else {
            $this->errorResponse('خطا در به‌روزرسانی اعلان');
        }
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        $result = $this->notificationService->markAllAsRead($userId);
        
        if ($result) {
            $this->successResponse([], 'همه اعلان‌ها به عنوان خوانده شده علامت‌گذاری شدند');
        } else {
            $this->errorResponse('خطا در به‌روزرسانی اعلان‌ها');
        }
    }

    /**
     * Delete notification
     */
    public function delete()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        $notificationId = Sanitizer::sanitize($_POST['notification_id'] ?? '');

        if (empty($notificationId)) {
            $this->errorResponse('شناسه اعلان الزامی است');
        }

        $result = $this->notificationService->delete($notificationId, $userId);
        
        if ($result) {
            $this->successResponse([], 'اعلان حذف شد');
        } else {
            $this->errorResponse('خطا در حذف اعلان');
        }
    }

    /**
     * Check reminders (for polling)
     */
    public function checkReminders()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        
        // Check and send reminders
        $newNotifications = $this->notificationService->checkActivityReminders();
        
        // Filter notifications for current user
        $userNotifications = array_filter($newNotifications, function($notif) use ($userId) {
            return $notif['user_id'] === $userId;
        });

        // Get all unread notifications for user
        $unreadNotifications = $this->notificationService->getUnread($userId, 20);
        $unreadCount = $this->notificationService->getUnreadCount($userId);

        $notificationsArray = array_map(function($notif) {
            return $notif->toArray();
        }, $unreadNotifications);

        // Filter popup notifications
        $popupNotifications = array_filter($userNotifications, function($notif) {
            return isset($notif['show_popup']) && $notif['show_popup'];
        });

        // Log for debugging
        error_log("Popup notifications count: " . count($popupNotifications));
        error_log("Popup notifications: " . json_encode($popupNotifications));

        $this->successResponse([
            'notifications' => $notificationsArray,
            'new_notifications' => array_values($userNotifications),
            'popup_notifications' => array_values($popupNotifications),
            'unread_count' => $unreadCount
        ]);
    }

    /**
     * Test notification (Admin only)
     */
    public function test()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('فقط مدیر سیستم می‌تواند اعلان را تست کند');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        if (empty($userId)) {
            $this->errorResponse('شناسه کاربر یافت نشد');
        }

        // Create a test notification
        $notificationId = $this->notificationService->create(
            $userId,
            'test',
            'تست اعلان',
            'این یک اعلان تستی است. اگر این پیام را می‌بینید، سیستم اعلان‌رسانی به درستی کار می‌کند.',
            'system',
            null
        );

        if ($notificationId) {
            // Get the created notification object
            // Ensure notificationId is an integer
            $notificationId = (int)$notificationId;
            $notification = $this->notificationService->getById($notificationId);
            
            if ($notification && is_object($notification) && method_exists($notification, 'toArray')) {
                $this->successResponse([
                    'notification' => $notification->toArray(),
                    'message' => 'اعلان تستی با موفقیت ایجاد شد'
                ]);
            } else {
                // Fallback: return notification ID if object retrieval failed
                $this->successResponse([
                    'notification_id' => $notificationId,
                    'message' => 'اعلان تستی با موفقیت ایجاد شد (جزئیات اعلان در دسترس نیست)'
                ]);
            }
        } else {
            $this->errorResponse('خطا در ایجاد اعلان تستی');
        }
    }

    /**
     * Get reminders list for UI
     */
    public function getRemindersList()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $userId = $_SESSION['didar_id'] ?? '';
        $isAdmin = $this->authService->isAdmin();

        $reminders = $this->notificationService->getRemindersList($userId, $isAdmin);

        $this->successResponse(['reminders' => $reminders]);
    }

    /**
     * Resend reminder notification
     */
    public function resendReminder()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $activityId = \App\Utils\Sanitizer::sanitize($_POST['activity_id'] ?? '');
        if (empty($activityId)) {
            $this->errorResponse('شناسه فعالیت الزامی است');
        }

        try {
            $result = $this->notificationService->resendReminder($activityId, $_SESSION['didar_id'] ?? '');
            $this->successResponse(['message' => 'اعلان مجدداً ارسال شد']);
        } catch (\Exception $e) {
            $this->errorResponse('خطا در ارسال مجدد اعلان: ' . $e->getMessage());
        }
    }

    /**
     * Mark reminder as done
     */
    public function markReminderDone()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $activityId = \App\Utils\Sanitizer::sanitize($_POST['activity_id'] ?? '');
        if (empty($activityId)) {
            $this->errorResponse('شناسه فعالیت الزامی است');
        }

        try {
            $result = $this->notificationService->markReminderDone($activityId, $_SESSION['didar_id'] ?? '');
            $this->successResponse(['message' => 'یادآوری به عنوان انجام شده علامت‌گذاری شد']);
        } catch (\Exception $e) {
            $this->errorResponse('خطا در علامت‌گذاری یادآوری: ' . $e->getMessage());
        }
    }

    /**
     * Snooze reminder
     */
    public function snoozeReminder()
    {
        if (!$this->authService->isAuthenticated()) {
            $this->errorResponse('لطفاً ابتدا وارد شوید');
        }

        $activityId = \App\Utils\Sanitizer::sanitize($_POST['activity_id'] ?? '');
        $minutes = (int)($_POST['minutes'] ?? 0);

        if (empty($activityId)) {
            $this->errorResponse('شناسه فعالیت الزامی است');
        }

        if ($minutes <= 0 || !in_array($minutes, [60, 240, 1440])) {
            $this->errorResponse('زمان تعویق نامعتبر است');
        }

        try {
            $result = $this->notificationService->snoozeReminder($activityId, $minutes, $_SESSION['didar_id'] ?? '');
            $timeLabels = [
                60 => '۱ ساعت',
                240 => '۴ ساعت',
                1440 => '۱ روز'
            ];
            $this->successResponse(['message' => 'یادآوری به مدت ' . $timeLabels[$minutes] . ' تعویق شد']);
        } catch (\Exception $e) {
            $this->errorResponse('خطا در تعویق یادآوری: ' . $e->getMessage());
        }
    }
}

