<?php
namespace App\Controllers;

use App\Repositories\UserRepository;
use App\Services\AuthService;
use App\Services\DidarApiService;
use App\Database\Connection;
use App\Utils\Sanitizer;
use PDO;

class UserController extends ApiController
{
    private $userRepo;
    private $authService;
    private $didarApi;
    private $db;

    public function __construct(UserRepository $userRepo, AuthService $authService, DidarApiService $didarApi = null, Connection $db = null)
    {
        $this->userRepo = $userRepo;
        $this->authService = $authService;
        
        // Get from global app if not provided (for backward compatibility)
        if ($didarApi === null || $db === null) {
            global $app;
            if (!isset($app)) {
                $app = $GLOBALS['app'] ?? null;
            }
            if ($app) {
                $this->didarApi = $didarApi ?? $app->services->didarApi;
                $this->db = $db ?? $app->db;
            }
        } else {
            $this->didarApi = $didarApi;
            $this->db = $db;
        }
    }

    /**
     * Public: Get login users list (for username dropdown on login page).
     * DOES NOT require authentication.
     */
    public function getLoginUsers()
    {
        $users = $this->userRepo->getLoginUsers();
        $this->successResponse(['users' => $users]);
    }

    public function getList()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('دسترسی محدود به مدیر سیستم');
        }
        
        $users = $this->userRepo->getAll();
        $usersArray = array_map(function($user) {
            return $user->toArray();
        }, $users);
        
        $this->successResponse(['users' => $usersArray]);
    }

    /**
     * Sync users from Didar API and update their didar_user_ids
     * This ensures OwnerId in activities is correct
     */
    public function syncDidarUsers()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('دسترسی محدود به مدیر سیستم');
        }
        
        if (!$this->didarApi) {
            $this->errorResponse('سرویس Didar API در دسترس نیست');
        }
        
        try {
            // Fetch users from Didar API
            $didarUsers = $this->didarApi->syncDidarUsers();
            
            if ($didarUsers === false) {
                $this->errorResponse('خطا در دریافت کاربران از Didar. لطفاً لاگ‌ها را بررسی کنید.');
            }
            
            if (!is_array($didarUsers) || empty($didarUsers)) {
                $this->errorResponse('هیچ کاربری از Didar دریافت نشد. لطفاً API Key را بررسی کنید.');
            }
            
            $updated = 0;
            $notFound = 0;
            
            // Update each user with their Didar UserId
            foreach ($didarUsers as $didarUser) {
                $userId = $didarUser['Id'] ?? null;
                $email = $didarUser['Email'] ?? null;
                $firstName = $didarUser['FirstName'] ?? '';
                $lastName = $didarUser['LastName'] ?? '';
                $userName = $didarUser['UserName'] ?? null;
                
                if (!$userId) {
                    continue;
                }
                
                // Try to find user by email first
                $user = null;
                if ($email) {
                    $user = $this->userRepo->findByEmail($email);
                }
                
                // Try to find by display name if not found by email
                if (!$user && !empty($firstName) && !empty($lastName)) {
                    $displayName = $firstName . ' ' . $lastName;
                    $user = $this->userRepo->findByUsernameOrDisplayName($displayName);
                }
                
                if ($user) {
                    // Update user with correct Didar UserId
                    $user->didar_user_id = $userId;
                    $this->userRepo->save($user);
                    $updated++;
                } else {
                    $notFound++;
                }
            }
            
            $this->successResponse([
                'total_didar_users' => count($didarUsers),
                'updated' => $updated,
                'not_found' => $notFound
            ], "کاربران با موفقیت همگام‌سازی شدند. $updated کاربر به‌روزرسانی شد.");
        } catch (\Exception $e) {
            \App\Utils\Logger::logError("Failed to sync Didar users", $e);
            $this->errorResponse('خطا در همگام‌سازی کاربران: ' . $e->getMessage());
        }
    }

    public function save()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('دسترسی محدود به مدیر سیستم');
        }
        
        $user = new \App\Models\User([
            'didar_user_id' => Sanitizer::sanitize($_POST['didar_user_id'] ?? ''),
            'email' => Sanitizer::sanitize($_POST['email'] ?? ''),
            'first_name' => Sanitizer::sanitize($_POST['first_name'] ?? ''),
            'last_name' => Sanitizer::sanitize($_POST['last_name'] ?? ''),
            'display_name' => Sanitizer::sanitize($_POST['display_name'] ?? ''),
            'role' => Sanitizer::sanitize($_POST['role'] ?? 'agent'),
            'is_active' => isset($_POST['is_active']) ? intval($_POST['is_active']) : 1
        ]);
        
        $this->userRepo->save($user);
        $this->successResponse([], 'کاربر با موفقیت ذخیره شد');
    }

    public function changePassword()
    {
        $userId = intval($_POST['user_id'] ?? 0);
        $currentPassword = $_POST['current_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        
        if (empty($userId) || empty($newPassword)) {
            $this->errorResponse('شناسه کاربر و رمز عبور جدید الزامی است');
        }
        
        if ($newPassword !== $confirmPassword) {
            $this->errorResponse('رمز عبور جدید و تأیید آن مطابقت ندارند');
        }
        
        if (strlen($newPassword) < 6) {
            $this->errorResponse('رمز عبور باید حداقل 6 کاراکتر باشد');
        }
        
        $user = $this->userRepo->findById($userId);
        if (!$user) {
            $this->errorResponse('کاربر یافت نشد');
        }
        
        // Check current password if not admin changing own password
        if ($userId != $_SESSION['user_id'] || !empty($currentPassword)) {
            if (empty($currentPassword) || !password_verify($currentPassword, $user->password)) {
                $this->errorResponse('رمز عبور فعلی اشتباه است');
            }
        }
        
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $this->userRepo->updatePassword($userId, $hashedPassword);
        
        $this->successResponse([], 'رمز عبور با موفقیت تغییر یافت');
    }

    public function updateRole()
    {
        if (!$this->authService->isAdmin()) {
            $this->errorResponse('دسترسی محدود به مدیر سیستم');
        }
        
        $userId = intval($_POST['user_id'] ?? 0);
        $role = Sanitizer::sanitize($_POST['role'] ?? '');
        
        if (empty($userId) || empty($role)) {
            $this->errorResponse('شناسه کاربر و نقش الزامی است');
        }
        
        if (!in_array($role, ['admin', 'agent', 'crm_specialist'])) {
            $this->errorResponse('نقش نامعتبر است');
        }
        
        $this->userRepo->updateRole($userId, $role);
        $this->successResponse([], 'نقش کاربر با موفقیت تغییر یافت');
    }

    /**
     * Get Didar users list for lead owner selection
     * Requires admin or CRM specialist role
     * Falls back to local database if Didar API fails
     */
    public function getDidarUsers()
    {
        if (!$this->authService->isAdminOrCrmSpecialist()) {
            $this->errorResponse('دسترسی محدود به مدیر و کارشناس CRM');
        }

        if (!$this->db) {
            // Try to get from global app
            global $app;
            if (!isset($app)) {
                $app = $GLOBALS['app'] ?? null;
            }
            if ($app) {
                $this->db = $this->db ?? $app->db;
                $this->didarApi = $this->didarApi ?? $app->services->didarApi;
            }
        }

        if (!$this->db) {
            $this->errorResponse('خطا در دسترسی به دیتابیس');
        }

        $pdo = $this->db->getPdo();
        // فقط از دیتابیس محلی و کاربران فعال بخوانیم (بدون همگام‌سازی دیدار)
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        $fields = $hasDisplayName ? "id, didar_user_id, display_name, email, is_active" : "id, didar_user_id, first_name, last_name, email, is_active";
        
        \App\Utils\Logger::logInfo('[DEBUG] getDidarUsers - Fetching users', [
            'hasDisplayName' => $hasDisplayName,
            'fields' => $fields
        ]);
        
        $stmtLocal = $pdo->prepare("SELECT $fields FROM users WHERE is_active = 1");
        $stmtLocal->execute();
        $rows = $stmtLocal->fetchAll(PDO::FETCH_ASSOC);
        
        \App\Utils\Logger::logInfo('[DEBUG] getDidarUsers - Found users', [
            'count' => count($rows),
            'rows' => $rows
        ]);

        $usersLocal = [];
        foreach ($rows as $u) {
            $displayName = $hasDisplayName
                ? ($u['display_name'] ?? '')
                : trim(($u['first_name'] ?? '') . ' ' . ($u['last_name'] ?? ''));
            $usersLocal[] = [
                'id' => $u['id'],
                'owner_id' => $u['didar_user_id'],
                'display_name' => $displayName,
                'email' => $u['email'] ?? '',
                'is_active' => 1
            ];
        }
        
        \App\Utils\Logger::logInfo('[DEBUG] getDidarUsers - Returning users', [
            'count' => count($usersLocal),
            'users' => $usersLocal
        ]);

        $this->successResponse([
            'users' => $usersLocal,
            'source' => 'local_only'
        ]);
        return;

        // legacy code below (will not run because of return)
        $users = [];
        $source = 'local'; // default source
        $totalFromDidar = 0;
        $skippedNoUserId = 0;
        $autoSynced = 0;
        $alreadyExists = 0;
        
        \App\Utils\Logger::logInfo("Fetching Didar users", ['role' => $_SESSION['role'] ?? 'unknown']);
        
        // Try to fetch from Didar API first
        $didarRes = null;
        if ($this->didarApi) {
            try {
                $didarRes = $this->didarApi->call('/User/List', 'POST');
            } catch (\Exception $e) {
                \App\Utils\Logger::logWarning("Didar API call failed, falling back to local database", ['error' => $e->getMessage()]);
                $didarRes = null;
            }
        }
        
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        
        // If Didar API returned data, process it
        if ($didarRes && !isset($didarRes['error']) && isset($didarRes['Response']) && is_array($didarRes['Response'])) {
            $source = 'didar';
            $totalFromDidar = count($didarRes['Response']);
            \App\Utils\Logger::logInfo("Received users from Didar", ['count' => $totalFromDidar]);
            
            foreach ($didarRes['Response'] as $u) {
                // بر اساس مستندات Didar: استفاده از Id به عنوان اولویت اول
                // UserId ممکن است 00000000-0000-0000-0000-000000000000 باشد
                $didarUserId = $u['Id'] ?? $u['UserId'] ?? null;
                
                // اگر UserId معتبر نیست (همه صفر) از Id استفاده کن
                if ($didarUserId === '00000000-0000-0000-0000-000000000000' || empty($didarUserId)) {
                    $didarUserId = $u['Id'] ?? null;
                }
                
                if (!$didarUserId) {
                    $skippedNoUserId++;
                    \App\Utils\Logger::logWarning("Skipping user without Id/UserId", ['user_data' => $u]);
                    continue;
                }
                
                // Check if user exists in local users table
                if ($hasDisplayName) {
                    $stmt = $pdo->prepare("SELECT didar_user_id, display_name, is_active FROM users WHERE didar_user_id = ? LIMIT 1");
                } else {
                    $stmt = $pdo->prepare("SELECT didar_user_id, is_active FROM users WHERE didar_user_id = ? LIMIT 1");
                }
                $stmt->execute([$didarUserId]);
                $localUser = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // If user doesn't exist, add it to users table (auto-sync)
                if (!$localUser) {
                    try {
                        $isActive = (isset($u['IsDisabled']) && $u['IsDisabled']) ? 0 : 1;
                        
                        if ($hasDisplayName) {
                            $pdo->prepare("INSERT INTO users (didar_user_id, email, first_name, last_name, display_name, role, is_active, last_sync) 
                                VALUES (?, ?, ?, ?, ?, 'agent', ?, NOW())
                                ON DUPLICATE KEY UPDATE 
                                    email = VALUES(email),
                                    first_name = VALUES(first_name),
                                    last_name = VALUES(last_name),
                                    display_name = VALUES(display_name),
                                    is_active = VALUES(is_active),
                                    last_sync = NOW()")
                                ->execute([
                                    $didarUserId,
                                    $u['UserName'] ?? '',
                                    $u['FirstName'] ?? '',
                                    $u['LastName'] ?? '',
                                    $u['DisplayName'] ?? '',
                                    $isActive
                                ]);
                        } else {
                            $pdo->prepare("INSERT INTO users (didar_user_id, email, first_name, last_name, role, is_active, last_sync) 
                                VALUES (?, ?, ?, ?, 'agent', ?, NOW())
                                ON DUPLICATE KEY UPDATE 
                                    email = VALUES(email),
                                    first_name = VALUES(first_name),
                                    last_name = VALUES(last_name),
                                    is_active = VALUES(is_active),
                                    last_sync = NOW()")
                                ->execute([
                                    $didarUserId,
                                    $u['UserName'] ?? '',
                                    $u['FirstName'] ?? '',
                                    $u['LastName'] ?? '',
                                    $isActive
                                ]);
                        }
                        
                        // Fetch the newly added user
                        if ($hasDisplayName) {
                            $stmt = $pdo->prepare("SELECT didar_user_id, display_name, is_active FROM users WHERE didar_user_id = ? LIMIT 1");
                        } else {
                            $stmt = $pdo->prepare("SELECT didar_user_id, is_active FROM users WHERE didar_user_id = ? LIMIT 1");
                        }
                        $stmt->execute([$didarUserId]);
                        $localUser = $stmt->fetch(PDO::FETCH_ASSOC);
                        $autoSynced++;
                        \App\Utils\Logger::logInfo("Auto-synced user from Didar", [
                            'user_id' => $didarUserId, 
                            'display_name' => $u['DisplayName'] ?? '',
                            'is_active' => $isActive
                        ]);
                    } catch (\Exception $e) {
                        \App\Utils\Logger::logError("Failed to auto-sync user", $e, ['user_id' => $didarUserId]);
                        continue;
                    }
                } else {
                    $alreadyExists++;
                }
                
                // نمایش همه کاربران (فعال و غیرفعال)
                // اگر $localUser null است، دوباره fetch کن
                if (!$localUser) {
                    if ($hasDisplayName) {
                        $stmt = $pdo->prepare("SELECT didar_user_id, display_name, is_active FROM users WHERE didar_user_id = ? LIMIT 1");
                    } else {
                        $stmt = $pdo->prepare("SELECT didar_user_id, is_active FROM users WHERE didar_user_id = ? LIMIT 1");
                    }
                    $stmt->execute([$didarUserId]);
                    $localUser = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$localUser) {
                        \App\Utils\Logger::logWarning("User still not found after re-fetch", [
                            'didar_user_id' => $didarUserId,
                            'display_name' => $u['DisplayName'] ?? ''
                        ]);
                    }
                }
                
                // همیشه کاربر را اضافه کن (حتی اگر $localUser null باشد، از داده‌های Didar استفاده کن)
                if (!empty($didarUserId)) {
                    $displayName = '';
                    $isActive = 1;
                    
                    if ($localUser) {
                        $displayName = $localUser['display_name'] ?? '';
                        $isActive = $localUser['is_active'] ?? 1;
                    }
                    
                    // اگر display_name خالی است، از Didar استفاده کن
                    if (empty($displayName)) {
                        $displayName = $u['DisplayName'] ?? (($u['FirstName'] ?? '') . ' ' . ($u['LastName'] ?? ''));
                    }
                    
                    // اگر هنوز خالی است، از UserName استفاده کن
                    if (empty($displayName)) {
                        $displayName = $u['UserName'] ?? $didarUserId;
                    }
                    
                    // برای OwnerId در contact/save، باید از UserId استفاده کنیم نه Id
                    // اما اگر UserId معتبر نیست (همه صفر)، از Id استفاده می‌کنیم
                    $ownerId = $u['UserId'] ?? $didarUserId;
                    if ($ownerId === '00000000-0000-0000-0000-000000000000' || empty($ownerId)) {
                        $ownerId = $didarUserId; // Use Id as fallback
                    }
                    
                    $users[] = [
                        'id' => $didarUserId, // برای ذخیره در دیتابیس
                        'owner_id' => $ownerId, // برای ارسال به Didar API (UserId)
                        'display_name' => trim($displayName),
                        'email' => $u['UserName'] ?? '',
                        'role' => $u['Role'] ?? '',
                        'is_active' => $isActive
                    ];
                }
            }
            
            \App\Utils\Logger::logInfo("Didar users processing complete", [
                'total_from_didar' => $totalFromDidar,
                'returned_to_client' => count($users),
                'auto_synced' => $autoSynced,
                'already_exists' => $alreadyExists,
                'skipped_no_userid' => $skippedNoUserId
            ]);
        } else {
            // Fallback: Get users from local database
            \App\Utils\Logger::logInfo("Falling back to local database for users", [
                'didar_error' => $didarRes['error'] ?? null,
                'has_response' => isset($didarRes['Response'])
            ]);
            
            try {
                if ($hasDisplayName) {
                    $stmt = $pdo->query("SELECT didar_user_id, display_name, email, is_active FROM users WHERE didar_user_id IS NOT NULL AND didar_user_id != '' ORDER BY display_name");
                } else {
                    $stmt = $pdo->query("SELECT didar_user_id, first_name, last_name, email, is_active FROM users WHERE didar_user_id IS NOT NULL AND didar_user_id != '' ORDER BY first_name, last_name");
                }
                
                $localUsers = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($localUsers as $localUser) {
                    $displayName = $localUser['display_name'] ?? trim(($localUser['first_name'] ?? '') . ' ' . ($localUser['last_name'] ?? ''));
                    
                    if (!empty($displayName) || !empty($localUser['email'])) {
                        $users[] = [
                            'id' => $localUser['didar_user_id'],
                            'display_name' => $displayName ?: $localUser['email'],
                            'email' => $localUser['email'] ?? '',
                            'role' => '',
                            'is_active' => $localUser['is_active'] ?? 1
                        ];
                    }
                }
                
                \App\Utils\Logger::logInfo("Loaded users from local database", ['count' => count($users)]);
            } catch (\Exception $e) {
                \App\Utils\Logger::logError("Failed to load users from local database", $e);
            }
        }
        
        $this->successResponse([
            'users' => $users,
            'debug' => [
                'source' => $source,
                'total_from_didar' => $totalFromDidar,
                'returned_count' => count($users),
                'auto_synced' => $autoSynced,
                'skipped_no_userid' => $skippedNoUserId
            ]
        ]);
    }
}
