<?php
namespace App\Database;

use App\Utils\Logger;
use PDOException;

/**
 * Database Migration Manager
 */
class Migration
{
    private $connection;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * Run all migrations
     */
    public function run()
    {
        $pdo = $this->connection->getPdo();
        $statements = Schema::getCreateTableStatements();

        foreach ($statements as $sql) {
            try {
                $pdo->exec($sql);
                Logger::logInfo("Table created/verified successfully");
            } catch (PDOException $e) {
                // Ignore table exists errors
                if (strpos($e->getMessage(), 'already exists') === false &&
                    strpos($e->getMessage(), 'Duplicate') === false) {
                    Logger::logError("DB Error creating table", $e, ['sql' => substr($sql, 0, 100)]);
                }
            }
        }

        // Run additional migrations for existing tables
        $this->migrateAdditionalColumns();

        // Seed default lookup values if empty
        $this->seedLookups();

        // Seed field configurations
        $this->seedFieldConfigurations();

        // Migrate payments from JSON to transactions table
        $this->migratePaymentsToTransactions();
    }

    private function ensureIndex(string $table, string $indexName, string $columns): void
    {
        $pdo = $this->connection->getPdo();
        try {
            $stmt = $pdo->prepare(
                "SELECT COUNT(*) FROM information_schema.statistics WHERE table_schema = DATABASE() AND table_name = ? AND index_name = ?"
            );
            $stmt->execute([$table, $indexName]);
            if ((int)$stmt->fetchColumn() > 0) {
                return;
            }
            $pdo->exec("CREATE INDEX {$indexName} ON {$table} ({$columns})");
        } catch (PDOException $e) {
            Logger::logWarning("Failed to create index", [
                'table' => $table,
                'index' => $indexName,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Migrate additional columns that might be missing
     */
    private function migrateAdditionalColumns()
    {
        $pdo = $this->connection->getPdo();
        
        // Additional columns for persons table
        $personsColumns = [
            'work_phone' => 'VARCHAR(50)',
            'secondary_mobile_phone' => 'VARCHAR(50)',
            'mobile_phone_3' => 'VARCHAR(50)',
            'mobile_phone_4' => 'VARCHAR(50)',
            'city' => 'VARCHAR(100)',
            'job_title' => 'VARCHAR(255)',
            'job_description' => 'TEXT',
            'acquaintance_duration' => 'VARCHAR(255)',
            'extra_info' => 'TEXT',
            'financial_level' => 'VARCHAR(50)',
            'asset_estimation' => 'VARCHAR(50)',
            'income_estimation' => 'VARCHAR(50)',
            'requested_services' => 'TEXT',
            'acquaintance_source' => 'VARCHAR(255)',
            'acquaintance_detail' => 'TEXT',
            'content_topic' => 'TEXT',
            'national_id' => 'VARCHAR(50)',
            'customer_products' => 'TEXT',
            'background_info' => 'TEXT',
            'previous_owner_id' => 'VARCHAR(255)',
            'sale_status' => 'TINYINT(1) DEFAULT 0',
            'register_time_jalali' => 'VARCHAR(50)',
        ];

        foreach ($personsColumns as $column => $type) {
            if (!$this->connection->hasColumn('persons', $column)) {
                try {
                    $pdo->exec("ALTER TABLE persons ADD COLUMN $column $type");
                    Logger::logInfo("Added column to persons table", ['column' => $column]);
                } catch (PDOException $e) {
                    Logger::logWarning("Failed to add column", ['column' => $column, 'error' => $e->getMessage()]);
                }
            }
        }

        // Additional columns for deals table
        $dealsColumns = [
	            'final_price' => 'DECIMAL(15,2)',
	            'is_paid' => 'TINYINT(1) DEFAULT 0',
	            'payment_short_link' => 'TEXT',
	            'requested_services' => 'TEXT',
	            'service_name' => 'VARCHAR(255)',
	            'service_cost' => 'DECIMAL(15,2)',
	            'price_list_code' => 'VARCHAR(255)',
	            'has_discount' => 'TINYINT(1) DEFAULT 0',
	            'discount_type' => 'VARCHAR(100)',
	            'discount_occasion' => 'VARCHAR(255)',
	            'discount_amount' => 'DECIMAL(15,2) DEFAULT 0',
	            'refund_amount' => 'DECIMAL(15,2) DEFAULT 0',
	            'refund_description' => 'TEXT',
	            'payable_amount' => 'DECIMAL(15,2)',
	            'payment_conditions' => 'TEXT',
	            'payment_amount' => 'DECIMAL(15,2) DEFAULT 0',
            'payment_method' => 'VARCHAR(100)',
            'payment_description' => 'TEXT',
            'payments' => 'TEXT',
            'is_legacy_didar' => 'TINYINT(1) DEFAULT 0',
            'pipeline_stage' => 'VARCHAR(100)',
	            'failure_reason_code' => 'VARCHAR(50)',
	            'failure_reason_text' => 'VARCHAR(255)',
	            'failure_reason_description' => 'TEXT',
	            'city' => 'VARCHAR(100)',
	            'job_title' => 'VARCHAR(255)',
	            'job_description' => 'TEXT',
	            'acquaintance_duration' => 'VARCHAR(255)',
	            'extra_info' => 'TEXT',
            'financial_level' => 'VARCHAR(50)',
            'asset_estimation' => 'VARCHAR(50)',
            'income_estimation' => 'VARCHAR(50)',
	            'has_previous_purchase' => 'TINYINT(1) NULL',
	            'person_id' => 'INT(11) NULL',
	        ];

        foreach ($dealsColumns as $column => $type) {
            if (!$this->connection->hasColumn('deals', $column)) {
                try {
                    $pdo->exec("ALTER TABLE deals ADD COLUMN $column $type");
                    Logger::logInfo("Added column to deals table", ['column' => $column]);
                } catch (PDOException $e) {
                    Logger::logWarning("Failed to add column", ['column' => $column, 'error' => $e->getMessage()]);
                }
            }
        }

        // Additional columns for sync_log table
        if (!$this->connection->hasColumn('sync_log', 'http_code')) {
            try {
                $pdo->exec("ALTER TABLE sync_log ADD COLUMN http_code INT");
            } catch (PDOException $e) {
                Logger::logWarning("Failed to add http_code column", ['error' => $e->getMessage()]);
            }
        }

        if (!$this->connection->hasColumn('sync_log', 'response_data')) {
            try {
                $pdo->exec("ALTER TABLE sync_log ADD COLUMN response_data LONGTEXT");
            } catch (PDOException $e) {
                Logger::logWarning("Failed to add response_data column", ['error' => $e->getMessage()]);
            }
        }

        // Additional columns for audit_logs table
        if ($this->connection->hasColumn('audit_logs', 'id')) {
            $auditColumns = [
                'actor_name' => 'VARCHAR(255) NULL',
                'actor_ip' => 'VARCHAR(100) NULL',
                'related_person_didar_id' => 'VARCHAR(255) NULL',
            ];
            foreach ($auditColumns as $column => $type) {
                if (!$this->connection->hasColumn('audit_logs', $column)) {
                    try {
                        $pdo->exec("ALTER TABLE audit_logs ADD COLUMN $column $type");
                        Logger::logInfo("Added column to audit_logs table", ['column' => $column]);
                    } catch (PDOException $e) {
                        Logger::logWarning("Failed to add column", ['table' => 'audit_logs', 'column' => $column, 'error' => $e->getMessage()]);
                    }
                }
            }
        }

	        // Additional columns for field_configurations
        $fcColumns = [
            'help_text' => 'TEXT',
            'default_value' => 'VARCHAR(255)',
            'validation_pattern' => 'VARCHAR(255)',
            'validation_message' => 'VARCHAR(255)',
            'conditional_logic' => 'TEXT',
            'group_name' => 'VARCHAR(100)',
            'module_name' => 'VARCHAR(100)',
            'is_editable_after_create' => 'TINYINT(1) DEFAULT 1',
            'delete_allow_roles' => 'VARCHAR(500)',
            'view_allow_roles' => 'VARCHAR(500)',
        ];

        foreach ($fcColumns as $column => $type) {
            if (!$this->connection->hasColumn('field_configurations', $column)) {
                try {
                    $pdo->exec("ALTER TABLE field_configurations ADD COLUMN $column $type");
                    Logger::logInfo("Added column to field_configurations", ['column' => $column]);
                } catch (PDOException $e) {
                    Logger::logWarning("Failed to add column to field_configurations", ['column' => $column, 'error' => $e->getMessage()]);
                }
            }
        }

        // Additional columns for activities table
	        $activitiesColumns = [
	            'is_scheduled' => 'TINYINT(1) DEFAULT 0',
	            'reminder_enabled' => 'TINYINT(1) DEFAULT 0',
	            'failure_reason_code' => 'VARCHAR(50)',
	            'failure_reason_description' => 'TEXT',
	        ];

        foreach ($activitiesColumns as $column => $type) {
            if (!$this->connection->hasColumn('activities', $column)) {
                try {
                    $pdo->exec("ALTER TABLE activities ADD COLUMN $column $type");
                    Logger::logInfo("Added column to activities table", ['column' => $column]);
                } catch (PDOException $e) {
                    Logger::logWarning("Failed to add column", ['column' => $column, 'error' => $e->getMessage()]);
                }
            }
        }

        // Add index for due_date if not exists
        try {
            $pdo->exec("CREATE INDEX IF NOT EXISTS idx_due_date ON activities(due_date)");
        } catch (PDOException $e) {
            // Index might already exist, ignore
        }

        try {
            $pdo->exec("CREATE INDEX IF NOT EXISTS idx_is_scheduled ON activities(is_scheduled)");
        } catch (PDOException $e) {
            // Index might already exist, ignore
        }

        // Add indexes for faster deal/person queries
        $this->ensureIndex('deals', 'idx_deals_legacy', 'is_legacy_didar');
        $this->ensureIndex('deals', 'idx_deals_contact_didar', 'contact_didar_id');
        $this->ensureIndex('deals', 'idx_deals_owner', 'owner_didar_id');
        $this->ensureIndex('deals', 'idx_deals_register_time', 'register_time');
        $this->ensureIndex('deals', 'idx_deals_status', 'status');
        $this->ensureIndex('deals', 'idx_deals_pipeline', 'pipeline_stage');
        $this->ensureIndex('deals', 'idx_deals_person_id', 'person_id');
        $this->ensureIndex('persons', 'idx_persons_didar_contact_id', 'didar_contact_id');

        // Relax enums to allow lookup-managed values
        $this->alterColumnType('users', 'role', "VARCHAR(50) DEFAULT 'agent'");
        $this->alterColumnType('deals', 'status', "VARCHAR(50) NULL");
        $this->alterColumnType('activities', 'direction', "VARCHAR(20) NULL");
        $this->alterColumnType('virtual_stages', 'stage_name', "VARCHAR(100) NULL");
        $this->alterColumnType('persons', 'customer_level', "VARCHAR(100) NULL");
    }

    private function alterColumnType(string $table, string $column, string $type): void
    {
        $pdo = $this->connection->getPdo();
        if (!$this->connection->hasColumn($table, $column)) {
            return;
        }
        try {
            $pdo->exec("ALTER TABLE {$table} MODIFY {$column} {$type}");
        } catch (PDOException $e) {
            Logger::logWarning("Failed to alter column type", [
                'table' => $table,
                'column' => $column,
                'error' => $e->getMessage()
            ]);
        }
    }

    private function seedLookups(): void
    {
        $pdo = $this->connection->getPdo();
        try {
            $count = (int)$pdo->query("SELECT COUNT(*) FROM lookup_groups")->fetchColumn();
        } catch (PDOException $e) {
            return;
        }
        if ($count > 0) {
            $this->ensureAdditionalLookups($pdo);
            return;
        }

        $groups = [
            ['code' => 'lead_sources', 'title' => 'شیوه‌های آشنایی', 'sort' => 10],
            ['code' => 'lead_source_details', 'title' => 'جزئیات شیوه‌های آشنایی', 'sort' => 20],
            ['code' => 'content_topics', 'title' => 'موضوعات محتوا', 'sort' => 30],
            ['code' => 'pipeline_stages', 'title' => 'مراحل پایپ‌لاین', 'sort' => 40],
            ['code' => 'allowed_services', 'title' => 'خدمات قابل ارائه', 'sort' => 50],
            ['code' => 'failure_reasons', 'title' => 'دلایل شکست', 'sort' => 60],
            ['code' => 'customer_levels', 'title' => 'سطوح مشتری', 'sort' => 70],
            ['code' => 'financial_levels', 'title' => 'سطوح مالی', 'sort' => 80],
            ['code' => 'asset_ranges', 'title' => 'بازه دارایی', 'sort' => 90],
            ['code' => 'income_ranges', 'title' => 'بازه درآمد', 'sort' => 100],
            ['code' => 'discount_types', 'title' => 'انواع تخفیف', 'sort' => 110],
            ['code' => 'discount_occasions', 'title' => 'مناسبت‌های تخفیف', 'sort' => 120],
            ['code' => 'user_roles', 'title' => 'نقش‌های کاربری', 'sort' => 130],
            ['code' => 'deal_statuses', 'title' => 'وضعیت معاملات', 'sort' => 140],
            ['code' => 'activity_stages', 'title' => 'مراحل فعالیت', 'sort' => 150],
            ['code' => 'reminder_hours', 'title' => 'ساعت‌های یادآوری', 'sort' => 160],
            ['code' => 'reminder_minutes', 'title' => 'دقیقه‌های یادآوری', 'sort' => 170],
            ['code' => 'activity_types_fallback', 'title' => 'انواع فعالیت (Fallback)', 'sort' => 180],
            ['code' => 'virtual_stages', 'title' => 'مراحل مجازی', 'sort' => 190],
            ['code' => 'activity_directions', 'title' => 'جهت فعالیت', 'sort' => 200]
        ];

        $groupStmt = $pdo->prepare("INSERT INTO lookup_groups (code, title, sort_order, is_active, is_system, created_at, updated_at)
                                    VALUES (?, ?, ?, 1, 1, NOW(), NOW())");
        $groupIds = [];
        foreach ($groups as $group) {
            $groupStmt->execute([$group['code'], $group['title'], $group['sort']]);
            $groupIds[$group['code']] = (int)$pdo->lastInsertId();
        }

        $items = [
            ['group' => 'lead_sources', 'value' => 'instagram', 'title' => 'اینستاگرام', 'sort' => 10],
            ['group' => 'lead_sources', 'value' => 'telegram', 'title' => 'تلگرام', 'sort' => 20],
            ['group' => 'lead_sources', 'value' => 'book', 'title' => 'کتاب', 'sort' => 30],
            ['group' => 'lead_sources', 'value' => 'website', 'title' => 'سایت', 'sort' => 40],
            ['group' => 'lead_sources', 'value' => 'whatsapp', 'title' => 'واتساپ', 'sort' => 50],
            ['group' => 'lead_sources', 'value' => 'campaign', 'title' => 'کمپین', 'sort' => 60],
            ['group' => 'lead_sources', 'value' => 'event', 'title' => 'ایونت', 'sort' => 70],
            ['group' => 'lead_sources', 'value' => 'support', 'title' => 'پشتیبانی', 'sort' => 80],
            ['group' => 'lead_sources', 'value' => 'incoming_call', 'title' => 'تماس ورودی', 'sort' => 90],
            ['group' => 'lead_sources', 'value' => 'visit', 'title' => 'مراجعه حضوری', 'sort' => 100],
            ['group' => 'lead_sources', 'value' => 'referral', 'title' => 'معرف', 'sort' => 110],
            ['group' => 'lead_sources', 'value' => 'other', 'title' => 'سایر', 'sort' => 120],

            ['group' => 'lead_source_details', 'value' => 'instagram_direct', 'title' => 'دایرکت', 'parent_value' => 'instagram', 'sort' => 10],
            ['group' => 'lead_source_details', 'value' => 'instagram_box', 'title' => 'باکس', 'parent_value' => 'instagram', 'sort' => 20],
            ['group' => 'lead_source_details', 'value' => 'instagram_highlight', 'title' => 'هایلایت', 'parent_value' => 'instagram', 'sort' => 30],
            ['group' => 'lead_source_details', 'value' => 'instagram_channel', 'title' => 'کانال', 'parent_value' => 'instagram', 'sort' => 40],
            ['group' => 'lead_source_details', 'value' => 'instagram_auto', 'title' => 'اتوحنفی', 'parent_value' => 'instagram', 'sort' => 50],
            ['group' => 'lead_source_details', 'value' => 'whatsapp_marketing', 'title' => 'مارکتینگ', 'parent_value' => 'whatsapp', 'sort' => 60],
            ['group' => 'lead_source_details', 'value' => 'website_signup', 'title' => 'ثبت نام', 'parent_value' => 'website', 'sort' => 70],
            ['group' => 'lead_source_details', 'value' => 'website_course_signup', 'title' => 'ثبت نام دوره', 'parent_value' => 'website', 'sort' => 80],
            ['group' => 'lead_source_details', 'value' => 'website_failed_purchase', 'title' => 'خرید ناموفق', 'parent_value' => 'website', 'sort' => 90],
            ['group' => 'lead_source_details', 'value' => 'campaign_sms', 'title' => 'پیامک', 'parent_value' => 'campaign', 'sort' => 100],
            ['group' => 'lead_source_details', 'value' => 'campaign_divar', 'title' => 'دیوار', 'parent_value' => 'campaign', 'sort' => 110],
            ['group' => 'lead_source_details', 'value' => 'campaign_porsline', 'title' => 'پرسلاین', 'parent_value' => 'campaign', 'sort' => 120],
            ['group' => 'lead_source_details', 'value' => 'campaign_web', 'title' => 'وب', 'parent_value' => 'campaign', 'sort' => 130],
            ['group' => 'lead_source_details', 'value' => 'event_live', 'title' => 'لایو', 'parent_value' => 'event', 'sort' => 140],
            ['group' => 'lead_source_details', 'value' => 'event_meetup', 'title' => 'دورهمی', 'parent_value' => 'event', 'sort' => 150],
            ['group' => 'lead_source_details', 'value' => 'event_course', 'title' => 'دوره', 'parent_value' => 'event', 'sort' => 160],
            ['group' => 'lead_source_details', 'value' => 'referral_internal', 'title' => 'همکاران داخلی', 'parent_value' => 'referral', 'sort' => 170],
            ['group' => 'lead_source_details', 'value' => 'referral_external', 'title' => 'همکاران خارجی', 'parent_value' => 'referral', 'sort' => 180],
            ['group' => 'lead_source_details', 'value' => 'referral_teacher', 'title' => 'استاد', 'parent_value' => 'referral', 'sort' => 190],
            ['group' => 'lead_source_details', 'value' => 'referral_customer', 'title' => 'مشتری قدیم', 'parent_value' => 'referral', 'sort' => 200],

            ['group' => 'content_topics', 'value' => 'car_intelligence', 'title' => 'هوش خودرو', 'sort' => 10],
            ['group' => 'content_topics', 'value' => 'in_person_inspection', 'title' => 'کارشناسی حضوری', 'sort' => 20],
            ['group' => 'content_topics', 'value' => 'business', 'title' => 'کسب و کار', 'sort' => 30],
            ['group' => 'content_topics', 'value' => 'mini_course', 'title' => 'مینی دوره هوش خودرو', 'sort' => 40],
            ['group' => 'content_topics', 'value' => 'vip_channel', 'title' => 'کانال VIP', 'sort' => 50],
            ['group' => 'content_topics', 'value' => 'budgeting', 'title' => 'بودجه بندی', 'sort' => 60],
            ['group' => 'content_topics', 'value' => 'online_exhibition', 'title' => 'نمایشگاه آنلاین', 'sort' => 70],
            ['group' => 'content_topics', 'value' => 'consulting', 'title' => 'مشاوره با استاد', 'sort' => 80],
            ['group' => 'content_topics', 'value' => 'inspection_device', 'title' => 'دستگاه کارشناسی', 'sort' => 90],
            ['group' => 'content_topics', 'value' => 'car_intelligence_test', 'title' => 'تست هوش خودرو', 'sort' => 100],
            ['group' => 'content_topics', 'value' => 'financial_personality', 'title' => 'تست تیپ شخصیت مالی', 'sort' => 110],
            ['group' => 'content_topics', 'value' => 'smart_trader', 'title' => 'معامله گر هوشمند', 'sort' => 120],
            ['group' => 'content_topics', 'value' => 'detailing_training', 'title' => 'آموزش دیتیلینگ', 'sort' => 130],
            ['group' => 'content_topics', 'value' => 'divar_car', 'title' => 'دیوار خودرو', 'sort' => 140],
            ['group' => 'content_topics', 'value' => 'inspection_test', 'title' => 'تست کارشناسی', 'sort' => 150],
            ['group' => 'content_topics', 'value' => 'book', 'title' => 'کتاب هوش خودرو', 'sort' => 160],
            ['group' => 'content_topics', 'value' => 'cip', 'title' => 'CIP', 'sort' => 170],

            ['group' => 'pipeline_stages', 'title' => 'بلاتکلیف', 'value' => 'بلاتکلیف', 'sort' => 10],
            ['group' => 'pipeline_stages', 'title' => 'معامله جدید', 'value' => 'معامله جدید', 'sort' => 20],
            ['group' => 'pipeline_stages', 'title' => 'تماس عدم پاسخ', 'value' => 'تماس عدم پاسخ', 'sort' => 30],
            ['group' => 'pipeline_stages', 'title' => 'تماس پیگیری', 'value' => 'تماس پیگیری', 'sort' => 40],
            ['group' => 'pipeline_stages', 'title' => 'ارسال لینک پرداخت', 'value' => 'ارسال لینک پرداخت', 'sort' => 50],
            ['group' => 'pipeline_stages', 'title' => 'پرداخت', 'value' => 'پرداخت', 'sort' => 60],
            ['group' => 'pipeline_stages', 'title' => 'پیگیری مانده حساب', 'value' => 'پیگیری مانده حساب', 'sort' => 70],
            ['group' => 'pipeline_stages', 'title' => 'برگزاری دوره', 'value' => 'برگزاری دوره', 'sort' => 80],
            ['group' => 'pipeline_stages', 'title' => 'پشتیبانی', 'value' => 'پشتیبانی', 'sort' => 90],
            ['group' => 'pipeline_stages', 'title' => 'ارجاع به CRM', 'value' => 'ارجاع به CRM', 'sort' => 100],

            ['group' => 'allowed_services', 'title' => 'هوش خودرو', 'value' => 'هوش خودرو', 'sort' => 10],
            ['group' => 'allowed_services', 'title' => 'کارشناسی حضوری', 'value' => 'کارشناسی حضوری', 'sort' => 20],
            ['group' => 'allowed_services', 'title' => 'کانال VIP', 'value' => 'کانال VIP', 'sort' => 30],
            ['group' => 'allowed_services', 'title' => 'دستگاه کارشناسی', 'value' => 'دستگاه کارشناسی', 'sort' => 40],
            ['group' => 'allowed_services', 'title' => 'مشاوره', 'value' => 'مشاوره', 'sort' => 50],
            ['group' => 'allowed_services', 'title' => 'پکیج مشاوره', 'value' => 'پکیج مشاوره', 'sort' => 60],
            ['group' => 'allowed_services', 'title' => 'چکاپ کسب و کار', 'value' => 'چکاپ کسب و کار', 'sort' => 70],
            ['group' => 'allowed_services', 'title' => 'شتاب کسب و کار', 'value' => 'شتاب کسب و کار', 'sort' => 80],
            ['group' => 'allowed_services', 'title' => 'مربی شخصی', 'value' => 'مربی شخصی', 'sort' => 90],
            ['group' => 'allowed_services', 'title' => 'آموزش دیتیلینگ', 'value' => 'آموزش دیتیلینگ', 'sort' => 100],
            ['group' => 'allowed_services', 'title' => 'نمایشگاه آنلاین', 'value' => 'نمایشگاه آنلاین', 'sort' => 110],
            ['group' => 'allowed_services', 'title' => 'دوره دیوار خودرو', 'value' => 'دوره دیوار خودرو', 'sort' => 120],
            ['group' => 'allowed_services', 'title' => 'بودجه بندی', 'value' => 'بودجه بندی', 'sort' => 130],
            ['group' => 'allowed_services', 'title' => 'کارشناسی برتر', 'value' => 'کارشناسی برتر', 'sort' => 140],
            ['group' => 'allowed_services', 'title' => 'مدرک کارشناسی', 'value' => 'مدرک کارشناسی', 'sort' => 150],
            ['group' => 'allowed_services', 'title' => 'CIP', 'value' => 'CIP', 'sort' => 160],

            ['group' => 'failure_reasons', 'value' => 'no_financial_ability', 'title' => 'عدم توان مالی', 'sort' => 10],
            ['group' => 'failure_reasons', 'value' => 'no_purchase_intent', 'title' => 'فعلا قصد خرید ندارد', 'sort' => 20],
            ['group' => 'failure_reasons', 'value' => 'competitor_purchase', 'title' => 'خرید از رقبا', 'sort' => 30],
            ['group' => 'failure_reasons', 'value' => 'wrong_number', 'title' => 'شماره اشتباه', 'sort' => 40],
            ['group' => 'failure_reasons', 'value' => 'irrelevant_lead', 'title' => 'لید نامرتبط', 'sort' => 50],
            ['group' => 'failure_reasons', 'value' => 'other', 'title' => 'سایر', 'sort' => 60],

            ['group' => 'customer_levels', 'value' => 'طلایی', 'title' => 'طلایی', 'sort' => 10],
            ['group' => 'customer_levels', 'value' => 'نقره‌ای', 'title' => 'نقره‌ای', 'sort' => 20],
            ['group' => 'customer_levels', 'value' => 'برنز', 'title' => 'برنز', 'sort' => 30],

            ['group' => 'financial_levels', 'value' => 'بالا', 'title' => 'بالا', 'sort' => 10],
            ['group' => 'financial_levels', 'value' => 'متوسط رو به بالا', 'title' => 'متوسط رو به بالا', 'sort' => 20],
            ['group' => 'financial_levels', 'value' => 'متوسط رو به پایین', 'title' => 'متوسط رو به پایین', 'sort' => 30],
            ['group' => 'financial_levels', 'value' => 'پایین', 'title' => 'پایین', 'sort' => 40],

            ['group' => 'asset_ranges', 'value' => '0-50', 'title' => '0 تا 50 میلیون تومان', 'sort' => 10],
            ['group' => 'asset_ranges', 'value' => '50-200', 'title' => '50 تا 200 میلیون تومان', 'sort' => 20],
            ['group' => 'asset_ranges', 'value' => '200-500', 'title' => '200 تا 500 میلیون تومان', 'sort' => 30],
            ['group' => 'asset_ranges', 'value' => '500-1000', 'title' => '500 تا 1000 میلیون تومان', 'sort' => 40],
            ['group' => 'asset_ranges', 'value' => '1000-plus', 'title' => 'بیش از 1 میلیارد تومان', 'sort' => 50],

            ['group' => 'income_ranges', 'value' => '0-20', 'title' => '0 تا 20 میلیون تومان', 'sort' => 10],
            ['group' => 'income_ranges', 'value' => '20-100', 'title' => '20 تا 100 میلیون تومان', 'sort' => 20],
            ['group' => 'income_ranges', 'value' => '100-200', 'title' => '100 تا 200 میلیون تومان', 'sort' => 30],
            ['group' => 'income_ranges', 'value' => '200-500', 'title' => '200 تا 500 میلیون تومان', 'sort' => 40],
            ['group' => 'income_ranges', 'value' => '500-plus', 'title' => 'بیش از 500 میلیون تومان', 'sort' => 50],

            ['group' => 'discount_types', 'value' => 'occasion', 'title' => 'مناسبتی', 'sort' => 10],
            ['group' => 'discount_types', 'value' => 'individual', 'title' => 'فردی', 'sort' => 20],
            ['group' => 'discount_types', 'value' => 'refund', 'title' => 'مرجوعی', 'sort' => 30],

            ['group' => 'discount_occasions', 'value' => 'black_friday', 'title' => 'بلک فرایدی', 'sort' => 10],
            ['group' => 'discount_occasions', 'value' => 'teacher_birthday', 'title' => 'تولد استاد', 'sort' => 20],
            ['group' => 'discount_occasions', 'value' => 'nowruz', 'title' => 'عید نوروز', 'sort' => 30],
            ['group' => 'discount_occasions', 'value' => 'other', 'title' => 'سایر', 'sort' => 40],

            ['group' => 'user_roles', 'value' => 'admin', 'title' => 'مدیر', 'sort' => 10],
            ['group' => 'user_roles', 'value' => 'agent', 'title' => 'کارشناس', 'sort' => 20],
            ['group' => 'user_roles', 'value' => 'crm_specialist', 'title' => 'کارشناس CRM', 'sort' => 30],

            ['group' => 'deal_statuses', 'value' => 'pending', 'title' => 'معامله جاری', 'sort' => 10],
            ['group' => 'deal_statuses', 'value' => 'won', 'title' => 'معامله موفق', 'sort' => 20],
            ['group' => 'deal_statuses', 'value' => 'lost', 'title' => 'معامله ناموفق', 'sort' => 30],

            ['group' => 'activity_stages', 'value' => 'تماس عدم پاسخ', 'title' => 'تماس عدم پاسخ', 'sort' => 10],
            ['group' => 'activity_stages', 'value' => 'تماس پیگیری', 'title' => 'تماس پیگیری', 'sort' => 20],
            ['group' => 'activity_stages', 'value' => 'ارسال لینک پرداخت', 'title' => 'ارسال لینک پرداخت', 'sort' => 30],
            ['group' => 'activity_stages', 'value' => 'پرداخت', 'title' => 'پرداخت', 'sort' => 40],
            ['group' => 'activity_stages', 'value' => 'پیگیری مانده حساب', 'title' => 'پیگیری مانده حساب', 'sort' => 50],
            ['group' => 'activity_stages', 'value' => 'برگزاری دوره', 'title' => 'برگزاری دوره', 'sort' => 60],
            ['group' => 'activity_stages', 'value' => 'پشتیبانی', 'title' => 'پشتیبانی', 'sort' => 70],
            ['group' => 'activity_stages', 'value' => 'ارجاع به CRM', 'title' => 'ارجاع به CRM', 'sort' => 80],
            ['group' => 'activity_stages', 'value' => 'deal_failed', 'title' => 'معامله ناموفق', 'sort' => 90],
            ['group' => 'activity_stages', 'value' => 'deal_success', 'title' => 'معامله موفق', 'sort' => 100],

            ['group' => 'reminder_hours', 'value' => '0', 'title' => '0 ساعت', 'sort' => 0],
            ['group' => 'reminder_hours', 'value' => '1', 'title' => '1 ساعت', 'sort' => 10],
            ['group' => 'reminder_hours', 'value' => '2', 'title' => '2 ساعت', 'sort' => 20],
            ['group' => 'reminder_hours', 'value' => '4', 'title' => '4 ساعت', 'sort' => 30],
            ['group' => 'reminder_hours', 'value' => '8', 'title' => '8 ساعت', 'sort' => 40],
            ['group' => 'reminder_hours', 'value' => '12', 'title' => '12 ساعت', 'sort' => 50],
            ['group' => 'reminder_hours', 'value' => '24', 'title' => '24 ساعت', 'sort' => 60],
            ['group' => 'reminder_hours', 'value' => '168', 'title' => '1 هفته', 'sort' => 70],

            ['group' => 'reminder_minutes', 'value' => '0', 'title' => '0 دقیقه', 'sort' => 0],
            ['group' => 'reminder_minutes', 'value' => '5', 'title' => '5 دقیقه', 'sort' => 10],
            ['group' => 'reminder_minutes', 'value' => '10', 'title' => '10 دقیقه', 'sort' => 20],
            ['group' => 'reminder_minutes', 'value' => '15', 'title' => '15 دقیقه', 'sort' => 30],
            ['group' => 'reminder_minutes', 'value' => '30', 'title' => '30 دقیقه', 'sort' => 40],

            ['group' => 'activity_types_fallback', 'value' => '2a335dc1-4834-484b-b6dc-9ddd97eef67e', 'title' => 'تماس ورودی', 'meta' => '{"direction":"incoming","category":"call"}', 'sort' => 10],
            ['group' => 'activity_types_fallback', 'value' => '790079e8-74ac-4760-9527-097627d3e7ec', 'title' => 'تماس خروجی', 'meta' => '{"direction":"outgoing","category":"call"}', 'sort' => 20],
            ['group' => 'activity_types_fallback', 'value' => '3b446dc2-5945-595c-c7ed-0eee08ff78fd', 'title' => 'چت ورودی', 'meta' => '{"direction":"incoming","category":"chat"}', 'sort' => 30],
            ['group' => 'activity_types_fallback', 'value' => '4c557ed3-6a56-606d-d8fe-1fff19gg89ge', 'title' => 'چت خروجی', 'meta' => '{"direction":"outgoing","category":"chat"}', 'sort' => 40],

            ['group' => 'virtual_stages', 'value' => 'with_purchase', 'title' => 'with_purchase', 'sort' => 10],
            ['group' => 'virtual_stages', 'value' => 'without_purchase', 'title' => 'without_purchase', 'sort' => 20],
            ['group' => 'virtual_stages', 'value' => 'new', 'title' => 'new', 'sort' => 30],
            ['group' => 'virtual_stages', 'value' => 'contact_no_answer', 'title' => 'contact_no_answer', 'sort' => 40],
            ['group' => 'virtual_stages', 'value' => 'contact_followup', 'title' => 'contact_followup', 'sort' => 50],
            ['group' => 'virtual_stages', 'value' => 'deal_registered', 'title' => 'deal_registered', 'sort' => 60],
            ['group' => 'virtual_stages', 'value' => 'payment', 'title' => 'payment', 'sort' => 70],
            ['group' => 'virtual_stages', 'value' => 'payment_followup', 'title' => 'payment_followup', 'sort' => 80],
            ['group' => 'virtual_stages', 'value' => 'course_delivered', 'title' => 'course_delivered', 'sort' => 90],
            ['group' => 'virtual_stages', 'value' => 'support', 'title' => 'support', 'sort' => 100],
            ['group' => 'virtual_stages', 'value' => 'refer_crm', 'title' => 'refer_crm', 'sort' => 110],

            ['group' => 'activity_directions', 'value' => 'incoming', 'title' => 'incoming', 'sort' => 10],
            ['group' => 'activity_directions', 'value' => 'outgoing', 'title' => 'outgoing', 'sort' => 20]
        ];

        $itemStmt = $pdo->prepare("INSERT INTO lookup_items (group_id, parent_id, code, title, value, meta_json, sort_order, is_active, is_system, created_at, updated_at)
                                   VALUES (?, ?, ?, ?, ?, ?, ?, 1, 1, NOW(), NOW())");
        $itemIndex = [];
        foreach ($items as $item) {
            $groupId = $groupIds[$item['group']] ?? null;
            if (!$groupId) {
                continue;
            }
            $parentId = null;
            if (!empty($item['parent_value'])) {
                $parentId = $itemIndex['lead_sources'][$item['parent_value']] ?? null;
            }
            $itemStmt->execute([
                $groupId,
                $parentId,
                $item['code'] ?? null,
                $item['title'],
                $item['value'] ?? $item['title'],
                $item['meta'] ?? null,
                $item['sort'] ?? 0
            ]);
            $insertedId = (int)$pdo->lastInsertId();
            if ($item['group'] === 'lead_sources') {
                $itemIndex['lead_sources'][$item['value']] = $insertedId;
            }
        }

        $this->ensureAdditionalLookups($pdo);
    }

    private function ensureAdditionalLookups($pdo): void
    {
        try {
            $rows = $pdo->query("SELECT id, code FROM lookup_groups")->fetchAll(\PDO::FETCH_ASSOC) ?: [];
        } catch (PDOException $e) {
            return;
        }

        $groupIds = [];
        foreach ($rows as $row) {
            $groupIds[$row['code']] = (int)$row['id'];
        }

        $ensureGroup = function (string $code, string $title, int $sort) use ($pdo, &$groupIds): int {
            if (isset($groupIds[$code])) {
                return $groupIds[$code];
            }
            $stmt = $pdo->prepare("INSERT INTO lookup_groups (code, title, sort_order, is_active, is_system, created_at, updated_at)
                                   VALUES (?, ?, ?, 1, 1, NOW(), NOW())");
            $stmt->execute([$code, $title, $sort]);
            $groupIds[$code] = (int)$pdo->lastInsertId();
            return $groupIds[$code];
        };

        $ensureItem = function (int $groupId, string $value, string $title, int $sort, ?string $meta = null, ?string $code = null) use ($pdo): void {
            if ($code !== null && $code !== '') {
                $stmt = $pdo->prepare("SELECT id FROM lookup_items WHERE group_id = ? AND (code = ? OR value = ?) LIMIT 1");
                $stmt->execute([$groupId, $code, $value]);
            } else {
                $stmt = $pdo->prepare("SELECT id FROM lookup_items WHERE group_id = ? AND value = ? LIMIT 1");
                $stmt->execute([$groupId, $value]);
            }
            if ($stmt->fetchColumn()) {
                return;
            }
            $insert = $pdo->prepare("INSERT INTO lookup_items (group_id, parent_id, code, title, value, meta_json, sort_order, is_active, is_system, created_at, updated_at)
                                     VALUES (?, NULL, ?, ?, ?, ?, ?, 1, 1, NOW(), NOW())");
            $insert->execute([$groupId, $code, $title, $value, $meta, $sort]);
        };

        $paymentMethodsGroupId = $ensureGroup('payment_methods', 'Payment Methods', 210);
        $ensureItem($paymentMethodsGroupId, 'card_reader', 'کارتخوان', 10, '{"requires_card_last4":true,"requires_bank_destination":true}');
        $ensureItem($paymentMethodsGroupId, 'card_to_card', 'کارت به کارت', 20, '{"requires_card_last4":true,"requires_bank_destination":true}');
        $ensureItem($paymentMethodsGroupId, 'card_to_card_lookup', 'کارت به کارت (Lookup)', 25, '{"requires_card_last4":true,"requires_bank_destination":true}');
        $ensureItem($paymentMethodsGroupId, 'check', 'چک', 30, '{"requires_card_last4":false,"requires_bank_destination":false}');

        $banksGroupId = $ensureGroup('payment_destination_banks', 'Payment Destination Banks', 220);
        $ensureItem($banksGroupId, 'parsian', 'پارسیان', 10);
        $ensureItem($banksGroupId, 'pasargad', 'پاسارگاد', 20);
        $ensureItem($banksGroupId, 'melli', 'ملی', 30);
        $ensureItem($banksGroupId, 'ayandeh', 'آینده', 40);
        $ensureItem($banksGroupId, 'mellat', 'ملت', 50);
        $ensureItem($banksGroupId, 'tejarat', 'تجارت', 60);
        $ensureItem($banksGroupId, 'other', 'سایر (درج در توضیحات)', 70);

        $rulesGroupId = $ensureGroup('field_rules', 'Field Rules', 230);
        $ensureItem($rulesGroupId, 'deal.payments.card_last4', '4 رقم آخر کارت پرداخت کننده', 10, '{"required":true,"editable":true,"editable_after_create":true,"editable_roles":[]}');
        $ensureItem($rulesGroupId, 'deal.payments.bank_destination', 'بانک مقصد پرداخت', 20, '{"required":false,"editable":true,"editable_after_create":true,"editable_roles":[]}');
        $ensureItem($rulesGroupId, 'deal.payments.method', 'نحوه پرداخت', 30, '{"required":true,"editable":true,"editable_after_create":true,"editable_roles":[]}');
    }

    private function seedFieldConfigurations(): void
    {
        $pdo = $this->connection->getPdo();

        // 1. Ensure Lookup Groups exist with unified codes
        $groups = [
            ['code' => 'payment_methods', 'title' => 'نحوه‌های پرداخت', 'sort' => 1],
            ['code' => 'payment_destination_banks', 'title' => 'بانک‌های مقصد پرداخت', 'sort' => 2]
        ];

        foreach ($groups as $g) {
            $stmt = $pdo->prepare("SELECT id FROM lookup_groups WHERE code = ?");
            $stmt->execute([$g['code']]);
            if (!$stmt->fetch()) {
                $pdo->prepare("INSERT INTO lookup_groups (code, title, sort_order, is_active, is_system) VALUES (?, ?, ?, 1, 0)")
                    ->execute([$g['code'], $g['title'], $g['sort']]);
            }
        }

        // 2. Ensure Lookup Items exist
        $paymentMethodsGroupId = (int)$pdo->query("SELECT id FROM lookup_groups WHERE code = 'payment_methods'")->fetchColumn();
        $methods = [
            ['code' => 'card_reader', 'title' => 'کارتخوان', 'value' => 'card_reader'],
            ['code' => 'card_to_card', 'title' => 'کارت به کارت', 'value' => 'card_to_card'],
            ['code' => 'card_to_card_lookup', 'title' => 'کارت به کارت (Lookup)', 'value' => 'card_to_card_lookup'],
            ['code' => 'payment_gateway', 'title' => 'درگاه پرداخت', 'value' => 'payment_gateway'],
            ['code' => 'check', 'title' => 'چک', 'value' => 'check'],
            ['code' => 'purchase_order', 'title' => 'سفارش خرید', 'value' => 'purchase_order'],
            ['code' => 'other', 'title' => 'سایر', 'value' => 'other'],
        ];
        foreach ($methods as $i => $m) {
            $stmt = $pdo->prepare("SELECT id FROM lookup_items WHERE group_id = ? AND (code = ? OR value = ?)");
            $stmt->execute([$paymentMethodsGroupId, $m['code'], $m['value']]);
            if (!$stmt->fetch()) {
                $pdo->prepare("INSERT INTO lookup_items (group_id, code, title, value, sort_order, is_active) VALUES (?, ?, ?, ?, ?, 1)")
                    ->execute([$paymentMethodsGroupId, $m['code'], $m['title'], $m['value'], $i+1]);
            }
        }

        $banksGroupId = (int)$pdo->query("SELECT id FROM lookup_groups WHERE code = 'payment_destination_banks'")->fetchColumn();
        $banks = [
            ['code' => 'parsian', 'title' => 'بانک پارسیان', 'value' => 'parsian'],
            ['code' => 'pasargad', 'title' => 'بانک پاسارگاد', 'value' => 'pasargad'],
            ['code' => 'melli', 'title' => 'بانک ملی', 'value' => 'melli'],
            ['code' => 'ayandeh', 'title' => 'بانک آینده', 'value' => 'ayandeh'],
            ['code' => 'melat', 'title' => 'بانک ملت', 'value' => 'melat'],
            ['code' => 'tejarat', 'title' => 'بانک تجارت', 'value' => 'tejarat'],
            ['code' => 'other', 'title' => 'سایر', 'value' => 'other'],
        ];
        foreach ($banks as $i => $b) {
            $stmt = $pdo->prepare("SELECT id FROM lookup_items WHERE group_id = ? AND (code = ? OR value = ?)");
            $stmt->execute([$banksGroupId, $b['code'], $b['value']]);
            if (!$stmt->fetch()) {
                $pdo->prepare("INSERT INTO lookup_items (group_id, code, title, value, sort_order, is_active) VALUES (?, ?, ?, ?, ?, 1)")
                    ->execute([$banksGroupId, $b['code'], $b['title'], $b['value'], $i+1]);
            }
        }

        // 3. Seed default field configurations with all new fields
        $configs = [
            [
                'field_name' => 'payment_method', 
                'field_label' => 'نحوه پرداخت', 
                'type' => 'select', 
                'lookup' => 'payment_methods', 
                'edit_roles' => ['admin', 'crm_specialist', 'agent'],
                'delete_roles' => ['admin'],
                'editable_after_create' => 1,
                'help_text' => 'نحوه دریافت وجه از مشتری'
            ],
            [
                'field_name' => 'payment_card_last4', 
                'field_label' => 'آخر ۴ رقم کارت', 
                'type' => 'text', 
                'lookup' => null, 
                'edit_roles' => ['admin', 'crm_specialist', 'agent'],
                'delete_roles' => ['admin'],
                'editable_after_create' => 1,
                'validation_pattern' => '^\d{4}$',
                'help_text' => '۴ رقم آخر شماره کارت پرداخت‌کننده',
                'conditional_logic' => '{"field":"payment_method","operator":"in","value":["card_to_card","card_reader","card_to_card_lookup"]}'
            ],
            [
                'field_name' => 'destination_bank', 
                'field_label' => 'بانک مقصد پرداخت', 
                'type' => 'select', 
                'lookup' => 'payment_destination_banks', 
                'edit_roles' => ['admin', 'crm_specialist'],
                'delete_roles' => ['admin'],
                'editable_after_create' => 1,
                'help_text' => 'بانکی که وجه به آن واریز می‌شود',
                'conditional_logic' => '{"field":"payment_method","operator":"in","value":["card_to_card","card_to_card_lookup"]}'
            ],
            [
                'field_name' => 'payment_delete', 
                'field_label' => 'حذف پرداخت', 
                'type' => 'text', 
                'lookup' => null, 
                'edit_roles' => ['admin'],
                'delete_roles' => ['admin'],
                'editable_after_create' => 1
            ],
            [
                'field_name' => 'deal_title', 
                'field_label' => 'عنوان معامله', 
                'type' => 'text', 
                'lookup' => null, 
                'edit_roles' => ['admin'],
                'delete_roles' => ['admin'],
                'editable_after_create' => 0,
                'help_text' => 'عنوان معامله پس از ثبت قابل ویرایش نیست'
            ],
        ];

        foreach ($configs as $i => $c) {
            $stmt = $pdo->prepare("SELECT id FROM field_configurations WHERE field_name = ?");
            $stmt->execute([$c['field_name']]);
            if (!$stmt->fetch()) {
                $pdo->prepare("INSERT INTO field_configurations 
                    (field_name, field_label, field_type, entity_type, is_required, is_editable, is_editable_after_create, is_active, is_system, edit_allow_roles, delete_allow_roles, lookup_group_code, sort_order, help_text, validation_pattern, conditional_logic) 
                    VALUES (?, ?, ?, 'deal', 0, 1, ?, 1, 0, ?, ?, ?, ?, ?, ?, ?)")
                    ->execute([
                        $c['field_name'], 
                        $c['field_label'], 
                        $c['type'], 
                        $c['editable_after_create'] ?? 1,
                        json_encode($c['edit_roles'] ?? []), 
                        json_encode($c['delete_roles'] ?? []),
                        $c['lookup'], 
                        $i + 1,
                        $c['help_text'] ?? null,
                        $c['validation_pattern'] ?? null,
                        $c['conditional_logic'] ?? null
                    ]);
            }
        }
        Logger::logInfo("Database seeded successfully with unified lookup codes");
    }

    /**
     * Migrate payments from JSON to transactions table
     */
    public function migratePaymentsToTransactions(): void
    {
        $pdo = $this->connection->getPdo();

        // Check if migration already ran
        $stmt = $pdo->query("SELECT COUNT(*) FROM transactions");
        $existingCount = $stmt->fetchColumn();

        if ($existingCount > 0) {
            Logger::logInfo("Payments migration already completed - skipped", ['existing_transactions' => $existingCount]);
            return;
        }

        // Get all deals with payments
        $stmt = $pdo->prepare("SELECT id, payments, won_time FROM deals WHERE payments IS NOT NULL AND payments != '' AND payments != 'null'");
        $stmt->execute();
        $deals = $stmt->fetchAll(\PDO::FETCH_ASSOC);

        $migrated = 0;
        $skipped = 0;
        $totalAmount = 0;

        Logger::logInfo("Starting payments migration", ['deals_with_payments' => count($deals)]);

        foreach ($deals as $deal) {
            try {
                $paymentsJson = $deal['payments'];
                $payments = json_decode($paymentsJson, true);

                if (!is_array($payments) || empty($payments)) {
                    $skipped++;
                    continue;
                }

                // Sort payments by date to find first payment
                usort($payments, function($a, $b) {
                    $dateA = $a['date'] ?? ($a['date_persian'] ?? '1970-01-01');
                    $dateB = $b['date'] ?? ($b['date_persian'] ?? '1970-01-01');
                    return strtotime($dateA) - strtotime($dateB);
                });

                foreach ($payments as $index => $payment) {
                    $amount = floatval($payment['amount'] ?? 0);
                    if ($amount <= 0) {
                        continue;
                    }

                    // Convert Persian date to Gregorian if needed
                    $dateStr = $payment['date'] ?? ($payment['date_persian'] ?? '');
                    if (preg_match('/^\d{4}\/\d{2}\/\d{2}$/', $dateStr)) {
                        // Persian date: convert
                        $parts = explode('/', $dateStr);
                        $gregorian = $this->jalaliToGregorian($parts[0], $parts[1], $parts[2]);
                        $dateStr = sprintf('%04d-%02d-%02d', $gregorian[0], $gregorian[1], $gregorian[2]);
                    } else {
                        // Already Gregorian or invalid - try to validate
                        $testDate = \DateTime::createFromFormat('Y-m-d', $dateStr);
                        if (!$testDate) {
                            // If invalid, use won_time or current date
                            if ($deal['won_time']) {
                                $dateStr = substr($deal['won_time'], 0, 10);
                            } else {
                                $dateStr = date('Y-m-d');
                            }
                        }
                    }

                    $isFirstPayment = ($index === 0) ? 1 : 0;

                    // Normalize payment fields
                    $method = $payment['method'] ?? $payment['payment_method'] ?? null;
                    $cardLast4 = $payment['card_last4'] ?? $payment['card_last_4'] ?? null;
                    $destinationBank = $payment['destination_bank'] ?? $payment['bank_destination'] ?? null;
                    $description = $payment['description'] ?? null;

                    $stmt = $pdo->prepare("
                        INSERT INTO transactions (deal_id, amount, payment_date, method, card_last4, destination_bank, description, is_first_payment)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    ");

                    $stmt->execute([
                        $deal['id'],
                        $amount,
                        $dateStr,
                        $method,
                        $cardLast4,
                        $destinationBank,
                        $description,
                        $isFirstPayment
                    ]);

                    $migrated++;
                    $totalAmount += $amount;
                }

            } catch (\Exception $e) {
                Logger::logError("Failed to migrate payments for deal {$deal['id']}", $e);
                $skipped++;
            }
        }

        Logger::logInfo("Payment migration completed", [
            'migrated' => $migrated,
            'skipped' => $skipped,
            'total_amount' => $totalAmount,
            'total_deals' => count($deals)
        ]);
    }

    /**
     * Convert Jalali date to Gregorian
     */
    private function jalaliToGregorian($jy, $jm, $jd): array
    {
        // Simple conversion - can be improved with proper library
        // For now, using basic algorithm
        $jy = (int) $jy;
        $jm = (int) $jm;
        $jd = (int) $jd;

        // Check if jy is 1404 or 1405 and convert approximately
        // 1404/09/01 ≈ 2024-11-21
        // This is a simplified approach
        if ($jy >= 1400) {
            $gy = $jy - 621;
            if ($jm > 3) {
                $gy++;
            }
        } else {
            $gy = $jy + 621;
        }

        // Approximate month conversion
        $gm = $jm;
        if ($gm > 3) {
            $gm = $gm - 3;
        } else {
            $gm = $gm + 9;
            $gy--;
        }

        $gd = min($jd, 31);

        // Validate and adjust
        $checkDate = \DateTime::createFromFormat('Y-m-d', sprintf('%04d-%02d-%02d', $gy, $gm, $gd));
        if (!$checkDate) {
            // Fallback to a reasonable date
            return [2024, 11, 1];
        }

        return [$gy, $gm, $gd];
    }
}