<?php
namespace App\Repositories;

use App\Database\Connection;
use PDO;

class KpiRepository
{
    private $db;

    public function __construct(Connection $db)
    {
        $this->db = $db;
    }

    /**
     * 1. فروش
     * تعداد معاملاتی که در بازه زمانی ساخته شده‌اند
     * مجموع مبلغ قابل پرداخت این معاملات
     * مجموع کل پرداخت‌های ثبت شده برای این معاملات (بدون محدودیت تاریخی)
     */
    public function getSales(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as deal_count,
                SUM(d.payable_amount) as total_contract_amount,
                SUM(COALESCE(tp.total_paid, 0)) as total_collected_amount
            FROM deals d
            LEFT JOIN (
                SELECT deal_id, SUM(amount) as total_paid
                FROM transactions
                WHERE status = 'confirmed'
                GROUP BY deal_id
            ) tp ON tp.deal_id = d.id
            WHERE {$where}
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * 2. تسویه حساب کامل
     * تعداد هر معامله که حداقل یک پرداخت ثبت شده دارد
     * و مجموع مبلغ پرداخت های ثبت شده آن با مبلغ قابل پرداخت آن معامله (name="payable_amount") برابر باشد
     * و مجموع مبلغ تسویه شده این معاملات تسویه شده
     */
    public function getSettledDeals(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $whereClause = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as deal_count,
                SUM(d.payable_amount) as total_amount,
                SUM(COALESCE(tp.total_paid, 0)) as total_collected_amount
            FROM deals d
            LEFT JOIN (
                SELECT deal_id, SUM(amount) as total_paid
                FROM transactions
                WHERE status = 'confirmed'
                GROUP BY deal_id
            ) tp ON tp.deal_id = d.id
            WHERE d.payable_amount IS NOT NULL
              AND d.payable_amount > 0
              AND COALESCE(tp.total_paid, 0) >= d.payable_amount
              AND {$whereClause}
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * 3. مانده حساب
     * تعداد معاملاتی که حداقل یک پرداخت ثبت شده دارد
     * و مجموع رقم پرداخت های ثبت شده آن با مبلغ قابل پرداخت آن معامله برابر نباشد
     * و مجموع مبلغ پرداخت شده این معاملات
     * و مجموع مبلغ پرداخت نشده این معاملات
     * بر اساس تفریق مبلغ قابل پرداخت از مجموع پرداخت های ثبت شده
     */
    public function getOutstandingBalance(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as deal_count,
                SUM(d.payable_amount) as total_contract_amount,
                SUM(COALESCE(tp.total_paid, 0)) as total_collected,
                SUM(d.payable_amount - COALESCE(tp.total_paid, 0)) as outstanding_amount
            FROM deals d
            LEFT JOIN (
                SELECT deal_id, SUM(amount) as total_paid
                FROM transactions
                WHERE status = 'confirmed'
                GROUP BY deal_id
            ) tp ON tp.deal_id = d.id
            WHERE d.payable_amount IS NOT NULL
              AND d.payable_amount > 0
              AND COALESCE(tp.total_paid, 0) < d.payable_amount
              AND {$where}
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * 4. فروش ناموفق
     * تعداد معاملاتی که در وضعیت ناموفق قرار دارند
     * و مجموع مبلغ قابل پرداخت معاملات ناموفق
     * همچنین دلایل ناموفق شدن معاملات باید تعداد هر دلیل مشخص باشه
     * قابل تفکیک باشد
     */
    public function getLostSales(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForLostSales($filters);
        $params = $this->buildParamsForLostSales($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as total_lost,
                d.failure_reason_code,
                COALESCE(
                    NULLIF(TRIM(d.failure_reason_text), ''),
                    NULLIF(TRIM(d.failure_reason_description), ''),
                    CASE TRIM(d.failure_reason_code)
                        WHEN 'no_financial_ability' THEN 'عدم توان مالی'
                        WHEN 'no_purchase_intent' THEN 'فعلا قصد خرید ندارد'
                        WHEN 'competitor_purchase' THEN 'خرید از رقبا'
                        WHEN 'wrong_number' THEN 'شماره اشتباه'
                        WHEN 'irrelevant_lead' THEN 'لید نامرتبط'
                        WHEN 'other' THEN 'سایر'
                        ELSE NULL
                    END,
                    NULLIF(TRIM(d.failure_reason_code), ''),
                    'نامشخص'
                ) as failure_reason_text,
                SUM(d.payable_amount) as total_amount
            FROM deals d
            WHERE d.status = 'Lost'
              AND d.lost_time IS NOT NULL
              AND {$where}
            GROUP BY d.failure_reason_code, failure_reason_text
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * 5. فروش جاری
     * تعداد و مجموع مبلغ قابل پرداخت همه معاملاتی که در وضعیت جاری قرار دارند
     */
    public function getOpenPipeline(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as deal_count,
                SUM(d.payable_amount) as total_potential_amount
            FROM deals d
            WHERE d.status NOT IN ('Won', 'Lost')
              AND {$where}
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * 5.5. معاملات بدون پرداخت
     * تعداد معاملاتی که payable_amount > 0 دارند اما هیچ transaction ندارند
     * این معاملات در Open Pipeline هم نمایش داده می‌شوند اما این بخش آن‌ها را جداگانه نشان می‌دهد
     */
    public function getDealsWithoutPayments(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as deal_count,
                SUM(d.payable_amount) as total_potential_amount
            FROM deals d
            LEFT JOIN (
                SELECT deal_id, SUM(amount) as total_paid
                FROM transactions
                WHERE status = 'confirmed'
                GROUP BY deal_id
            ) tp ON tp.deal_id = d.id
            WHERE d.status NOT IN ('Won', 'Lost')
              AND d.payable_amount IS NOT NULL
              AND d.payable_amount > 0
              AND (tp.total_paid IS NULL OR tp.total_paid = 0)
              AND {$where}
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * تعداد فروش محصولاتی که حداقل یک پرداخت در بازه زمانی دارند
     */
    public function getProductSalesCounts(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $parts = [];
        $params = [];

        if ($this->db->hasColumn('deals', 'is_legacy_didar')) {
            $parts[] = "COALESCE(d.is_legacy_didar, 0) = 0";
        }
        if (!empty($filters['date_from'])) {
            $parts[] = "DATE(COALESCE(t.payment_date, t.payment_time)) >= ?";
            $params[] = $filters['date_from'];
        }
        if (!empty($filters['date_to'])) {
            $parts[] = "DATE(COALESCE(t.payment_date, t.payment_time)) <= ?";
            $params[] = $filters['date_to'];
        }
        if (!empty($filters['owner_didar_id']) && trim($filters['owner_didar_id']) !== '') {
            $parts[] = "d.owner_didar_id = ?";
            $params[] = trim($filters['owner_didar_id']);
        }
        if (!empty($filters['product_id'])) {
            $parts[] = "p.id = ?";
            $params[] = $filters['product_id'];
        }

        $where = !empty($parts) ? implode(' AND ', $parts) : '1=1';

        $sql = "
            SELECT
                p.id AS product_id,
                p.name AS product_name,
                COUNT(DISTINCT d.id) AS deal_count
            FROM transactions t
            INNER JOIN deals d ON d.id = t.deal_id
            INNER JOIN deal_products dp ON dp.deal_id = d.id
            INNER JOIN products p ON p.id = dp.product_id
            INNER JOIN lookup_items i ON COALESCE(i.value, i.title) = p.name
            INNER JOIN lookup_groups g ON g.id = i.group_id AND g.code = 'allowed_services'
            WHERE t.status = 'confirmed'
              AND i.is_active = 1
              AND {$where}
            GROUP BY p.id, p.name
            ORDER BY deal_count DESC, p.name ASC
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * 6. نرخ تبدیل و نرخ عدم تبدیل لید جدید
     * تعداد معامله ثبت لید های جدید (یعنی لیدی که در آن بازه تاریخی برای اولین بار در دیتابیس ثبت شده)
     * برای هر کاربر
     * به همراه نرخ عدم تبدیل برای همین معاملات و لیدها بر اساس معاملاتی که هنوز براشون پرداختی ثبت نشده
     * 
     * SIMPLIFIED VERSION: Uses person.register_time as basis for "new lead" definition
     */
    public function getDealConversionRate(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(DISTINCT d.id) as total_deals,
                COUNT(DISTINCT CASE WHEN COALESCE(tp.total_paid, 0) > 0 THEN d.id END) as converted_deals
            FROM deals d
            LEFT JOIN (
                SELECT deal_id, SUM(amount) as total_paid
                FROM transactions
                WHERE status = 'confirmed'
                GROUP BY deal_id
            ) tp ON tp.deal_id = d.id
            WHERE {$where}
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $row = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        $totalDeals = (int)($row['total_deals'] ?? 0);
        $convertedDeals = (int)($row['converted_deals'] ?? 0);
        $conversionRate = $totalDeals > 0 ? round(($convertedDeals / $totalDeals) * 100, 2) : 0;
        $nonConversionRate = $totalDeals > 0 ? round((($totalDeals - $convertedDeals) / $totalDeals) * 100, 2) : 0;

        return [
            'total_deals' => $totalDeals,
            'converted_deals' => $convertedDeals,
            'conversion_rate' => $conversionRate,
            'non_conversion_rate' => $nonConversionRate
        ];
    }

    /**
     * نرخ تبدیل لید جدید (persons)
     * لید جدید: persons.register_time در بازه
     * تبدیل: حداقل یک معامله با پرداخت confirmed (بدون محدودیت تاریخ پرداخت)
     */
    public function getLeadConversionRate(array $filters): array
    {
        $pdo = $this->db->getPdo();

        $dateFrom = $filters['date_from'] ?? null;
        $dateTo = $filters['date_to'] ?? null;

        $personWhereParts = [];
        $personParams = [];

        if (!empty($dateFrom)) {
            $personWhereParts[] = "DATE(p.register_time) >= ?";
            $personParams[] = $dateFrom;
        }
        if (!empty($dateTo)) {
            $personWhereParts[] = "DATE(p.register_time) <= ?";
            $personParams[] = $dateTo;
        }
        if (!empty($filters['owner_didar_id'])) {
            $personWhereParts[] = "p.owner_didar_id = ?";
            $personParams[] = $filters['owner_didar_id'];
        }

        $personWhere = !empty($personWhereParts) ? implode(' AND ', $personWhereParts) : '1=1';

        $leadsSql = "
            SELECT COUNT(DISTINCT p.didar_contact_id) as total_new_leads
            FROM persons p
            WHERE {$personWhere}
        ";
        $stmt = $pdo->prepare($leadsSql);
        $stmt->execute($personParams);
        $totalNewLeads = (int) $stmt->fetchColumn();

        $convertedWhereParts = $personWhereParts;
        $convertedParams = $personParams;

        if (!empty($filters['owner_didar_id'])) {
            $convertedWhereParts[] = "d.owner_didar_id = ?";
            $convertedParams[] = $filters['owner_didar_id'];
        }
        if (!empty($filters['product_id'])) {
            $convertedWhereParts[] = "EXISTS (SELECT 1 FROM deal_products dp WHERE dp.deal_id = d.id AND dp.product_id = ?)";
            $convertedParams[] = $filters['product_id'];
        }

        $convertedWhere = !empty($convertedWhereParts) ? implode(' AND ', $convertedWhereParts) : '1=1';

        $convertedSql = "
            SELECT COUNT(DISTINCT p.didar_contact_id) as converted_leads
            FROM persons p
            INNER JOIN deals d ON d.contact_didar_id = p.didar_contact_id
            INNER JOIN transactions t ON d.id = t.deal_id
            WHERE t.status = 'confirmed'
              AND {$convertedWhere}
        ";
        $stmt = $pdo->prepare($convertedSql);
        $stmt->execute($convertedParams);
        $convertedLeads = (int) $stmt->fetchColumn();

        $conversionRate = $totalNewLeads > 0 ? round(($convertedLeads / $totalNewLeads) * 100, 2) : 0;
        $nonConversionRate = $totalNewLeads > 0 ? round((($totalNewLeads - $convertedLeads) / $totalNewLeads) * 100, 2) : 0;

        return [
            'total_new_leads' => $totalNewLeads,
            'converted_leads' => $convertedLeads,
            'conversion_rate' => $conversionRate,
            'non_conversion_rate' => $nonConversionRate
        ];
    }

    /**
     * Backward compatibility
     */
    public function getNewLeadConversionRate(array $filters): array
    {
        return $this->getLeadConversionRate($filters);
    }

    /**
     * 7. نرخ زمان تبدیل
     * نرخ مدت زمانی که طول کشیده از زمان ثبت یک معامله تا زمان تبدیل شدن آن معامله
     * یعنی از created_at تا اولین پرداخت
     * 
     * FIXED: Now calculates time from FIRST PAYMENT DATE to FIRST PAYMENT DATE (always 0)
     * This correctly represents that conversion happens at registration time (when first payment is received)
     */
    public function getTimeToConversion(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                COUNT(*) as deal_count,
                AVG(DATEDIFF(first_payment_date, deal_date)) as avg_conversion_days
            FROM (
                SELECT
                    d.id,
                    DATE(COALESCE(d.register_time, d.created_at)) as deal_date,
                    MIN(COALESCE(t.payment_date, DATE(t.payment_time))) as first_payment_date
                FROM deals d
                INNER JOIN transactions t ON d.id = t.deal_id
                WHERE t.status = 'confirmed'
                  AND {$where}
                GROUP BY d.id
            ) s
            WHERE s.first_payment_date IS NOT NULL
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * فروش روزانه در ماه جاری
     */
    public function getDailySales(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDeals($filters, 'd');
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                DATE(COALESCE(t.payment_date, t.payment_time)) as date,
                COUNT(DISTINCT d.id) as deal_count,
                SUM(d.payable_amount) as total_amount,
                SUM(t.amount) as collected_amount
            FROM deals d
            INNER JOIN transactions t ON d.id = t.deal_id
            WHERE t.status = 'confirmed'
              AND {$where}
            GROUP BY DATE(COALESCE(t.payment_date, t.payment_time))
            ORDER BY DATE(COALESCE(t.payment_date, t.payment_time))
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * هدف KPI برای کاربر
     */
    public function getKpiTarget(int $userId, int $year, int $month): ?array
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            SELECT * FROM kpi_targets
            WHERE user_id = ? AND year = ? AND month = ? AND is_active = 1
        ");
        $stmt->execute([$userId, $year, $month]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ?: null;
    }

    /**
     * لیست معاملات برای جدول KPI
     */
    public function getKpiDealDetails(array $filters): array
    {
        $pdo = $this->db->getPdo();
        $where = $this->buildWhereClauseForDealDetails($filters);
        $params = $this->buildParamsForDeals($filters);

        $sql = "
            SELECT
                d.id,
                d.didar_deal_id,
                d.title,
                d.status,
                d.payable_amount,
                d.register_time,
                d.won_time,
                d.lost_time,
                u.display_name as owner_name,
                u.didar_user_id as owner_didar_id,
                (
                    SELECT COALESCE(SUM(t2.amount), 0)
                    FROM transactions t2
                    WHERE t2.deal_id = d.id AND t2.status = 'confirmed'
                ) as collected_amount,
                (
                    SELECT MIN(t3.payment_date)
                    FROM transactions t3
                    WHERE t3.deal_id = d.id AND t3.status = 'confirmed'
                ) as first_payment_date,
                (
                    SELECT COUNT(*)
                    FROM transactions t4
                    WHERE t4.deal_id = d.id AND t4.status = 'confirmed'
                ) as payment_count
            FROM deals d
            LEFT JOIN users u ON d.owner_didar_id = u.didar_user_id
            WHERE {$where}
            ORDER BY d.register_time DESC
            LIMIT 500
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * ساخت where clause بر اساس فیلترها
     */
    private function buildWhereClause(array $filters): string
    {
        return $this->buildWhereClauseForDeals($filters, 'd');
    }

    /**
     * ساخت where clause برای فیلترهای مبتنی بر معامله
     */
    private function buildWhereClauseForDeals(array $filters, string $alias = 'd'): string
    {
        $parts = [];

        if ($this->db->hasColumn('deals', 'is_legacy_didar')) {
            $parts[] = "COALESCE({$alias}.is_legacy_didar, 0) = 0";
        }

        if (!empty($filters['date_from'])) {
            $parts[] = "DATE(COALESCE({$alias}.register_time, {$alias}.created_at)) >= ?";
        }
        if (!empty($filters['date_to'])) {
            $parts[] = "DATE(COALESCE({$alias}.register_time, {$alias}.created_at)) <= ?";
        }

        if (!empty($filters['owner_didar_id']) && trim($filters['owner_didar_id']) !== '') {
            $parts[] = "{$alias}.owner_didar_id = ?";
        }

        if (!empty($filters['product_id'])) {
            $parts[] = "EXISTS (SELECT 1 FROM deal_products dp WHERE dp.deal_id = {$alias}.id AND dp.product_id = ?)";
        }

        return !empty($parts) ? implode(' AND ', $parts) : '1=1';
    }

    /**
     * ساخت where clause برای Sales (بدون محدودیت status معامله)
     */
    private function buildWhereClauseForSales(array $filters): string
    {
        return $this->buildWhereClauseForDeals($filters, 'd');
    }

    /**
     * ساخت where clause برای Lost Sales
     */
    private function buildWhereClauseForLostSales(array $filters): string
    {
        $parts = [];

        if ($this->db->hasColumn('deals', 'is_legacy_didar')) {
            $parts[] = "COALESCE(d.is_legacy_didar, 0) = 0";
        }

        // فیلتر زمانی بر اساس lost_time
        if (!empty($filters['date_from'])) {
            $parts[] = "DATE(d.lost_time) >= ?";
        }
        if (!empty($filters['date_to'])) {
            $parts[] = "DATE(d.lost_time) <= ?";
        }

        // فیلتر کاربر
        if (!empty($filters['owner_didar_id'])) {
            $parts[] = "d.owner_didar_id = ?";
        }

        // فیلتر محصول
        if (!empty($filters['product_id'])) {
            $parts[] = "EXISTS (SELECT 1 FROM deal_products dp WHERE dp.deal_id = d.id AND dp.product_id = ?)";
        }

        return !empty($parts) ? implode(' AND ', $parts) : '1=1';
    }

    /**
     * ساخت where clause بدون transactions (برای Open Pipeline و Persons)
     */
    private function buildWhereClauseWithoutTransactions(array $filters): string
    {
        return $this->buildWhereClauseForDeals($filters, 'd');
    }

    /**
     * ساخت where clause برای Persons
     * NOTE: persons جدول product_id ندارد
     */
    private function buildWhereClauseForPersons(array $filters): string
    {
        $parts = [];

        // فیلتر زمانی
        if (!empty($filters['date_from'])) {
            $parts[] = "DATE(p.register_time) >= ?";
        }
        if (!empty($filters['date_to'])) {
            $parts[] = "DATE(p.register_time) <= ?";
        }

        // فیلتر کاربر
        if (!empty($filters['owner_didar_id'])) {
            $parts[] = "p.owner_didar_id = ?";
        }

        // نکته: product_id را برای persons نادیده می‌گیریم چون persons جدول product_id ندارد

        return !empty($parts) ? implode(' AND ', $parts) : '1=1';
    }

    /**
     * ساخت where clause برای جزئیات معاملات KPI
     * فیلتر تاریخ بر اساس اولین پرداخت (t5)
     */
    private function buildWhereClauseForDealDetails(array $filters): string
    {
        $parts = [];

        if ($this->db->hasColumn('deals', 'is_legacy_didar')) {
            $parts[] = "COALESCE(d.is_legacy_didar, 0) = 0";
        }

        if (!empty($filters['date_from'])) {
            $parts[] = "DATE(COALESCE(d.register_time, d.created_at)) >= ?";
        }
        if (!empty($filters['date_to'])) {
            $parts[] = "DATE(COALESCE(d.register_time, d.created_at)) <= ?";
        }
        if (!empty($filters['owner_didar_id'])) {
            $parts[] = "d.owner_didar_id = ?";
        }
        if (!empty($filters['product_id'])) {
            $parts[] = "EXISTS (SELECT 1 FROM deal_products dp WHERE dp.deal_id = d.id AND dp.product_id = ?)";
        }

        return !empty($parts) ? implode(' AND ', $parts) : '1=1';
    }

    private function buildWhereClauseForDailySales(array $filters): string
    {
        return $this->buildWhereClauseForDeals($filters, 'd');
    }

    /**
     * ساخت پارامترها برای کوئری
     */
    private function buildParams(array $filters): array
    {
        return $this->buildParamsForDeals($filters);
    }

    /**
     * ساخت پارامترها برای Sales
     */
    private function buildParamsForSales(array $filters): array
    {
        return $this->buildParamsForDeals($filters);
    }

    /**
     * ساخت پارامترها برای Lost Sales
     */
    private function buildParamsForLostSales(array $filters): array
    {
        $params = [];

        if (!empty($filters['date_from'])) {
            $params[] = $filters['date_from'];
        }
        if (!empty($filters['date_to'])) {
            $params[] = $filters['date_to'];
        }
        if (!empty($filters['owner_didar_id'])) {
            $params[] = $filters['owner_didar_id'];
        }
        if (!empty($filters['product_id'])) {
            $params[] = $filters['product_id'];
        }

        return $params;
    }

    /**
     * ساخت پارامترها بدون transactions
     */
    private function buildParamsWithoutTransactions(array $filters): array
    {
        return $this->buildParamsForDeals($filters);
    }

    /**
     * ساخت پارامترها برای فیلترهای مبتنی بر معامله
     */
    private function buildParamsForDeals(array $filters): array
    {
        $params = [];

        if (!empty($filters['date_from'])) {
            $params[] = $filters['date_from'];
        }
        if (!empty($filters['date_to'])) {
            $params[] = $filters['date_to'];
        }
        if (!empty($filters['owner_didar_id']) && trim($filters['owner_didar_id']) !== '') {
            $params[] = trim($filters['owner_didar_id']);
        }
        if (!empty($filters['product_id'])) {
            $params[] = $filters['product_id'];
        }

        return $params;
    }

    /**
     * ساخت پارامترها برای Persons
     */
    private function buildParamsForPersons(array $filters): array
    {
        $params = [];

        if (!empty($filters['date_from'])) {
            $params[] = $filters['date_from'];
        }
        if (!empty($filters['date_to'])) {
            $params[] = $filters['date_to'];
        }
        if (!empty($filters['owner_didar_id'])) {
            $params[] = $filters['owner_didar_id'];
        }

        return $params;
    }
}
