<?php
namespace App\Repositories;

use App\Database\Connection;
use App\Models\Notification;
use App\Utils\Logger;
use PDO;
use PDOException;

class NotificationRepository
{
    private $db;

    public function __construct(Connection $db)
    {
        $this->db = $db;
    }

    public function create(array $data)
    {
        $pdo = $this->db->getPdo();
        $sql = "INSERT INTO notifications
                (user_didar_id, type, title, message, entity_type, entity_id, show_popup, is_read, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, 0, NOW())";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            $data['user_didar_id'],
            $data['type'],
            $data['title'],
            $data['message'] ?? null,
            $data['entity_type'] ?? null,
            $data['entity_id'] ?? null,
            isset($data['show_popup']) ? ($data['show_popup'] ? 1 : 0) : 1
        ]);

        return $pdo->lastInsertId();
    }

    public function findUnreadByUser($userId, $limit = 50)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            SELECT * FROM notifications 
            WHERE user_didar_id = ? AND is_read = 0 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->execute([$userId, $limit]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map(function($data) {
            return new Notification($data);
        }, $results);
    }

    public function findAllByUser($userId, $limit = 50, $offset = 0)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            SELECT * FROM notifications 
            WHERE user_didar_id = ? 
            ORDER BY created_at DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$userId, $limit, $offset]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map(function($data) {
            return new Notification($data);
        }, $results);
    }

    public function countUnread($userId)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM notifications 
            WHERE user_didar_id = ? AND is_read = 0
        ");
        $stmt->execute([$userId]);
        return (int)$stmt->fetchColumn();
    }

    public function markAsRead($id, $userId)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            UPDATE notifications 
            SET is_read = 1, read_at = NOW() 
            WHERE id = ? AND user_didar_id = ?
        ");
        return $stmt->execute([$id, $userId]);
    }

    public function markAllAsRead($userId)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            UPDATE notifications 
            SET is_read = 1, read_at = NOW() 
            WHERE user_didar_id = ? AND is_read = 0
        ");
        return $stmt->execute([$userId]);
    }

    public function delete($id, $userId)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            DELETE FROM notifications 
            WHERE id = ? AND user_didar_id = ?
        ");
        return $stmt->execute([$id, $userId]);
    }

    public function findById($id)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("SELECT * FROM notifications WHERE id = ? LIMIT 1");
        $stmt->execute([$id]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        return $data ? new Notification($data) : null;
    }
}

