<?php
namespace App\Repositories;

use App\Database\Connection;
use App\Models\Product;
use PDO;

class ProductRepository
{
    private $db;

    public function __construct(Connection $db)
    {
        $this->db = $db;
    }

    public function all()
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->query("SELECT * FROM products ORDER BY name ASC");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map(fn($r) => new Product($r), $rows);
    }

    public function allUsedInDeals()
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->query("
            SELECT DISTINCT p.*
            FROM products p
            INNER JOIN deal_products dp ON dp.product_id = p.id
            ORDER BY p.name ASC
        ");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map(fn($r) => new Product($r), $rows);
    }

    public function allAllowedServices()
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->query("
            SELECT DISTINCT p.*
            FROM products p
            INNER JOIN lookup_items i ON COALESCE(i.value, i.title) = p.name
            INNER JOIN lookup_groups g ON g.id = i.group_id AND g.code = 'allowed_services'
            WHERE i.is_active = 1
            ORDER BY p.name ASC
        ");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return array_map(fn($r) => new Product($r), $rows);
    }

    public function getIdsByNames(array $names): array
    {
        $names = array_values(array_filter(array_map(function($name) {
            return trim((string)$name);
        }, $names)));
        if (empty($names)) {
            return [];
        }

        $placeholders = implode(',', array_fill(0, count($names), '?'));
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("SELECT id FROM products WHERE name IN ($placeholders)");
        $stmt->execute($names);
        return array_map('intval', $stmt->fetchAll(PDO::FETCH_COLUMN));
    }

    public function create(string $name)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("INSERT INTO products (name) VALUES (?)");
        $stmt->execute([$name]);
        return $pdo->lastInsertId();
    }

    public function delete(int $id)
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("DELETE FROM products WHERE id = ?");
        return $stmt->execute([$id]);
    }

    public function getNamesByIds(array $ids): array
    {
        $ids = array_values(array_filter(array_map('intval', $ids)));
        if (empty($ids)) {
            return [];
        }
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("SELECT name FROM products WHERE id IN ($placeholders)");
        $stmt->execute($ids);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }

    public function matchIdsByText(array $texts): array
    {
        $texts = array_values(array_filter(array_map(function($text) {
            return trim((string)$text);
        }, $texts)));
        if (empty($texts)) {
            return [];
        }

        $pdo = $this->db->getPdo();
        $stmt = $pdo->query("SELECT id, name FROM products WHERE name IS NOT NULL AND name <> ''");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $normalize = function($value) {
            $value = trim((string)$value);
            $value = preg_replace('/[|،,]+/', ' ', $value);
            $value = preg_replace('/\s+/', ' ', $value);
            if (function_exists('mb_strtolower')) {
                return mb_strtolower($value, 'UTF-8');
            }
            return strtolower($value);
        };

        $normalizedTexts = array_map($normalize, $texts);
        $matches = [];
        foreach ($rows as $row) {
            $name = trim((string)($row['name'] ?? ''));
            if ($name === '') {
                continue;
            }
            $needle = $normalize($name);
            if ($needle === '') {
                continue;
            }
            foreach ($normalizedTexts as $text) {
                if ($text === '') {
                    continue;
                }
                $found = function_exists('mb_stripos')
                    ? (mb_stripos($text, $needle, 0, 'UTF-8') !== false)
                    : (stripos($text, $needle) !== false);
                if ($found) {
                    $matches[] = (int)($row['id'] ?? 0);
                    break;
                }
            }
        }

        $matches = array_values(array_filter(array_unique($matches)));
        return $matches;
    }
}
