<?php
namespace App\Services;

use App\Repositories\ActivityRepository;
use App\Repositories\PersonRepository;
use App\Repositories\ConfigRepository;
use App\Services\DidarApiService;
use App\Services\VirtualStageService;
use App\Services\NotificationService;
use App\Utils\Sanitizer;
use App\Utils\Logger;

/**
 * Activity Service
 */
class ActivityService
{
    private $activityRepo;
    private $personRepo;
    private $didarApi;
    private $virtualStageService;
    private $notificationService;
    private ConfigRepository $configRepo;

    public function __construct(
        ActivityRepository $activityRepo,
        PersonRepository $personRepo,
        DidarApiService $didarApi,
        ConfigRepository $configRepo,
        VirtualStageService $virtualStageService,
        NotificationService $notificationService = null
    ) {
        $this->activityRepo = $activityRepo;
        $this->personRepo = $personRepo;
        $this->didarApi = $didarApi;
        $this->configRepo = $configRepo;
        $this->virtualStageService = $virtualStageService;
        $this->notificationService = $notificationService;
    }

    public function getActivitiesByContact($contactId, $limit = 50)
    {
        return $this->activityRepo->findByContactId($contactId, $limit);
    }

    public function getActivitiesByDeal($dealId, $limit = 100)
    {
        return $this->activityRepo->findByDealId($dealId, $limit);
    }

    private function isLocalContactId(?string $contactId): bool
    {
        return !empty($contactId) && strpos($contactId, 'local_') === 0;
    }

    private function isDidarWriteEnabled(): bool
    {
        return false;
    }

    /**
     * Returns true if there is at least one local activity for the given deal ID.
     */
    public function hasActivityForDeal(string $dealId): bool
    {
        return $this->activityRepo->hasActivityForDeal($dealId);
    }

    public function saveActivity($data, $virtualStage = null, $currentStage = null, $pipelineStage = null)
    {
        $didarId = $_SESSION['didar_id'] ?? '';
        $contactId = $data['contact_didar_id'] ?? '';
        $isLocalLead = $this->isLocalContactId($contactId);
        $activityTypeId = $data['activity_type_id'] ?? '';
        $note = $data['note'] ?? '';
	        $resultNote = $data['result_note'] ?? '';
	        $failureReasonCode = $data['failure_reason'] ?? null;
	        $failureReasonDesc = $data['failure_other_description'] ?? null;
        $dealId = $data['deal_didar_id'] ?? '';
        $requestedServices = $data['requested_services'] ?? '';
        $city = $data['city'] ?? null;
        $jobTitle = $data['job_title'] ?? null;
        $jobDescription = $data['job_description'] ?? null;
        $acquaintanceDuration = $data['acquaintance_duration'] ?? null;
        $extraInfo = $data['extra_info'] ?? null;
        
        // Check if activity is scheduled for future
        $dueDate = $data['due_date'] ?? null;
        $isScheduled = isset($data['is_scheduled']) && $data['is_scheduled'] == '1';
        $reminderEnabled = isset($data['reminder_enabled']) && $data['reminder_enabled'] == '1';
        $reminderOffset = isset($data['reminder_offset_minutes']) ? (int)$data['reminder_offset_minutes'] : 1440; // Default: 24 hours
        
        // If not explicitly set, check if due date is in the future
        if (!$isScheduled && !empty($dueDate)) {
            $dueDateTime = strtotime($dueDate);
            $now = time();
            if ($dueDateTime > $now) {
                $isScheduled = true;
            }
        }
        
        // For scheduled activities, is_done must be 0
        $isDone = $isScheduled ? 0 : ($data['is_done'] ?? 0);
        
        Logger::logInfo("Activity is_done determination", [
            'is_scheduled' => $isScheduled,
            'is_done_from_data' => $data['is_done'] ?? 0,
            'final_is_done' => $isDone
        ]);
        
        // targetStage no longer used for leads; pipelineStage controls deal stage
        
        // Prepare activity data for Didar API
        // Format due date for Didar API (ISO 8601 with Z)
        $formattedDueDate = null;
        if (!empty($dueDate)) {
            try {
                // Parse the date string - format: YYYY-MM-DD HH:mm:ss
                // Try different formats
                $dateTime = null;
                
                // Try YYYY-MM-DD HH:mm:ss format
                if (preg_match('/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2}):(\d{2})$/', $dueDate, $matches)) {
                    $dateTime = \DateTime::createFromFormat(
                        'Y-m-d H:i:s',
                        $dueDate,
                        new \DateTimeZone('Asia/Tehran')
                    );
                }
                // Try YYYY-MM-DDTHH:mm:ss format
                elseif (preg_match('/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})/', $dueDate, $matches)) {
                    $dateTime = \DateTime::createFromFormat(
                        'Y-m-d\TH:i:s',
                        $dueDate,
                        new \DateTimeZone('Asia/Tehran')
                    );
                }
                // Try YYYY-MM-DD HH:mm format
                elseif (preg_match('/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2})$/', $dueDate, $matches)) {
                    $dateTime = \DateTime::createFromFormat(
                        'Y-m-d H:i',
                        $dueDate,
                        new \DateTimeZone('Asia/Tehran')
                    );
                }
                // Fallback: use strtotime
                else {
                    $timestamp = strtotime($dueDate);
                    if ($timestamp !== false) {
                        $dateTime = new \DateTime('@' . $timestamp);
                        $dateTime->setTimezone(new \DateTimeZone('Asia/Tehran'));
                    }
                }
                
                if (!$dateTime) {
                    throw new \Exception("Could not parse date: " . $dueDate);
                }
                
                // Convert to UTC for Didar API (Z means UTC)
                $dateTime->setTimezone(new \DateTimeZone('UTC'));
                $formattedDueDate = $dateTime->format('Y-m-d\TH:i:s.000\Z');
                
                Logger::logInfo("Formatted due date for Didar", [
                    'original' => $dueDate,
                    'formatted' => $formattedDueDate
                ]);
            } catch (\Exception $e) {
                Logger::logError("Error formatting due date", $e, ['due_date' => $dueDate]);
                // Final fallback: try strtotime
                try {
                    $timestamp = strtotime($dueDate);
                    if ($timestamp !== false) {
                        $formattedDueDate = gmdate('Y-m-d\TH:i:s.000\Z', $timestamp);
                    } else {
                        throw new \Exception("Could not parse date: " . $dueDate);
                    }
                } catch (\Exception $e2) {
                    Logger::logError("Fallback date formatting also failed", $e2);
                    throw new \Exception("خطا در فرمت تاریخ: " . $dueDate . " - " . $e2->getMessage());
                }
            }
        }
        
        // Get the correct Didar User ID (OwnerId)
        // OwnerId must be the Didar UserId of the current user, not the API key owner
        // $_SESSION['didar_id'] is set from $user->didar_user_id during login (AuthService.php:53)
        // But we need to use the synced didar_user_id from database to ensure it's correct
        $didarUserId = $didarId;
        
        // Get user from database to use the synced didar_user_id
        // Try to find by didar_id first, then by email if not found
        $userRepo = new \App\Repositories\UserRepository($this->personRepo->getDb());
        $currentUser = $userRepo->findByDidarId($didarId);
        
        // If not found by didar_id, try to find by email from session
        if (!$currentUser && !empty($_SESSION['email'])) {
            $currentUser = $userRepo->findByEmail($_SESSION['email']);
        }
        
        if ($currentUser && $currentUser->didar_user_id) {
            // Use the synced didar_user_id from database (this is the correct one from Didar)
            $didarUserId = $currentUser->didar_user_id;
            Logger::logInfo("Using synced didar_user_id from database as OwnerId", [
                'session_didar_id' => $didarId,
                'database_didar_user_id' => $didarUserId,
                'user_email' => $currentUser->email ?? 'unknown',
                'user_name' => ($currentUser->first_name ?? '') . ' ' . ($currentUser->last_name ?? ''),
                'ids_match' => ($didarId === $didarUserId) ? 'yes' : 'no',
                'found_by' => $currentUser->didar_user_id === $didarId ? 'didar_id' : 'email'
            ]);
        } else {
            // User not found or didar_user_id not set - use session didar_id as fallback
            Logger::logWarning("User not found or didar_user_id not set in database. Using session didar_id as OwnerId. Please sync users from Didar first!", [
                'didar_id' => $didarId,
                'session_email' => $_SESSION['email'] ?? 'not_set',
                'user_found' => $currentUser ? 'yes' : 'no',
                'has_didar_user_id' => ($currentUser && $currentUser->didar_user_id) ? 'yes' : 'no'
            ]);
        }
        
        // Get contact owner name to add to Note
        $leadOwnerName = '';
        try {
            $contact = $this->personRepo->findByDidarId($contactId);
            if ($contact && $contact->owner_didar_id) {
                $ownerUser = $userRepo->findByDidarId($contact->owner_didar_id);
                if ($ownerUser) {
                    $leadOwnerName = trim(($ownerUser->first_name ?? '') . ' ' . ($ownerUser->last_name ?? ''));
                    if (empty($leadOwnerName) && isset($ownerUser->display_name)) {
                        $leadOwnerName = $ownerUser->display_name;
                    }
                }
            }
        } catch (\Exception $e) {
            Logger::logWarning("Could not get lead owner name for activity note", [
                'contact_id' => $contactId,
                'error' => $e->getMessage()
            ]);
        }
        
        // Add lead owner name to Note if available
        $finalNote = $note;
        if (!empty($leadOwnerName)) {
            $ownerInfo = "📌 صاحب لید: " . $leadOwnerName;
            if (!empty($note)) {
                $finalNote = $note . "\n\n" . $ownerInfo;
            } else {
                $finalNote = $ownerInfo;
            }
            Logger::logInfo("Added lead owner name to activity note", [
                'lead_owner_name' => $leadOwnerName,
                'contact_id' => $contactId,
                'note_was_empty' => empty($note)
            ]);
        }
        
        $activityData = [
            'Activity' => [
                'ActivityTypeId' => $activityTypeId,
                'Title' => 'فعالیت',
                'OwnerId' => $didarUserId,
                'CreatorId' => $didarUserId, // شناسه شخص ایجاد کننده فعالیت
                'Note' => $finalNote,
                'ResultNote' => $resultNote,
                'IsDone' => $isDone,
                'ContactIds' => [$contactId],
                'DueDate' => $formattedDueDate ?: date('Y-m-d\TH:i:s.000\Z'),
                'DueDateType' => !empty($formattedDueDate) ? 'Default' : 'NoTime',
                'DoneDateType' => 'Notime'
            ],
            'SetDone' => $isDone
        ];
        
        if (!empty($dealId)) {
            $activityData['Activity']['DealIds'] = [$dealId];
        }

        // Persist requested services on deal locally (no Didar API support)
        if (!empty($dealId) && !empty($requestedServices)) {
            try {
                $dealRepo = new \App\Repositories\DealRepository($this->personRepo->getDb());
                $servicesValue = $requestedServices;
                if (is_array($servicesValue)) {
                    $servicesValue = json_encode(array_values(array_filter($servicesValue)), JSON_UNESCAPED_UNICODE);
                }
                $dealRepo->updateRequestedServices($dealId, $servicesValue);
                // ذخیره در فیلدهای اضافی معامله برای دسترسی بعدی
                $dealRepo->updateExtraFields($dealId, ['requested_services' => $servicesValue]);
            } catch (\Exception $e) {
                Logger::logWarning("Failed to update requested services for deal", [
                    'deal_id' => $dealId,
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Update person/deal extra fields (city, job, etc.)
        if (!empty($contactId)) {
            $pdo = $this->personRepo->getDb()->getPdo();
            $fields = [];
            $params = [];
            $map = [
                'city' => $city,
                'job_title' => $jobTitle,
                'job_description' => $jobDescription,
                'acquaintance_duration' => $acquaintanceDuration,
                'extra_info' => $extraInfo,
                'financial_level' => $data['financial_level'] ?? null,
                'asset_estimation' => $data['asset_estimation'] ?? null,
                'income_estimation' => $data['income_estimation'] ?? null,
                'requested_services' => !empty($requestedServices) ? (is_array($requestedServices) ? json_encode(array_values(array_filter($requestedServices)), JSON_UNESCAPED_UNICODE) : $requestedServices) : null,
                'has_previous_purchase' => ($data['has_previous_purchase'] ?? $data['has_previous_purchase_other_number'] ?? '') !== '' ? (int)($data['has_previous_purchase'] ?? $data['has_previous_purchase_other_number']) : null
            ];
            foreach ($map as $col => $val) {
                if ($val !== null) {
                    $fields[] = "$col = ?";
                    $params[] = $val;
                }
            }
            if (!empty($fields)) {
                $params[] = $contactId;
                try {
                    $stmt = $pdo->prepare("UPDATE persons SET " . implode(', ', $fields) . " WHERE didar_contact_id = ?");
                    $stmt->execute($params);
                } catch (\Exception $e) {
                    Logger::logWarning("Failed to update person extra fields", ['contact_id' => $contactId, 'error' => $e->getMessage()]);
                }
            }
        }

        if (!empty($dealId)) {
            try {
                $dealRepo = isset($dealRepo) ? $dealRepo : new \App\Repositories\DealRepository($this->personRepo->getDb());
                $dealRepo->updateExtraFields($dealId, [
                    'city' => $city,
                    'job_title' => $jobTitle,
                    'job_description' => $jobDescription,
                    'acquaintance_duration' => $acquaintanceDuration,
                    'extra_info' => $extraInfo,
                    'financial_level' => $data['financial_level'] ?? null,
                    'asset_estimation' => $data['asset_estimation'] ?? null,
                    'income_estimation' => $data['income_estimation'] ?? null,
                    'has_previous_purchase' => ($data['has_previous_purchase'] ?? $data['has_previous_purchase_other_number'] ?? '') !== '' ? (int)($data['has_previous_purchase'] ?? $data['has_previous_purchase_other_number']) : null
                ]);
            } catch (\Exception $e) {
                Logger::logWarning("Failed to update deal extra fields", ['deal_id' => $dealId, 'error' => $e->getMessage()]);
            }
        }
        
        // Add Notifies if reminder is enabled and activity is scheduled
        if ($isScheduled && $reminderEnabled) {
            $notifies = [];
            $notifyOffset = $reminderOffset; // minutes before due date
            $notifies[] = [
                'Id' => '00000000-0000-0000-0000-000000000000',
                'ActivityId' => '00000000-0000-0000-0000-000000000000',
                'Type' => 'Notification',
                'Group' => 'Owner',
                'Data' => null,
                'NotifyOffset' => $notifyOffset,
                'SentDate' => null
            ];
            $activityData['Activity']['Notifies'] = $notifies;
        }
        
        $shouldSyncToDidar = $this->isDidarWriteEnabled() && !$isLocalLead;

        if (!$shouldSyncToDidar) {
            if (!$isLocalLead) {
                Logger::logInfo("Didar write disabled; saving activity locally", [
                    'contact_id' => $contactId,
                    'owner_id' => $didarUserId
                ]);
            }
            $activityId = uniqid('local_activity_', true);
            $activityTypeTitle = $data['activity_type_title'] ?? 'فعالیت';
            $res = [
                'Response' => [
                    'Id' => $activityId,
                    'ActivityTypeTitle' => $activityTypeTitle,
                    'Duration' => 0,
                    'LocalOnly' => true
                ]
            ];
        } else {
            // Call Didar API
            Logger::logInfo("Calling Didar API to save activity", [
                'contact_id' => $contactId,
                'owner_id' => $didarUserId,
                'creator_id' => $didarUserId,
                'activity_data' => [
                    'ActivityTypeId' => $activityTypeId,
                    'OwnerId' => $didarUserId,
                    'CreatorId' => $didarUserId,
                    'IsDone' => $isDone,
                    'IsScheduled' => $isScheduled
                ]
            ]);
            $res = $this->didarApi->call('/activity/save', 'POST', $activityData);
        }
        
        // Log the response to verify OwnerId and CreatorId were accepted
        if (isset($res['Response']['OwnerId']) || isset($res['Response']['Owner']) || isset($res['Response']['CreatorId']) || isset($res['Response']['Creator'])) {
            Logger::logInfo("Activity saved with OwnerId and CreatorId from Didar response", [
                'requested_owner_id' => $didarUserId,
                'requested_creator_id' => $didarUserId,
                'response_owner_id' => $res['Response']['OwnerId'] ?? 'not_set',
                'response_creator_id' => $res['Response']['CreatorId'] ?? 'not_set',
                'owner_name' => $res['Response']['Owner']['DisplayName'] ?? ($res['Response']['OwnerName'] ?? 'unknown'),
                'creator_name' => $res['Response']['Creator']['DisplayName'] ?? ($res['Response']['CreatorName'] ?? 'unknown'),
                'owner_match' => (isset($res['Response']['OwnerId']) && $didarUserId === $res['Response']['OwnerId']) ? 'yes' : 'no',
                'creator_match' => (isset($res['Response']['CreatorId']) && $didarUserId === $res['Response']['CreatorId']) ? 'yes' : 'no'
            ]);
        }
        
        if ($shouldSyncToDidar && (isset($res['error']) || !isset($res['Response']['Id']))) {
            $errorMsg = $res['error'] ?? ($res['message'] ?? 'Unknown error');
            
            // Check for CURL errors (network issues)
            if (isset($res['curl_error']) || (isset($res['curl_errno']) && $res['curl_errno'] > 0)) {
                $curlError = $res['curl_error'] ?? 'خطا در اتصال به سرور دیدار';
                Logger::logError("Network error connecting to Didar API", null, [
                    'error' => $curlError,
                    'curl_errno' => $res['curl_errno'] ?? null,
                    'response' => $res
                ]);
                throw new \Exception("خطا در اتصال به سرور دیدار. لطفاً اتصال اینترنت و DNS سرور را بررسی کنید: " . $curlError);
            }
            
            Logger::logError("Failed to save activity to Didar", null, [
                'error' => $errorMsg,
                'response' => $res
            ]);
            throw new \Exception("خطا در ثبت فعالیت در دیدار: " . $errorMsg);
        }
        
        $activityId = $res['Response']['Id'];
        
        // Update deal pipeline stage / status if provided
        if (!empty($pipelineStage) && !empty($dealId)) {
            try {
                $dealRepo = new \App\Repositories\DealRepository($this->personRepo->getDb());
                $personRepo = $this->personRepo;
                $deal = $dealRepo->findByDidarId($dealId);
                $statusChange = null;
                $stageNormalized = $pipelineStage;
                $isSuccessStage = in_array($pipelineStage, ['deal_success', 'اتمام', 'موفق', 'معامله موفق', 'پرداخت']);
                $isFailedStage = in_array($pipelineStage, ['deal_failed', 'معامله ناموفق', 'ناموفق']);

                if ($isSuccessStage) {
                    // Ensure payments exist for success
                    $payments = [];
                    if ($deal && !empty($deal->payments)) {
                        $payments = json_decode($deal->payments, true) ?: [];
                    }
                    $hasValidPayment = false;
                    foreach ($payments as $p) {
                        $amount = isset($p['amount']) ? floatval($p['amount']) : 0;
                        $date = trim((string)($p['date'] ?? ''));
                        $datePersian = trim((string)($p['date_persian'] ?? ''));
                        $method = trim((string)($p['method'] ?? ''));
                        if ($amount > 0 && ($date !== '' || $datePersian !== '') && $method !== '') {
                            $hasValidPayment = true;
                            break;
                        }
                    }
                    if (!$hasValidPayment) {
                        Logger::logWarning("Pipeline stage change to success blocked: no payments", [
                            'deal_id' => $dealId,
                            'contact_id' => $contactId
                        ]);
                        throw new \Exception('برای ثبت معامله موفق، پرداخت کامل ثبت نشده است');
                    }
                    $statusChange = 'Won';
                    $stageNormalized = 'اتمام';
                } elseif ($isFailedStage) {
                    $statusChange = 'Lost';
                    $stageNormalized = 'معامله ناموفق';
                }

                if ($statusChange) {
                    $dealRepo->updateStatus($dealId, $statusChange, $statusChange === 'Won' ? 'won_time' : 'lost_time');

                    if ($statusChange === 'Lost' && (!empty($failureReasonCode) || !empty($failureReasonDesc))) {
                        $reasonMap = [
                            'no_financial_ability' => 'عدم توان مالی',
                            'no_purchase_intent' => 'فعلا قصد خرید ندارد',
                            'competitor_purchase' => 'خرید از رقبا',
                            'wrong_number' => 'شماره اشتباه',
                            'irrelevant_lead' => 'لید نامرتبط',
                            'other' => 'سایر',
                        ];
                        $code = !empty($failureReasonCode) ? trim((string)$failureReasonCode) : null;
                        $desc = !empty($failureReasonDesc) ? trim((string)$failureReasonDesc) : null;
                        $text = null;

                        if ($code) {
                            if (isset($reasonMap[$code])) {
                                $text = $reasonMap[$code];
                            } else {
                                // Older clients might send Persian text in failure_reason
                                $text = $code;
                                $code = null;
                            }
                        }

                        $dealRepo->updateFailureReason($dealId, $code, $text, $desc);
                    } elseif ($statusChange === 'Won') {
                        $dealRepo->updateFailureReason($dealId, null, null, null);
                    }
                }

                $dealRepo->updatePipelineStage($dealId, $stageNormalized);

        // Sync lead sale status unless already successful
        if (!empty($contactId)) {
            // Update virtual stage immediately after activity save to ensure consistency
            try {
                $this->virtualStageService->update($contactId);
                Logger::logInfo("Virtual stage updated after activity save", ['contact_id' => $contactId]);
            } catch (\Exception $e) {
                Logger::logWarning("Failed to update virtual stage after activity save", [
                    'contact_id' => $contactId,
                    'error' => $e->getMessage()
                ]);
            }

            if ($statusChange) {
                $person = $personRepo->findByDidarId($contactId);
                $currentSaleStatus = $person ? (int)$person->sale_status : null;
                $saleStatus = $statusChange === 'Won' ? 1 : 0;
                $shouldUpdate = true;
                if ($saleStatus === 0 && $currentSaleStatus === 1) {
                    $shouldUpdate = false;
                }
                if ($shouldUpdate) {
                    $personRepo->updateSaleStatus($contactId, $saleStatus);
                }
            }
        }

                Logger::logInfo("Pipeline stage updated after activity save", [
                    'deal_id' => $dealId,
                    'stage' => $stageNormalized,
                    'status_change' => $statusChange
                ]);
            } catch (\Exception $e) {
                Logger::logError("Error updating deal pipeline stage after activity", $e, [
                    'deal_id' => $dealId,
                    'stage' => $pipelineStage
                ]);
                throw $e;
            }
        }
        
        // Save to local DB
        $savedDueDate = !empty($dueDate) ? date('Y-m-d H:i:s', strtotime($dueDate)) : date('Y-m-d H:i:s');
        
	        $activity = new \App\Models\Activity([
	            'didar_activity_id' => $activityId,
	            'activity_type_id' => $activityTypeId,
	            'activity_type_title' => $res['Response']['ActivityTypeTitle'] ?? 'فعالیت',
	            'owner_didar_id' => $didarId,
	            'deal_didar_id' => $dealId ?: null,
	            'contact_didar_id' => $contactId,
	            'title' => 'فعالیت',
	            'note' => $note,
	            'result_note' => $resultNote,
	            'failure_reason_code' => $failureReasonCode ?: null,
	            'failure_reason_description' => $failureReasonDesc ?: null,
	            'is_done' => $isDone,
	            'due_date' => $savedDueDate,
	            'done_date' => $isDone ? date('Y-m-d H:i:s') : null,
	            'duration' => $res['Response']['Duration'] ?? 0,
            'direction' => $data['direction'] ?? 'outgoing',
            'stage' => $pipelineStage ?? '',
            'is_scheduled' => $isScheduled ? 1 : 0,
            'reminder_enabled' => $reminderEnabled ? 1 : 0,
            'register_date' => date('Y-m-d H:i:s'),
            'last_sync' => date('Y-m-d H:i:s'),
            'raw_json' => json_encode($res['Response'], JSON_UNESCAPED_UNICODE)
        ]);
        
        $localId = $this->activityRepo->save($activity);
        
        // Schedule reminder if enabled
        if ($isScheduled && $reminderEnabled && $this->notificationService) {
            Logger::logInfo("Attempting to schedule activity reminder", [
                'activity_id' => $activityId,
                'is_scheduled' => $isScheduled,
                'reminder_enabled' => $reminderEnabled,
                'saved_due_date' => $savedDueDate,
                'reminder_offset' => $reminderOffset,
                'notification_service_available' => !is_null($this->notificationService)
            ]);
            $reminderResult = $this->notificationService->scheduleActivityReminder($activityId, $savedDueDate, $reminderOffset);
            Logger::logInfo("Reminder scheduling result", [
                'activity_id' => $activityId,
                'result' => $reminderResult
            ]);
        } else {
            Logger::logWarning("Reminder not scheduled", [
                'activity_id' => $activityId,
                'is_scheduled' => $isScheduled,
                'reminder_enabled' => $reminderEnabled,
                'notification_service_available' => !is_null($this->notificationService)
            ]);
        }
        
        // Notify lead owner if activity is for someone else's lead
        if ($this->notificationService && !empty($contactId)) {
            $person = $this->personRepo->findByDidarId($contactId);
            if ($person && $person->owner_didar_id && $person->owner_didar_id !== $didarId) {
                $contactName = trim(($person->first_name ?? '') . ' ' . ($person->last_name ?? ''));
                $this->notificationService->notifyNewActivity(
                    $person->owner_didar_id,
                    $activityId,
                    $res['Response']['ActivityTypeTitle'] ?? 'فعالیت',
                    $contactName ?: 'لید'
                );
            }
        }
        
        Logger::logInfo("Activity saved successfully", ['activity_id' => $activityId]);
        
        // CRITICAL: Trigger virtual stage update after activity save to ensure leads list is accurate
        try {
            if (!empty($contactId)) {
                $this->virtualStageService->update($contactId);
                Logger::logInfo("Virtual stage updated after activity save", ['contact_id' => $contactId]);
            }
        } catch (\Exception $e) {
            Logger::logWarning("Failed to update virtual stage after activity save", [
                'contact_id' => $contactId,
                'error' => $e->getMessage()
            ]);
        }
        
        return $activityId;
    }

    public function updateScheduledActivity($activityId, $data)
    {
        $didarId = $_SESSION['didar_id'] ?? '';
        $activityTypeId = $data['activity_type_id'] ?? '';
        $dueDate = $data['due_date'] ?? null;
        $reminderEnabled = isset($data['reminder_enabled']) && $data['reminder_enabled'] == '1';
        $reminderOffset = isset($data['reminder_offset_minutes']) ? (int)$data['reminder_offset_minutes'] : 1440;
        
        if (empty($activityTypeId)) {
            throw new \Exception('نوع فعالیت الزامی است');
        }
        
        // Get existing activity
        $existingActivity = $this->activityRepo->findByDidarId($activityId);
        if (!$existingActivity) {
            throw new \Exception('فعالیت یافت نشد');
        }

        $isLocalLead = $this->isLocalContactId($existingActivity->contact_didar_id ?? '');
        $isLocalActivity = is_string($activityId) && strpos($activityId, 'local_activity_') === 0;
        if ($isLocalLead || $isLocalActivity) {
            $savedDueDate = !empty($dueDate) ? date('Y-m-d H:i:s', strtotime($dueDate)) : ($existingActivity->due_date ?? date('Y-m-d H:i:s'));
            $activity = new \App\Models\Activity([
                'didar_activity_id' => $activityId,
                'activity_type_id' => $activityTypeId,
                'activity_type_title' => $existingActivity->activity_type_title ?? 'فعالیت',
                'owner_didar_id' => $existingActivity->owner_didar_id ?? $didarId,
                'contact_didar_id' => $existingActivity->contact_didar_id,
                'title' => $existingActivity->title ?? 'فعالیت',
                'note' => $existingActivity->note ?? '',
                'result_note' => $existingActivity->result_note ?? '',
                'is_done' => $existingActivity->is_done ?? 0,
                'due_date' => $savedDueDate,
                'is_scheduled' => !empty($dueDate) ? 1 : 0,
                'reminder_enabled' => $reminderEnabled ? 1 : 0,
                'last_update_time' => date('Y-m-d H:i:s'),
                'last_sync' => date('Y-m-d H:i:s'),
                'raw_json' => json_encode(['local_only' => true, 'updated_at' => date('Y-m-d H:i:s')], JSON_UNESCAPED_UNICODE)
            ]);
            $this->activityRepo->save($activity);

            if ($this->notificationService) {
                $this->notificationService->deleteActivityReminders($activityId);
                if ($reminderEnabled && !empty($dueDate)) {
                    $this->notificationService->scheduleActivityReminder($activityId, $savedDueDate, $reminderOffset);
                }
            }

            Logger::logInfo("Local-only activity updated successfully", ['activity_id' => $activityId]);
            return $activityId;
        }
        
        // Format due date for Didar API
        $formattedDueDate = null;
        if (!empty($dueDate)) {
            try {
                $dateString = str_replace(' ', 'T', $dueDate);
                if (strpos($dateString, '+') === false && strpos($dateString, 'Z') === false) {
                    $dateTime = new \DateTime($dateString, new \DateTimeZone('Asia/Tehran'));
                } else {
                    $dateTime = new \DateTime($dateString);
                }
                $dateTime->setTimezone(new \DateTimeZone('UTC'));
                $formattedDueDate = $dateTime->format('Y-m-d\TH:i:s.000\Z');
            } catch (\Exception $e) {
                Logger::logError("Error formatting due date for update", $e, ['due_date' => $dueDate]);
                $timestamp = strtotime($dueDate);
                if ($timestamp !== false) {
                    $formattedDueDate = gmdate('Y-m-d\TH:i:s.000\Z', $timestamp);
                } else {
                    throw new \Exception("خطا در فرمت تاریخ: " . $dueDate);
                }
            }
        }
        
        // Get the correct Didar User ID (OwnerId)
        $didarUserId = $didarId;
        $userRepo = new \App\Repositories\UserRepository($this->personRepo->getDb());
        $currentUser = $userRepo->findByDidarId($didarId);
        
        if ($currentUser && $currentUser->didar_user_id && $currentUser->didar_user_id !== $didarId) {
            $didarUserId = $currentUser->didar_user_id;
        }
        
        // Prepare activity data for Didar API
        $activityData = [
            'Activity' => [
                'Id' => $activityId,
                'ActivityTypeId' => $activityTypeId,
                'Title' => 'فعالیت',
                'OwnerId' => $didarUserId,
                'CreatorId' => $didarUserId, // شناسه شخص ایجاد کننده فعالیت
                'Note' => $existingActivity->note ?? '',
                'ResultNote' => $existingActivity->result_note ?? '',
                'IsDone' => $existingActivity->is_done ?? 0,
                'ContactIds' => [$existingActivity->contact_didar_id],
                'DueDate' => $formattedDueDate ?: date('Y-m-d\TH:i:s.000\Z'),
                'DueDateType' => !empty($formattedDueDate) ? 'Default' : 'NoTime',
                'DoneDateType' => 'Notime'
            ],
            'SetDone' => $existingActivity->is_done ?? 0
        ];
        
        // Add Notifies if reminder is enabled
        if ($reminderEnabled && !empty($formattedDueDate)) {
            $notifies = [];
            $notifies[] = [
                'Id' => '00000000-0000-0000-0000-000000000000',
                'ActivityId' => '00000000-0000-0000-0000-000000000000',
                'Type' => 'Notification',
                'Group' => 'Owner',
                'Data' => null,
                'NotifyOffset' => $reminderOffset,
                'SentDate' => null
            ];
            $activityData['Activity']['Notifies'] = $notifies;
        }
        
        // Call Didar API
        Logger::logInfo("Calling Didar API to update activity", ['activity_id' => $activityId]);
        $res = $this->didarApi->call('/activity/save', 'POST', $activityData);
        
        if (isset($res['error']) || !isset($res['Response']['Id'])) {
            $errorMsg = $res['error'] ?? ($res['message'] ?? 'Unknown error');
            Logger::logError("Failed to update activity in Didar", null, [
                'error' => $errorMsg,
                'response' => $res
            ]);
            throw new \Exception("خطا در به‌روزرسانی فعالیت در دیدار: " . $errorMsg);
        }
        
        // Update local database
        $savedDueDate = !empty($dueDate) ? date('Y-m-d H:i:s', strtotime($dueDate)) : date('Y-m-d H:i:s');
	        $activity = new Activity([
	            'didar_activity_id' => $activityId,
	            'activity_type_id' => $activityTypeId,
	            'activity_type_title' => $res['Response']['ActivityTypeTitle'] ?? '',
	            'owner_didar_id' => $didarId,
	            'contact_didar_id' => $existingActivity->contact_didar_id,
	            'title' => $res['Response']['Title'] ?? 'فعالیت',
	            'note' => $existingActivity->note ?? '',
	            'result_note' => $existingActivity->result_note ?? '',
	            'failure_reason_code' => $failureReasonCode ?: ($existingActivity->failure_reason_code ?? null),
	            'failure_reason_description' => $failureReasonDesc ?: ($existingActivity->failure_reason_description ?? null),
	            'is_done' => $existingActivity->is_done ?? 0,
	            'due_date' => $savedDueDate,
	            'is_scheduled' => !empty($dueDate) ? 1 : 0,
	            'reminder_enabled' => $reminderEnabled ? 1 : 0,
	            'last_update_time' => date('Y-m-d H:i:s'),
            'last_sync' => date('Y-m-d H:i:s'),
            'raw_json' => json_encode($res['Response'], JSON_UNESCAPED_UNICODE)
        ]);
        
        $this->activityRepo->save($activity);
        
        // Update reminder if enabled
        if ($this->notificationService) {
            // Delete old reminders
            $this->notificationService->deleteActivityReminders($activityId);
            
            // Create new reminder if enabled
            if ($reminderEnabled && !empty($dueDate)) {
                $this->notificationService->scheduleActivityReminder($activityId, $savedDueDate, $reminderOffset);
            }
        }
        
        Logger::logInfo("Activity updated successfully", ['activity_id' => $activityId]);
        
        return $activityId;
    }

    public function deleteActivity($activityId)
    {
        return $this->activityRepo->delete($activityId);
    }
}
