<?php
namespace App\Services;

use App\Repositories\UserRepository;
use App\Utils\Logger;

/**
 * Authentication Service
 */
class AuthService
{
    private $userRepo;

    public function __construct(UserRepository $userRepo)
    {
        $this->userRepo = $userRepo;
    }

    public function login($identifier, $password)
    {
        try {
            // Try to find user by email first
            $user = $this->userRepo->findByEmail($identifier);
            
            // If not found by email, try by username or display_name
            if (!$user) {
                $user = $this->userRepo->findByUsernameOrDisplayName($identifier);
            }
            
            if (!$user) {
                Logger::logWarning("Login attempt failed - user not found", ['identifier' => $identifier]);
                return ['success' => false, 'message' => 'نام کاربری یا رمز عبور اشتباه است'];
            }
            
            if (!$user->is_active) {
                Logger::logWarning("Login attempt failed - user inactive", [
                    'user_id' => $user->id,
                    'identifier' => $identifier
                ]);
                return ['success' => false, 'message' => 'حساب کاربری شما غیرفعال است'];
            }
            
            if (empty($user->password) || !password_verify($password, $user->password)) {
                Logger::logWarning("Login attempt failed - wrong password", [
                    'user_id' => $user->id,
                    'identifier' => $identifier
                ]);
                return ['success' => false, 'message' => 'نام کاربری یا رمز عبور اشتباه است'];
            }
            
            // Set session
            $_SESSION['user_id'] = $user->id;
            $_SESSION['didar_id'] = $user->didar_user_id;
            $_SESSION['email'] = $user->email;
            $_SESSION['role'] = $user->role;
            $_SESSION['first_name'] = $user->first_name;
            $_SESSION['last_name'] = $user->last_name;
            $_SESSION['display_name'] = $user->display_name ?? '';
            $_SESSION['name'] = $user->display_name ?? trim(($user->first_name ?? '') . ' ' . ($user->last_name ?? ''));
            
            Logger::logInfo("User logged in", [
                'user_id' => $user->id,
                'email' => $user->email,
                'role' => $user->role
            ]);
            
            return ['success' => true, 'user' => $user->toArray()];
        } catch (\Exception $e) {
            Logger::logError("Login error", $e, ['identifier' => $identifier]);
            return ['success' => false, 'message' => 'خطا در ورود: ' . $e->getMessage()];
        }
    }

    public function logout()
    {
        session_destroy();
        return ['success' => true];
    }

    public function isAuthenticated()
    {
        return isset($_SESSION['user_id']);
    }

    public function getCurrentUser()
    {
        if (!$this->isAuthenticated()) {
            return null;
        }
        
        return $this->userRepo->findById($_SESSION['user_id']);
    }

    public function isAdmin()
    {
        return isset($_SESSION['role']) && $_SESSION['role'] === 'admin';
    }

    public function isCrmSpecialist()
    {
        return isset($_SESSION['role']) && $_SESSION['role'] === 'crm_specialist';
    }

    public function isAdminOrCrmSpecialist()
    {
        $role = $_SESSION['role'] ?? '';
        return $role === 'admin' || $role === 'crm_specialist';
    }

    public function canEditPhone()
    {
        return $this->isAdmin();
    }

    public function canAccessSatisfaction()
    {
        return $this->isAdminOrCrmSpecialist();
    }
    
    /**
     * Check if user can manage support module
     * Only admin and CRM specialist can access support
     */
    public function canAccessSupport()
    {
        return $this->isAdminOrCrmSpecialist();
    }
    
    /**
     * Check if user can edit support entries
     * Only admin can edit support entries (CRM specialist can only create)
     */
    public function canEditSupport()
    {
        return $this->isAdmin();
    }
}

