<?php

namespace App\Services;

use App\Models\Deal;
use App\Repositories\DealRepository;
use App\Repositories\PersonRepository;
use App\Repositories\ActivityRepository;
use App\Utils\Logger;

class DealService
{
    private $dealRepo;
    private $personRepo;
    private $didarApiService;
    private $productRepo;
    private $db;
    private $lookupRepo;

    public function __construct(DealRepository $dealRepo, PersonRepository $personRepo, $didarApiService, $productRepo, $db, $lookupRepo = null)
    {
        $this->dealRepo = $dealRepo;
        $this->personRepo = $personRepo;
        $this->didarApiService = $didarApiService;
        $this->productRepo = $productRepo;
        $this->db = $db;
        $this->lookupRepo = $lookupRepo;
    }

    public function getDeals(array $filters = [])
    {
        return $this->dealRepo->getAll($filters);
    }

    public function getDeal(string $dealId): ?Deal
    {
        return $this->dealRepo->findByDidarId($dealId);
    }

    public function saveDeal(array $data): Deal
    {
        // Ensure deal owner matches lead owner
        $contactId = $data['contact_didar_id'] ?? '';
        if (!empty($contactId)) {
            // IMPORTANT: Here we KNOW $contactId is a Didar ID (GUID),
            // so we MUST use findByDidarId to avoid collision with local numeric IDs
            $person = $this->personRepo->findByDidarId($contactId);
            if (!$person) {
                throw new \Exception('شناسه لید نامعتبر است');
            }
            $leadOwner = $person->owner_didar_id;
            if (empty($leadOwner)) {
                throw new \Exception('مسئول لید نامشخص است');
            }
            if (!empty($data['owner_didar_id']) && $data['owner_didar_id'] !== $leadOwner) {
                throw new \Exception('مسئول معامله باید با مسئول لید یکسان باشد');
            }
            // Force owner to align
            $data['owner_didar_id'] = $leadOwner;
            // Ensure we use the real GUID for the contact_didar_id field
            $data['contact_didar_id'] = $person->didar_contact_id;
            // Store the local person id for robust joining
            $data['local_person_id'] = $person->id;
        }

        // For new deals, generate title automatically if not provided
        $isNewDeal = empty($data['didar_deal_id']) || !isset($data['didar_deal_id']);
        if ($isNewDeal && !empty($contactId) && empty($data['title'])) {
            try {
                $data['title'] = $this->generateDealTitle($contactId);
                Logger::logInfo('Generated deal title for new deal', [
                    'contact_id' => $contactId,
                    'generated_title' => $data['title']
                ]);
            } catch (\Exception $e) {
                // If title generation fails, continue with empty title (will be caught by validation)
                Logger::logError('Failed to generate deal title', $e, ['contact_id' => $contactId]);
            }
        }

        // Generate unique local didar_deal_id if missing
        if (empty($data['didar_deal_id'])) {
            $data['didar_deal_id'] = uniqid('local_', true);
        }
        if (empty($data['status'])) {
            throw new \Exception('وضعیت معامله الزامی است');
        }
        if (empty($data['pipeline_stage'])) {
            throw new \Exception('مرحله کاریز معامله الزامی است');
        }
        if (empty($data['register_time'])) {
            $data['register_time'] = date('Y-m-d H:i:s');
        }
        if (empty($data['last_sync'])) {
            $data['last_sync'] = date('Y-m-d H:i:s');
        }

        // Set default values for financial fields to prevent NULL constraint violations
        if (!isset($data['payable_amount']) || $data['payable_amount'] === '') {
            $data['payable_amount'] = 0;
        }
        if (!isset($data['payment_amount']) || $data['payment_amount'] === '') {
            $data['payment_amount'] = 0;
        }

        // Normalize requested services: allow array or comma-separated
        if (isset($data['requested_services']) && is_array($data['requested_services'])) {
            $data['requested_services'] = json_encode(array_values(array_filter($data['requested_services'])), JSON_UNESCAPED_UNICODE);
        }
        $deal = new \App\Models\Deal($data);
        // Normalize status casing
        $statusMap = [
            'pending' => 'Pending',
            'Pending' => 'Pending',
            'won' => 'Won',
            'Won' => 'Won',
            'lost' => 'Lost',
            'Lost' => 'Lost'
        ];
        if (!empty($deal->status) && isset($statusMap[$deal->status])) {
            $deal->status = $statusMap[$deal->status];
        }
        $now = date('Y-m-d H:i:s');
        if ($deal->status === 'Won') {
            if (empty($deal->won_time)) {
                $deal->won_time = $now;
            }
            $deal->lost_time = null;
        } elseif ($deal->status === 'Lost') {
            if (empty($deal->lost_time)) {
                $deal->lost_time = $now;
            }
            $deal->won_time = null;
        } else {
            $deal->won_time = null;
            $deal->lost_time = null;
        }

        $this->dealRepo->save($deal);

        // Sync deal products if provided
        if (!empty($data['products']) && is_array($data['products'])) {
            $this->syncDealProducts($deal->didar_deal_id, $data['products']);
        }

        // If deal is successful, update lead sale status and products
        if (!empty($deal->contact_didar_id) && $deal->status === 'Won') {
            $this->personRepo->updateSaleStatus($deal->contact_didar_id, 1);
            $this->handleSuccessfulDeal($deal->contact_didar_id, $data['products'] ?? [], $deal->title ?? '');
        }

        return $deal;
    }

    public function syncTransactionsFromPayments(string $dealId, array $payments): void
    {
        if (empty($payments)) {
            return;
        }

        $internalDealId = ctype_digit((string)$dealId) ? (int)$dealId : null;
        if (!$internalDealId) {
            $internalDealId = (int)$this->dealRepo->findInternalIdByDidarId($dealId);
        }
        if (!$internalDealId) {
            Logger::logWarning('syncTransactionsFromPayments: deal not found', ['deal_id' => $dealId]);
            return;
        }

        $normalized = array_values(array_filter(array_map(function($payment) {
            if (!is_array($payment)) {
                return null;
            }
            $amount = isset($payment['amount']) ? floatval($payment['amount']) : 0;
            $date = trim((string)($payment['date'] ?? ''));
            $datePersian = trim((string)($payment['date_persian'] ?? ''));
            $method = trim((string)($payment['method'] ?? $payment['payment_method'] ?? ''));
            $description = trim((string)($payment['description'] ?? ''));
            $cardLast4 = trim((string)($payment['card_last4'] ?? $payment['card_last_4'] ?? ''));
            $bank = trim((string)($payment['destination_bank'] ?? $payment['bank_destination'] ?? ''));

            if ($amount <= 0 || ($date === '' && $datePersian === '')) {
                return null;
            }

            $dateFinal = $this->normalizePaymentDate($date, $datePersian);
            if (!$dateFinal) {
                // Fallback to current date if normalization fails (prevents NULL payment_date)
                $dateFinal = date('Y-m-d');
            }

            return [
                'amount' => $amount,
                'payment_date' => $dateFinal,
                'method' => $method,
                'description' => $description,
                'card_last4' => $cardLast4,
                'destination_bank' => $bank
            ];
        }, $payments)));

        if (empty($normalized)) {
            return;
        }

        // Sort by timestamp to ensure chronological order (not string comparison)
        usort($normalized, function($a, $b) {
            $timeA = strtotime($a['payment_date']);
            $timeB = strtotime($b['payment_date']);
            return $timeA - $timeB;
        });

        $pdo = $this->db->getPdo();
        $delete = $pdo->prepare("DELETE FROM transactions WHERE deal_id = ?");
        $delete->execute([$internalDealId]);

        $insert = $pdo->prepare("
            INSERT INTO transactions (deal_id, amount, payment_date, payment_time, method, card_last4, destination_bank, description, is_first_payment, status, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'confirmed', NOW())
        ");

        foreach ($normalized as $idx => $payment) {
            $insert->execute([
                $internalDealId,
                $payment['amount'],
                $payment['payment_date'],
                $payment['payment_date'] . ' 00:00:00',
                $payment['method'],
                $payment['card_last4'],
                $payment['destination_bank'],
                $payment['description'],
                0  // is_first_payment - set to 0 initially, will be updated below
            ]);
        }

        $resetFirstPayment = $pdo->prepare("UPDATE transactions SET is_first_payment = 0 WHERE deal_id = ?");
        $resetFirstPayment->execute([$internalDealId]);

        $firstIdStmt = $pdo->prepare("
            SELECT id
            FROM transactions
            WHERE deal_id = ?
            ORDER BY COALESCE(payment_date, DATE(payment_time)) ASC, id ASC
            LIMIT 1
        ");
        $firstIdStmt->execute([$internalDealId]);
        $firstId = $firstIdStmt->fetchColumn();
        if ($firstId) {
            $markFirst = $pdo->prepare("UPDATE transactions SET is_first_payment = 1 WHERE id = ?");
            $markFirst->execute([$firstId]);
        }

        try {
            $deal = $this->dealRepo->findById($internalDealId);
            if ($deal) {
                $this->ensureDealProductsFromText(
                    (string)$internalDealId,
                    $deal->requested_services ?? '',
                    $deal->title ?? ''
                );
            }
        } catch (\Exception $e) {
            Logger::logWarning('syncTransactionsFromPayments: ensureDealProductsFromText failed', [
                'deal_id' => $dealId,
                'error' => $e->getMessage()
            ]);
        }
    }

    public function updateStatus($dealId, $status, $timeField = null)
    {
        return $this->dealRepo->updateStatus($dealId, $status, $timeField);
    }

    public function deleteDeal($dealId)
    {
        return $this->dealRepo->delete($dealId);
    }

    public function updatePipelineStage(string $dealId, string $stage)
    {
        return $this->dealRepo->updatePipelineStage($dealId, $stage);
    }

    public function generateDealTitle(string $contactId): string
    {
        // CRITICAL: This method MUST ONLY use findByDidarId
        // It should receive a Didar GUID from the controller, NOT a local numeric ID
        $person = $this->personRepo->findByDidarId($contactId);
        
        if (!$person) {
            Logger::logError('generateDealTitle: person not found in Didar - CRITICAL ERROR', [
                'contact_id' => $contactId,
                'expected_format' => 'Didar GUID (didar_contact_id)',
                'note' => 'This method should NEVER receive a local numeric ID'
            ]);
            throw new \Exception("شخص در سیستم دیدار یافت نشد");
        }

        if (empty($person->mobile_phone)) {
            Logger::logError('generateDealTitle: no mobile phone', [
                'contact_id' => $contactId,
                'person_name' => trim(($person->first_name ?? '') . ' ' . ($person->last_name ?? ''))
            ]);
            throw new \Exception("شماره تماس اصلی لید پیدا نشد");
        }

        // Always use contact_didar_id (GUID) for counting to ensure consistency
        // This matches the logic in createLocalDealForContact which also uses countDealsByContactId
        $dealCount = $this->dealRepo->countDealsByContactId($contactId);

        // Generate ordinal number (اول، دوم، سوم، ...)
        $ordinals = [
            0 => "اول",
            1 => "دوم",
            2 => "سوم",
            3 => "چهارم",
            4 => "پنجم",
            5 => "ششم",
            6 => "هفتم",
            7 => "هشتم",
            8 => "نهم",
            9 => "دهم"
        ];

        $ordinal = isset($ordinals[$dealCount])
            ? $ordinals[$dealCount]
            : ($dealCount + 1) . "ام";
        $title = "معامله {$ordinal} {$person->mobile_phone}";

        Logger::logInfo('generateDealTitle result', [
            'contact_id' => $contactId,
            'person_name' => trim(($person->first_name ?? '') . ' ' . ($person->last_name ?? '')),
            'mobile' => $person->mobile_phone,
            'existing_deals' => $dealCount,
            'ordinal' => $ordinal,
            'title' => $title
        ]);

        return $title;
    }

    /**
     * Get allowed services list
     */
    public function getAllowedServices(): array
    {
        if ($this->lookupRepo) {
            $items = $this->lookupRepo->getItemsByGroupCode('allowed_services');
            if (!empty($items)) {
                $values = [];
                foreach ($items as $item) {
                    $key = $item['code'] ?? $item['value'] ?? null;
                    $val = $item['value'] ?? $item['title'] ?? null;
                    if ($key !== null && $key !== '') {
                        $values[$key] = $val ?? $key;
                    }
                }
                return $values;
            }
        }
        return [
            'consultation' => 'مشاوره',
            'training' => 'آموزش',
            'development' => 'توسعه',
            'support' => 'پشتیبانی'
        ];
    }

    /**
     * Sync deal products from form
     */
    private function syncDealProducts(string $dealId, array $products): void
    {
        if (empty($products)) {
            return;
        }

        $internalDealId = ctype_digit((string)$dealId) ? (int)$dealId : null;
        if (!$internalDealId) {
            $internalDealId = (int)$this->dealRepo->findInternalIdByDidarId($dealId);
        }
        if (!$internalDealId) {
            Logger::logWarning('syncDealProducts: deal not found', ['deal_id' => $dealId]);
            return;
        }

        $pdo = $this->db->getPdo();

        // First, delete existing products for this deal
        $stmt = $pdo->prepare("DELETE FROM deal_products WHERE deal_id = ?");
        $stmt->execute([$internalDealId]);

        // Insert new products
        $stmt = $pdo->prepare("
            INSERT INTO deal_products (deal_id, product_id, created_at)
            VALUES (?, ?, NOW())
        ");

        foreach ($products as $productId) {
            $stmt->execute([$internalDealId, $productId]);
        }
    }

    private function normalizePaymentDate(string $gregorianDate, string $persianDate): ?string
    {
        $gregorianDate = trim($this->normalizePersianDigits($gregorianDate));
        if ($gregorianDate !== '') {
            if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $gregorianDate)) {
                return $gregorianDate;
            }
            if (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}/', $gregorianDate)) {
                return substr($gregorianDate, 0, 10);
            }
        }

        $persianDate = trim($this->normalizePersianDigits($persianDate));
        if (preg_match('/^\d{4}\/\d{1,2}\/\d{1,2}$/', $persianDate)) {
            [$jy, $jm, $jd] = array_map('intval', explode('/', $persianDate));
            
            // Validate parsed values
            if ($jy < 1300 || $jy > 1500 || $jm < 1 || $jm > 12 || $jd < 1 || $jd > 31) {
                // Invalid Persian date, return current date as fallback
                return date('Y-m-d');
            }
            
            $greg = $this->jalaliToGregorian($jy, $jm, $jd);
            
            // Validate converted date
            if ($greg[0] < 1900 || $greg[0] > 2100) {
                // Invalid conversion result, return current date as fallback
                return date('Y-m-d');
            }
            
            return sprintf('%04d-%02d-%02d', $greg[0], $greg[1], $greg[2]);
        }

        // If no valid date found, return current date (prevents NULL payment_date)
        return date('Y-m-d');
    }

    private function normalizePersianDigits(string $value): string
    {
        if ($value === '') {
            return $value;
        }

        $map = [
            '۰' => '0', '۱' => '1', '۲' => '2', '۳' => '3', '۴' => '4',
            '۵' => '5', '۶' => '6', '۷' => '7', '۸' => '8', '۹' => '9',
            '٠' => '0', '١' => '1', '٢' => '2', '٣' => '3', '٤' => '4',
            '٥' => '5', '٦' => '6', '٧' => '7', '٨' => '8', '٩' => '9',
        ];

        return strtr($value, $map);
    }

    /**
     * Convert Jalali (Persian) date to Gregorian date
     * Uses accurate algorithm for date conversion
     */
    private function jalaliToGregorian(int $jy, int $jm, int $jd): array
    {
        $jy = (int)$jy;
        $jm = (int)$jm;
        $jd = (int)$jd;

        // Validate input
        if ($jy < 1 || $jy > 3000 || $jm < 1 || $jm > 12 || $jd < 1 || $jd > 31) {
            // Invalid date, return current date as fallback
            $now = getdate();
            return [$now['year'], $now['mon'], $now['mday']];
        }

        // Accurate Jalali to Gregorian conversion using standard algorithm
        // Source: https://www.fourmilab.ch/documents/calendar/
        
        // Step 1: Calculate Julian Day Number for Jalali date
        $jy = (int)$jy;
        $jm = (int)$jm;
        $jd = (int)$jd;
        
        // Jalali epoch: 226894 (March 21, 622 AD)
        $julianDay = $jy < 0 ? (($jy + 1) * 365) + floor(($jy + 1) / 4) - floor(($jy + 1) / 128) : 
                   ($jy * 365) + floor($jy / 4) - floor($jy / 128) + 226894;
        
        // Add days for months
        if ($jm <= 6) {
            $julianDay += ($jm - 1) * 31;
        } else {
            $julianDay += 186 + (($jm - 7) * 30);
        }
        
        // Add day
        $julianDay += $jd - 1;
        
        // Step 2: Convert Julian Day to Gregorian date
        $j = $julianDay + 1402;
        $k = floor(($j - 1) / 1461);
        $l = $j - (1461 * $k);
        $n = floor(($l - 1) / 365) - floor($l / 1461);
        $i = $l - (365 * $n) + 30;
        $gy = (4 * $k) + $n + $jy - 621;
        $gm = floor($i / 31);
        $gd = $i - (31 * $gm);
        
        // Adjust month and day
        if ($gm < 0 || $gm > 11) {
            $gy += floor($gm / 12);
            $gm = (($gm % 12) + 12) % 12;
        }
        $gm++; // Month is 1-12
        $gd++; // Day is 1-31
        
        // Validate and fix day if needed
        $daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
        // Check for leap year
        if ($gm == 2 && (($gy % 4 == 0 && $gy % 100 != 0) || ($gy % 400 == 0))) {
            $daysInMonth[1] = 29;
        }
        if ($gd > $daysInMonth[$gm - 1]) {
            $gd = $daysInMonth[$gm - 1];
        }
        
        // Final validation
        if ($gy < 1900 || $gy > 2100 || $gm < 1 || $gm > 12 || $gd < 1 || $gd > 31) {
            // Invalid result, use current date as fallback
            $now = getdate();
            return [$now['year'], $now['mon'], $now['mday']];
        }
        
        return [$gy, $gm, $gd];
    }

    public function ensureDealProductsFromText(string $dealId, $requestedServices, string $title): array
    {
        $internalDealId = ctype_digit((string)$dealId) ? (int)$dealId : null;
        if (!$internalDealId) {
            $internalDealId = (int)$this->dealRepo->findInternalIdByDidarId($dealId);
        }
        if (!$internalDealId) {
            Logger::logWarning('ensureDealProductsFromText: deal not found', ['deal_id' => $dealId]);
            return [];
        }

        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM deal_products WHERE deal_id = ?");
        $stmt->execute([$internalDealId]);
        $hasProducts = (int)$stmt->fetchColumn() > 0;
        if ($hasProducts) {
            return [];
        }

        $services = [];
        if (is_array($requestedServices)) {
            $services = $requestedServices;
        } else {
            $raw = trim((string)$requestedServices);
            if ($raw !== '') {
                $decoded = json_decode($raw, true);
                if (is_array($decoded)) {
                    $services = $decoded;
                } else {
                    $services = array_filter(array_map('trim', preg_split('/[,،]+/', $raw)));
                }
            }
        }

        $candidateTexts = array_values(array_filter(array_map('trim', $services)));
        if (empty($candidateTexts)) {
            $title = trim((string)$title);
            if ($title !== '') {
                if (strpos($title, '|') !== false) {
                    $parts = array_map('trim', explode('|', $title));
                    $candidateTexts[] = end($parts);
                }
                $candidateTexts[] = $title;
            }
        }

        $candidateTexts = array_values(array_unique(array_filter($candidateTexts)));
        if (empty($candidateTexts)) {
            return [];
        }

        $matchedProductIds = $this->productRepo->getIdsByNames($candidateTexts);
        if (empty($matchedProductIds)) {
            $matchedProductIds = $this->productRepo->matchIdsByText($candidateTexts);
        }
        if (empty($matchedProductIds)) {
            Logger::logInfo('ensureDealProductsFromText: no product match', [
                'deal_id' => $dealId,
                'texts' => $candidateTexts
            ]);
            return [];
        }

        $this->syncDealProducts((string)$internalDealId, $matchedProductIds);
        Logger::logInfo('ensureDealProductsFromText: products synced', [
            'deal_id' => $dealId,
            'product_ids' => $matchedProductIds
        ]);

        return $matchedProductIds;
    }

    /**
     * Handle successful deal: update lead status and products
     */
    private function handleSuccessfulDeal(string $contactId, array $products, string $dealTitle): void
    {
        $person = $this->personRepo->findByDidarId($contactId);
        if (!$person) {
            Logger::logWarning('handleSuccessfulDeal: person not found', ['contact_id' => $contactId]);
            return;
        }

        // Add products to customer_products
        if (!empty($products)) {
            $this->personRepo->appendProducts($contactId, $products);
        }

        Logger::logInfo('Deal success processed', [
            'contact_id' => $contactId,
            'person_name' => trim(($person->first_name ?? '') . ' ' . ($person->last_name ?? '')),
            'products_count' => count($products)
        ]);
    }

    public function reactivateDeal(string $dealId, string $newStage, string $reason = ''): void
    {
        $pdo = $this->db->getPdo();
        $stmt = $pdo->prepare("
            UPDATE deals
            SET status = 'Pending',
                pipeline_stage = ?,
                won_time = NULL,
                lost_time = NULL,
                failure_reason_code = NULL,
                failure_reason_text = NULL,
                failure_reason_description = NULL,
                last_update_time = NOW(),
                last_sync = NOW()
            WHERE didar_deal_id = ?
        ");
        $stmt->execute([$newStage, $dealId]);

        Logger::logInfo('Deal successfully reactivated', [
            'deal_id' => $dealId,
            'new_stage' => $newStage,
            'reason' => $reason
        ]);
    }

    public function getLookupItems(string $groupCode, bool $includeInactive = false): array
    {
        if (!$this->lookupRepo) {
            return [];
        }
        return $this->lookupRepo->getItemsByGroupCode($groupCode, $includeInactive);
    }
}
