<?php
namespace App\Services;

use App\Database\Connection;
use App\Utils\Logger;
use PDO;

class ExportService
{
    private Connection $db;

    public function __construct(Connection $db)
    {
        $this->db = $db;
    }

    public function getCatalog(): array
    {
        $owners = $this->getOwnerOptions();
        $dealStages = [
            'بلاتکلیف', 'معامله جدید', 'تماس عدم پاسخ', 'تماس پیگیری',
            'ارسال لینک پرداخت', 'پرداخت', 'پیگیری مانده حساب', 'برگزاری دوره',
            'پشتیبانی', 'ارجاع به CRM', 'اتمام'
        ];

        // Fetch custom field configurations to add to export columns
        $pdo = $this->db->getPdo();
        $customConfigs = $pdo->query("SELECT field_name, field_label, entity_type FROM field_configurations WHERE is_active = 1")->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $leadsColumns = [
            ['key' => 'lead_id', 'label' => 'شناسه لید'],
            ['key' => 'lead_code', 'label' => 'کد'],
            ['key' => 'first_name', 'label' => 'نام'],
            ['key' => 'last_name', 'label' => 'نام خانوادگی'],
            ['key' => 'mobile_phone', 'label' => 'موبایل'],
            ['key' => 'secondary_mobile_phone', 'label' => 'موبایل ۲'],
            ['key' => 'mobile_phone_3', 'label' => 'موبایل ۳'],
            ['key' => 'mobile_phone_4', 'label' => 'موبایل ۴'],
            ['key' => 'work_phone', 'label' => 'تلفن ثابت/کار'],
            ['key' => 'email', 'label' => 'ایمیل'],
            ['key' => 'city', 'label' => 'شهر'],
            ['key' => 'job_title', 'label' => 'شغل'],
            ['key' => 'job_description', 'label' => 'شرح شغل'],
            ['key' => 'acquaintance_source', 'label' => 'شیوه آشنایی'],
            ['key' => 'acquaintance_detail', 'label' => 'جزئیات آشنایی'],
            ['key' => 'acquaintance_duration', 'label' => 'مدت آشنایی'],
            ['key' => 'content_topic', 'label' => 'موضوع محتوا'],
            ['key' => 'national_id', 'label' => 'کد ملی'],
            ['key' => 'has_previous_purchase', 'label' => 'خرید قبلی'],
            ['key' => 'sale_status', 'label' => 'وضعیت فروش لید'],
            ['key' => 'register_time_jalali', 'label' => 'تاریخ ثبت (شمسی)'],
            ['key' => 'register_time', 'label' => 'تاریخ ثبت (میلادی)'],
            ['key' => 'owner_name', 'label' => 'مسئول'],
            ['key' => 'previous_owner_name', 'label' => 'مسئول قبلی'],
            ['key' => 'kariz_active_stage', 'label' => 'مرحله کاریز (فعال)'],
            ['key' => 'kariz_special_stage', 'label' => 'مرحله کاریز (ویژه)'],
            ['key' => 'kariz_most_recent_stage', 'label' => 'آخرین مرحله کاریز'],
            ['key' => 'customer_products', 'label' => 'محصولات/خدمات خریداری‌شده'],
            ['key' => 'requested_services', 'label' => 'خدمات درخواستی'],
        ];

        // Add custom person/lead fields
        foreach ($customConfigs as $conf) {
            if ($conf['entity_type'] === 'person') {
                $leadsColumns[] = ['key' => $conf['field_name'], 'label' => $conf['field_label'] . ' (سفارشی)'];
            }
        }
        $leadsColumns[] = ['key' => 'raw_json', 'label' => 'دیتای خام (raw_json)'];

        $dealsColumns = [
            ['key' => 'deal_id', 'label' => 'شناسه معامله'],
            ['key' => 'deal_internal_id', 'label' => 'شناسه داخلی'],
            ['key' => 'title', 'label' => 'عنوان'],
            ['key' => 'code', 'label' => 'کد'],
            ['key' => 'status', 'label' => 'وضعیت'],
            ['key' => 'pipeline_stage', 'label' => 'مرحله کاریز'],
            ['key' => 'owner_name', 'label' => 'مسئول'],
            ['key' => 'contact_id', 'label' => 'شناسه لید'],
            ['key' => 'contact_name', 'label' => 'نام لید'],
            ['key' => 'contact_mobile', 'label' => 'موبایل لید'],
            ['key' => 'products', 'label' => 'محصولات معامله'],
            ['key' => 'service_cost', 'label' => 'هزینه خدمات'],
            ['key' => 'payable_amount', 'label' => 'مبلغ قابل پرداخت'],
            ['key' => 'payment_amount', 'label' => 'مبلغ پرداخت‌شده'],
            ['key' => 'payment_method', 'label' => 'روش پرداخت'],
            ['key' => 'payment_description', 'label' => 'توضیحات پرداخت'],
            ['key' => 'payments', 'label' => 'پرداخت‌ها (JSON)'],
            ['key' => 'is_paid', 'label' => 'پرداخت شده'],
            ['key' => 'payment_short_link', 'label' => 'لینک پرداخت'],
            ['key' => 'has_discount', 'label' => 'تخفیف دارد'],
            ['key' => 'discount_type', 'label' => 'نوع تخفیف'],
            ['key' => 'discount_amount', 'label' => 'مبلغ تخفیف'],
            ['key' => 'refund_amount', 'label' => 'مبلغ مرجوعی'],
            ['key' => 'refund_description', 'label' => 'توضیحات مرجوعی'],
            ['key' => 'failure_reason_code', 'label' => 'کد دلیل ناموفق'],
            ['key' => 'failure_reason_text', 'label' => 'عنوان دلیل ناموفق'],
            ['key' => 'failure_reason_description', 'label' => 'توضیحات دلیل ناموفق'],
            ['key' => 'register_time', 'label' => 'زمان ثبت'],
            ['key' => 'won_time', 'label' => 'زمان موفق'],
            ['key' => 'lost_time', 'label' => 'زمان ناموفق'],
        ];

        // Add custom deal fields
        foreach ($customConfigs as $conf) {
            if ($conf['entity_type'] === 'deal') {
                $dealsColumns[] = ['key' => $conf['field_name'], 'label' => $conf['field_label'] . ' (سفارشی)'];
            }
        }
        $dealsColumns[] = ['key' => 'raw_json', 'label' => 'دیتای خام (raw_json)'];

        return [
            'datasets' => [
                [
                    'id' => 'leads',
                    'title' => 'خروجی لیدها (اشخاص)',
                    'formats' => ['csv', 'xlsx'],
                    'filters' => [
                        ['key' => 'sale_status', 'label' => 'وضعیت لید', 'type' => 'select', 'options' => [
                            ['value' => 'all', 'label' => 'همه'],
                            ['value' => 'success', 'label' => 'موفق'],
                            ['value' => 'failed', 'label' => 'ناموفق'],
                        ]],
                        ['key' => 'owner_didar_id', 'label' => 'مسئول', 'type' => 'select', 'options' => array_merge([['value' => '', 'label' => 'همه']], $owners)],
                        ['key' => 'register_date_from_jalali', 'label' => 'از تاریخ ثبت (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/01'],
                        ['key' => 'register_date_to_jalali', 'label' => 'تا تاریخ ثبت (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/30'],
                        ['key' => 'include_raw_json', 'label' => 'raw_json', 'type' => 'checkbox'],
                    ],
                    'columns' => $leadsColumns,
                ],
                [
                    'id' => 'deals',
                    'title' => 'گزارش معاملات',
                    'formats' => ['csv', 'xlsx'],
                    'filters' => [
                        ['key' => 'status', 'label' => 'وضعیت معامله', 'type' => 'select', 'options' => [
                            ['value' => 'all', 'label' => 'همه'],
                            ['value' => 'Pending', 'label' => 'جاری'],
                            ['value' => 'Won', 'label' => 'موفق'],
                            ['value' => 'Lost', 'label' => 'ناموفق'],
                        ]],
                        ['key' => 'pipeline_stage', 'label' => 'مرحله کاریز', 'type' => 'select', 'options' => array_merge([['value' => 'all', 'label' => 'همه']], array_map(fn($s) => ['value' => $s, 'label' => $s], $dealStages))],
                        ['key' => 'owner_didar_id', 'label' => 'مسئول', 'type' => 'select', 'options' => array_merge([['value' => '', 'label' => 'همه']], $owners)],
                        ['key' => 'register_date_from_jalali', 'label' => 'از تاریخ ثبت معامله (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/01'],
                        ['key' => 'register_date_to_jalali', 'label' => 'تا تاریخ ثبت معامله (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/30'],
                        ['key' => 'include_legacy', 'label' => 'شامل معاملات قدیمی (دیدار)', 'type' => 'checkbox'],
                        ['key' => 'include_raw_json', 'label' => 'raw_json', 'type' => 'checkbox'],
                    ],
                    'columns' => $dealsColumns,
                ],
                [
                    'id' => 'activities',
                    'title' => 'گزارش فعالیت‌ها',
                    'formats' => ['csv', 'xlsx'],
                    'filters' => [
                        ['key' => 'is_done', 'label' => 'وضعیت انجام', 'type' => 'select', 'options' => [
                            ['value' => 'all', 'label' => 'همه'],
                            ['value' => '1', 'label' => 'انجام شده'],
                            ['value' => '0', 'label' => 'انجام نشده'],
                        ]],
                        ['key' => 'owner_didar_id', 'label' => 'مسئول', 'type' => 'select', 'options' => array_merge([['value' => '', 'label' => 'همه']], $owners)],
                        ['key' => 'register_date_from_jalali', 'label' => 'از تاریخ ثبت فعالیت (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/01'],
                        ['key' => 'register_date_to_jalali', 'label' => 'تا تاریخ ثبت فعالیت (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/30'],
                        ['key' => 'include_raw_json', 'label' => 'raw_json', 'type' => 'checkbox'],
                    ],
                    'columns' => [
                        ['key' => 'activity_id', 'label' => 'شناسه فعالیت'],
                        ['key' => 'activity_type_title', 'label' => 'نوع فعالیت'],
                        ['key' => 'title', 'label' => 'عنوان'],
                        ['key' => 'note', 'label' => 'یادداشت'],
                        ['key' => 'result_note', 'label' => 'نتیجه'],
                        ['key' => 'failure_reason_code', 'label' => 'کد دلیل ناموفق'],
                        ['key' => 'failure_reason_description', 'label' => 'توضیحات دلیل ناموفق'],
                        ['key' => 'is_done', 'label' => 'انجام شده'],
                        ['key' => 'due_date', 'label' => 'زمان سررسید'],
                        ['key' => 'done_date', 'label' => 'زمان انجام'],
                        ['key' => 'duration', 'label' => 'مدت'],
                        ['key' => 'direction', 'label' => 'جهت'],
                        ['key' => 'stage', 'label' => 'مرحله (در فعالیت)'],
                        ['key' => 'owner_name', 'label' => 'مسئول'],
                        ['key' => 'contact_id', 'label' => 'شناسه لید'],
                        ['key' => 'contact_name', 'label' => 'نام لید'],
                        ['key' => 'deal_id', 'label' => 'شناسه معامله'],
                        ['key' => 'register_date', 'label' => 'زمان ثبت'],
                        ['key' => 'raw_json', 'label' => 'دیتای خام (raw_json)'],
                    ],
                ],
                [
                    'id' => 'payments',
                    'title' => 'گزارش پرداخت‌ها (از داخل معاملات)',
                    'formats' => ['csv', 'xlsx'],
                    'filters' => [
                        ['key' => 'payment_date_from_persian', 'label' => 'از تاریخ پرداخت (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/01'],
                        ['key' => 'payment_date_to_persian', 'label' => 'تا تاریخ پرداخت (شمسی)', 'type' => 'date_jalali', 'placeholder' => 'مثال: 1403/10/30'],
                        ['key' => 'deal_status', 'label' => 'وضعیت معامله', 'type' => 'select', 'options' => [
                            ['value' => 'all', 'label' => 'همه'],
                            ['value' => 'Pending', 'label' => 'جاری'],
                            ['value' => 'Won', 'label' => 'موفق'],
                            ['value' => 'Lost', 'label' => 'ناموفق'],
                        ]],
                        ['key' => 'owner_didar_id', 'label' => 'مسئول', 'type' => 'select', 'options' => array_merge([['value' => '', 'label' => 'همه']], $owners)],
                        ['key' => 'include_legacy', 'label' => 'شامل معاملات قدیمی (دیدار)', 'type' => 'checkbox'],
                    ],
                    'columns' => [
                        ['key' => 'payment_date_persian', 'label' => 'تاریخ پرداخت (شمسی)'],
                        ['key' => 'payment_date', 'label' => 'تاریخ پرداخت (میلادی/متن)'],
                        ['key' => 'amount', 'label' => 'مبلغ'],
                        ['key' => 'method', 'label' => 'روش'],
                        ['key' => 'card_last4', 'label' => 'Card Last 4'],
                        ['key' => 'bank_destination', 'label' => 'Bank Destination'],
                        ['key' => 'description', 'label' => 'توضیحات'],
                        ['key' => 'deal_id', 'label' => 'شناسه معامله'],
                        ['key' => 'deal_title', 'label' => 'عنوان معامله'],
                        ['key' => 'deal_status', 'label' => 'وضعیت معامله'],
                        ['key' => 'pipeline_stage', 'label' => 'مرحله کاریز'],
                        ['key' => 'contact_id', 'label' => 'شناسه لید'],
                        ['key' => 'contact_name', 'label' => 'نام لید'],
                        ['key' => 'owner_name', 'label' => 'مسئول'],
                    ],
                ],
            ],
        ];
    }

    public function export(string $dataset, array $columns, array $filters, string $format): array
    {
        $format = strtolower($format);
        if (!in_array($format, ['csv', 'xlsx'], true)) {
            throw new \Exception('فرمت خروجی نامعتبر است');
        }

        $includeRawJson = !empty($filters['include_raw_json']);

        if ($dataset === 'leads') {
            return $this->exportLeads($columns, $filters, $format, $includeRawJson);
        }
        if ($dataset === 'deals') {
            return $this->exportDeals($columns, $filters, $format, $includeRawJson);
        }
        if ($dataset === 'activities') {
            return $this->exportActivities($columns, $filters, $format, $includeRawJson);
        }
        if ($dataset === 'payments') {
            return $this->exportPayments($columns, $filters, $format);
        }

        throw new \Exception('نوع خروجی نامعتبر است');
    }

    private function getDatasetLabelMap(string $datasetId): array
    {
        try {
            $catalog = $this->getCatalog();
            $datasets = $catalog['datasets'] ?? [];
            foreach ($datasets as $d) {
                if (($d['id'] ?? '') !== $datasetId) continue;
                $map = [];
                foreach (($d['columns'] ?? []) as $c) {
                    $key = $c['key'] ?? null;
                    if (!$key) continue;
                    $map[$key] = $c['label'] ?? $key;
                }
                return $map;
            }
        } catch (\Throwable $e) {
            Logger::logWarning('Failed to build label map for export', ['dataset' => $datasetId, 'error' => $e->getMessage()]);
        }
        return [];
    }

    private function getOwnerOptions(): array
    {
        $pdo = $this->db->getPdo();
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        $nameExpr = $hasDisplayName ? "COALESCE(display_name, '')" : "COALESCE(CONCAT(first_name,' ',last_name),'')";
        $stmt = $pdo->query("SELECT didar_user_id, $nameExpr AS name FROM users WHERE is_active = 1 ORDER BY name ASC");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $out = [];
        foreach ($rows as $r) {
            $out[] = [
                'value' => $r['didar_user_id'] ?? '',
                'label' => ($r['name'] ?? '') !== '' ? $r['name'] : ($r['didar_user_id'] ?? ''),
            ];
        }
        return $out;
    }

    private function normalizePersianDateForCompare(string $date): string
    {
        $d = trim($date);
        if ($d === '') return '';
        $d = strtr($d, [
            '۰' => '0', '۱' => '1', '۲' => '2', '۳' => '3', '۴' => '4',
            '۵' => '5', '۶' => '6', '۷' => '7', '۸' => '8', '۹' => '9',
            '٠' => '0', '١' => '1', '٢' => '2', '٣' => '3', '٤' => '4',
            '٥' => '5', '٦' => '6', '٧' => '7', '٨' => '8', '٩' => '9',
        ]);
        $d = str_replace('-', '/', $d);
        $parts = explode('/', $d);
        if (count($parts) < 3) return '';
        $y = preg_replace('/\\D+/', '', $parts[0] ?? '');
        $m = preg_replace('/\\D+/', '', $parts[1] ?? '');
        $day = preg_replace('/\\D+/', '', $parts[2] ?? '');
        if ($y === '' || $m === '' || $day === '') return '';
        $y = str_pad(substr($y, 0, 4), 4, '0', STR_PAD_LEFT);
        $m = str_pad(substr($m, 0, 2), 2, '0', STR_PAD_LEFT);
        $day = str_pad(substr($day, 0, 2), 2, '0', STR_PAD_LEFT);
        return $y . '/' . $m . '/' . $day;
    }

    private function writeCsv(string $filename, array $headers, \Generator $rows): void
    {
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');

        $out = fopen('php://output', 'w');
        // UTF-8 BOM for Excel
        fwrite($out, "\xEF\xBB\xBF");
        fputcsv($out, $headers);
        foreach ($rows as $row) {
            fputcsv($out, $row);
        }
        fclose($out);
    }

    private function writeXlsx(string $filename, array $headers, array $rows): void
    {
        if (!class_exists('\\PhpOffice\\PhpSpreadsheet\\Spreadsheet')) {
            throw new \Exception('کتابخانه Excel نصب نیست');
        }
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->fromArray($headers, null, 'A1');
        if (!empty($rows)) {
            $sheet->fromArray($rows, null, 'A2');
        }

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');

        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save('php://output');
    }

    private function exportLeads(array $columns, array $filters, string $format, bool $includeRawJson): array
    {
        $pdo = $this->db->getPdo();
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        $ownerNameExpr = $hasDisplayName ? "COALESCE(u1.display_name,'')" : "COALESCE(CONCAT(u1.first_name,' ',u1.last_name),'')";
        $prevOwnerNameExpr = $hasDisplayName ? "COALESCE(u2.display_name,'')" : "COALESCE(CONCAT(u2.first_name,' ',u2.last_name),'')";

        $selectMap = [
            'lead_id' => 'p.didar_contact_id',
            'lead_code' => 'p.code',
            'first_name' => 'p.first_name',
            'last_name' => 'p.last_name',
            'mobile_phone' => 'p.mobile_phone',
            'secondary_mobile_phone' => 'p.secondary_mobile_phone',
            'mobile_phone_3' => 'p.mobile_phone_3',
            'mobile_phone_4' => 'p.mobile_phone_4',
            'work_phone' => 'p.work_phone',
            'email' => 'p.email',
            'city' => 'p.city',
            'job_title' => 'p.job_title',
            'job_description' => 'p.job_description',
            'acquaintance_source' => 'p.acquaintance_source',
            'acquaintance_detail' => 'p.acquaintance_detail',
            'acquaintance_duration' => 'p.acquaintance_duration',
            'content_topic' => 'p.content_topic',
            'national_id' => 'p.national_id',
            'has_previous_purchase' => 'p.has_previous_purchase',
            'sale_status' => 'p.sale_status',
            'register_time_jalali' => 'p.register_time_jalali',
            'register_time' => 'p.register_time',
            'owner_name' => $ownerNameExpr,
            'previous_owner_name' => $prevOwnerNameExpr,
            'kariz_active_stage' => "(SELECT stage_name FROM virtual_stages vs WHERE vs.contact_didar_id = p.didar_contact_id AND vs.stage_name NOT IN ('with_purchase','without_purchase','refer_crm') ORDER BY vs.entered_at DESC LIMIT 1)",
            'kariz_special_stage' => "(SELECT stage_name FROM virtual_stages vs WHERE vs.contact_didar_id = p.didar_contact_id AND vs.stage_name IN ('with_purchase','without_purchase','refer_crm') ORDER BY vs.entered_at DESC LIMIT 1)",
            'kariz_most_recent_stage' => "(SELECT stage_name FROM virtual_stages vs WHERE vs.contact_didar_id = p.didar_contact_id ORDER BY vs.entered_at DESC LIMIT 1)",
            'customer_products' => 'p.customer_products',
            'requested_services' => 'p.requested_services',
            'raw_json' => 'p.raw_json',
        ];

        if (!$includeRawJson) {
            unset($selectMap['raw_json']);
        }

        $columns = array_values(array_filter($columns, fn($c) => isset($selectMap[$c])));
        if (empty($columns)) {
            $columns = ['lead_id', 'first_name', 'last_name', 'mobile_phone', 'owner_name', 'sale_status', 'kariz_most_recent_stage', 'register_time_jalali'];
        }

        $selectParts = array_map(fn($c) => $selectMap[$c] . " AS `$c`", $columns);
        $sql = "SELECT " . implode(', ', $selectParts) . "
                FROM persons p
                LEFT JOIN users u1 ON p.owner_didar_id = u1.didar_user_id
                LEFT JOIN users u2 ON p.previous_owner_id = u2.didar_user_id
                WHERE p.didar_contact_id IS NOT NULL";

        $where = [];
        $params = [];
        if (!empty($filters['owner_didar_id'])) {
            $where[] = "p.owner_didar_id = ?";
            $params[] = $filters['owner_didar_id'];
        }
        if (!empty($filters['sale_status']) && $filters['sale_status'] !== 'all') {
            if ($filters['sale_status'] === 'success') $where[] = "p.sale_status = 1";
            if ($filters['sale_status'] === 'failed') $where[] = "p.sale_status = 0";
        }
        if (!empty($filters['register_date_from_jalali'])) {
            $where[] = "REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(p.register_time_jalali,'۰','0'),'۱','1'),'۲','2'),'۳','3'),'۴','4'),'۵','5'),'۶','6'),'۷','7'),'۸','8'),'۹','9') >= ?";
            $params[] = $this->normalizePersianDateForCompare($filters['register_date_from_jalali']);
        }
        if (!empty($filters['register_date_to_jalali'])) {
            $where[] = "REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(REPLACE(p.register_time_jalali,'۰','0'),'۱','1'),'۲','2'),'۳','3'),'۴','4'),'۵','5'),'۶','6'),'۷','7'),'۸','8'),'۹','9') <= ?";
            $params[] = $this->normalizePersianDateForCompare($filters['register_date_to_jalali']);
        }
        if (!empty($where)) {
            $sql .= " AND " . implode(' AND ', $where);
        }
        $sql .= " ORDER BY p.register_time DESC, p.created_at DESC";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $labelMap = $this->getDatasetLabelMap('leads');
        $headers = array_map(fn($c) => $labelMap[$c] ?? $c, $columns);
        $rowsGenerator = (function () use ($stmt, $columns) {
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $out = [];
                foreach ($columns as $c) {
                    $out[] = $row[$c] ?? '';
                }
                yield $out;
            }
        })();

        $filenameBase = 'leads_' . date('Ymd_His');
        if ($format === 'csv') {
            $this->writeCsv($filenameBase . '.csv', $headers, $rowsGenerator);
            return ['ok' => true];
        }

        // XLSX: cap to 5000 rows
        $rows = [];
        $i = 0;
        foreach ($rowsGenerator as $r) {
            $rows[] = $r;
            $i++;
            if ($i > 5000) {
                throw new \Exception('برای خروجی Excel، حداکثر 5000 ردیف مجاز است. لطفاً CSV بگیرید یا فیلتر دقیق‌تر بزنید.');
            }
        }
        $this->writeXlsx($filenameBase . '.xlsx', $headers, $rows);
        return ['ok' => true];
    }

    private function exportDeals(array $columns, array $filters, string $format, bool $includeRawJson): array
    {
        $pdo = $this->db->getPdo();
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        $ownerNameExpr = $hasDisplayName ? "COALESCE(u.display_name,'')" : "COALESCE(CONCAT(u.first_name,' ',u.last_name),'')";

        $selectMap = [
            'deal_id' => 'd.didar_deal_id',
            'deal_internal_id' => 'd.id',
            'title' => 'd.title',
            'code' => 'd.code',
            'status' => 'd.status',
            'pipeline_stage' => 'd.pipeline_stage',
            'owner_name' => $ownerNameExpr,
            'contact_id' => 'd.contact_didar_id',
            'contact_name' => "CONCAT(p.first_name,' ',p.last_name)",
            'contact_mobile' => 'p.mobile_phone',
            'products' => "(SELECT GROUP_CONCAT(DISTINCT pr.name ORDER BY pr.name SEPARATOR '، ')
                           FROM deal_products dp
                           JOIN products pr ON pr.id = dp.product_id
                           WHERE dp.deal_id = d.id)",
            'service_cost' => 'd.service_cost',
            'payable_amount' => 'd.payable_amount',
            'payment_amount' => 'd.payment_amount',
            'payment_method' => 'd.payment_method',
            'payment_description' => 'd.payment_description',
            'payments' => 'd.payments',
            'is_paid' => 'd.is_paid',
            'payment_short_link' => 'd.payment_short_link',
            'has_discount' => 'd.has_discount',
            'discount_type' => 'd.discount_type',
            'discount_amount' => 'd.discount_amount',
            'refund_amount' => 'd.refund_amount',
            'refund_description' => 'd.refund_description',
            'failure_reason_code' => 'd.failure_reason_code',
            'failure_reason_text' => 'd.failure_reason_text',
            'failure_reason_description' => 'd.failure_reason_description',
            'register_time' => 'd.register_time',
            'won_time' => 'd.won_time',
            'lost_time' => 'd.lost_time',
            'raw_json' => 'd.raw_json',
        ];
        if (!$includeRawJson) unset($selectMap['raw_json']);

        $columns = array_values(array_filter($columns, fn($c) => isset($selectMap[$c])));
        if (empty($columns)) {
            $columns = ['deal_id', 'title', 'status', 'pipeline_stage', 'owner_name', 'contact_name', 'payment_amount', 'register_time'];
        }

        $selectParts = array_map(fn($c) => $selectMap[$c] . " AS `$c`", $columns);
        $sql = "SELECT " . implode(', ', $selectParts) . "
                FROM deals d
                LEFT JOIN persons p ON d.contact_didar_id = p.didar_contact_id
                LEFT JOIN users u ON d.owner_didar_id = u.didar_user_id
                WHERE 1=1";
        $where = [];
        $params = [];
        if (!empty($filters['owner_didar_id'])) {
            $where[] = "d.owner_didar_id = ?";
            $params[] = $filters['owner_didar_id'];
        }
        if (!empty($filters['status']) && $filters['status'] !== 'all') {
            $where[] = "d.status = ?";
            $params[] = $filters['status'];
        }
        if (!empty($filters['pipeline_stage']) && $filters['pipeline_stage'] !== 'all') {
            $where[] = "d.pipeline_stage = ?";
            $params[] = $filters['pipeline_stage'];
        }
        if (!empty($filters['register_date_from'])) {
            $where[] = "d.register_time >= ?";
            $params[] = $filters['register_date_from'];
        }
        if (!empty($filters['register_date_to'])) {
            $where[] = "d.register_time <= ?";
            $params[] = $filters['register_date_to'];
        }
        
        // Handle Legacy Deals filter
        $includeLegacy = !empty($filters['include_legacy']);
        if (!$includeLegacy) {
            $where[] = "d.is_legacy_didar = 0";
        }

        if (!empty($where)) {
            $sql .= " AND " . implode(' AND ', $where);
        }
        $sql .= " ORDER BY d.register_time DESC, d.created_at DESC";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $labelMap = $this->getDatasetLabelMap('deals');
        $headers = array_map(fn($c) => $labelMap[$c] ?? $c, $columns);
        $rowsGenerator = (function () use ($stmt, $columns) {
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $out = [];
                foreach ($columns as $c) {
                    $out[] = $row[$c] ?? '';
                }
                yield $out;
            }
        })();

        $filenameBase = 'deals_' . date('Ymd_His');
        if ($format === 'csv') {
            $this->writeCsv($filenameBase . '.csv', $headers, $rowsGenerator);
            return ['ok' => true];
        }

        $rows = [];
        $i = 0;
        foreach ($rowsGenerator as $r) {
            $rows[] = $r;
            $i++;
            if ($i > 5000) {
                throw new \Exception('برای خروجی Excel، حداکثر 5000 ردیف مجاز است. لطفاً CSV بگیرید یا فیلتر دقیق‌تر بزنید.');
            }
        }
        $this->writeXlsx($filenameBase . '.xlsx', $headers, $rows);
        return ['ok' => true];
    }

    private function exportActivities(array $columns, array $filters, string $format, bool $includeRawJson): array
    {
        $pdo = $this->db->getPdo();
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        $ownerNameExpr = $hasDisplayName ? "COALESCE(u.display_name,'')" : "COALESCE(CONCAT(u.first_name,' ',u.last_name),'')";

        $selectMap = [
            'activity_id' => 'a.didar_activity_id',
            'activity_type_title' => 'a.activity_type_title',
            'title' => 'a.title',
            'note' => 'a.note',
            'result_note' => 'a.result_note',
            'failure_reason_code' => 'a.failure_reason_code',
            'failure_reason_description' => 'a.failure_reason_description',
            'is_done' => 'a.is_done',
            'due_date' => 'a.due_date',
            'done_date' => 'a.done_date',
            'duration' => 'a.duration',
            'direction' => 'a.direction',
            'stage' => 'a.stage',
            'owner_name' => $ownerNameExpr,
            'contact_id' => 'a.contact_didar_id',
            'contact_name' => "CONCAT(p.first_name,' ',p.last_name)",
            'deal_id' => 'a.deal_didar_id',
            'register_date' => 'a.register_date',
            'raw_json' => 'a.raw_json',
        ];
        if (!$includeRawJson) unset($selectMap['raw_json']);

        $columns = array_values(array_filter($columns, fn($c) => isset($selectMap[$c])));
        if (empty($columns)) {
            $columns = ['activity_id', 'activity_type_title', 'contact_name', 'deal_id', 'is_done', 'register_date'];
        }

        $selectParts = array_map(fn($c) => $selectMap[$c] . " AS `$c`", $columns);
        $sql = "SELECT " . implode(', ', $selectParts) . "
                FROM activities a
                LEFT JOIN persons p ON a.contact_didar_id = p.didar_contact_id
                LEFT JOIN users u ON a.owner_didar_id = u.didar_user_id
                WHERE 1=1";
        $where = [];
        $params = [];
        if (!empty($filters['owner_didar_id'])) {
            $where[] = "a.owner_didar_id = ?";
            $params[] = $filters['owner_didar_id'];
        }
        if (isset($filters['is_done']) && $filters['is_done'] !== 'all' && $filters['is_done'] !== '') {
            $where[] = "a.is_done = ?";
            $params[] = (int)$filters['is_done'];
        }
        if (!empty($filters['register_date_from'])) {
            $where[] = "a.register_date >= ?";
            $params[] = $filters['register_date_from'];
        }
        if (!empty($filters['register_date_to'])) {
            $where[] = "a.register_date <= ?";
            $params[] = $filters['register_date_to'];
        }
        if (!empty($where)) $sql .= " AND " . implode(' AND ', $where);
        $sql .= " ORDER BY a.register_date DESC, a.created_at DESC";

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $labelMap = $this->getDatasetLabelMap('activities');
        $headers = array_map(fn($c) => $labelMap[$c] ?? $c, $columns);
        $rowsGenerator = (function () use ($stmt, $columns) {
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $out = [];
                foreach ($columns as $c) {
                    $out[] = $row[$c] ?? '';
                }
                yield $out;
            }
        })();

        $filenameBase = 'activities_' . date('Ymd_His');
        if ($format === 'csv') {
            $this->writeCsv($filenameBase . '.csv', $headers, $rowsGenerator);
            return ['ok' => true];
        }

        $rows = [];
        $i = 0;
        foreach ($rowsGenerator as $r) {
            $rows[] = $r;
            $i++;
            if ($i > 5000) {
                throw new \Exception('برای خروجی Excel، حداکثر 5000 ردیف مجاز است. لطفاً CSV بگیرید یا فیلتر دقیق‌تر بزنید.');
            }
        }
        $this->writeXlsx($filenameBase . '.xlsx', $headers, $rows);
        return ['ok' => true];
    }

    private function exportPayments(array $columns, array $filters, string $format): array
    {
        $pdo = $this->db->getPdo();
        $hasDisplayName = $this->db->hasColumn('users', 'display_name');
        $ownerNameExpr = $hasDisplayName ? "COALESCE(u.display_name,'')" : "COALESCE(CONCAT(u.first_name,' ',u.last_name),'')";

        $selectMap = [
            'payment_date_persian' => null,
            'payment_date' => null,
            'amount' => null,
            'method' => null,
            'card_last4' => null,
            'bank_destination' => null,
            'description' => null,
            'deal_id' => null,
            'deal_title' => null,
            'deal_status' => null,
            'pipeline_stage' => null,
            'contact_id' => null,
            'contact_name' => null,
            'owner_name' => null,
        ];
        $columns = array_values(array_filter($columns, fn($c) => array_key_exists($c, $selectMap)));
        if (empty($columns)) {
            $columns = ['payment_date_persian', 'amount', 'method', 'card_last4', 'bank_destination', 'description', 'deal_id', 'deal_title', 'deal_status', 'pipeline_stage', 'contact_name', 'owner_name'];
        }

        $sql = "SELECT
                    d.didar_deal_id AS deal_id,
                    d.title AS deal_title,
                    d.status AS deal_status,
                    d.pipeline_stage AS pipeline_stage,
                    d.contact_didar_id AS contact_id,
                    CONCAT(p.first_name,' ',p.last_name) AS contact_name,
                    $ownerNameExpr AS owner_name,
                    d.payments AS payments_json
                FROM deals d
                LEFT JOIN persons p ON d.contact_didar_id = p.didar_contact_id
                LEFT JOIN users u ON d.owner_didar_id = u.didar_user_id
                WHERE d.payments IS NOT NULL AND d.payments <> ''";

        $where = [];
        $params = [];
        if (!empty($filters['owner_didar_id'])) {
            $where[] = "d.owner_didar_id = ?";
            $params[] = $filters['owner_didar_id'];
        }
        if (!empty($filters['deal_status']) && $filters['deal_status'] !== 'all') {
            $where[] = "d.status = ?";
            $params[] = $filters['deal_status'];
        }
        
        // Handle Legacy Deals filter
        $includeLegacy = !empty($filters['include_legacy']);
        if (!$includeLegacy) {
            $where[] = "d.is_legacy_didar = 0";
        }

        if (!empty($where)) $sql .= " AND " . implode(' AND ', $where);
        $sql .= " ORDER BY d.register_time DESC";

        $from = $this->normalizePersianDateForCompare((string)($filters['payment_date_from_persian'] ?? ''));
        $to = $this->normalizePersianDateForCompare((string)($filters['payment_date_to_persian'] ?? ''));

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        $labelMap = $this->getDatasetLabelMap('payments');
        $headers = array_map(fn($c) => $labelMap[$c] ?? $c, $columns);
        $rowsGenerator = (function () use ($stmt, $columns, $from, $to) {
            while ($dealRow = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $payments = json_decode($dealRow['payments_json'] ?? '', true);
                if (!is_array($payments)) {
                    continue;
                }
                foreach ($payments as $p) {
                    $datePersian = trim((string)($p['date_persian'] ?? ''));
                    $datePersianKey = $datePersian !== '' ? $this->normalizePersianDateForCompare($datePersian) : '';
                    if ($from !== '' && ($datePersianKey === '' || $datePersianKey < $from)) {
                        continue;
                    }
                    if ($to !== '' && ($datePersianKey === '' || $datePersianKey > $to)) {
                        continue;
                    }
                    $flat = [
                        'payment_date_persian' => $datePersian,
                        'payment_date' => (string)($p['date'] ?? ''),
                        'amount' => (string)($p['amount'] ?? ''),
                        'method' => (string)($p['method'] ?? ''),
                        'card_last4' => (string)($p['card_last4'] ?? ''),
                        'bank_destination' => (string)($p['bank_destination'] ?? ''),
                        'description' => (string)($p['description'] ?? ''),
                        'deal_id' => (string)($dealRow['deal_id'] ?? ''),
                        'deal_title' => (string)($dealRow['deal_title'] ?? ''),
                        'deal_status' => (string)($dealRow['deal_status'] ?? ''),
                        'pipeline_stage' => (string)($dealRow['pipeline_stage'] ?? ''),
                        'contact_id' => (string)($dealRow['contact_id'] ?? ''),
                        'contact_name' => (string)($dealRow['contact_name'] ?? ''),
                        'owner_name' => (string)($dealRow['owner_name'] ?? ''),
                    ];
                    $out = [];
                    foreach ($columns as $c) {
                        $out[] = $flat[$c] ?? '';
                    }
                    yield $out;
                }
            }
        })();

        $filenameBase = 'payments_' . date('Ymd_His');
        if ($format === 'csv') {
            $this->writeCsv($filenameBase . '.csv', $headers, $rowsGenerator);
            return ['ok' => true];
        }

        $rows = [];
        $i = 0;
        foreach ($rowsGenerator as $r) {
            $rows[] = $r;
            $i++;
            if ($i > 5000) {
                throw new \Exception('برای خروجی Excel، حداکثر 5000 ردیف مجاز است. لطفاً CSV بگیرید یا فیلتر دقیق‌تر بزنید.');
            }
        }
        $this->writeXlsx($filenameBase . '.xlsx', $headers, $rows);
        return ['ok' => true];
    }
}
