<?php
namespace App\Services;

use App\Repositories\KpiRepository;
use App\Repositories\UserRepository;
use App\Utils\Logger;

class KpiService
{
    private $kpiRepo;
    private $userRepo;

    public function __construct(
        KpiRepository $kpiRepo,
        UserRepository $userRepo
    ) {
        $this->kpiRepo = $kpiRepo;
        $this->userRepo = $userRepo;
    }

    /**
     * محاسبه تمام شاخص‌های KPI
     * پشتیبانی ۳ حالت:
     * 1. Admin بدون فیلتر: KPI سازمانی (global)
     * 2. Admin با فیلتر user_id: KPI آن کاربر خاص
     * 3. Agent بدون فیلتر: KPI خودش
     */
    public function calculateKPI(array $filters): array
    {
        Logger::logInfo('KPI calculation started', [
            'filters' => $filters,
            'user_role' => $_SESSION['role'] ?? 'unknown',
            'session_didar_id' => $_SESSION['didar_id'] ?? null
        ]);

        $kpi = [];

        // دریافت نقش کاربر جاری
        $userRole = $_SESSION['role'] ?? '';
        $sessionDidarId = $_SESSION['didar_id'] ?? '';
        $isAgent = ($userRole === 'agent');
        $isAdmin = ($userRole === 'admin');

        // بررسی فیلتر کاربر (owner_didar_id یا user_id)
        $hasUserIdFilter = !empty($filters['user_id']) && $filters['user_id'] !== null;
        $hasOwnerFilter = !empty($filters['owner_didar_id']);
        $hasAnyUserFilter = $hasUserIdFilter || $hasOwnerFilter;

        // حالت 1: Admin بدون فیلتر = KPI سازمانی
        $isGlobalKpi = $isAdmin && !$hasAnyUserFilter;

        // حالت 2: Admin با فیلتر user_id = KPI آن کاربر
        $isUserSpecificKpi = $isAdmin && $hasAnyUserFilter;

        // حالت 3: Agent بدون فیلتر = KPI خودش
        $isSelfKpi = $isAgent && !$hasAnyUserFilter;

        Logger::logInfo('KPI Mode Determined', [
            'is_global_kpi' => $isGlobalKpi,
            'is_user_specific_kpi' => $isUserSpecificKpi,
            'is_self_kpi' => $isSelfKpi,
            'has_user_id_filter' => $hasUserIdFilter,
            'has_owner_filter' => $hasOwnerFilter,
            'filter_user_id' => $filters['user_id'] ?? null,
            'filter_owner_id' => $filters['owner_didar_id'] ?? null,
            'session_didar_id' => $sessionDidarId
        ]);

        // اگر CRM Specialist بود، خطا بده
        if ($userRole === 'crm_specialist') {
            Logger::logWarning('CRM Specialist tried to access KPI - Access Denied');
            throw new \Exception('دسترسی محدود');
        }

        // به‌روزرسانی فیلترها برای Repository
        $repoFilters = $filters;
        unset($repoFilters['user_id']); // حذف چون Repository owner_didar_id می‌خواد

        // اگر حالت Agent بود، فقط خودش را فیلتر کن
        if ($isSelfKpi && !$hasAnyUserFilter) {
            $repoFilters['owner_didar_id'] = $sessionDidarId;
        }

        // اگر حالت Admin بدون فیلتر بود، فیلتر را پاک کن (برای KPI سازمانی)
        if ($isGlobalKpi) {
            $repoFilters['owner_didar_id'] = null;
        }

        // 1. فروش
        $sales = $this->kpiRepo->getSales($repoFilters);
        $kpi['sales'] = $sales;

        // 2. تسویه کامل
        $settled = $this->kpiRepo->getSettledDeals($repoFilters);
        $kpi['settled'] = $settled;

        // 3. مانده حساب
        $outstanding = $this->kpiRepo->getOutstandingBalance($repoFilters);
        $kpi['outstanding'] = $outstanding;

        // 4. فروش ناموفق
        $lost = $this->kpiRepo->getLostSales($repoFilters);
        $kpi['lost_sales'] = $lost;

        // 5. فروش جاری
        $openPipeline = $this->kpiRepo->getOpenPipeline($repoFilters);
        $kpi['open_pipeline'] = $openPipeline;

        // 5.5. معاملات بدون پرداخت
        $dealsWithoutPayments = $this->kpiRepo->getDealsWithoutPayments($repoFilters);
        $kpi['deals_without_payments'] = $dealsWithoutPayments;

        // 5.6. تعداد فروش هر محصول (بر اساس پرداخت در بازه)
        $kpi['product_sales'] = $this->kpiRepo->getProductSalesCounts($repoFilters);

        // 6. نرخ تبدیل لید جدید
        $dealConversion = $this->kpiRepo->getDealConversionRate($repoFilters);
        $kpi['deal_conversion'] = $dealConversion;

        $leadConversion = $this->kpiRepo->getLeadConversionRate($repoFilters);
        $kpi['lead_conversion'] = $leadConversion;

        // 7. میانگین زمان تبدیل
        $timeToConversion = $this->kpiRepo->getTimeToConversion($repoFilters);
        $kpi['time_to_conversion'] = $timeToConversion;

        // 8. فروش روزانه
        $dailySales = $this->kpiRepo->getDailySales($repoFilters);
        $kpi['daily_sales'] = $dailySales;

        // 9. محاسبه درصد تحقق هدف
        // فقط اگر فیلتر user_id داشت (برای KPI آن کاربر خاص) یا اگر حالت Agent بود
        if ($hasAnyUserFilter || $isSelfKpi) {
            $kpi['target_achievement'] = $this->calculateTargetAchievement($filters, $sales, $repoFilters);
        } else {
            // بدون فیلتر کاربر، هدف سازمانی وجود ندارد
            $kpi['target_achievement'] = [
                'has_target' => false,
                'target_amount' => 0,
                'actual_amount' => $sales['total_collected_amount'] ?? 0,
                'achievement_percent' => 0,
                'target_deal_count' => 0,
                'actual_deal_count' => $sales['deal_count'] ?? 0,
                'deal_achievement_percent' => 0
            ];
        }

        // افزودن حالت KPI به پاسخ
        $kpi['kpi_mode'] = $isGlobalKpi ? 'global' : ($isUserSpecificKpi ? 'user_specific' : 'self');

        Logger::logInfo('KPI calculation completed', [
            'kpi_mode' => $kpi['kpi_mode'],
            'kpi_keys' => array_keys($kpi)
        ]);

        return $kpi;
    }

    /**
     * محاسبه درصد تحقق هدف
     * پشتیبانی ۲ حالت:
     * 1. Admin با فیلتر user_id: هدف آن کاربر
     * 2. Agent: هدف کاربر خودش
     */
    private function calculateTargetAchievement(array $filters, array $sales, array $repoFilters): array
    {
        // پیدا کردن user_id از owner_didar_id
        $userId = null;
        if (!empty($filters['owner_didar_id'])) {
            $user = $this->userRepo->findByDidarId($filters['owner_didar_id']);
            $userId = $user ? $user->id : null;
        }

        // اگر کاربر پیدا نشد، هدف وجود ندارد
        if (!$userId) {
            return [
                'has_target' => false,
                'target_amount' => 0,
                'actual_amount' => $sales['total_collected_amount'] ?? 0,
                'achievement_percent' => 0,
                'target_deal_count' => 0,
                'actual_deal_count' => $sales['deal_count'] ?? 0,
                'deal_achievement_percent' => 0
            ];
        }

        // پیدا کردن سال و ماه از فیلترها
        $dateFrom = !empty($filters['date_from']) ? new \DateTime($filters['date_from']) : new \DateTime();
        $year = (int) $dateFrom->format('Y');
        $month = (int) $dateFrom->format('m');

        // دریافت هدف
        $target = $this->kpiRepo->getKpiTarget($userId, $year, $month);

        if (!$target) {
            return [
                'has_target' => false,
                'target_amount' => 0,
                'actual_amount' => $sales['total_collected_amount'] ?? 0,
                'achievement_percent' => 0,
                'target_deal_count' => 0,
                'actual_deal_count' => $sales['deal_count'] ?? 0,
                'deal_achievement_percent' => 0
            ];
        }

        $actualAmount = floatval($sales['total_collected_amount'] ?? 0);
        $targetAmount = floatval($target['target_amount']);
        $achievementPercent = $targetAmount > 0 ? round(($actualAmount / $targetAmount) * 100, 2) : 0;

        $actualDealCount = intval($sales['deal_count'] ?? 0);
        $targetDealCount = intval($target['target_deal_count'] ?? 0);
        $dealAchievementPercent = $targetDealCount > 0 ? round(($actualDealCount / $targetDealCount) * 100, 2) : 0;

        return [
            'has_target' => true,
            'target_amount' => $targetAmount,
            'actual_amount' => $actualAmount,
            'achievement_percent' => $achievementPercent,
            'target_deal_count' => $targetDealCount,
            'actual_deal_count' => $actualDealCount,
            'deal_achievement_percent' => $dealAchievementPercent
        ];
    }

    /**
     * ثبت هدف جدید (Admin فقط)
     */
    public function setKpiTarget(int $userId, int $year, int $month, float $targetAmount, int $targetDealCount = 0): bool
    {
        $pdo = $this->kpiRepo->db->getPdo();

        try {
            $stmt = $pdo->prepare("
                INSERT INTO kpi_targets (user_id, year, month, target_amount, target_deal_count)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    target_amount = VALUES(target_amount),
                    target_deal_count = VALUES(target_deal_count),
                    updated_at = CURRENT_TIMESTAMP
            ");

            $stmt->execute([$userId, $year, $month, $targetAmount, $targetDealCount]);

            Logger::logInfo('KPI target set', [
                'user_id' => $userId,
                'year' => $year,
                'month' => $month,
                'target_amount' => $targetAmount,
                'target_deal_count' => $targetDealCount
            ]);

            return true;
        } catch (\Exception $e) {
            Logger::logError('Failed to set KPI target', $e, [
                'user_id' => $userId,
                'year' => $year,
                'month' => $month
            ]);
            return false;
        }
    }

    /**
     * دریافت لیست تمام اهداف KPI (فقط Admin)
     */
    public function getAllKpiTargets(): array
    {
        $pdo = $this->kpiRepo->db->getPdo();
        $stmt = $pdo->prepare("
            SELECT
                kt.*,
                u.first_name,
                u.last_name,
                u.display_name,
                u.didar_user_id,
                u.role
            FROM kpi_targets kt
            INNER JOIN users u ON kt.user_id = u.id
            WHERE kt.is_active = 1
            ORDER BY kt.year DESC, kt.month DESC, u.first_name ASC
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * دریافت جزئیات معاملات KPI (Admin فقط)
     */
    public function getKpiDealDetails(array $filters): array
    {
        return $this->kpiRepo->getKpiDealDetails($filters);
    }
}
