﻿
// APP_USER will be defined in layout
const APP_USER = window.APP_USER || { role: 'guest', didar_id: null, name: '' };

// Tehran Timezone Helper Functions
function getTehranNow() {
    // Get current date/time in Tehran timezone
    const now = new Date();
    const formatter = new Intl.DateTimeFormat('en-CA', {
        timeZone: 'Asia/Tehran',
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        hour12: false
    });
    const parts = formatter.formatToParts(now);
    
    const year = parseInt(parts.find(p => p.type === 'year').value);
    const month = parseInt(parts.find(p => p.type === 'month').value) - 1; // JS months are 0-indexed
    const day = parseInt(parts.find(p => p.type === 'day').value);
    const hour = parseInt(parts.find(p => p.type === 'hour').value);
    const minute = parseInt(parts.find(p => p.type === 'minute').value);
    const second = parseInt(parts.find(p => p.type === 'second').value);
    
    return new Date(year, month, day, hour, minute, second);
}

function getTehranDate() {
    // Get current date in Tehran timezone (without time)
    const now = new Date();
    const formatter = new Intl.DateTimeFormat('en-CA', {
        timeZone: 'Asia/Tehran',
        year: 'numeric',
        month: '2-digit',
        day: '2-digit'
    });
    const parts = formatter.formatToParts(now);
    
    const year = parseInt(parts.find(p => p.type === 'year').value);
    const month = parseInt(parts.find(p => p.type === 'month').value) - 1;
    const day = parseInt(parts.find(p => p.type === 'day').value);
    
    return new Date(year, month, day);
}

function getTehranPersianDate() {
    // Use moment-jalaali for accurate date conversion (fixes leap year bug)
    if (typeof moment !== 'undefined') {
        const testMoment = moment();
        if (typeof testMoment.jYear === 'function') {
            // Get current date in Tehran timezone
            const now = new Date();
            const formatter = new Intl.DateTimeFormat('en-CA', {
                timeZone: 'Asia/Tehran',
                year: 'numeric',
                month: '2-digit',
                day: '2-digit'
            });
            const parts = formatter.formatToParts(now);

            const gYear = parseInt(parts.find(p => p.type === 'year').value);
            const gMonth = parseInt(parts.find(p => p.type === 'month').value);
            const gDay = parseInt(parts.find(p => p.type === 'day').value);

            // Use moment-jalaali to convert gregorian to persian
            const gregorianMoment = moment(`${gYear}-${String(gMonth).padStart(2, '0')}-${String(gDay).padStart(2, '0')}`, 'YYYY-MM-DD');

            // Get Persian date components using moment-jalaali API
            const persianYear = gregorianMoment.jYear();
            const persianMonth = gregorianMoment.jMonth() + 1; // moment-jalaali months are 0-indexed
            const persianDay = gregorianMoment.jDate();

            // Create persianDate object from array format for compatibility with persianDatepicker
            if (typeof persianDate !== 'undefined') {
                const pDate = new persianDate([persianYear, persianMonth, persianDay]);

                // Debug log (only once)
                if (!window._tehranDateLogged) {
                    console.log('Tehran Gregorian:', gYear, gMonth, gDay, '-> Persian (moment-jalaali):', pDate.format('YYYY/MM/DD'));
                    window._tehranDateLogged = true;
                    setTimeout(() => { window._tehranDateLogged = false; }, 1000);
                }

                return pDate;
            }
        }
    }

    // Fallback to persianDate if moment-jalaali not available
    if (typeof persianDate === 'undefined') {
        console.warn('Neither jalali-moment nor persianDate available');
        return null;
    }

    // Get current date in Tehran timezone
    const now = new Date();
    const formatter = new Intl.DateTimeFormat('en-CA', {
        timeZone: 'Asia/Tehran',
        year: 'numeric',
        month: '2-digit',
        day: '2-digit'
    });
    const parts = formatter.formatToParts(now);

    const gYear = parseInt(parts.find(p => p.type === 'year').value);
    const gMonth = parseInt(parts.find(p => p.type === 'month').value);
    const gDay = parseInt(parts.find(p => p.type === 'day').value);

    // Use persianDate directly (fallback)
    return new persianDate([gYear, gMonth, gDay]);
}

function toPersianDate(dateString) {
    if (!dateString) return '';
    try {
        // Use moment-jalaali for accurate conversion (fixes leap year bug)
        if (typeof moment !== 'undefined') {
            const testMoment = moment();
            if (typeof testMoment.jYear === 'function') {
                let date = new Date(dateString);
                
                if (dateString && dateString.includes(' ') && !dateString.includes('T')) {
                    const parts = dateString.split(' ');
                    const dateParts = parts[0].split('-');
                    const timeParts = parts[1] ? parts[1].split(':') : ['00', '00', '00'];
                    date = new Date(
                        parseInt(dateParts[0]),
                        parseInt(dateParts[1]) - 1,
                        parseInt(dateParts[2]),
                        parseInt(timeParts[0]) || 0,
                        parseInt(timeParts[1]) || 0,
                        parseInt(timeParts[2]) || 0
                    );
                }
                
                const m = moment(date);
                return m.format('jYYYY/jMM/jDD').replace(/\//g, '/');
            }
        }
        
        // Fallback to persianDate
        if (typeof persianDate === 'undefined') {
            console.warn('persianDate library not loaded');
            return dateString.substring(0, 10);
        }
        
        // Parse the date - handle both ISO and local formats
        let date = new Date(dateString);
        
        // If dateString contains time info but is in local format (not ISO), adjust parsing
        if (dateString && dateString.includes(' ') && !dateString.includes('T')) {
            // Format: YYYY-MM-DD HH:mm:ss (local timezone)
            const parts = dateString.split(' ');
            const dateParts = parts[0].split('-');
            const timeParts = parts[1] ? parts[1].split(':') : ['00', '00', '00'];
            date = new Date(
                parseInt(dateParts[0]),
                parseInt(dateParts[1]) - 1,
                parseInt(dateParts[2]),
                parseInt(timeParts[0]) || 0,
                parseInt(timeParts[1]) || 0,
                parseInt(timeParts[2]) || 0
            );
        }
        
        const pDate = new persianDate(date);
        return pDate.format('YYYY/MM/DD');
    } catch(e) {
        console.warn('Error in toPersianDate:', e, dateString);
        return dateString.substring(0, 10);
    }
}

function toPersianDateTime(dateString) {
    if (!dateString) return '';
    try {
        // Use moment-jalaali for accurate conversion (fixes leap year bug)
        if (typeof moment !== 'undefined' && typeof moment.jYear === 'function') {
            let date = new Date(dateString);
            
            if (dateString && dateString.includes(' ') && !dateString.includes('T')) {
                const parts = dateString.split(' ');
                const dateParts = parts[0].split('-');
                const timeParts = parts[1] ? parts[1].split(':') : ['00', '00', '00'];
                date = new Date(
                    parseInt(dateParts[0]),
                    parseInt(dateParts[1]) - 1,
                    parseInt(dateParts[2]),
                    parseInt(timeParts[0]) || 0,
                    parseInt(timeParts[1]) || 0,
                    parseInt(timeParts[2]) || 0
                );
            }
            
            const m = moment(date);
            return m.format('jYYYY/jMM/jDD HH:mm').replace(/\//g, '/');
        }
        
        // Fallback to persianDate
        if (typeof persianDate === 'undefined') {
            console.warn('Neither jalali-moment nor persianDate available');
            return dateString;
        }
        
        let date = new Date(dateString);
        if (dateString && dateString.includes(' ') && !dateString.includes('T')) {
            const parts = dateString.split(' ');
            const dateParts = parts[0].split('-');
            const timeParts = parts[1] ? parts[1].split(':') : ['00', '00', '00'];
            date = new Date(
                parseInt(dateParts[0]),
                parseInt(dateParts[1]) - 1,
                parseInt(dateParts[2]),
                parseInt(timeParts[0]) || 0,
                parseInt(timeParts[1]) || 0,
                parseInt(timeParts[2]) || 0
            );
        }
        
        const pDate = new persianDate(date);
        return pDate.format('YYYY/MM/DD HH:mm');
    } catch(e) {
        console.warn('Error in toPersianDateTime:', e, dateString);
        return dateString;
    }
}

function persianToGregorian(jalaliDate, time = '00:00') {
    if (!jalaliDate || !jalaliDate.trim()) return null;
    try {
        // Check if persianDate is available
        if (typeof persianDate === 'undefined') {
            console.warn('persianDate library not loaded');
            return null;
        }
        
        const parts = jalaliDate.trim().split('/');
        if (parts.length !== 3) return null;
        const year = parseInt(parts[0]);
        const month = parseInt(parts[1]);
        const day = parseInt(parts[2]);
        
        // Validate date parts
        if (isNaN(year) || isNaN(month) || isNaN(day)) return null;
        if (year < 1300 || year > 1500) return null; // Reasonable Persian year range
        if (month < 1 || month > 12) return null;
        if (day < 1 || day > 31) return null;
        
        // Create Persian date object
        let pDate;
        try {
            pDate = new persianDate([year, month, day]);
            
            // Convert to native JavaScript Date object (this handles the conversion automatically)
            const nativeDate = pDate.toDate();
            
            // Validate the date
            if (!nativeDate || isNaN(nativeDate.getTime())) {
                return null;
            }
            
            // Extract time parts
            const timeParts = time.split(':');
            const hours = parseInt(timeParts[0] || 0);
            const minutes = parseInt(timeParts[1] || 0);
            
            // IMPORTANT: Create final date using local timezone
            // The date should be treated as local time, not UTC
            // Format: YYYY-MM-DD HH:mm:ss (for local timezone storage)
            const year4 = nativeDate.getFullYear();
            const month2 = String(nativeDate.getMonth() + 1).padStart(2, '0');
            const day2 = String(nativeDate.getDate()).padStart(2, '0');
            const hour2 = String(hours).padStart(2, '0');
            const min2 = String(minutes).padStart(2, '0');
            
            // Return in local timezone format (not ISO/UTC)
            return `${year4}-${month2}-${day2} ${hour2}:${min2}:00`;
        } catch(e) {
            console.error('Error in persianDate conversion:', e);
            return null;
        }
    } catch(e) {
        console.error('Error converting Persian date:', e, 'Input:', jalaliDate);
        return null;
    }
}

function parseOffset(offsetValue) {
    const match = offsetValue.match(/GMT([+-])(\d{1,2}):?(\d{2})?/);
    if (!match) return null;
    const sign = match[1] === '-' ? -1 : 1;
    const hours = parseInt(match[2], 10) || 0;
    const minutes = parseInt(match[3] || '0', 10);
    return sign * (hours * 60 + minutes);
}

function getTehranNow() {
    const formatter = new Intl.DateTimeFormat('en-US', {
        timeZone: 'Asia/Tehran',
        hour12: false,
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit',
        timeZoneName: 'shortOffset'
    });

    const now = new Date();
    const parts = formatter.formatToParts(now);
    const mapped = {};
    let offsetMinutes = null;

    for (const part of parts) {
        if (part.type === 'year' || part.type === 'month' || part.type === 'day' ||
            part.type === 'hour' || part.type === 'minute' || part.type === 'second') {
            mapped[part.type] = parseInt(part.value, 10);
        }
        if (part.type === 'timeZoneName' && offsetMinutes === null) {
            offsetMinutes = parseOffset(part.value);
        }
    }

    const tehranOffset = offsetMinutes !== null ? offsetMinutes : -210;
    const localOffset = now.getTimezoneOffset();
    const diff = tehranOffset - localOffset;
    const tehranTimestamp = now.getTime() + diff * 60000;

    return new Date(tehranTimestamp);
}

// Validation Helper Functions
function validatePersianText(text, fieldName) {
    if (!text || text.trim() === '') return { valid: true }; // Allow empty
    // فقط حروف فارسی، فاصله و اعراب
    const persianRegex = /^[\u0600-\u06FF\s\u200C\u200D]+$/;
    if (!persianRegex.test(text.trim())) {
        return { valid: false, message: `${fieldName} باید فقط شامل حروف فارسی باشد` };
    }
    return { valid: true };
}

function validateNationalId(nationalId) {
    if (!nationalId || nationalId.trim() === '') return { valid: true }; // Allow empty
    const cleaned = nationalId.trim().replace(/\D/g, ''); // فقط اعداد
    if (cleaned.length !== 10) {
        return { valid: false, message: 'کد ملی باید دقیقاً 10 رقم باشد' };
    }
    return { valid: true };
}

function validatePhone(phone, fieldName) {
    if (!phone || phone.trim() === '') return { valid: true }; // Allow empty
    const cleaned = phone.trim().replace(/\D/g, ''); // فقط اعداد
    if (cleaned.length !== 10) {
        return { valid: false, message: `${fieldName} باید دقیقاً 10 رقم باشد (بدون صفر اول)` };
    }
    if (cleaned.startsWith('0')) {
        return { valid: false, message: `${fieldName} نباید با صفر شروع شود` };
    }
    return { valid: true };
}

// تبدیل اعداد فارسی به انگلیسی
function toEnglishDigits(str = '') {
    const map = {
        '۰':'0','۱':'1','۲':'2','۳':'3','۴':'4','۵':'5','۶':'6','۷':'7','۸':'8','۹':'9',
        '٠':'0','١':'1','٢':'2','٣':'3','٤':'4','٥':'5','٦':'6','٧':'7','٨':'8','٩':'9'
    };
    return String(str).replace(/[۰-۹٠-٩]/g, d => map[d] || d);
}

// نرمال‌سازی تاریخ به فرمت YYYY/MM/DD با ارقام انگلیسی
function normalizeDate(val = '') {
    const clean = toEnglishDigits(val).trim().replace(/-/g, '/');
    if (!clean) return '';
    if (/^\d{4}\/\d{1,2}\/\d{1,2}$/.test(clean)) return clean;
    return '';
}

// Money formatting helpers (###,### while typing)
function parseMoneyValue(val) {
    const s = toEnglishDigits(String(val ?? '')).replace(/,/g, '').trim();
    if (!s) return 0;
    const cleaned = s.replace(/[^\d.]/g, '');
    const n = parseFloat(cleaned);
    return Number.isFinite(n) ? n : 0;
}

function formatMoneyString(val) {
    const s = toEnglishDigits(String(val ?? '')).replace(/,/g, '');
    const cleaned = s.replace(/[^\d]/g, '');
    if (!cleaned) return '';
    return cleaned.replace(/\B(?=(\d{3})+(?!\d))/g, ',');
}

function formatMoneyInput(el) {
    if (!el) return;
    const raw = String(el.value ?? '');
    const start = typeof el.selectionStart === 'number' ? el.selectionStart : raw.length;
    const digitsLeft = raw.slice(0, start).replace(/[^\d]/g, '').length;
    const formatted = formatMoneyString(raw);
    el.value = formatted;
    if (typeof el.setSelectionRange === 'function') {
        let pos = 0;
        let digitCount = 0;
        while (pos < formatted.length && digitCount < digitsLeft) {
            if (/\d/.test(formatted[pos])) digitCount++;
            pos++;
        }
        el.setSelectionRange(pos, pos);
    }
}

const app = {
	    state: { 
	        user: APP_USER,
        leads: [],
        leadsPagination: {
            currentPage: 1,
            perPage: 50,
            totalCount: 0,
            totalPages: 0
        }, 
        deals: [],
        activityTypes: [],
        didarUsers: [],
        referrals: [],
        currentLead: null,
        currentDeals: [],
        dealHasActivity: false,
        originalDealTitleOptions: null,
        crmOwnerId: null,
        crmOwnerName: '',
        lookups: {},
        lookupsLoaded: false,
        lookupGroups: [],
        lookupItems: [],
        lookupSelectedGroup: '',
        didarDeals: [], // Cache for Didar deals to avoid API inconsistency
        products: [],
        kpiUsers: [],
        kpiProducts: [],
        kpiFilters: {
            date_from: '',
            date_to: '',
            owner_didar_id: '',
            product_id: ''
        },
        showKpiDealDetails: false,
        activityStagePrev: '',
        saleStatusMap: {
            success: { title: 'موفق', color: 'border-green-500', bg: 'bg-green-600', icon: '✅' },
            failed: { title: 'ناموفق', color: 'border-red-500', bg: 'bg-red-600', icon: '❌' },
            unknown: { title: 'نامشخص', color: 'border-gray-500', bg: 'bg-gray-600', icon: '➖' }
        },
        pipelineStages: [
            'بلاتکلیف',
            'معامله جدید',
            'تماس عدم پاسخ',
            'تماس پیگیری',
            'ارسال لینک پرداخت',
            'پرداخت',
            'پیگیری مانده حساب',
            'برگزاری دوره',
            'پشتیبانی',
            'ارجاع به CRM'
        ],
	        paymentsLocked: false,
	        lastPaymentsCache: [],
	        exportCatalog: null,
	        exportCurrentDatasetId: null
	    },

    /**
     * بررسی وجود فعالیت برای معامله (دیتابیس داخلی)
     */
    async checkDealHasActivity(dealId) {
        const fallbackId =
            dealId ||
            this.state.currentDeal?.didar_deal_id ||
            this.state.currentDeal?.id ||
            this.state.currentDeal?.deal_id ||
            document.getElementById('deal-id')?.value ||
            '';
        if (!fallbackId) return false;
        try {
            const res = await this.req('has_deal_activity', { deal_id: fallbackId });
            const has = res.status === 'success' && !!res.has_activity;
            this.state.dealHasActivity = has;
            return has;
        } catch (e) {
            console.warn('has_deal_activity failed', e);
            this.state.dealHasActivity = false;
            return false;
        }
    },

    toggleDealActivityForm(hasActivity) {
        this.state.dealHasActivity = !!hasActivity;
        const formWrapper = document.getElementById('deal-activity-form-wrapper');
        const lockMsg = document.getElementById('deal-activity-locked-message');
        const submitBtn = document.getElementById('deal-activity-submit-btn');
        const form = document.getElementById('deal-activity-form');
        const fields = form ? form.querySelectorAll('input, select, textarea, button') : [];
        const chips = document.getElementById('deal-activity-service-chips');
        const svcSelect = document.getElementById('deal-activity-service-select');
        const svcAdd = document.getElementById('deal-activity-service-add');

        if (hasActivity) {
            lockMsg?.classList.remove('hidden');
            formWrapper?.classList.remove('hidden'); // نمایش برای مشاهده
            fields.forEach(el => {
                if (el === submitBtn) {
                    el.classList.add('hidden');
                    return;
                }
                if (el.tagName === 'BUTTON') {
                    el.disabled = true;
                    el.classList.add('cursor-not-allowed', 'opacity-60');
                } else {
                    if ('readOnly' in el) el.readOnly = true;
                    if ('disabled' in el) el.disabled = true;
                    el.classList.add('bg-gray-900', 'cursor-not-allowed');
                }
            });
            if (chips) chips.classList.add('pointer-events-none', 'opacity-70');
            if (svcSelect) {
                svcSelect.disabled = true;
                svcSelect.classList.add('cursor-not-allowed', 'bg-gray-900');
            }
            if (svcAdd) {
                svcAdd.disabled = true;
                svcAdd.classList.add('cursor-not-allowed', 'opacity-60');
            }
        } else {
            lockMsg?.classList.add('hidden');
            formWrapper?.classList.remove('hidden');
            fields.forEach(el => {
                if (el === submitBtn) {
                    el.classList.remove('hidden');
                    return;
                }
                if (el.tagName === 'BUTTON') {
                    el.disabled = false;
                    el.classList.remove('cursor-not-allowed', 'opacity-60');
                } else {
                    if ('readOnly' in el) el.readOnly = false;
                    if ('disabled' in el) el.disabled = false;
                    el.classList.remove('bg-gray-900', 'cursor-not-allowed');
                }
            });
            if (chips) chips.classList.remove('pointer-events-none', 'opacity-70');
            if (svcSelect) {
                svcSelect.disabled = false;
                svcSelect.classList.remove('cursor-not-allowed', 'bg-gray-900');
            }
            if (svcAdd) {
                svcAdd.disabled = false;
                svcAdd.classList.remove('cursor-not-allowed', 'opacity-60');
            }
        }
    },

    // UTILITIES
    toEnglishDigits(str = '') {
        const map = {
            '۰':'0','۱':'1','۲':'2','۳':'3','۴':'4','۵':'5','۶':'6','۷':'7','۸':'8','۹':'9',
            '٠':'0','١':'1','٢':'2','٣':'3','٤':'4','٥':'5','٦':'6','٧':'7','٨':'8','٩':'9'
        };
        return String(str).replace(/[۰-۹٠-٩]/g, d => map[d] || d);
    },

    normalizeDate(val = '') {
        const clean = this.toEnglishDigits(val).trim().replace(/-/g, '/');
        if (!clean) return '';
        if (/^\d{4}\/\d{1,2}\/\d{1,2}$/.test(clean)) return clean;
        return '';
    },

    quickPipelineChange(dealId, stage, status) {
        if (!stage) return;
        // برای مراحل دوره/پشتیبانی/اتمام/مرجوعی از مودال استفاده کن
        if (stage === 'برگزاری دوره' || stage === 'پشتیبانی' || stage === 'اتمام' || stage === 'مرجوعی') {
            // نیاز به اطلاعات معامله فعلی برای contact_id
            if (this.state.currentDeal && (this.state.currentDeal.didar_deal_id === dealId || String(this.state.currentDeal.id) === String(dealId))) {
                this.openCourseModal(this.state.currentDeal, stage);
            } else {
                // اگر در state نبود، از API بگیر
                this.req('get_deal', {deal_id: dealId}).then(res => {
                    if (res.status === 'success' && res.deal) {
                        this.openCourseModal(res.deal, stage);
                    }
                });
            }
            return;
        }
        // سایر مراحل سریع
        this.req('set_deal_pipeline_stage', {deal_id: dealId, pipeline_stage: stage}).then(res => {
            if (res.status === 'success') {
                Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'مرحله به‌روزرسانی شد', timer:1500, showConfirmButton:false});
                this.loadDeals();
            } else {
                Swal.fire('خطا', res.message || 'به‌روزرسانی مرحله انجام نشد', 'error');
            }
        });
    },

    openCourseModal(deal, presetStage = '') {
        if (!deal) return;
        const modal = document.getElementById('modalCourse');
        if (!modal) return;
        document.getElementById('course-deal-id').value = deal.didar_deal_id || deal.id || '';
        document.getElementById('course-contact-id').value = deal.local_person_id || deal.contact_didar_id || '';
        document.getElementById('course-virtual').checked = false;
        document.getElementById('course-online').checked = false;
        document.getElementById('course-access-created').checked = false;
        document.getElementById('course-link-sent').checked = false;
        document.getElementById('course-date-jalali').value = '';
        document.getElementById('course-refund').checked = false;
        document.getElementById('course-refund-reason').value = '';
        document.getElementById('refund-section').classList.add('hidden');
        const nextStageSelect = document.getElementById('course-next-stage');
        if (presetStage === 'پشتیبانی') nextStageSelect.value = 'پشتیبانی';
        else nextStageSelect.value = 'اتمام';
        this.openModal('modalCourse');
    },

    toggleRefundSection() {
        const refundCheckbox = document.getElementById('course-refund');
        const section = document.getElementById('refund-section');
        if (refundCheckbox && section) {
            if (refundCheckbox.checked) section.classList.remove('hidden');
            else section.classList.add('hidden');
        }
    },

    async submitCourseStatus(e) {
        e.preventDefault();
        const form = e.target;
        const dealId = form.deal_id.value;
        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله نامعتبر است', 'error');
            return;
        }
        let courseDate = form.course_date.value || '';
        let gregDate = '';
        if (courseDate) {
            gregDate = persianToGregorian(courseDate);
        }
        const data = {
            deal_id: dealId,
            contact_id: form.contact_id.value || '',
            course_virtual: form.course_virtual.checked ? '1' : '',
            course_online: form.course_online.checked ? '1' : '',
            access_created: form.access_created.checked ? '1' : '',
            link_sent: form.link_sent.checked ? '1' : '',
            course_date: gregDate,
            next_stage: form.next_stage.value,
            refund: form.refund.checked ? '1' : '',
            refund_reason: form.refund_reason.value || ''
        };
        const res = await this.req('set_course_status', data);
        if (res.status === 'success') {
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'به‌روز شد', timer:1500, showConfirmButton:false});
            this.closeModal('modalCourse');
            this.loadDeals();
        } else {
            Swal.fire('خطا', res.message || 'ذخیره نشد', 'error');
        }
    },

    async addProductPrompt() {
        const { value: name } = await Swal.fire({
            title: 'افزودن محصول',
            input: 'text',
            inputLabel: 'نام محصول',
            inputPlaceholder: 'نام محصول را وارد کنید',
            showCancelButton: true,
            confirmButtonText: 'ذخیره',
            cancelButtonText: 'لغو',
            inputValidator: (value) => {
                if (!value || !value.trim()) {
                    return 'نام محصول الزامی است';
                }
            }
        });
        if (!name) return;
        try {
            const res = await this.req('save_product', { name });
            if (res.status === 'success') {
                await this.loadProducts();
                Swal.fire('موفق', 'محصول ذخیره شد', 'success');
            } else {
                Swal.fire('خطا', res.message || 'ذخیره محصول ناموفق بود', 'error');
            }
        } catch (e) {
            Swal.fire('خطا', 'ذخیره محصول ناموفق بود', 'error');
        }
    },

    renderProductOptions(selectedIds = []) {
        const container = document.getElementById('deal-products-list');
        if (!container) return;
        const selectedSet = new Set((selectedIds || []).map(id => String(id)));
        if (!this.state.products || this.state.products.length === 0) {
            container.innerHTML = '<div class="text-xs text-gray-500">محصولی ثبت نشده است.</div>';
            return;
        }
        container.innerHTML = this.state.products.map(p => {
            const checked = selectedSet.has(String(p.id)) ? 'checked' : '';
            return `
                <label class="flex items-center gap-2 text-sm text-gray-300">
                    <input type="checkbox" name="product_ids[]" value="${p.id}" class="form-checkbox text-yellow-500" ${checked}>
                    <span>${p.name}</span>
                </label>
            `;
        }).join('');
    },

    renderDealTitleOptions(selectedValue = null) {
        const select = document.getElementById('deal-title');
        if (!select) return;
        const current = selectedValue !== null ? selectedValue : (select.value || '');
        const generated = Array.from(select.options || []).filter(opt => opt.dataset && opt.dataset.generated === '1');

        select.innerHTML = '<option value="">-- انتخاب خدمات قابل ارائه --</option>';
        const products = Array.isArray(this.state.products) ? this.state.products : [];
        products.forEach(p => {
            if (!p || !p.name) return;
            const opt = document.createElement('option');
            opt.value = p.name;
            opt.textContent = p.name;
            opt.dataset.productId = p.id;
            select.appendChild(opt);
        });

        generated.forEach(opt => {
            if (!opt.value) return;
            const exists = Array.from(select.options || []).some(o => o.value === opt.value);
            if (exists) return;
            const clone = document.createElement('option');
            clone.value = opt.value;
            clone.textContent = opt.textContent;
            clone.dataset.generated = '1';
            select.appendChild(clone);
        });

        if (current) {
            const hasCurrent = Array.from(select.options || []).some(o => o.value === current);
            if (hasCurrent) {
                select.value = current;
            }
        }
    },

    renderLeadProductFilter() {
        const select = document.getElementById('leads-product-filter');
        if (!select) return;
        const current = select.value;
        select.innerHTML = '<option value="">محصول / خدمت</option>';

        const appendOption = (value, label) => {
            const opt = document.createElement('option');
            opt.value = value;
            opt.textContent = label;
            select.appendChild(opt);
        };

        const products = this.state.products && this.state.products.length ? this.state.products : null;
        if (products) {
            products.forEach(p => appendOption(p.id, p.name));
        }

        if (current) {
            select.value = current;
        }
    },

    initLeadFiltersDatePickers() {
        const ids = ['leads-register-from', 'leads-register-to', 'leads-activity-from', 'leads-activity-to'];
        const initInput = (input) => {
            if (!input) return;
            if (typeof $ === 'undefined' || typeof $.fn.persianDatepicker === 'undefined') {
                setTimeout(() => initInput(input), 200);
                return;
            }
            try {
                if ($(input).data('persianDatepicker')) {
                    $(input).persianDatepicker('destroy');
                }
            } catch (_) {}
            $(input).persianDatepicker({
                format: 'YYYY/MM/DD',
                initialValue: false,
                autoClose: true,
                observer: true
            });
        };
        ids.forEach(id => initInput(document.getElementById(id)));
    },

    async renderDealActivitiesInline(contactId) {
        const container = document.getElementById('deal-activities-inline');
        if (!container) return;
        container.innerHTML = '<div class="text-center text-gray-500 py-6">در حال بارگذاری...</div>';
        try {
            const dealId =
                document.getElementById('deal-id')?.value ||
                this.state.currentDeal?.didar_deal_id ||
                this.state.currentDeal?.id ||
                this.state.currentDeal?.deal_id ||
                '';

            if (dealId) {
                const res = await this.req('get_deal_activities', { deal_id: dealId });
                if (res.status === 'success') {
                    const acts = res.activities || [];
                    this.renderDidarActivities(acts, container);
                    this.updateDealActivitiesBadge(acts.length || 0);
                return;
            }
            }

            // برای معاملات جدید، فعالیت‌ها نمایش داده نشود
            container.innerHTML = '<div class="text-center text-gray-500 py-6">فعالیتی یافت نشد</div>';
            this.updateDealActivitiesBadge(0);
        } catch (e) {
            console.error('Error rendering inline activities:', e);
            container.innerHTML = '<div class="text-center text-red-400 py-6">خطا در دریافت فعالیت‌ها</div>';
            this.updateDealActivitiesBadge(0);
        }
    },
    leadSources: [
        {value:'instagram', label:'اینستاگرام'},
        {value:'telegram', label:'تلگرام'},
        {value:'book', label:'کتاب'},
        {value:'website', label:'سایت'},
        {value:'whatsapp', label:'واتساپ'},
        {value:'campaign', label:'کمپین'},
        {value:'event', label:'ایونت (event)'},
        {value:'support', label:'پشتیبانی'},
        {value:'incoming_call', label:'تماس ورودی'},
        {value:'visit', label:'مراجعه حضوری'},
        {value:'referral', label:'معرف'},
        {value:'other', label:'سایر'}
    ],
    leadSourceDetails: {
        instagram: [
            {value:'instagram_direct', label:'اینستاگرام - دایرکت'},
            {value:'instagram_box', label:'اینستاگرام - باکس'},
            {value:'instagram_highlight', label:'اینستاگرام - هایلایت'},
            {value:'instagram_channel', label:'اینستاگرام - کانال'},
            {value:'instagram_auto', label:'اینستاگرام - اتوحنفی'}
        ],
        whatsapp: [
            {value:'whatsapp_marketing', label:'واتساپ مارکتینگ'}
        ],
        website: [
            {value:'website_signup', label:'سایت - ثبت نام در سایت'},
            {value:'website_course_signup', label:'سایت - ثبت نام دوره'},
            {value:'website_failed_purchase', label:'سایت - خرید ناموفق'}
        ],
        campaign: [
            {value:'campaign_sms', label:'کمپین - پیامک'},
            {value:'campaign_divar', label:'کمپین - دیوار'},
            {value:'campaign_porsline', label:'کمپین - پرسلاین'},
            {value:'campaign_web', label:'کمپین وب'}
        ],
        event: [
            {value:'event_live', label:'ایونت - لایو'},
            {value:'event_meetup', label:'ایونت - دورهمی'},
            {value:'event_course', label:'ایونت - دوره'}
        ],
        referral: [
            {value:'referral_internal', label:'معرف - همکاران داخلی'},
            {value:'referral_external', label:'معرف - همکاران خارج از سازمان'},
            {value:'referral_teacher', label:'معرف - استاد مطلبی'},
            {value:'referral_customer', label:'معرف - مشتری قدیم'}
        ],
        default: [
            {value:'generic', label:'عمومی'}
        ]
    },
    contentTopics: [
        {value:'car_intelligence', label:'هوش خودرو'},
        {value:'in_person_inspection', label:'کارشناسی حضوری'},
        {value:'business', label:'کسب و کار'},
        {value:'mini_course', label:'مینی دوره هوش خودرو'},
        {value:'vip_channel', label:'کانال VIP'},
        {value:'budgeting', label:'بودجه بندی'},
        {value:'online_exhibition', label:'نمایشگاه آنلاین'},
        {value:'consulting', label:'مشاوره با استاد'},
        {value:'inspection_device', label:'دستگاه کارشناسی'},
        {value:'car_intelligence_test', label:'تست هوش خودرو'},
        {value:'financial_personality', label:'تست تیپ شخصیت مالی'},
        {value:'smart_trader', label:'معامله گر هوشمند'},
        {value:'detailing_training', label:'آموزش دیتیلینگ'},
        {value:'divar_car', label:'دیوار خودرو'},
        {value:'inspection_test', label:'تست کارشناسی'},
        {value:'book', label:'کتاب هوش خودرو'},
        {value:'cip', label:'CIP'}
    ],
        getLeadSourceLabel(value) {
        return this.getLookupLabel('lead_sources', value);
    },
    getLeadDetailLabel(value) {
        return this.getLookupLabel('lead_source_details', value);
    },
    getContentTopicLabel(value) {
        return this.getLookupLabel('content_topics', value);
    },
    escapeHtml(str) {
        if(!str) return '';
        return str.replace(/&/g, '&amp;')
                  .replace(/</g, '&lt;')
                  .replace(/>/g, '&gt;')
                  .replace(/"/g, '&quot;')
                  .replace(/'/g, '&#39;');
    },
    
    /**
     * Strip HTML tags and convert HTML entities to plain text
     * @param {string} html - HTML string to clean
     * @param {boolean} preserveLineBreaks - If true, convert <br> and <p> to line breaks
     * @returns {string} Clean text
     */
    stripHtml(html, preserveLineBreaks = true) {
        if (!html) return '';
        
        // Create a temporary div element to parse HTML
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = html;
        
        // Get text content (automatically strips HTML tags)
        let text = tempDiv.textContent || tempDiv.innerText || '';
        
        // Replace common HTML entities
        text = text.replace(/&nbsp;/g, ' ')
                   .replace(/&amp;/g, '&')
                   .replace(/&lt;/g, '<')
                   .replace(/&gt;/g, '>')
                   .replace(/&quot;/g, '"')
                   .replace(/&#39;/g, "'")
                   .replace(/&apos;/g, "'");
        
        // Clean up multiple spaces and line breaks
        if (preserveLineBreaks) {
            // Replace multiple spaces with single space, but preserve line breaks
            text = text.replace(/[ \t]+/g, ' ')
                       .replace(/\n\s*\n/g, '\n\n') // Preserve paragraph breaks
                       .trim();
        } else {
            // Replace all whitespace with single space
            text = text.replace(/\s+/g, ' ').trim();
        }
        
        return text;
    },

    isAdmin() {
        return (this.state.user?.role === 'admin');
    },
    isAdminOrCrmSpecialist() {
        const role = this.state.user?.role;
        return role === 'admin' || role === 'crm_specialist';
    },
    isLeadOwner(lead) {
        if(!lead || !lead.owner_didar_id) return false;
        return this.state.user?.didar_id && this.state.user.didar_id === lead.owner_didar_id;
    },
    async req(action, data = {}) {
        const fd = new FormData();
        for(let k in data) {
            if(data[k] !== null && data[k] !== undefined) {
                fd.append(k, data[k]);
            }
        }
        console.log(`[DEBUG] API Request: ${action}`, data);
        try {
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 180000); // 3 minutes timeout
            
            const r = await fetch('?api=' + action, {
                method:'POST', 
                body:fd,
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            
            if (!r.ok) {
                // Try to parse detailed error from server
                try {
                    const errRes = await r.json();
                    if (errRes && errRes.message) {
                        console.log(`[DEBUG] API Error Response: ${action}`, errRes);
                        return { status: 'error', message: errRes.message };
                    }
                } catch(_) {}
                throw new Error(`HTTP ${r.status}: ${r.statusText}`);
            }
            
            const response = await r.json();
            console.log(`[DEBUG] API Response: ${action}`, response);
            return response;
        } catch(e) { 
            console.error(`[DEBUG] API Error for ${action}:`, e);
            
            // Check if it's a network error or timeout
            if (e.name === 'AbortError' || e.message.includes('fetch')) {
                throw new Error('NETWORK_ERROR'); // Throw to be caught by setupSync
            }
            
            return {status:'error', message: 'خطا در ارتباط با سرور: ' + e.message}; 
        }
    },

    toggleSidebar() {
        const sidebar = document.getElementById('main-sidebar');
        const backdrop = document.getElementById('sidebar-backdrop');
        if (!sidebar || !backdrop) return;
        
        const isClosed = sidebar.classList.contains('translate-x-full');
        if (isClosed) {
            sidebar.classList.remove('translate-x-full');
            sidebar.classList.add('translate-x-0');
            backdrop.classList.remove('hidden');
        } else {
            sidebar.classList.add('translate-x-full');
            sidebar.classList.remove('translate-x-0');
            backdrop.classList.add('hidden');
        }
    },

    closeSidebar() {
        const sidebar = document.getElementById('main-sidebar');
        const backdrop = document.getElementById('sidebar-backdrop');
        if (!sidebar || !backdrop) return;
        
        // Only if it's open (mobile)
        if (!sidebar.classList.contains('translate-x-full')) {
            sidebar.classList.add('translate-x-full');
            sidebar.classList.remove('translate-x-0');
            backdrop.classList.add('hidden');
        }
    },

    initTheme() {
        const theme = localStorage.getItem('theme') || 'dark';
        const body = document.body;
        const icon = document.getElementById('theme-icon');
        
        if (theme === 'light') {
            body.classList.add('light-theme');
            if (icon) {
                icon.classList.remove('fa-sun');
                icon.classList.add('fa-moon');
            }
        } else {
            body.classList.remove('light-theme');
            if (icon) {
                icon.classList.remove('fa-moon');
                icon.classList.add('fa-sun');
            }
        }
    },

    toggleTheme() {
        const body = document.body;
        const icon = document.getElementById('theme-icon');
        const isLight = body.classList.contains('light-theme');
        
        if (isLight) {
            body.classList.remove('light-theme');
            localStorage.setItem('theme', 'dark');
            if (icon) {
                icon.classList.remove('fa-moon');
                icon.classList.add('fa-sun');
            }
        } else {
            body.classList.add('light-theme');
            localStorage.setItem('theme', 'light');
            if (icon) {
                icon.classList.remove('fa-sun');
                icon.classList.add('fa-moon');
            }
        }
    },

    async init() {
        if (typeof Logger !== 'undefined') Logger.init();
        console.log('[DEBUG] App init started');
        console.log('[DEBUG] Current user state:', this.state.user);
        console.log('[DEBUG] isAdmin:', this.isAdmin());
        console.log('[DEBUG] isAdminOrCrmSpecialist:', this.isAdminOrCrmSpecialist());
        
        await this.loadLookups();
        this.applyLookupsToStaticSelects();
        this.initMoneyInputFormatting();
        this.initTheme();
        this.loadActivityTypes();
        
        if (this.isAdminOrCrmSpecialist()) {
            console.log('[DEBUG] User is admin or CRM specialist, loading Didar users...');
            await this.loadDidarUsers();
        } else {
            console.log('[DEBUG] User is NOT admin or CRM specialist, skipping Didar users');
        }
        
        // Load dashboard after users are loaded (so filters can be populated)
        this.loadDashboard();
        if (this.isAdmin()) {
            this.loadReferrals();
        }
        
        console.log('[DEBUG] App init complete');
        
        // Initialize notifications
        this.loadNotifications();
        this.startNotificationPolling();

        // Request notification permission after a short delay
        setTimeout(() => {
            if ('Notification' in window) {
                console.log('Notification support detected, permission:', Notification.permission);

                // Check if we're on HTTPS (required for notifications in some browsers)
                if (location.protocol !== 'https:' && location.hostname !== 'localhost' && location.hostname !== '127.0.0.1') {
                    console.log('⚠️ Desktop notifications require HTTPS. Current protocol:', location.protocol);
                }

                if (Notification.permission === 'default') {
                    console.log('Requesting notification permission on startup');
                    this.requestNotificationPermission();
                } else if (Notification.permission === 'denied') {
                    console.log('Notifications are blocked by user');
                } else {
                    console.log('Notifications already granted');
                }
            } else {
                console.log('This browser does not support desktop notifications');
            }
        }, 2000);

        // Load notification settings (default to enabled)
        this.notificationSoundEnabled = localStorage.getItem('notificationSoundEnabled') !== 'false';
        this.notificationPopupEnabled = localStorage.getItem('notificationPopupEnabled') !== 'false';
        this.notificationBrowserEnabled = localStorage.getItem('notificationBrowserEnabled') !== 'false';

        // Update sound button
        this.updateSoundButton();

        // Initialize reminder offset calculators
        this.initReminderOffsetListeners();

        console.log('Notification settings loaded:', {
            sound: this.notificationSoundEnabled,
            popup: this.notificationPopupEnabled,
            browser: this.notificationBrowserEnabled
        });
        
        // Close notifications dropdown when clicking outside
        document.addEventListener('click', (e) => {
            const container = document.getElementById('notifications-container');
            const dropdown = document.getElementById('notifications-dropdown');
            if (container && dropdown && !container.contains(e.target)) {
                dropdown.classList.add('hidden');
            }
        });
    },

    initMoneyInputFormatting() {
        if (this.state._moneyInputFormattingInitialized) return;
        this.state._moneyInputFormattingInitialized = true;

        document.addEventListener('input', (e) => {
            const el = e.target;
            if (!el || !(el instanceof HTMLElement)) return;
            if (!el.classList || !el.classList.contains('money-input')) return;
            formatMoneyInput(el);
        });
    },
    
    // SETUP
    async setupInit(e) {
        e.preventDefault();
        const apiKey = document.getElementById('setup-api-key').value;
        Swal.showLoading();
        const res = await this.req('setup_init', {api_key: apiKey});
        Swal.close();
        
        if(res.status === 'success') {
            document.getElementById('setup-step1').classList.add('hidden');
            const step2 = document.getElementById('setup-step2');
            step2.classList.remove('hidden');
            
            const list = document.getElementById('setup-users-list');
            list.innerHTML = '';
            res.users.forEach(u => {
                const div = document.createElement('div');
                div.className = 'p-3 bg-gray-800 rounded cursor-pointer hover:bg-gray-700 transition';
                div.innerHTML = `
                    <input type="radio" name="admin_user" value="${u.Id || u.UserId}" id="user-${u.Id || u.UserId}" class="mr-2">
                    <label for="user-${u.Id || u.UserId}" class="cursor-pointer">${u.DisplayName} (${u.UserName})</label>
                `;
                list.appendChild(div);
            });
            
            window.setupApiKey = apiKey;
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    async setupComplete() {
        const selected = document.querySelector('input[name="admin_user"]:checked');
        if(!selected) {
            Swal.fire('خطا', 'لطفاً کاربر ادمین را انتخاب کنید', 'error');
            return;
        }
        
        Swal.showLoading();
        const res = await this.req('setup_complete', {
            api_key: window.setupApiKey,
            admin_didar_id: selected.value
        });
        
        if(res.status === 'success') {
            document.getElementById('setup-step2').classList.add('hidden');
            document.getElementById('setup-step3').classList.remove('hidden');
            
            // Check if there's existing data to resume
            const statusRes = await this.req('get_sync_status', {});
            if(statusRes.status === 'success' && statusRes.counts.deals > 0) {
                const resumeBtn = document.getElementById('setup-resume-btn');
                resumeBtn.classList.remove('hidden');
                resumeBtn.innerHTML = `<i class="fa-solid fa-play ml-1"></i> ادامه همگام‌سازی (${statusRes.counts.deals} معامله sync شده)`;
            } else {
                this.setupSync();
            }
        } else {
            Swal.close();
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    async setupSync(step = 'users', from = 0) {
        try {
            const res = await this.req('setup_sync', {step, from});
            
        if(res.status === 'success') {
                document.getElementById('setup-progress-bar').style.width = res.progress + '%';
                document.getElementById('setup-progress-text').textContent = res.message;
                
                // نمایش پیشرفت دقیق‌تر
                let progressDetail = `مرحله: ${step} | پیشرفت: ${res.progress}%`;
                if (res.total_count && res.from !== undefined) {
                    const synced = Math.min(res.from + (res.limit || 500), res.total_count);
                    progressDetail += ` | ${synced} از ${res.total_count}`;
                }
                document.getElementById('setup-progress-detail').textContent = progressDetail;
                
                // Hide resume button during successful sync
                document.getElementById('setup-resume-btn').classList.add('hidden');
                
                if(res.next_step === 'complete') {
                    // Calculate virtual stages in batch after sync completes
                    Swal.fire({icon:'info', title:'در حال محاسبه مراحل مجازی...', text:'لطفاً صبر کنید', allowOutsideClick: false});
                    try {
                        // virtual stages deprecated
                    } catch (e) {
                        console.error('Error updating virtual stages:', e);
                    }
                    Swal.fire({icon:'success', title:'نصب تکمیل شد', text:'در حال انتقال...', timer:2000});
                    setTimeout(() => location.reload(), 2000);
                } else {
                    // Fix: محاسبه nextFrom برای جلوگیری از لوپ بی‌نهایت
                    let nextFrom = 0;
                    if(res.next_step === step) {
                        // اگر همان step است، from را افزایش بده
                        nextFrom = (res.from || 0) + (res.limit || 500);
                    }
                    
                    setTimeout(() => this.setupSync(res.next_step, nextFrom), 500);
                }
            } else {
                // Show resume button on API error
                document.getElementById('setup-resume-btn').classList.remove('hidden');
                Swal.fire('خطا', res.message, 'error');
            }
        } catch (error) {
            // Network error handling
            console.error('Network error in setupSync:', error);
            document.getElementById('setup-resume-btn').classList.remove('hidden');
            
            // Check if it's a network error
            const isNetworkError = error.message === 'NETWORK_ERROR' || 
                                   error.message.includes('fetch') || 
                                   error.message.includes('network') ||
                                   error.message.includes('Failed to fetch');
            
            Swal.fire({
                icon: 'warning',
                title: 'خطا در ارتباط',
                html: isNetworkError 
                    ? 'ارتباط با سرور قطع شد.<br>داده‌های sync شده حفظ شده‌اند.<br>می‌توانید با دکمه "ادامه همگام‌سازی" از همانجا ادامه دهید.'
                    : 'خطا در پردازش درخواست.<br>لطفاً دوباره تلاش کنید.',
                showCancelButton: true,
                confirmButtonText: 'ادامه همگام‌سازی',
                cancelButtonText: 'بستن',
                allowOutsideClick: false
            }).then((result) => {
                if (result.isConfirmed) {
                    this.resumeSync();
                }
            });
        }
    },
    
    async resumeSync() {
        try {
            const res = await this.req('resume_sync', {});
            if (res.status === 'success') {
                this.setupSync(res.step || 'users', res.from || 0);
            } else {
                Swal.fire('خطا', res.message || 'خطا در ادامه همگام‌سازی', 'error');
            }
        } catch (error) {
            console.error('Error in resumeSync:', error);
            Swal.fire('خطا', 'خطا در ادامه همگام‌سازی: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    // LOGIN
    async login(e) {
        e.preventDefault();
        const username = document.getElementById('login-user-select').value;
        const password = document.getElementById('login-password').value;
        
        if (!username || !password) {
            Swal.fire('خطا', 'لطفاً نام کاربری و رمز عبور را وارد کنید', 'error');
            return;
        }
        
        const res = await this.req('login', {username: username, password: password});
        if(res.status === 'success') {
            location.reload();
        } else {
            Swal.fire('خطا', res.message || 'نام کاربری یا رمز عبور اشتباه است', 'error');
        }
    },

    refresh() {
        const active = document.querySelector('.sidebar-item.active');
        if(active) {
            const view = active.id.replace('menu-', '');
            if(view === 'dashboard') this.loadDashboard();
            if(view === 'leads') this.loadLeads();
            if(view === 'deals') this.loadDeals();
            if(view === 'referrals') this.loadReferrals();
            if(view === 'persons') this.loadPersons();
            if(view === 'settings') this.loadSettings();
        }
    },
    
    // SETTINGS
    loadSettings() {
        // Settings view is static, no need to load data
        // Just make sure the form is reset
        const form = document.querySelector('#view-settings form');
        if (form) {
            form.reset();
            document.getElementById('settings-change-password-user-id').value = window.CURRENT_USER_ID || 0;
        }
        if (this.isAdmin()) {
            this.loadDidarWriteSettings();
        }
    },

    async loadDidarWriteSettings() {
        try {
            const res = await this.req('get_didar_write_settings');
            if (res.status !== 'success') return;
            const enabled = String(res.didar_write_enabled || '1') === '1';
            const checkbox = document.getElementById('settings-didar-write-enabled');
            const status = document.getElementById('settings-didar-write-status');
            if (checkbox) checkbox.checked = enabled;
            if (status) {
                status.textContent = enabled
                    ? 'ارسال به دیدار فعال است.'
                    : 'ارسال به دیدار غیرفعال است.';
            }
        } catch (e) {
            console.warn('Failed to load Didar write settings', e);
        }
    },

    async saveDidarWriteSettings() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'دسترسی محدود به مدیر سیستم', 'error');
            return;
        }
        const checkbox = document.getElementById('settings-didar-write-enabled');
        const enabled = checkbox ? (checkbox.checked ? '1' : '0') : '1';
        const res = await this.req('save_didar_write_settings', { didar_write_enabled: enabled });
        if (res.status === 'success') {
            const status = document.getElementById('settings-didar-write-status');
            if (status) {
                status.textContent = enabled === '1'
                    ? 'ارسال به دیدار فعال است.'
                    : 'ارسال به دیدار غیرفعال است.';
            }
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:res.message || 'تنظیمات ذخیره شد', showConfirmButton:false, timer:2000});
        } else {
            Swal.fire('خطا', res.message || 'ذخیره تنظیمات با خطا مواجه شد', 'error');
        }
    },

    // REFERRALS
    async loadReferrals() {
        try {
            const res = await this.req('get_referrals');
            if (res.status === 'success') {
                this.state.referrals = res.referrals || [];
                this.renderReferrals();
            }
        } catch (e) {
            console.warn('Failed to load referrals', e);
            this.state.referrals = [];
            this.renderReferrals();
        }
    },

    renderReferrals() {
        const tbody = document.getElementById('referrals-list');
        if (!tbody) return;

        if (this.state.referrals.length === 0) {
            tbody.innerHTML = '<tr><td colspan="5" class="text-center py-8 text-gray-500">هیچ ارجاعی یافت نشد</td></tr>';
            return;
        }

        tbody.innerHTML = this.state.referrals.map(r => `
            <tr class="hover:bg-gray-800/30">
                <td class="p-3">
                    <div class="text-sm text-gray-200">${r.first_name || ''} ${r.last_name || ''}</div>
                    <div class="text-xs text-gray-500 mt-1">${r.mobile_phone || ''}</div>
                </td>
                <td class="p-3 text-xs text-gray-300">${r.phones ? JSON.parse(r.phones).join(' ، ') : ''}</td>
                <td class="p-3 text-xs text-gray-300">${r.referrer_name || '---'}</td>
                <td class="p-3 text-xs text-gray-400">${r.created_at ? new Date(r.created_at).toLocaleString('fa-IR') : ''}</td>
                <td class="p-3 text-xs text-gray-300 whitespace-pre-wrap">${r.note || ''}</td>
            </tr>
        `).join('');
    },
    
    // DASHBOARD
    async loadDashboard() {
        const ownerFilter = document.getElementById('dashboard-owner-filter')?.value || '';
        const statusFilter = document.getElementById('dashboard-status-filter')?.value || 'all';

        const res = await this.req('get_dashboard_stats', {
            owner_didar_id: ownerFilter,
            status: statusFilter
        });
        if(res.status === 'success') {
            const stats = res.stats;
            const content = document.getElementById('dashboard-content');
            
            content.innerHTML = `
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 mb-6">
                    <div class="glass p-4 border-r-4 border-blue-500 cursor-pointer hover:bg-gray-800/50 transition" onclick="app.goToLeadsWithFilter('all')">
                        <div class="text-sm text-gray-400 mb-1">کل لید</div>
                        <div class="text-2xl font-bold text-white">${stats.total_leads || stats.total_persons || 0}</div>
                        </div>
                    <div class="glass p-4 border-r-4 border-green-500 cursor-pointer hover:bg-gray-800/50 transition" onclick="app.goToLeadsWithFilter('success')">
                        <div class="text-sm text-gray-400 mb-1">لید های موفق</div>
                        <div class="text-2xl font-bold text-white">${stats.successful_leads || 0}</div>
                    </div>
                    <div class="glass p-4 border-r-4 border-red-500 cursor-pointer hover:bg-gray-800/50 transition" onclick="app.goToLeadsWithFilter('failed')">
                        <div class="text-sm text-gray-400 mb-1">لید های ناموفق</div>
                        <div class="text-2xl font-bold text-white">${stats.failed_leads || 0}</div>
                    </div>
                </div>
                
                <div class="glass p-6">
                    <h3 class="text-lg font-bold mb-4 text-yellow-500">کاریز معاملات</h3>
                    <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 xl:grid-cols-7 gap-3">
                        ${this.state.pipelineStages.map(stage => `
                            <div class="text-center p-3 bg-gray-800 rounded cursor-pointer hover:bg-gray-700 transition" onclick="app.goToDealsWithFilter('${stage}')">
                                <div class="text-xs text-gray-400 mb-1">${stage}</div>
                                <div class="text-xl font-bold text-white">${(stats.deals_by_stage && stats.deals_by_stage[stage]) ? stats.deals_by_stage[stage] : 0}</div>
                            </div>
                        `).join('')}
                    </div>
                    <div class="mt-4 text-xs text-gray-400">کل جاری (Pending): ${stats.ongoing_deals || 0}</div>
                </div>
            `;
        }
    },
    
    // ACTIVITY TYPES
    async loadActivityTypes() {
        const res = await this.req('get_activity_types');
        if(res.status === 'success') {
            this.state.activityTypes = res.types || [];
        }
    },

    async loadProducts() {
        try {
            const res = await this.req('get_products');
            if (res.status === 'success') {
                this.state.products = res.products || [];
                this.renderProductOptions();
                this.renderLeadProductFilter();
                this.renderDealTitleOptions();
            }
        } catch (e) {
            console.error('Failed to load products', e);
        }
    },
    
	    async loadDidarUsers() {
	        try {
	            console.log('[DEBUG] Starting loadDidarUsers...');
	            const res = await this.req('get_didar_users');
	            console.log('[DEBUG] get_didar_users response:', res);
	            
	            if(res.status === 'success') {
	                this.state.didarUsers = res.users || [];
	                console.log('[DEBUG] Didar users loaded successfully:', this.state.didarUsers.length, 'users');
	                console.log('[DEBUG] Users data:', JSON.stringify(this.state.didarUsers, null, 2));
	                
	                this.populateLeadOwnerSelect();
	                this.populateOwnerFilterSelects();
	                
	                // اگر لیست خالی است، به کاربر اطلاع بده
	                if (this.state.didarUsers.length === 0) {
	                    console.warn('[DEBUG] No Didar users found. This might indicate an issue with API or all users are inactive.');
	                }
	            } else {
	                // نمایش خطا به کاربر
	                console.error('[DEBUG] Failed to load Didar users:', res.message || 'Unknown error', res);
	                if (res.message) {
	                    Swal.fire({
	                        icon: 'warning',
	                        title: 'هشدار',
	                        text: 'خطا در دریافت لیست کاربران دیدار: ' + res.message,
	                        confirmButtonText: 'باشه',
	                        timer: 5000
	                    });
	                }
	                this.state.didarUsers = [];
	                this.populateLeadOwnerSelect();
	                this.populateOwnerFilterSelects();
	            }
	        } catch (error) {
	            console.error('[DEBUG] Error loading Didar users:', error);
	            this.state.didarUsers = [];
	            this.populateLeadOwnerSelect();
	            this.populateOwnerFilterSelects();
	        }
	    },

    async loadReminders() {
        const container = document.getElementById('reminders-list-container');
        if (!container) return;

        container.innerHTML = '<div class="text-center py-20 col-span-full"><div class="loading-spinner mb-4 mx-auto"></div><p class="text-gray-400">در حال بارگذاری معاملات گذشته...</p></div>';

        try {
            const res = await this.req('get_reminders_list');
            if (res.status === 'success') {
                const reminders = res.reminders || [];
                if (reminders.length === 0) {
                    container.innerHTML = '<div class="text-center py-20 col-span-full"><i class="fa-solid fa-clock-rotate-left text-4xl text-gray-700 mb-4 block"></i><p class="text-gray-500">هیچ یادآوری فعالی یافت نشد.</p></div>';
                    return;
                }

                // Priority levels configuration
                const priorityLevels = {
                    'high': {
                        color: 'border-red-600 bg-red-500/10',
                        icon: 'fa-exclamation-triangle',
                        badge: 'bg-red-500/20 text-red-400',
                        text: 'اولویت بالا'
                    },
                    'medium': {
                        color: 'border-yellow-600 bg-yellow-500/10',
                        icon: 'fa-clock',
                        badge: 'bg-yellow-500/20 text-yellow-400',
                        text: 'اولویت متوسط'
                    },
                    'low': {
                        color: 'border-blue-600 bg-blue-500/10',
                        icon: 'fa-info-circle',
                        badge: 'bg-blue-500/20 text-blue-400',
                        text: 'اولویت پایین'
                    }
                };

                container.innerHTML = reminders.map(rem => {
                    const contactId = rem.contact_didar_id || rem.contact_id || (rem.contact ? rem.contact.didar_contact_id : '') || rem.local_person_id;
                    const isOverdue = new Date(rem.reminder_time) < new Date();
                    const statusColor = rem.reminder_sent ? 'border-gray-500' : (isOverdue ? 'border-red-500' : 'border-blue-500');
                    const statusText = rem.reminder_sent ? 'ارسال شده' : (isOverdue ? 'زمان گذشته' : 'در انتظار');
                    const badgeClass = rem.reminder_sent ? 'bg-gray-500/20 text-gray-400' : (isOverdue ? 'bg-red-500/20 text-red-400' : 'bg-blue-500/20 text-blue-400');

                    const timeToReminder = new Date(rem.reminder_time) - new Date();
                    const hoursToReminder = timeToReminder / (1000 * 60 * 60);
                    let priority = 'low';
                    if (isOverdue || hoursToReminder < 1) priority = 'high';
                    else if (hoursToReminder < 24) priority = 'medium';

                    const priorityData = priorityLevels[priority];
                    
                    const timeRemaining = isOverdue ? 
                        `${Math.abs(Math.floor(hoursToReminder))} ساعت پیش` : 
                        `${Math.floor(hoursToReminder)} ساعت دیگر`;

                    const customerName = (rem.first_name || '') + ' ' + (rem.last_name || '').trim();
                    const avatarInitials = customerName.length > 0 ? 
                        customerName.split(' ').map(n => n[0]).join('').substring(0, 2).toUpperCase() : 
                        '؟؟';

                    return `
                        <div class="glass p-5 rounded-2xl border border-gray-700/50 hover:border-blue-500/50 transition-all duration-500 group flex flex-col h-full relative overflow-hidden">
                            <!-- Top Status Bar -->
                            <div class="flex justify-between items-center mb-4">
                                <span class="text-[9px] px-2.5 py-1 rounded-full font-bold tracking-tight ${priorityData.badge} flex items-center gap-1.5">
                                    <i class="fa-solid ${priorityData.icon} text-[8px]"></i>
                                    ${priorityData.text}
                                </span>
                                <span class="text-[9px] px-2.5 py-1 rounded-full font-bold ${badgeClass}">
                                    ${statusText}
                                </span>
                            </div>

                            <!-- Main Content -->
                            <div class="flex items-center gap-4 mb-5">
                                <div class="w-12 h-12 rounded-xl bg-gradient-to-br from-blue-600/20 to-purple-600/20 border border-blue-500/20 flex items-center justify-center text-blue-400 font-black text-sm shadow-inner group-hover:scale-110 transition-transform">
                                    ${avatarInitials}
                                </div>
                                <div class="flex-1 min-w-0">
                                    <h4 class="text-white font-bold text-sm truncate mb-0.5">${this.escapeHtml(customerName || 'مشتری نامشخص')}</h4>
                                    <p class="text-[10px] text-gray-400 font-medium truncate opacity-80">${this.escapeHtml(rem.activity_title || 'فعالیت بدون عنوان')}</p>
                                </div>
                            </div>

                            <!-- Timing Info -->
                            <div class="grid grid-cols-2 gap-2 mb-5">
                                <div class="bg-gray-800/40 p-2 rounded-lg border border-gray-700/30">
                                    <span class="block text-[8px] text-gray-500 mb-0.5">زمان یادآوری</span>
                                    <span class="text-[10px] text-gray-300 font-bold font-mono">${toPersianDateTime(rem.reminder_time)}</span>
                                </div>
                                <div class="bg-gray-800/40 p-2 rounded-lg border border-gray-700/30">
                                    <span class="block text-[8px] text-gray-500 mb-0.5">وضعیت زمانی</span>
                                    <span class="text-[10px] ${isOverdue ? 'text-red-400' : 'text-green-400'} font-bold">${timeRemaining}</span>
                                </div>
                            </div>

                            <!-- Progress Section -->
                            <div class="mb-6">
                                <div class="flex justify-between text-[9px] mb-1.5 px-0.5">
                                    <span class="text-gray-500">پیشرفت فعالیت</span>
                                    <span class="text-blue-400 font-bold">${rem.reminder_sent ? '100%' : 'در جریان'}</span>
                                </div>
                                <div class="w-full bg-gray-900/50 rounded-full h-1.5 overflow-hidden border border-gray-800">
                                    <div class="bg-gradient-to-r from-blue-600 to-indigo-500 h-full rounded-full transition-all duration-700 ease-out"
                                         style="width: ${rem.reminder_sent ? '100' : Math.max(15, Math.min(85, (new Date() - new Date(rem.due_date)) / (new Date(rem.reminder_time) - new Date(rem.due_date)) * 100))}%">
                                    </div>
                                </div>
                            </div>

                            <!-- Actions Group -->
                            <div class="mt-auto space-y-3">
                                <div class="grid grid-cols-2 gap-2">
                                    <button onclick="app.openPersonProfile('${contactId}')"
                                        class="text-[10px] bg-blue-600/10 hover:bg-blue-600 text-blue-400 hover:text-white py-2 rounded-xl transition-all border border-blue-500/20 flex items-center justify-center gap-2">
                                        <i class="fa-solid fa-user-tag"></i> مشاهده لید
                                    </button>
                                    <button onclick="app.markReminderDone('${rem.activity_didar_id}')"
                                        class="text-[10px] bg-emerald-600/10 hover:bg-emerald-600 text-emerald-400 hover:text-white py-2 rounded-xl transition-all border border-emerald-500/20 flex items-center justify-center gap-2">
                                        <i class="fa-solid fa-check-circle"></i> انجام شد
                                    </button>
                                </div>
                                
                                <div class="flex justify-between items-center gap-1.5 pt-3 border-t border-gray-800">
                                    <button onclick="app.snoozeReminder('${rem.activity_didar_id}', 60)" class="flex-1 text-[9px] bg-gray-800 hover:bg-yellow-600/20 text-gray-400 hover:text-yellow-500 py-1.5 rounded-lg border border-gray-700/50 transition-colors">+۱س</button>
                                    <button onclick="app.snoozeReminder('${rem.activity_didar_id}', 240)" class="flex-1 text-[9px] bg-gray-800 hover:bg-yellow-600/20 text-gray-400 hover:text-yellow-500 py-1.5 rounded-lg border border-gray-700/50 transition-colors">+۴س</button>
                                    <button onclick="app.snoozeReminder('${rem.activity_didar_id}', 1440)" class="flex-1 text-[9px] bg-gray-800 hover:bg-yellow-600/20 text-gray-400 hover:text-yellow-500 py-1.5 rounded-lg border border-gray-700/50 transition-colors">+۱روز</button>
                                    <div class="w-px h-4 bg-gray-800 mx-1"></div>
                                    <button onclick="app.resendReminder('${rem.activity_didar_id}')" class="p-2 text-gray-500 hover:text-blue-400 transition-colors" title="ارسال مجدد">
                                        <i class="fa-solid fa-bell"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    `;
                }).join('');
            } else {
                container.innerHTML = `<div class="text-center py-20 col-span-full text-red-400">${res.message || 'خطا در دریافت یادآوری‌ها'}</div>`;
            }
        } catch (error) {
            console.error('Reminders load failed:', error);
            container.innerHTML = '<div class="text-center py-20 col-span-full text-red-400">خطا در ارتباط با سرور</div>';
        }
    },

    // Reminder Actions
    async editReminder(activityId) {
        // TODO: Implement edit reminder functionality
        Swal.fire({
            title: 'ویرایش یادآوری',
            text: 'این قابلیت به زودی اضافه خواهد شد',
            icon: 'info',
            confirmButtonText: 'باشه'
        });
    },

    async resendReminder(activityId) {
        try {
            const result = await Swal.fire({
                title: 'ارسال مجدد اعلان',
                text: 'آیا مطمئن هستید که می‌خواهید اعلان را مجدداً ارسال کنید؟',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'بله، ارسال کن',
                cancelButtonText: 'لغو'
            });

            if (result.isConfirmed) {
                const res = await this.req('resend_reminder', { activity_id: activityId });
                if (res.status === 'success') {
                    Swal.fire({
                        title: 'موفق!',
                        text: 'اعلان مجدداً ارسال شد',
                        icon: 'success',
                        timer: 2000
                    });
                    this.loadReminders();
                } else {
                    throw new Error(res.message);
                }
            }
        } catch (error) {
            console.error('Resend reminder failed:', error);
            Swal.fire({
                title: 'خطا!',
                text: error.message || 'خطا در ارسال مجدد اعلان',
                icon: 'error'
            });
        }
    },

    async markReminderDone(activityId) {
        try {
            const result = await Swal.fire({
                title: 'علامت‌گذاری به عنوان انجام شده',
                text: 'این یادآوری به عنوان انجام شده علامت‌گذاری خواهد شد',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'بله، انجام شد',
                cancelButtonText: 'لغو'
            });

            if (result.isConfirmed) {
                const res = await this.req('mark_reminder_done', { activity_id: activityId });
                if (res.status === 'success') {
                    Swal.fire({
                        title: 'موفق!',
                        text: 'یادآوری به عنوان انجام شده علامت‌گذاری شد',
                        icon: 'success',
                        timer: 2000
                    });
                    this.loadReminders();
                } else {
                    throw new Error(res.message);
                }
            }
        } catch (error) {
            console.error('Mark reminder done failed:', error);
            Swal.fire({
                title: 'خطا!',
                text: error.message || 'خطا در علامت‌گذاری یادآوری',
                icon: 'error'
            });
        }
    },

    async snoozeReminder(activityId, minutes) {
        try {
            const timeLabels = {
                60: '۱ ساعت',
                240: '۴ ساعت',
                1440: '۱ روز'
            };

            const result = await Swal.fire({
                title: 'تعویق یادآوری',
                text: `یادآوری به مدت ${timeLabels[minutes]} تعویق خواهد شد`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'بله، تعویق کن',
                cancelButtonText: 'لغو'
            });

            if (result.isConfirmed) {
                const res = await this.req('snooze_reminder', {
                    activity_id: activityId,
                    minutes: minutes
                });
                if (res.status === 'success') {
                    Swal.fire({
                        title: 'موفق!',
                        text: `یادآوری به مدت ${timeLabels[minutes]} تعویق شد`,
                        icon: 'success',
                        timer: 2000
                    });
                    this.loadReminders();
                } else {
                    throw new Error(res.message);
                }
            }
        } catch (error) {
            console.error('Snooze reminder failed:', error);
            Swal.fire({
                title: 'خطا!',
                text: error.message || 'خطا در تعویق یادآوری',
                icon: 'error'
            });
        }
    },

	    populateOwnerFilterSelects() {
	        const leadsOwnerSelect = document.getElementById('leads-owner-filter');
	        const dealsOwnerSelect = document.getElementById('deals-owner-filter');
	        const legacyDealsOwnerSelect = document.getElementById('legacy-deals-owner-filter');
	        const dashboardOwnerSelect = document.getElementById('dashboard-owner-filter');
	        
	        console.log('[DEBUG] populateOwnerFilterSelects called');
	        console.log('[DEBUG] leadsOwnerSelect exists:', !!leadsOwnerSelect);
	        console.log('[DEBUG] dealsOwnerSelect exists:', !!dealsOwnerSelect);
	        console.log('[DEBUG] legacyDealsOwnerSelect exists:', !!legacyDealsOwnerSelect);
	        console.log('[DEBUG] dashboardOwnerSelect exists:', !!dashboardOwnerSelect);
	        console.log('[DEBUG] didarUsers count:', this.state.didarUsers?.length || 0);

	        const render = (select, isDashboard = false) => {
	            if (!select) {
	                console.log('[DEBUG] Select element is null, skipping render');
	                return;
	            }
	            
	            const current = select.value || '';
	            console.log('[DEBUG] Rendering select, current value:', current, 'isDashboard:', isDashboard);
	            
	            select.innerHTML = isDashboard ? '<option value="">همه مسئول‌ها (کل سیستم)</option>' : '<option value="">همه مسئول‌ها</option>';
	            
	            console.log('[DEBUG] Starting to add user options. Users array:', this.state.didarUsers);
	            
	            (this.state.didarUsers || []).forEach((user, index) => {
	                // Use owner_id (didar_id) if available, otherwise fallback to internal id
	                const value = user.owner_id || user.id;
	                const label = user.display_name || user.email || ('کاربر ' + user.id);
	                
	                console.log(`[DEBUG] User ${index}:`, { value, label, user });
	                
	                if (!value) {
	                    console.log(`[DEBUG] User ${index} skipped - no value`);
	                    return;
	                }
	                
	                const opt = document.createElement('option');
	                opt.value = value;
	                opt.textContent = label;
	                select.appendChild(opt);
	                console.log(`[DEBUG] User ${index} added to select with value: ${value}, label: ${label}`);
	            });
	            
	            console.log('[DEBUG] Total options in select:', select.options.length);
	            // Restore previous value if it still exists in the new list
	            select.value = current;
	        };

	        render(leadsOwnerSelect);
	        render(dealsOwnerSelect);
	        render(legacyDealsOwnerSelect);
	        render(dashboardOwnerSelect, true);
	        
	        console.log('[DEBUG] populateOwnerFilterSelects complete');
	    },
    
    populateLeadOwnerSelect() {
        const select = document.getElementById('lead-owner-select');
        const prevSelect = document.getElementById('lead-previous-owner-select');
        if(!select && !prevSelect) {
            console.warn('lead owner selects not found');
            return;
        }
        if (select) {
            select.innerHTML = '<option value="">-- انتخاب مسئول از دیدار --</option>';
        }
        if (prevSelect) {
            prevSelect.innerHTML = '<option value="">-- انتخاب مسئول قبلی --</option>';
        }
        
        if (!this.state.didarUsers || this.state.didarUsers.length === 0) {
            console.warn('No Didar users available to populate select');
            if (select) {
                const opt = document.createElement('option');
                opt.value = '';
                opt.textContent = '-- هیچ کاربری یافت نشد --';
                opt.disabled = true;
                select.appendChild(opt);
            }
            if (prevSelect) {
                const opt2 = document.createElement('option');
                opt2.value = '';
                opt2.textContent = '-- هیچ کاربری یافت نشد --';
                opt2.disabled = true;
                prevSelect.appendChild(opt2);
            }
            return;
        }
        
        let addedCount = 0;
        this.state.didarUsers.forEach(user => {
            if(!user.id) {
                console.warn('User without id found:', user);
                return;
            }
            const value = user.owner_id || user.id;
            const label = user.display_name || user.email || user.id;
            if (select) {
                const opt = document.createElement('option');
                opt.value = value;
                opt.textContent = label;
                select.appendChild(opt);
            }
            if (prevSelect) {
                const optPrev = document.createElement('option');
                optPrev.value = value;
                optPrev.textContent = label;
                prevSelect.appendChild(optPrev);
            }
            addedCount++;
        });
        
        console.log(`Populated lead owner select with ${addedCount} users`);
    },
    
        populateSelectOptions(selectId, options, placeholderOverride = null) {
        const select = document.getElementById(selectId);
        if(!select) return;
        const current = select.value;
        let placeholder = placeholderOverride;
        if (!placeholder) {
            const existingPlaceholder = select.querySelector('option[value=""]')?.textContent;
            if (existingPlaceholder) {
                placeholder = existingPlaceholder;
            }
        }
        if (!placeholder) {
            // Get default placeholder based on select ID
            placeholder = '-- انتخاب --';
            if (selectId === 'lead-source') {
                placeholder = 'شیوه آشنایی';
            } else if (selectId === 'lead-content-topic') {
                placeholder = 'موضوع محتوا';
            }
        }
        select.innerHTML = `<option value="">${placeholder}</option>`;
        options.forEach(opt => {
            const option = document.createElement('option');
            option.value = opt.value;
            option.textContent = opt.label ?? opt.title ?? opt.value;
            select.appendChild(option);
        });
        if (current) {
            select.value = current;
        }
    },

    async loadLookups() {
        try {
            const res = await this.req('get_lookups', {});
            if (res.status === 'success' && res.lookups) {
                this.state.lookups = res.lookups || {};
                this.state.lookupsLoaded = true;
                const pipeline = this.getLookupValues('pipeline_stages');
                if (pipeline.length) {
                    this.state.pipelineStages = pipeline;
                }
                await this.loadLookupItems(this.state.lookupSelectedGroup);
            }
        } catch (error) {
            console.warn('Failed to load lookups:', error);
        }
    },

    getLookupItems(groupCode) {
        const group = this.state.lookups?.[groupCode];
        if (group && Array.isArray(group.items) && group.items.length) {
            return group.items;
        }
        if (groupCode === 'lead_sources') return this.leadSources || [];
        if (groupCode === 'content_topics') return this.contentTopics || [];
        return [];
    },

    getLookupOptions(groupCode) {
        const items = this.getLookupItems(groupCode);
        return items.map(item => ({
            value: item.value ?? item.title ?? item.label ?? '',
            label: item.title ?? item.label ?? item.value ?? ''
        })).filter(opt => opt.value !== '' || opt.label !== '');
    },

    getLookupValues(groupCode) {
        return this.getLookupOptions(groupCode).map(opt => opt.value || opt.label);
    },

    getLookupLabel(groupCode, value) {
        if (!value) return '-';
        const items = this.getLookupItems(groupCode);
        for (const item of items) {
            const itemValue = item.value ?? item.title ?? item.label;
            if (itemValue === value) {
                return item.title ?? item.label ?? itemValue;
            }
        }
        return value;
    },

    getLookupChildOptions(groupCode, parentValue) {
        const group = this.state.lookups?.[groupCode];
        if (group && Array.isArray(group.items) && group.items.length) {
            return group.items
                .filter(item => (item.parent_value || '') === parentValue)
                .map(item => ({
                    value: item.value ?? item.title ?? '',
                    label: item.title ?? item.value ?? ''
                }));
        }
        if (groupCode === 'lead_source_details') {
            const fallback = this.leadSourceDetails?.[parentValue] || this.leadSourceDetails?.default || [];
            return fallback.map(item => ({ value: item.value, label: item.label }));
        }
        return [];
    },
    getLookupMeta(groupCode, value) {
        if (!value) return {};
        const items = this.getLookupItems(groupCode) || [];
        for (const item of items) {
            const itemValue = item.value ?? item.code ?? item.title ?? item.label;
            if (itemValue === value) {
                if (!item.meta_json) return {};
                try {
                    const parsed = JSON.parse(item.meta_json);
                    return parsed && typeof parsed === 'object' ? parsed : {};
                } catch (_) {
                    return {};
                }
            }
        }
        return {};
    },
    getFallbackActivityTypes() {
        const items = this.getLookupItems('activity_types_fallback');
        if (!items.length) return [];
        return items.map(item => {
            let meta = {};
            if (item.meta_json) {
                try { meta = JSON.parse(item.meta_json); } catch (_) { meta = {}; }
            }
            return {
                Id: item.value ?? item.title,
                Title: item.title ?? item.value,
                Direction: meta.direction || 'outgoing',
                Category: meta.category || 'other',
                IsDisabled: false
            };
        });
    },

    applyLookupsToStaticSelects() {
        this.populateSelectOptions('lead-source', this.getLookupOptions('lead_sources'));
        this.populateSelectOptions('lead-content-topic', this.getLookupOptions('content_topics'));

        this.renderDealTitleOptions();
        this.populateSelectOptions('deal-status', this.getLookupOptions('deal_statuses'));
        this.populateSelectOptions('deal-pipeline-stage', this.getLookupOptions('pipeline_stages'));
        this.populateSelectOptions('deals-pipeline-filter', [
            { value: 'all', label: 'همه مراحل' },
            ...this.getLookupOptions('pipeline_stages')
        ]);
        this.populateSelectOptions('deals-status-filter', [
            { value: 'all', label: 'همه وضعیت‌ها' },
            ...this.getLookupOptions('deal_statuses')
        ]);
        this.populateSelectOptions('activity-virtual-stage', this.getLookupOptions('activity_stages'));

        this.populateSelectOptions('lead-customer-level', this.getLookupOptions('customer_levels'));
        this.populateSelectOptions('leads-customer-level-filter', this.getLookupOptions('customer_levels'));
        this.populateSelectOptions('lead-activity-financial-level', this.getLookupOptions('financial_levels'));
        this.populateSelectOptions('lead-activity-asset', this.getLookupOptions('asset_ranges'));
        this.populateSelectOptions('lead-activity-income', this.getLookupOptions('income_ranges'));

        this.populateSelectOptions('deal-discount-type', this.getLookupOptions('discount_types'));
        this.populateSelectOptions('deal-discount-occasion', this.getLookupOptions('discount_occasions'));

        this.populateSelectOptions('activity-failure-reason', this.getLookupOptions('failure_reasons'));
        this.populateSelectOptions('deal-failure-reason', this.getLookupOptions('failure_reasons'));
        this.populateSelectOptions('failure-reason', this.getLookupOptions('failure_reasons'));

        this.populateSelectOptions('activity-reminder-hours', this.getLookupOptions('reminder_hours'));
        this.populateSelectOptions('edit-activity-reminder-hours', this.getLookupOptions('reminder_hours'));
        this.populateSelectOptions('activity-reminder-minutes', this.getLookupOptions('reminder_minutes'));
        this.populateSelectOptions('edit-activity-reminder-minutes', this.getLookupOptions('reminder_minutes'));

        this.populateSelectOptions('edit-role-select', this.getLookupOptions('user_roles'));

        const roleOptions = this.getLookupOptions('user_roles');
        document.querySelectorAll('select[name="role"]').forEach(sel => {
            const current = sel.value;
            sel.innerHTML = '<option value="">Select role</option>';
            roleOptions.forEach(opt => {
                const option = document.createElement('option');
                option.value = opt.value;
                option.textContent = opt.label || opt.value;
                sel.appendChild(option);
            });
            if (current) sel.value = current;
        });

        this.handleSourceChange();
    },

    handleSourceChange() {
        const source = document.getElementById('lead-source')?.value || '';
        const detailSelect = document.getElementById('lead-source-detail');
        if(!detailSelect) return;
        let options = this.getLookupChildOptions('lead_source_details', source);
        if (!options.length) {
            options = [{ value: 'general', label: 'عمومی' }];
        }
        detailSelect.innerHTML = '<option value="">جزئیات شیوه آشنایی</option>';
        options.forEach(opt => {
            const el = document.createElement('option');
            el.value = opt.value;
            el.textContent = opt.label;
            detailSelect.appendChild(el);
        });
        detailSelect.disabled = false;
        detailSelect.required = false;
        detailSelect.value = options[0]?.value || '';
    },

    handlePurchaseOtherNumberChange() {
        const configs = [
            {
                select: document.getElementById('activity-has-purchase'),
                field: document.getElementById('previous-purchase-number-field'),
                input: document.querySelector('input[name="previous_purchase_number"]')
            },
            {
                select: document.getElementById('deal-activity-has-purchase'),
                field: document.getElementById('deal-previous-purchase-number-field'),
                input: document.getElementById('deal-activity-previous-number')
            },
            {
                select: document.getElementById('lead-activity-has-purchase'),
                field: document.getElementById('lead-previous-purchase-number-field'),
                input: document.getElementById('lead-activity-previous-number')
            }
        ];
        configs.forEach(cfg => {
            if (!cfg.select || !cfg.field) return;
            const hasPurchase = cfg.select.value;
            if (hasPurchase === '1') {
                cfg.field.classList.remove('hidden');
                if (cfg.input) cfg.input.required = false;
            } else {
                cfg.field.classList.add('hidden');
                if (cfg.input) {
                    cfg.input.required = false;
                    cfg.input.value = '';
                }
            }

            // پس از اولین انتخاب، فیلد قفل شود
            if (hasPurchase !== '' && cfg.select.dataset.locked !== 'true') {
                cfg.select.dataset.locked = 'true';
                cfg.select.disabled = true;
                cfg.select.classList.add('bg-gray-900', 'cursor-not-allowed');
            }
        });
    },
    
    checkIfPurchaseQuestionAnswered(contactId) {
        // بررسی اینکه آیا لید قبلاً به این سوال پاسخ داده (و پاسخ "نامشخص" نباشد)
        if (!contactId || !this.state.currentLeadActivities) return false;
        
        for (let act of this.state.currentLeadActivities) {
            const note = act.result_note || '';
            if ((note.includes('🟢 این لید قبلاً خرید دیگری') || 
                 note.includes('⚪ این لید خرید قبلی با شماره دیگر نداشته است')) &&
                !note.includes('نامشخص')) {
                return true;
            }
        }
        return false;
    },
    
    handleActivityFormVisibility() {
        try {
            const virtualStage = document.getElementById('activity-virtual-stage')?.value;
            const contactId = document.getElementById('activity-contact-id')?.value;
            // The purchase question container has been removed from Activity Modal as per user request.
            // const purchaseField = document.getElementById('activity-has-purchase');
            // const purchaseFieldContainer = document.getElementById('purchase-question-container');

            // No longer checking for these elements in the activity modal context.
            // if (!purchaseField || !purchaseFieldContainer) {
            //     console.warn('Purchase field or container not found');
            //     return;
            // }

            // The logic to handle purchase question visibility is now primarily in openLeadModal and saveLead.

        } catch(error) {
            console.error('Error in handleActivityFormVisibility:', error);
        }
    },

    handleActivityDealSelection() {
        const dealSelect = document.getElementById('activity-deal-select');
        const pipelineContainer = document.getElementById('activity-pipeline-stage-container');
        const pipelineSelect = document.getElementById('activity-virtual-stage');
        const dealId = dealSelect?.value;

        if (!pipelineContainer || !pipelineSelect) return;

        if (dealId) {
            // Deal selected - show and require pipeline stage
            pipelineContainer.style.display = 'block';
            pipelineSelect.required = true;
            // Update label to show it's required
            const label = pipelineContainer.querySelector('label');
            if (label) {
                label.innerHTML = 'مرحله کاریز معامله پس از فعالیت <span class="text-red-500">*</span>';
            }
        } else {
            // No deal selected - hide pipeline stage
            pipelineContainer.style.display = 'none';
            pipelineSelect.required = false;
            pipelineSelect.value = '';
        }
    },
    
    handleActivityStageChange() {
        const stage = document.getElementById('activity-virtual-stage')?.value;
        const extra = document.getElementById('referral-extra');
        const noteInput = document.querySelector('textarea[name="referral_note"]');
        const failureExtra = document.getElementById('failure-reason-extra');
        const failureReason = document.getElementById('activity-failure-reason');
        const failureOtherDesc = document.getElementById('activity-failure-other-description');
        
        const isCrmReferral = stage === 'ارجاع به CRM';
        const isFailed = stage === 'deal_failed' || stage === 'معامله ناموفق' || stage === 'ناموفق';
        const stageMeta = this.getLookupMeta('activity_stages', stage);
        const showFailureFromMeta = stageMeta.show_failure_reason === true || stageMeta.showFailureReason === true;

        // مدیریت بخش ارجاع به CRM
        if(extra) {
            if(isCrmReferral) {
                extra.classList.remove('hidden');
                if (noteInput) noteInput.required = true;
            } else {
                extra.classList.add('hidden');
                if (noteInput) noteInput.required = false;
            }
        }
        
        // مدیریت بخش دلیل ناموفق
        if (failureExtra) {
            const shouldShowFailure = isCrmReferral || isFailed || showFailureFromMeta;
            if (shouldShowFailure) {
                failureExtra.classList.remove('hidden');
                if (failureReason) failureReason.required = true;
                if (failureOtherDesc) {
                    failureOtherDesc.required = true;
                    failureOtherDesc.placeholder = 'لطفاً توضیحات دلیل عدم موفقیت را وارد کنید';
                }
                // نمایش فیلد توضیحات
                const otherDiv = document.getElementById('activity-failure-other-reason');
                if (otherDiv) otherDiv.classList.remove('hidden');
            } else {
                failureExtra.classList.add('hidden');
                if (failureReason) {
                    failureReason.required = false;
                    failureReason.value = '';
                }
                if (failureOtherDesc) {
                    failureOtherDesc.required = false;
                    failureOtherDesc.value = '';
                }
            }
        }
        
        // فراخوانی handleActivityFormVisibility برای به‌روزرسانی وضعیت required فیلد خرید قبلی
        this.handleActivityFormVisibility();
    },
    
    toggleActivityFailureOtherReason() {
        const reason = document.getElementById('activity-failure-reason')?.value;
        const otherDiv = document.getElementById('activity-failure-other-reason');
        const otherDesc = document.getElementById('activity-failure-other-description');
        
        if (!otherDiv || !otherDesc) return;
        
        const virtualStage = document.getElementById('activity-virtual-stage')?.value;
        const isFailed = virtualStage === 'deal_failed' || virtualStage === 'معامله ناموفق';
        const show = isFailed; // در حالت ناموفق همیشه نمایش بده
        otherDiv.classList.toggle('hidden', !show);
        otherDesc.required = show;
        if (!show) {
            otherDesc.value = '';
        } else {
            otherDesc.placeholder = 'لطفاً توضیحات دلیل عدم موفقیت را وارد کنید';
        }
    },
    
	    async openLeadModal(lead = null) {
	        const isEdit = !!lead;
        if (!this.isAdminOrCrmSpecialist() && !isEdit) {
            Swal.fire('خطا', 'فقط مدیر و کارشناس CRM می‌توانند لید جدید ایجاد کنند', 'error');
            return;
        }
        if (isEdit && !this.isAdminOrCrmSpecialist() && !this.isLeadOwner(lead)) {
            Swal.fire('خطا', 'دسترسی به ویرایش این لید را ندارید', 'error');
            return;
        }
        const form = document.getElementById('lead-form');
        if(!form) return;
        const titleEl = document.getElementById('lead-modal-title');
        if (titleEl) {
            titleEl.textContent = isEdit ? 'ویرایش اطلاعات لید' : 'افزودن لید جدید';
        }
	        form.reset();
	        form.dataset.mode = isEdit ? 'edit' : 'create';
	        form.dataset.contactId = lead?.didar_contact_id || '';
	        const deleteBtn = document.getElementById('lead-delete-btn');
	        if (deleteBtn) {
	            const show = isEdit && this.isAdmin();
	            deleteBtn.classList.toggle('hidden', !show);
	            deleteBtn.disabled = !show;
	        }
        this.populateSelectOptions('lead-source', this.getLookupOptions('lead_sources'));
        this.populateSelectOptions('lead-content-topic', this.getLookupOptions('content_topics'));
        this.handleSourceChange();
        const leadActivitySection = document.getElementById('lead-activity-section');
        if (leadActivitySection) {
            leadActivitySection.classList.toggle('hidden', !isEdit);
        }
        
        // Ensure Didar users are loaded before populating the select
        if (this.isAdminOrCrmSpecialist() && (!this.state.didarUsers || this.state.didarUsers.length === 0)) {
            await this.loadDidarUsers();
        } else {
            this.populateLeadOwnerSelect();
        }
        
	        if (isEdit && lead) {
            const fullName = `${lead.first_name || ''} ${lead.last_name || ''}`.trim();
            document.querySelector('#lead-form [name="full_name"]').value = fullName;
            document.querySelector('#lead-form [name="national_id"]').value = lead.national_id || '';
            document.querySelector('#lead-form [name="mobile_phone"]').value = lead.mobile_phone || '';
            document.querySelector('#lead-form [name="secondary_mobile_phone"]').value = lead.secondary_mobile_phone || '';
            const phone3 = document.querySelector('#lead-form [name="mobile_phone_3"]');
            const phone4 = document.querySelector('#lead-form [name="mobile_phone_4"]');
            if (phone3) phone3.value = lead.mobile_phone_3 || '';
            if (phone4) phone4.value = lead.mobile_phone_4 || '';
            document.querySelector('#lead-form [name="city"]').value = lead.city || '';
            document.querySelector('#lead-form [name="job_title"]').value = lead.job_title || '';
            document.querySelector('#lead-form [name="background_info"]').value = lead.background_info || '';
            document.querySelector('#lead-form [name="owner_didar_id"]').value = lead.owner_didar_id || '';
            const prevOwnerSelect = document.getElementById('lead-previous-owner-select');
            if (prevOwnerSelect) prevOwnerSelect.value = lead.previous_owner_id || '';
            document.getElementById('lead-source').value = lead.acquaintance_source || '';
            this.handleSourceChange();
            document.getElementById('lead-source-detail').value = lead.acquaintance_detail || '';
            document.getElementById('lead-content-topic').value = lead.content_topic || '';

            // Activity-style extra fields
            const setVal = (id, val) => { const el = document.getElementById(id); if (el) el.value = val || ''; };
            // setVal('lead-activity-city', lead.city || ''); // Removed (Issue 4)
            // setVal('lead-activity-job', lead.job_title || ''); // Removed (Issue 4)
            setVal('lead-activity-job-desc', lead.job_description || '');
            setVal('lead-activity-extra-info', lead.extra_info || '');
            setVal('lead-activity-financial-level', lead.financial_level || '');
            setVal('lead-activity-asset', lead.asset_estimation || '');
            setVal('lead-activity-income', lead.income_estimation || '');
            const acqVal = lead.acquaintance_duration || '';
            const acqYears = document.getElementById('lead-activity-acq-years');
            const acqMonths = document.getElementById('lead-activity-acq-months');
            const acqLess = document.getElementById('lead-activity-acq-less');
            if (acqLess && acqYears && acqMonths) {
                if (acqVal && acqVal.includes('کمتر')) {
                    acqLess.checked = true;
                    acqYears.value = '0';
                    acqMonths.value = '0';
                } else {
                    const match = acqVal.match(/(\d+)\s*سال.*?(\d+)\s*ماه/);
                    acqYears.value = match ? match[1] : '0';
                    acqMonths.value = match ? match[2] : '0';
                    acqLess.checked = false;
                }
            }
            const purchaseSelect = document.getElementById('lead-activity-has-purchase');
            if (purchaseSelect) {
                if (lead.has_previous_purchase === 0 || lead.has_previous_purchase === 1 || lead.has_previous_purchase === '0' || lead.has_previous_purchase === '1') {
                    purchaseSelect.value = String(lead.has_previous_purchase);
                    // Issue 2: Hide or Lock?
                    // "The question ... should appear only once during first edit."
                    // If it has a value, we can lock it or hide it. 
                    // Let's lock it for now to be consistent with previous logic, 
                    // but the user requirement implies it might not need to be shown if answered.
                    // However, for profile completeness visibility, keeping it locked is better than hiding.
                    purchaseSelect.disabled = true;
                    purchaseSelect.dataset.locked = 'true';
                    purchaseSelect.classList.add('bg-gray-900', 'cursor-not-allowed');
                } else {
                    purchaseSelect.value = '';
                    purchaseSelect.disabled = false;
                    purchaseSelect.dataset.locked = '';
                    purchaseSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
                }
            }
        } else {
            const prevOwnerSelect = document.getElementById('lead-previous-owner-select');
            if (prevOwnerSelect) prevOwnerSelect.value = '';
            ['lead-activity-job-desc','lead-activity-extra-info'].forEach(id => {
                const el = document.getElementById(id);
                if (el) el.value = '';
            });
            ['lead-activity-financial-level','lead-activity-asset','lead-activity-income','lead-activity-has-purchase'].forEach(id => {
                const el = document.getElementById(id);
                if (el) { el.value=''; el.disabled=false; el.dataset.locked=''; el.classList.remove('bg-gray-900','cursor-not-allowed'); }
            });
            const acqYears = document.getElementById('lead-activity-acq-years');
            const acqMonths = document.getElementById('lead-activity-acq-months');
            const acqLess = document.getElementById('lead-activity-acq-less');
            if (acqYears) acqYears.value = '0';
            if (acqMonths) acqMonths.value = '0';
            if (acqLess) acqLess.checked = false;
            const phone3 = document.querySelector('#lead-form [name="mobile_phone_3"]');
            const phone4 = document.querySelector('#lead-form [name="mobile_phone_4"]');
            if (phone3) phone3.value = '';
            if (phone4) phone4.value = '';
        }
        const regDisplay = document.getElementById('lead-register-date-display');
        if (regDisplay) {
            if (isEdit && lead) {
                const regDate = lead.register_time_jalali || (lead.register_time ? toPersianDate(lead.register_time) : '');
                regDisplay.value = regDate || '---';
            } else {
                let today = '';
                if (typeof persianDate !== 'undefined') {
                    try {
                        today = new persianDate().format('YYYY/MM/DD');
                    } catch (e) {}
                }
                regDisplay.value = today || (toPersianDate(new Date().toISOString()) || '');
            }
        }
        
        this.applyLeadFormPermissions(lead, isEdit);
        
        this.openModal('modalLead');
    },

    /**
     * ذخیره فیلدهای سبک فعالیت از مودال لید در دیتابیس داخلی
     */
    async saveLeadActivityExtras(contactId, form) {
        if (!contactId || !form) return;
        const payload = { contact_id: contactId };
        const getVal = name => form.querySelector(`[name="${name}"]`)?.value?.trim() || '';

        // const city = getVal('activity_city'); // Removed (Issue 4)
        // const job = getVal('activity_job_title'); // Removed (Issue 4)
        const jobDesc = getVal('activity_job_desc');
        const extraInfo = getVal('activity_extra_info');
        const finLevel = getVal('activity_financial_level');
        const asset = getVal('activity_asset');
        const income = getVal('activity_income');
        const acqYears = form.querySelector('[name="activity_acq_years"]')?.value || '0';
        const acqMonths = form.querySelector('[name="activity_acq_months"]')?.value || '0';
        const acqLess = form.querySelector('[name="activity_acq_less"]')?.checked;
        const hasPurchaseEl = form.querySelector('[name="activity_has_purchase"]');
        const prevPurchase = getVal('activity_previous_number');

        // if (city) payload.city = city;
        // if (job) payload.job_title = job;
        if (jobDesc) payload.job_description = jobDesc;
        if (extraInfo) payload.extra_info = extraInfo;
        if (finLevel) payload.financial_level = finLevel;
        if (asset) payload.asset_estimation = asset;
        if (income) payload.income_estimation = income;

        if (acqLess) {
            payload.acquaintance_duration = 'کمتر از 1 ماه';
        } else if (acqYears || acqMonths) {
            payload.acquaintance_duration = `${acqYears || '0'} سال ${acqMonths || '0'} ماه`;
        }

        if (hasPurchaseEl && hasPurchaseEl.dataset.locked !== 'true' && hasPurchaseEl.value !== '') {
            payload.has_previous_purchase = hasPurchaseEl.value;
        } else if (hasPurchaseEl && hasPurchaseEl.dataset.locked === 'true') {
            payload.has_previous_purchase = hasPurchaseEl.value;
        }
        if (prevPurchase) {
            payload.previous_purchase_number = prevPurchase.replace(/\D/g, '').slice(-10);
        }

        if (Object.keys(payload).length <= 1) return;

        try {
            await this.req('save_deal_info', payload);
        } catch (e) {
            console.warn('saveLeadActivityExtras failed', e);
        }
    },
    
    applyLeadFormPermissions(lead, isEdit) {
        const form = document.getElementById('lead-form');
        if(!form) return;
        const isAdmin = this.isAdmin();
        const isCrmSpecialist = this.state.user?.role === 'crm_specialist';
        const isAdminOrCrm = this.isAdminOrCrmSpecialist();
        const isOwner = isEdit && this.isLeadOwner(lead);
        
        const primary = form.querySelector('[name="mobile_phone"]');
        const secondary = form.querySelector('[name="secondary_mobile_phone"]');
        const phone3 = form.querySelector('[name="mobile_phone_3"]');
        const phone4 = form.querySelector('[name="mobile_phone_4"]');
        const ownerSelect = form.querySelector('[name="owner_didar_id"]');
        const prevOwnerSelect = form.querySelector('[name="previous_owner_id"]');
        const sourceSelect = document.getElementById('lead-source');
        const detailSelect = document.getElementById('lead-source-detail');
        const topicSelect = document.getElementById('lead-content-topic');
        const adminFieldsDiv = document.getElementById('lead-admin-fields');
        const phoneFieldsDiv = primary?.closest('.grid');
        const phoneExtraDiv = phone3?.closest('.grid');
        
        const disable = (el, state) => { if(el) el.disabled = state; };
        const setRequired = (el, state) => { if(el) el.required = state; };
        const hide = (el, state) => { if(el) el.style.display = state ? 'none' : ''; };
        const lockIfFilled = (el, value) => {
            if (!el) return;
            const hasVal = value && String(value).trim() !== '';
            if (!isAdmin && hasVal) {
                if ('readOnly' in el) el.readOnly = true;
                if ('disabled' in el) el.disabled = true;
                el.classList.add('bg-gray-900','cursor-not-allowed');
            } else if (!isAdmin) {
                if ('readOnly' in el) el.readOnly = false;
                if ('disabled' in el) el.disabled = false;
                el.classList.remove('bg-gray-900','cursor-not-allowed');
            } else {
                // ادمین آزاد است؛ اما اگر مقدار دارد، قفل نرم
                if ('readOnly' in el) el.readOnly = false;
                if ('disabled' in el) el.disabled = false;
                el.classList.remove('bg-gray-900','cursor-not-allowed');
            }
        };
        
        // Set all admin fields as required
        setRequired(ownerSelect, true);
        setRequired(primary, true);
        // فیلدهای آشنایی: اختیاری هستند
        setRequired(sourceSelect, false);
        setRequired(detailSelect, false);
        setRequired(topicSelect, false);
        if (prevOwnerSelect) prevOwnerSelect.required = false;

        // Admin یا CRM Specialist: می‌تواند همه را ویرایش کند
        if (isAdminOrCrm) {
            hide(phoneFieldsDiv, false);
            hide(phoneExtraDiv, false);
            if (adminFieldsDiv) adminFieldsDiv.style.display = '';
            return;
        }

        // Agent (کاربر عادی): فیلدهای admin پنهان می‌شوند
        if (adminFieldsDiv) adminFieldsDiv.style.display = 'none';
        hide(phoneFieldsDiv, false);
        hide(phoneExtraDiv, false);
        
        // فیلدهای آشنایی در حالت edit غیرالزامی هستند
        if (isEdit) {
            setRequired(sourceSelect, false);
            setRequired(detailSelect, false);
            setRequired(topicSelect, false);
        }
        
        lockIfFilled(primary, lead?.mobile_phone);
        lockIfFilled(secondary, lead?.secondary_mobile_phone);
        lockIfFilled(phone3, lead?.mobile_phone_3);
        lockIfFilled(phone4, lead?.mobile_phone_4);
        lockIfFilled(form.querySelector('[name="national_id"]'), lead?.national_id);
        lockIfFilled(form.querySelector('[name="city"]'), lead?.city);
        lockIfFilled(form.querySelector('[name="job_title"]'), lead?.job_title);
        lockIfFilled(form.querySelector('[name="background_info"]'), lead?.background_info);

        // Prevent changing ownership/previous owner
        disable(ownerSelect, true);
        disable(prevOwnerSelect, true);
    },
    
    editLeadInfo() {
        if(!this.state.currentLead) return;
        
        // CRM specialist is view-only
        const isAdmin = window.APP_USER?.role === 'admin';
        const isCrm = window.APP_USER?.role === 'crm_specialist';
        
        if (isCrm) {
            Swal.fire('خطا', 'کارشناس CRM فقط مجاز به مشاهده است', 'error');
            return;
        }
        
        if (!isAdmin && !this.isLeadOwner(this.state.currentLead)) {
            Swal.fire('خطا', 'دسترسی به ویرایش این لید را ندارید', 'error');
            return;
        }
        
        this.openLeadModal(this.state.currentLead);
    },

    async saveLead(e) {
        e.preventDefault();
        const form = e.target;
        const mode = form.dataset.mode || 'create';
        const data = Object.fromEntries(new FormData(form).entries());
        
        // اعتبارسنجی فیلدها
        const fullName = data.full_name || '';
        const nationalId = data.national_id || '';
        const mobilePhone = data.mobile_phone || '';
        const secondaryMobilePhone = data.secondary_mobile_phone || '';
        const mobilePhone3 = data.mobile_phone_3 || '';
        const mobilePhone4 = data.mobile_phone_4 || '';
        const city = data.city || '';
        const jobTitle = data.job_title || '';
        
        // اعتبارسنجی نام و نام خانوادگی
        if (fullName.trim()) {
            const nameValidation = validatePersianText(fullName, 'نام و نام خانوادگی');
            if (!nameValidation.valid) {
                Swal.fire('خطا', nameValidation.message, 'error');
                return;
            }
        }
        
        // اعتبارسنجی کد ملی
        if (nationalId.trim()) {
            const nationalIdValidation = validateNationalId(nationalId);
            if (!nationalIdValidation.valid) {
                Swal.fire('خطا', nationalIdValidation.message, 'error');
                return;
            }
            // فقط اعداد را ذخیره کن
            data.national_id = nationalId.trim().replace(/\D/g, '');
        }
        
        // اعتبارسنجی شماره تماس اصلی
        if (mobilePhone.trim()) {
            const mobileValidation = validatePhone(mobilePhone, 'شماره تماس اصلی');
            if (!mobileValidation.valid) {
                Swal.fire('خطا', mobileValidation.message, 'error');
                return;
            }
            // فقط اعداد را ذخیره کن
            data.mobile_phone = mobilePhone.trim().replace(/\D/g, '');
        }
        
        // اعتبارسنجی شماره تماس دوم
        if (secondaryMobilePhone.trim()) {
            const secondaryValidation = validatePhone(secondaryMobilePhone, 'شماره تماس دوم');
            if (!secondaryValidation.valid) {
                Swal.fire('خطا', secondaryValidation.message, 'error');
                return;
            }
            // فقط اعداد را ذخیره کن
            data.secondary_mobile_phone = secondaryMobilePhone.trim().replace(/\D/g, '');
        }
        if (mobilePhone3.trim()) {
            const mobile3Validation = validatePhone(mobilePhone3, 'شماره تماس سوم');
            if (!mobile3Validation.valid) {
                Swal.fire('خطا', mobile3Validation.message, 'error');
                return;
            }
            data.mobile_phone_3 = mobilePhone3.trim().replace(/\D/g, '');
        }
        if (mobilePhone4.trim()) {
            const mobile4Validation = validatePhone(mobilePhone4, 'شماره تماس چهارم');
            if (!mobile4Validation.valid) {
                Swal.fire('خطا', mobile4Validation.message, 'error');
                return;
            }
            data.mobile_phone_4 = mobilePhone4.trim().replace(/\D/g, '');
        }

        // بررسی تکراری بودن شماره‌های تلفن
        const phoneNumbers = [];
        const phoneLabels = [];

        if (data.mobile_phone) {
            phoneNumbers.push(data.mobile_phone);
            phoneLabels.push('شماره تماس اصلی');
        }
        if (data.secondary_mobile_phone) {
            phoneNumbers.push(data.secondary_mobile_phone);
            phoneLabels.push('شماره تماس دوم');
        }
        if (data.mobile_phone_3) {
            phoneNumbers.push(data.mobile_phone_3);
            phoneLabels.push('شماره تماس سوم');
        }
        if (data.mobile_phone_4) {
            phoneNumbers.push(data.mobile_phone_4);
            phoneLabels.push('شماره تماس چهارم');
        }

        // بررسی تکراری بودن شماره‌ها
        for (let i = 0; i < phoneNumbers.length; i++) {
            for (let j = i + 1; j < phoneNumbers.length; j++) {
                if (phoneNumbers[i] === phoneNumbers[j]) {
                    Swal.fire('خطا', `شماره ${phoneLabels[i]} با ${phoneLabels[j]} تکراری است! لطفاً شماره‌های منحصر به فرد وارد کنید.`, 'error');
                    return;
                }
            }
        }

        // اعتبارسنجی شهر
        if (city.trim()) {
            const cityValidation = validatePersianText(city, 'شهر');
            if (!cityValidation.valid) {
                Swal.fire('خطا', cityValidation.message, 'error');
                return;
            }
        }
        
        // اعتبارسنجی شغل
        if (jobTitle.trim()) {
            const jobValidation = validatePersianText(jobTitle, 'شغل');
            if (!jobValidation.valid) {
                Swal.fire('خطا', jobValidation.message, 'error');
                return;
            }
        }
        
        if (mode === 'edit') {
            data.contact_id = form.dataset.contactId;
            if (!data.contact_id) {
                Swal.fire('خطا', 'شناسه لید نامعتبر است', 'error');
                return;
            }
            if (!this.isAdminOrCrmSpecialist() && (!this.state.currentLead || !this.isLeadOwner(this.state.currentLead))) {
                Swal.fire('خطا', 'دسترسی به ویرایش این لید را ندارید', 'error');
                return;
            }

            // Issue 5: Mandatory fields on First Edit
            // Check if name is generic (New Lead)
            const currentName = ((this.state.currentLead?.first_name || '') + ' ' + (this.state.currentLead?.last_name || '')).trim();
            const isFirstEdit = currentName === 'لید جدید' || currentName === 'New Lead';
            
            if (isFirstEdit) {
                if (!fullName || fullName === 'لید جدید' || fullName === 'New Lead') {
                    Swal.fire('خطا', 'لطفاً نام و نام خانوادگی لید را وارد کنید.', 'error');
                    return;
                }
                
                const hasPurchaseSelect = form.querySelector('[name="activity_has_purchase"]');
                if (hasPurchaseSelect && !hasPurchaseSelect.value) {
                     Swal.fire('خطا', 'لطفاً وضعیت خرید قبلی با شماره دیگر را مشخص کنید.', 'error');
                     return;
                }
            }
        } else if (!this.isAdminOrCrmSpecialist()) {
            Swal.fire('خطا', 'فقط مدیر و کارشناس CRM می‌توانند لید جدید اضافه کنند', 'error');
            return;
        }

        // Collect Additional Fields (Issue 3)
        // job_description
        const jobDesc = form.querySelector('[name="activity_job_desc"]')?.value;
        if (jobDesc) data.job_description = jobDesc;
        
        // extra_info
        const extraInfo = form.querySelector('[name="activity_extra_info"]')?.value;
        if (extraInfo) data.extra_info = extraInfo;
        
        // financial_level
        const finLevel = form.querySelector('[name="activity_financial_level"]')?.value;
        if (finLevel) data.financial_level = finLevel;
        
        // asset_estimation
        const assetEst = form.querySelector('[name="activity_asset"]')?.value;
        if (assetEst) data.asset_estimation = assetEst;
        
        // income_estimation
        const incomeEst = form.querySelector('[name="activity_income"]')?.value;
        if (incomeEst) data.income_estimation = incomeEst;
        
        // customer_level و customer_level_notes
        const customerLevelEl = form.querySelector('[name="customer_level"]');
        const customerLevelNotesEl = form.querySelector('[name="customer_level_notes"]');
        const customerLevel = customerLevelEl?.value?.trim() || '';
        const customerLevelNotes = customerLevelNotesEl?.value?.trim() || '';
        
        if (customerLevel) {
            data.customer_level = customerLevel;
            // اگر سطح مشتری انتخاب شده باشد، توضیحات الزامی است
            if (!customerLevelNotes) {
                Swal.fire('خطا', 'لطفاً توضیحات سطح مشتری را وارد کنید', 'error');
                return;
            }
            data.customer_level_notes = customerLevelNotes;
        }
        
        // acquaintance_duration
        const acqYears = form.querySelector('[name="activity_acq_years"]')?.value || '0';
        const acqMonths = form.querySelector('[name="activity_acq_months"]')?.value || '0';
        const acqLess = form.querySelector('[name="activity_acq_less"]')?.checked;
        
        if (acqLess) {
            data.acquaintance_duration = 'کمتر از 1 ماه';
        } else if (acqYears !== '0' || acqMonths !== '0') {
            data.acquaintance_duration = `${acqYears} سال ${acqMonths} ماه`;
        }
        
        // has_previous_purchase
        const hasPurchaseEl = form.querySelector('[name="activity_has_purchase"]');
        if (hasPurchaseEl && hasPurchaseEl.value !== '') {
            data.has_previous_purchase = hasPurchaseEl.value;
        }
        
        // previous_purchase_number
        const prevPurchaseNum = form.querySelector('[name="activity_previous_number"]')?.value;
        if (prevPurchaseNum) {
            data.previous_purchase_number = prevPurchaseNum.replace(/\D/g, '');
        }
        
        // ثبت تاریخ جلالی/میلادی برای مرتب‌سازی
        if (typeof persianDate !== 'undefined') {
            try {
                const p = new persianDate();
                const jalaliStr = p.format('YYYY/MM/DD');
                data.register_time_jalali = jalaliStr;
                const greg = persianToGregorian(jalaliStr);
                if (greg) {
                    data.register_time = greg;
                }
            } catch (e) {
                console.warn('Failed to set jalali date for lead', e);
            }
        }
        
        // Backend will automatically check for duplicate phone and create activity if needed
        // No need to check here - backend handles it gracefully
        
        const res = await this.req(mode === 'edit' ? 'update_lead' : 'create_lead', data);
        if(res.status === 'success') {
            // Duplicate flow
            if (res.is_duplicate) {
                console.log('Duplicate response:', res); // Debug log
                const matches = Array.isArray(res.matches) ? res.matches : [];
                // اولویت: استفاده از owner_name و owner_id از پاسخ backend
                const currentOwnerName = res.owner_name || 'نامشخص';
                const currentOwnerId = res.owner_id || '';
                const contactName = res.contact_name || '---';
                const matchedPhone = res.matched_phone || (data.mobile_phone || '---');
                const existingDeals = Array.isArray(res.deal_ids) ? res.deal_ids.length : 0;
                const nextDealIndex = existingDeals + 1;

                const result = await Swal.fire({
                    icon: 'warning',
                    title: 'شماره تکراری',
                    html: `
                        <div class="text-right">
                            <div class="mb-2"><strong>لید:</strong> ${contactName}</div>
                            <div class="mb-2"><strong>شماره تکراری:</strong> ${matchedPhone}</div>
                            <div class="mb-2"><strong>مسئول فعلی:</strong> ${currentOwnerName || res.owner_id}</div>
                            ${matches.length ? `
                              <div class="mt-2 text-sm">
                                ${matches.map(m => `<div>شماره ${m.phone} → ${m.contact_name || '---'} (${m.owner_name || m.owner_id || '---'})</div>`).join('')}
                              </div>` : ''}
                            <div class="mt-3 text-sm text-gray-300">مسئول قابل تغییر نیست؛ معامله جدید برای همین مسئول ثبت می‌شود.</div>
                            <div class="mt-2 text-sm">عنوان معامله بعدی بر اساس شماره اصلی ساخته می‌شود.</div>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: 'ثبت معامله جدید',
                    cancelButtonText: 'انصراف',
                    confirmButtonColor: '#3085d6',
                    cancelButtonColor: '#d33',
                    focusConfirm: false
                });

                if (result.isConfirmed) {
                    // فقط برای همان مسئول فعلی معامله جدید بساز؛ عنوان را از سرور بگیر
                    const targetContactId = res.didar_contact_id || res.contact_id;
                    let dealTitle = '';
                    try {
                        const titleRes = await this.req('generate_deal_title', {contact_id: targetContactId});
                        if (titleRes?.status === 'success' && titleRes.title) {
                            dealTitle = titleRes.title;
                        }
                    } catch (err) {
                        console.error('generate_deal_title failed:', err);
                    }
                    // fallback در صورت بروز خطا
                    if (!dealTitle) {
                        const ordinals = ['اول', 'دوم', 'سوم', 'چهارم', 'پنجم', 'ششم', 'هفتم', 'هشتم', 'نهم', 'دهم'];
                        const idx = nextDealIndex - 1;
                        const ordinal = ordinals[idx] || `${nextDealIndex}م`;
                        dealTitle = `معامله ${ordinal} ${res.matched_phone || ''}`.trim();
                    }
                    if (!targetContactId || !currentOwnerId) {
                        Swal.fire('خطا', 'اطلاعات لید تکراری یا مسئول یافت نشد', 'error');
                        return;
                    }

                    const dealRes = await this.req('save_deal', {
                        contact_id: targetContactId,
                        title: dealTitle,
                        pipeline_stage: 'معامله جدید',
                        deal_status: 'Pending'
                    });
                    if (dealRes.status !== 'success') {
                        Swal.fire('خطا', dealRes.message || 'خطا در ثبت معامله', 'error');
                        return;
                    }

                    Swal.fire({
                        icon: 'success',
                        title: 'ثبت شد',
                        html: `این شماره قبلاً ثبت شده بود. معامله جدید با عنوان "${dealTitle}" برای مسئول فعلی ثبت شد.`,
                        confirmButtonText: 'باشه'
                    });

                    this.closeModal('modalLead');
                    form.reset();
                    form.dataset.contactId = '';
                    form.dataset.mode = 'create';
                    this.loadLeads();

                    if (targetContactId) {
                        try {
                            this.openPersonProfile(targetContactId);
                        } catch (e) {
                            console.warn('Failed to load duplicate lead profile', e);
                        }
                    }
                } else {
                    Swal.fire({
                        icon: 'info',
                        title: 'لغو شد',
                        text: 'ثبت لید جدید انجام نشد',
                        timer: 2000,
                        showConfirmButton: false
                    });
                }

                return;
            }

            // Non-duplicate flow
            // Note: `saveLeadActivityExtras` call removed because `update_lead` API now handles
            // saving additional fields (job_desc, extra_info, etc.) directly.
            // Refresh current profile and leads list without needing manual refresh
            if (this.state.currentLead && this.state.currentLead.didar_contact_id === data.contact_id) {
                await this.openPersonProfile(data.contact_id);
            }
            // Always refresh leads list to show ownership changes
            this.loadLeads();

            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:res.message || 'انجام شد', showConfirmButton:false, timer:2000});
            this.closeModal('modalLead');
            form.reset();
            form.dataset.contactId = '';
            form.dataset.mode = 'create';
            
            if (mode === 'edit' && res.person) {
                this.state.currentLead = res.person;
                this.openPersonProfile(res.person.didar_contact_id);
            }

            // Hide duplicate warning if shown
            const warningDiv = document.getElementById('duplicate-phone-warning');
            if (warningDiv) {
                warningDiv.classList.add('hidden');
            }
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    async createNewRequestActivity(contactId = null) {
        // Function functionality pending implementation or restoration
        console.warn('createNewRequestActivity called but logic is missing.');
        if (!contactId) {
             const warningDiv = document.getElementById('duplicate-phone-warning');
             if (warningDiv && !warningDiv.classList.contains('hidden')) {
                 // Try to find contact ID from warning context if needed
             }
        }
        // Placeholder
    },
    
    // LOAD LEADS (restored)
	    async loadLeads(page = 1) {
	        try {
	            const search = document.getElementById('leads-search')?.value || '';
	            const saleStatus = document.getElementById('leads-stage-filter')?.value || 'all';
	            const ownerId = document.getElementById('leads-owner-filter')?.value || '';
	            const productRaw = document.getElementById('leads-product-filter')?.value || '';
	            const customerLevel = document.getElementById('leads-customer-level-filter')?.value || '';
	            const registerFromRaw = document.getElementById('leads-register-from')?.value || '';
	            const registerToRaw = document.getElementById('leads-register-to')?.value || '';
	            const activityFromRaw = document.getElementById('leads-activity-from')?.value || '';
	            const activityToRaw = document.getElementById('leads-activity-to')?.value || '';

	            const payload = {
	                search,
	                sale_status: saleStatus,
	                bucket: 'all',
	                page,
	                per_page: this.state.leadsPagination?.perPage || 20
	            };
	            if (ownerId) {
	                payload.owner_id = ownerId;
	            }
	            if (customerLevel) {
	                payload.customer_level = customerLevel;
	            }

                const registerFrom = normalizeDate(registerFromRaw);
                const registerTo = normalizeDate(registerToRaw);
                const activityFrom = normalizeDate(activityFromRaw);
                const activityTo = normalizeDate(activityToRaw);
                if (registerFrom) payload.register_date_from = registerFrom;
                if (registerTo) payload.register_date_to = registerTo;
                if (registerFromRaw) payload.register_date_from_jalali = toEnglishDigits(registerFromRaw.trim());
                if (registerToRaw) payload.register_date_to_jalali = toEnglishDigits(registerToRaw.trim());
                if (activityFrom) payload.activity_date_from = activityFrom;
                if (activityTo) payload.activity_date_to = activityTo;
                if (productRaw) {
                    if (/^\d+$/.test(productRaw)) payload.product_id = productRaw;
                    else payload.product_name = productRaw;
                }

                const res = await this.req('get_leads', payload);
                if (res.status === 'success') {
                    this.state.leads = res.leads || [];
                    if (res.pagination) {
                        this.state.leadsPagination = {
                            currentPage: res.pagination.current_page,
                            perPage: res.pagination.per_page,
                            totalCount: res.pagination.total_count,
                            totalPages: res.pagination.total_pages
                        };
                    }
                    this.renderLeads();
                    this.renderLeadsPagination();
                } else {
                    console.error('Error loading leads:', res.message || 'Unknown error');
                    this.state.leads = [];
                    const container = document.getElementById('leads-list');
                    if (container) {
                        container.innerHTML = `<div class="text-center py-8 text-red-500">خطا در بارگذاری لیدها: ${this.escapeHtml(res.message || 'خطای نامشخص')}</div>`;
                    }
                }
        } catch (error) {
            console.error('Exception loading leads:', error);
            this.state.leads = [];
            const container = document.getElementById('leads-list');
            if (container) {
                container.innerHTML = `<div class="text-center py-8 text-red-500">خطا در بارگذاری لیدها: ${this.escapeHtml(error.message || 'خطای اتصال')}</div>`;
            }
        }
    },

    async loadBlataklifLeads(page = 1) {
        const search = document.getElementById('blataklif-leads-search')?.value || '';
        const saleStatus = document.getElementById('blataklif-leads-sale-status-filter')?.value || 'all';

        const payload = {
            search,
            sale_status: saleStatus,
            bucket: 'blataklif',
            page,
            per_page: this.state.blataklifLeadsPagination?.perPage || 50
        };

        const res = await this.req('get_leads', payload);
        if (res.status === 'success') {
            this.state.blataklifLeads = res.leads || [];
            if (res.pagination) {
                this.state.blataklifLeadsPagination = {
                    currentPage: res.pagination.current_page,
                    perPage: res.pagination.per_page,
                    totalCount: res.pagination.total_count,
                    totalPages: res.pagination.total_pages
                };
            }
            this.renderBlataklifLeads();
            this.renderBlataklifLeadsPagination();
        }
    },

    renderBlataklifLeads() {
        const container = document.getElementById('blataklif-leads-list');
        if (!container) return;

        const list = Array.isArray(this.state.blataklifLeads) ? this.state.blataklifLeads : [];
        if (list.length === 0) {
            container.innerHTML = '<div class="text-center text-gray-500 py-8">لیدی یافت نشد</div>';
            return;
        }

        container.innerHTML = list.map(lead => {
            const statusKey = lead.sale_status === 1 ? 'success' : (lead.sale_status === 0 ? 'failed' : 'unknown');
            const statusMeta = this.state.saleStatusMap[statusKey] || this.state.saleStatusMap.unknown;
            const regDate = lead.register_time_jalali || (lead.register_time ? toPersianDate(lead.register_time) : '');
            const products = this.escapeHtml(lead.customer_products || '-') || '-';

            return `
                <div class="glass p-4 hover:bg-gray-800/50 transition cursor-pointer" onclick="app.openPersonProfile('${lead.didar_contact_id || lead.id || ''}')">
                    <div class="flex justify-between items-start mb-2">
                        <div>
                            <div class="font-bold text-white text-sm">${lead.first_name || ''} ${lead.last_name || ''}</div>
                            <div class="text-xs text-gray-400 mt-1">${lead.mobile_phone || '-'} ${lead.email ? '| ' + lead.email : ''}</div>
                            <div class="text-[11px] text-gray-500 mt-1">مسئول: ${lead.owner_name || '---'}</div>
                        </div>
                        <div class="flex flex-col items-end gap-1">
                            <span class="text-xs px-2 py-1 rounded ${statusMeta.bg} text-white">${statusMeta.icon} ${statusMeta.title}</span>
                        </div>
                    </div>
                    <div class="text-[11px] text-gray-400 mt-1">محصولات: ${products}</div>
                    <div class="flex justify-between items-center mt-2 text-xs text-gray-500">
                        <span>تاریخ ثبت: ${regDate || '-'}</span>
                        <span>${statusMeta.title}</span>
                    </div>
                </div>
            `;
        }).join('');
    },

    renderBlataklifLeadsPagination() {
        const container = document.getElementById('blataklif-leads-pagination');
        if (!container) return;

        const p = this.state.blataklifLeadsPagination || { currentPage: 1, perPage: 50, totalCount: 0, totalPages: 0 };
        if (p.totalPages <= 1) {
            container.innerHTML = '';
            return;
        }

        const maxVisiblePages = 5;
        let startPage = Math.max(1, p.currentPage - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(p.totalPages, startPage + maxVisiblePages - 1);
        if (endPage - startPage + 1 < maxVisiblePages) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        let paginationHTML = '<div class="flex justify-center items-center gap-2 mt-4">';

        if (p.currentPage > 1) {
            paginationHTML += `<button onclick="app.loadBlataklifLeads(${p.currentPage - 1})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">قبلی</button>`;
        } else {
            paginationHTML += `<button disabled class="px-3 py-1 bg-gray-800 text-gray-600 rounded text-sm cursor-not-allowed">قبلی</button>`;
        }

        if (startPage > 1) {
            paginationHTML += `<button onclick="app.loadBlataklifLeads(1)" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">1</button>`;
            if (startPage > 2) paginationHTML += '<span class="text-gray-500">...</span>';
        }

        for (let i = startPage; i <= endPage; i++) {
            const activeClass = i === p.currentPage ? 'bg-yellow-600 text-white' : 'bg-gray-700 hover:bg-gray-600 text-white';
            paginationHTML += `<button onclick="app.loadBlataklifLeads(${i})" class="px-3 py-1 ${activeClass} rounded text-sm transition">${i}</button>`;
        }

        if (endPage < p.totalPages) {
            if (endPage < p.totalPages - 1) paginationHTML += '<span class="text-gray-500">...</span>';
            paginationHTML += `<button onclick="app.loadBlataklifLeads(${p.totalPages})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">${p.totalPages}</button>`;
        }

        if (p.currentPage < p.totalPages) {
            paginationHTML += `<button onclick="app.loadBlataklifLeads(${p.currentPage + 1})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">بعدی</button>`;
        } else {
            paginationHTML += `<button disabled class="px-3 py-1 bg-gray-800 text-gray-600 rounded text-sm cursor-not-allowed">بعدی</button>`;
        }

        paginationHTML += `<span class="text-xs text-gray-400 ml-4">نمایش ${(p.currentPage - 1) * p.perPage + 1} تا ${Math.min(p.currentPage * p.perPage, p.totalCount)} از ${p.totalCount}</span>`;
        paginationHTML += '</div>';
        container.innerHTML = paginationHTML;
    },

	    renderLeads() {
	        const container = document.getElementById('leads-list');
	        if(!container) return;
        
        if(this.state.leads.length === 0) {
            const searchValue = document.getElementById('leads-search')?.value || '';
            
            if (searchValue.trim()) {
                container.innerHTML = `
                    <div class="text-center py-12">
                        <div class="mb-4">
                            <i class="fa-solid fa-search text-4xl text-gray-600"></i>
                        </div>
                        <p class="text-gray-400 font-semibold text-lg">لیدی با این مشخصات یافت نشد</p>
                        <p class="text-gray-500 text-sm mt-3">جستجو: <span class="text-yellow-500 font-mono">${this.escapeHtml(searchValue)}</span></p>
                        <p class="text-gray-600 text-xs mt-4">
                            💡 احتمالاً این لید برای مسئول دیگری است یا هنوز ثبت نشده است
                        </p>
                        <button onclick="document.getElementById('leads-search').value = ''; app.loadLeads();" class="mt-4 px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm rounded transition">
                            پاک کردن جستجو
                        </button>
                    </div>
                `;
            } else {
                container.innerHTML = '<div class="text-center text-gray-500 py-8">لیدی یافت نشد</div>';
            }
            return;
        }
        
	        container.innerHTML = this.state.leads.map(lead => {
	            const statusKey = lead.sale_status === 1 ? 'success' : (lead.sale_status === 0 ? 'failed' : 'unknown');
	            const statusMeta = this.state.saleStatusMap[statusKey] || this.state.saleStatusMap.unknown;
	            const regDate = lead.register_time_jalali || (lead.register_time ? toPersianDate(lead.register_time) : '');
	            const products = this.escapeHtml(lead.customer_products || '-') || '-';
	            
	            return `
	                <div class="glass p-4 hover:bg-gray-800/50 transition cursor-pointer" onclick="app.openPersonProfile('${lead.didar_contact_id || lead.id || ''}')">
	                    <div class="flex justify-between items-start mb-2">
	                        <div>
                            <div class="font-bold text-white text-sm">${lead.first_name || ''} ${lead.last_name || ''}</div>
                            <div class="text-xs text-gray-400 mt-1">${lead.mobile_phone || '-'} ${lead.email ? '| ' + lead.email : ''}</div>
                            <div class="text-[11px] text-gray-500 mt-1">مسئول: ${lead.owner_name || '---'}</div>
	                        </div>
	                        <div class="flex flex-col items-end gap-1">
	                            <span class="text-xs px-2 py-1 rounded ${statusMeta.bg} text-white">${statusMeta.icon} ${statusMeta.title}</span>
	                        </div>
	                    </div>
	                    <div class="text-[11px] text-gray-400 mt-1">محصولات: ${products}</div>
	                    <div class="flex justify-between items-center mt-2 text-xs text-gray-500">
	                        <span>تاریخ ثبت: ${regDate || '-'}</span>
                        <span>${statusMeta.title}</span>
                    </div>
                </div>
            `;
        }).join('');
    },

    // DELETE LEAD (admin only)
	    async deleteLead(contactId) {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند لید را حذف کند', 'error');
            return;
        }

        if (!contactId) {
            Swal.fire('خطا', 'شناسه لید مشخص نشده است', 'error');
            return;
        }

        const result = await Swal.fire({
            title: 'آیا مطمئن هستید؟',
            text: 'این لید و اطلاعات مرتبط (فعالیت‌ها و معاملات محلی) حذف خواهد شد',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'بله، حذف کن',
            cancelButtonText: 'لغو',
            reverseButtons: true
        });

        if (!result.isConfirmed) {
            return;
        }

        Swal.fire({
            title: 'در حال حذف...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await this.req('delete_lead', { contact_id: contactId });
            Swal.close();
            if (res.status === 'success') {
                Swal.fire('موفق', res.message || 'لید با موفقیت حذف شد', 'success');
                // Refresh leads list
                if (document.getElementById('leads-list')) {
                    this.loadLeads(1);
                }
                // Refresh deals list if open
                if (document.getElementById('deals-container')) {
                    this.loadDeals();
                }
                // Close profile modal if this lead is open
                if (window.currentContactId && window.currentContactId === contactId) {
                    this.closeModal('modalProfile');
                    window.currentContactId = null;
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در حذف لید', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در حذف لید: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    renderLeadsPagination() {
        const container = document.getElementById('leads-pagination');
        if(!container) return;
        
        const p = this.state.leadsPagination;
        
        // Calculate page range to show
        const maxVisiblePages = 5;
        let startPage = Math.max(1, p.currentPage - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(p.totalPages, startPage + maxVisiblePages - 1);
        
        // Adjust if we're near the end
        if (endPage - startPage < maxVisiblePages - 1) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }
        // Fix for negative startPage if totalPages is small
        startPage = Math.max(1, startPage);
        
        let paginationHTML = '<div class="flex items-center justify-center gap-2 mt-4 flex-wrap">';
        
        // Info text
        const from = p.totalCount > 0 ? (p.currentPage - 1) * p.perPage + 1 : 0;
        const to = Math.min(p.currentPage * p.perPage, p.totalCount);
        paginationHTML += `<span class="text-xs text-gray-400 ml-4 order-last md:order-first w-full md:w-auto text-center">نمایش ${from} تا ${to} از ${p.totalCount}</span>`;
        
        // Previous button
        if (p.currentPage > 1) {
            paginationHTML += `<button onclick="app.loadLeads(${p.currentPage - 1})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">قبلی</button>`;
        } else {
            paginationHTML += `<button disabled class="px-3 py-1 bg-gray-800 text-gray-600 rounded text-sm cursor-not-allowed">قبلی</button>`;
        }
        
        // First page
        if (startPage > 1) {
            paginationHTML += `<button onclick="app.loadLeads(1)" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">1</button>`;
            if (startPage > 2) {
                paginationHTML += `<span class="text-gray-500">...</span>`;
            }
        }
        
        // Page numbers
        for (let i = startPage; i <= endPage; i++) {
            if (i === p.currentPage) {
                paginationHTML += `<button class="px-3 py-1 bg-blue-600 text-white rounded text-sm font-bold">${i}</button>`;
            } else {
                paginationHTML += `<button onclick="app.loadLeads(${i})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">${i}</button>`;
            }
        }
        
        // Last page
        if (endPage < p.totalPages) {
            if (endPage < p.totalPages - 1) {
                paginationHTML += `<span class="text-gray-500">...</span>`;
            }
            paginationHTML += `<button onclick="app.loadLeads(${p.totalPages})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">${p.totalPages}</button>`;
        }
        
        // Next button
        if (p.currentPage < p.totalPages) {
            paginationHTML += `<button onclick="app.loadLeads(${p.currentPage + 1})" class="px-3 py-1 bg-gray-700 hover:bg-gray-600 text-white rounded text-sm transition">بعدی</button>`;
        } else {
            paginationHTML += `<button disabled class="px-3 py-1 bg-gray-800 text-gray-600 rounded text-sm cursor-not-allowed">بعدی</button>`;
        }
        
        paginationHTML += '</div>';
        container.innerHTML = paginationHTML;
    },


    // DEALS
    async loadDeals() {
	        const container = document.getElementById('deals-container');
	        if(!container) return;
	        container.innerHTML = '<div class="text-center py-20 col-span-full"><div class="loading-spinner mb-4 mx-auto"></div><p class="text-gray-400">در حال دریافت معاملات...</p></div>';
	        
	        const pipelineStageFilter = document.getElementById('deals-pipeline-filter')?.value || 'all';
	        const statusFilterRaw = document.getElementById('deals-status-filter')?.value || 'all';
	        const ownerId = document.getElementById('deals-owner-filter')?.value || '';
            const search = document.getElementById('deals-search')?.value || '';
            
	        const statusMap = { pending: 'Pending', won: 'Won', lost: 'Lost', all: 'all' };
	        const statusFilter = statusMap[statusFilterRaw] || 'all';
	        
            const payload = { 
                pipeline_stage: pipelineStageFilter, 
                status: statusFilter,
                search: search,
                legacy: '0'
            };
	        if (ownerId) payload.owner_id = ownerId;
            
	        const res = await this.req('get_deals', payload);
	        container.innerHTML = '';
	        
	        if(res.status === 'success' && res.deals && res.deals.length) {
                res.deals.forEach(d => {
                    const price = d.estimated_price ? parseInt(d.estimated_price).toLocaleString() : '0';
                    
                    const statusConfig = {
                        'Won': { text: 'موفق', badge: 'bg-emerald-500/20 text-emerald-400 border-emerald-500/30' },
                        'Lost': { text: 'ناموفق', badge: 'bg-red-500/20 text-red-400 border-red-500/30' },
                        'Pending': { text: 'جاری', badge: 'bg-blue-500/20 text-blue-400 border-blue-500/30' }
                    };
                    const status = statusConfig[d.status] || statusConfig['Pending'];
                    
                    const pipelineStage = d.pipeline_stage || 'معامله جدید';
                    const stageBadge = (()=>{
                        if (pipelineStage === 'برگزاری دوره') return '<span class="px-1.5 py-0.5 rounded text-[9px] bg-amber-500/20 text-amber-300 border border-amber-500/30 ml-1">دوره</span>';
                        if (pipelineStage === 'پشتیبانی') return '<span class="px-1.5 py-0.5 rounded text-[9px] bg-indigo-500/20 text-indigo-200 border border-indigo-500/30 ml-1">پشتیبانی</span>';
                        if (pipelineStage === 'مرجوعی' || (d.status === 'Lost' && (d.refund === '1' || d.refund_reason))) return '<span class="px-1.5 py-0.5 rounded text-[9px] bg-red-500/20 text-red-300 border border-red-500/30 ml-1">مرجوعی</span>';
                        return '';
                    })();

                    const productBadges = (d.products || []).map(p => 
                        `<span class="px-2 py-0.5 rounded-md text-[9px] bg-gray-700/50 text-gray-300 border border-gray-600/30 whitespace-nowrap">${this.escapeHtml(p.name)}</span>`
                    ).join('');

                    const contactId = d.contact_didar_id || d.contact_id || d.contactId || (d.contact ? d.contact.didar_contact_id : '') || d.local_person_id;
                    const dealId = d.didar_deal_id || d.id;
                    
                    container.innerHTML += `
                    <div class="glass p-5 rounded-2xl border border-gray-700/50 hover:border-yellow-500/40 transition-all duration-500 group flex flex-col h-full relative overflow-hidden">
                        <!-- Top Row: Status & Price -->
                        <div class="flex justify-between items-start mb-4">
                            <span class="text-[9px] px-2.5 py-1 rounded-full font-bold border ${status.badge}">
                                ${status.text}
                            </span>
                            <div class="text-left">
                                <span class="block text-[8px] text-gray-500 mb-0.5">مبلغ معامله</span>
                                <span class="text-xs font-black text-yellow-500 font-mono">${price} <span class="text-[9px] font-normal opacity-70">T</span></span>
                            </div>
                        </div>

                        <!-- Customer Info -->
                        <div class="mb-4 cursor-pointer" onclick="app.openDealModal('${dealId}')">
                            <h4 class="text-white font-bold text-sm mb-1 group-hover:text-yellow-500 transition-colors truncate" title="${this.escapeHtml(d.contact_name || '')}">
                                ${this.escapeHtml(d.contact_name || 'نامشخص')}
                            </h4>
                            <div class="flex items-center gap-2 text-[10px] text-gray-400">
                                <i class="fa-solid fa-briefcase text-[8px] opacity-50"></i>
                                <span class="truncate opacity-80" title="${this.escapeHtml(d.title || '')}">${this.escapeHtml(d.title || 'بدون عنوان')}</span>
                            </div>
                        </div>

                        <!-- Details Section -->
                        <div class="space-y-2 mb-4">
                            <!-- Pipeline Stage: Full Width for better visibility -->
                            <div class="bg-gray-900/60 p-2.5 rounded-xl border border-gray-800/50 flex flex-col gap-1">
                                <span class="text-[8px] text-gray-500 font-bold flex items-center gap-1">
                                    <i class="fa-solid fa-layer-group opacity-50"></i>
                                    مرحله کاریز
                                </span>
                                <div class="flex items-center gap-2">
                                    <span class="text-[11px] text-gray-200 font-bold">${pipelineStage}</span>
                                    ${stageBadge}
                                </div>
                            </div>
                            
                            <!-- Sales Owner -->
                            <div class="bg-gray-900/40 p-2 rounded-xl border border-gray-800/50 flex items-center justify-between">
                                <span class="text-[8px] text-gray-500 font-bold">مسئول فروش</span>
                                <div class="flex items-center gap-1.5">
                                    <div class="w-4 h-4 rounded-full bg-blue-500/20 flex items-center justify-center text-[8px] text-blue-400 font-bold border border-blue-500/30">
                                        ${(d.owner_name || '?').charAt(0)}
                                    </div>
                                    <span class="text-[10px] text-gray-300 font-bold">${this.escapeHtml(d.owner_name || '---')}</span>
                                </div>
                            </div>
                        </div>

                        <!-- Products List -->
                        ${productBadges ? `
                        <div class="mb-4 px-1">
                            <div class="flex flex-wrap gap-1.5">
                                ${productBadges}
                            </div>
                        </div>` : ''}

                        <!-- Footer Actions -->
                        <div class="mt-auto pt-4 border-t border-gray-800/50 flex justify-between items-center gap-2">
                            <div class="flex items-center gap-1 text-[9px] text-gray-500 italic">
                                <i class="fa-solid fa-calendar-day opacity-50"></i>
                                ${d.register_time ? toPersianDate(d.register_time) : '---'}
                            </div>
                            <div class="flex gap-1.5">
                                <button onclick="app.openDealModal('${dealId}')" 
                                    class="text-[10px] bg-amber-500/10 hover:bg-amber-500 text-amber-500 hover:text-black px-3 py-1.5 rounded-xl border border-amber-500/20 transition-all font-bold flex items-center gap-1">
                                    <i class="fa-solid fa-edit text-[9px]"></i>
                                    ویرایش
                                </button>
                                <button onclick="app.openPersonProfile('${contactId}')" 
                                    class="text-[10px] bg-blue-600/10 hover:bg-blue-600 text-blue-400 hover:text-white px-3 py-1.5 rounded-xl border border-blue-500/20 transition-all font-bold flex items-center gap-1">
                                    <i class="fa-solid fa-user text-[9px]"></i>
                                    مشاهده لید
                                </button>
                            </div>
                        </div>
                    </div>`;
                });
	        } else {
	            container.innerHTML = `
                <div class="col-span-full py-20 text-center">
                    <i class="fa-solid fa-sack-dollar text-5xl text-gray-800 mb-4 block"></i>
                    <p class="text-gray-500">معامله‌ای یافت نشد</p>
                </div>`;
	        }
	    },

        // Helper for quick filters
        setQuickStatusFilter(status) {
            const statusFilter = document.getElementById('deals-status-filter');
            if (statusFilter) {
                statusFilter.value = status;
                
                // Update chip active state
                document.querySelectorAll('.quick-filter-chip').forEach(chip => {
                    if (chip.getAttribute('data-status') === status) {
                        chip.classList.add('active');
                    } else {
                        chip.classList.remove('active');
                    }
                });
                
                this.loadDeals();
            }
        },
    
    // PERSONS
    async loadPersons() {
        const res = await this.req('get_leads', {});
        if(res.status === 'success') {
            this.state.leads = res.leads || [];
            this.renderLeads();
        }
    },

    async loadLegacyDeals() {
        const container = document.getElementById('legacy-deals-container');
        if (!container) return;
        container.innerHTML = '<div class="text-center py-20 col-span-full"><div class="loading-spinner mb-4 mx-auto"></div><p class="text-gray-400">\u062f\u0631 \u062d\u0627\u0644 \u0628\u0627\u0631\u06af\u0630\u0627\u0631\u06cc \u0645\u0639\u0627\u0645\u0644\u0627\u062a \u06af\u0630\u0634\u062a\u0647...</p></div>';

        const statusFilterRaw = document.getElementById('legacy-deals-status-filter')?.value || 'all';
        const ownerId = document.getElementById('legacy-deals-owner-filter')?.value || '';
        const search = document.getElementById('legacy-deals-search')?.value || '';

        const statusMap = { pending: 'Pending', won: 'Won', lost: 'Lost', all: 'all' };
        const statusFilter = statusMap[statusFilterRaw] || 'all';

        const payload = {
            status: statusFilter,
            search: search,
            legacy: '1'
        };
        if (ownerId) payload.owner_id = ownerId;

        const res = await this.req('get_deals', payload);
        container.innerHTML = '';

        if (res.status === 'success' && res.deals && res.deals.length) {
            res.deals.forEach(d => {
                const price = d.estimated_price ? parseInt(d.estimated_price).toLocaleString() : '0';

                const statusConfig = {
                    'Won': { text: '\u0645\u0648\u0641\u0642', badge: 'bg-emerald-500/20 text-emerald-400 border-emerald-500/30' },
                    'Lost': { text: '\u0646\u0627\u0645\u0648\u0641\u0642', badge: 'bg-red-500/20 text-red-400 border-red-500/30' },
                    'Pending': { text: '\u062f\u0631 \u062c\u0631\u06cc\u0627\u0646', badge: 'bg-blue-500/20 text-blue-400 border-blue-500/30' }
                };
                const status = statusConfig[d.status] || statusConfig['Pending'];

                const pipelineStage = d.pipeline_stage || d.pipeline_stage_title || '\u0645\u0631\u062d\u0644\u0647 \u0646\u0627\u0645\u0634\u062e\u0635';
                const dealTitle = d.title || d.pipeline_stage_title || d.pipeline_stage || d.contact_name || '\u0646\u0627\u0645\u0634\u062e\u0635';
                const contactId = d.contact_didar_id || d.contact_id || d.contactId || (d.contact ? d.contact.didar_contact_id : '') || d.local_person_id;
                const dealId = d.didar_deal_id || d.id;

                container.innerHTML += `
                <div class="glass p-5 rounded-2xl border border-gray-700/50 hover:border-yellow-500/40 transition-all duration-500 group flex flex-col h-full relative overflow-hidden">
                    <div class="flex justify-between items-start mb-4">
                        <div class="flex items-center gap-2">
                            <span class="text-[9px] px-2.5 py-1 rounded-full font-bold border ${status.badge}">
                                ${status.text}
                            </span>
                            <span class="text-[9px] px-2 py-1 rounded-full border border-purple-500/30 bg-purple-500/10 text-purple-300">\u0645\u0639\u0627\u0645\u0644\u0647 \u06af\u0630\u0634\u062a\u0647</span>
                        </div>
                        <div class="text-left">
                            <span class="block text-[8px] text-gray-500 mb-0.5">\u0627\u0631\u0632\u0634 \u0645\u0639\u0627\u0645\u0644\u0647</span>
                            <span class="text-xs font-black text-yellow-500 font-mono">${price} <span class="text-[9px] font-normal opacity-70">T</span></span>
                        </div>
                    </div>

                    <div class="mb-4 cursor-pointer" onclick="app.openDealModal('${dealId}')">
                        <h4 class="text-white font-bold text-sm mb-1 group-hover:text-yellow-500 transition-colors truncate" title="${this.escapeHtml(d.contact_name || '')}">
                            ${this.escapeHtml(d.contact_name || '\u0646\u0627\u0645\u0634\u062e\u0635')}
                        </h4>
                        <div class="flex items-center gap-2 text-[10px] text-gray-400">
                            <i class="fa-solid fa-briefcase text-[8px] opacity-50"></i>
                            <span class="truncate opacity-80" title="${this.escapeHtml(dealTitle)}">${this.escapeHtml(dealTitle)}</span>
                        </div>
                    </div>

                    <div class="space-y-2 mb-4">
                        <div class="bg-gray-900/60 p-2.5 rounded-xl border border-gray-800/50 flex flex-col gap-1">
                            <span class="text-[8px] text-gray-500 font-bold flex items-center gap-1">
                                <i class="fa-solid fa-layer-group opacity-50"></i>
                                \u0645\u0631\u062d\u0644\u0647 \u0641\u0631\u0648\u0634
                            </span>
                            <div class="flex items-center gap-2">
                                <span class="text-[11px] text-gray-200 font-bold">${this.escapeHtml(pipelineStage)}</span>
                            </div>
                        </div>

                        <div class="bg-gray-900/40 p-2 rounded-xl border border-gray-800/50 flex items-center justify-between">
                            <span class="text-[8px] text-gray-500 font-bold">\u0645\u0633\u0626\u0648\u0644</span>
                            <div class="flex items-center gap-1.5">
                                <div class="w-4 h-4 rounded-full bg-blue-500/20 flex items-center justify-center text-[8px] text-blue-400 font-bold border border-blue-500/30">
                                    ${(d.owner_name || '?').charAt(0)}
                                </div>
                                <span class="text-[10px] text-gray-300 font-bold">${this.escapeHtml(d.owner_name || '---')}</span>
                            </div>
                        </div>
                    </div>

                    <div class="mt-auto pt-4 border-t border-gray-800/50 flex justify-between items-center gap-2">
                        <div class="flex items-center gap-1 text-[9px] text-gray-500 italic">
                            <i class="fa-solid fa-calendar-day opacity-50"></i>
                            <span>${d.register_time ? toPersianDateTime(d.register_time) : '\u0646\u0627\u0645\u0634\u062e\u0635'}</span>
                        </div>
                        ${contactId ? `<button class="text-[10px] text-yellow-500 hover:text-yellow-300" onclick="event.stopPropagation(); app.openPersonProfile('${contactId}')">\u0645\u0634\u0627\u0647\u062f\u0647 \u0644\u06cc\u062f</button>` : ''}
                    </div>
                </div>
                `;
            });
        } else {
            container.innerHTML = '<div class="text-center py-12 text-gray-500">\u0647\u06cc\u0686 \u0645\u0639\u0627\u0645\u0644\u0647 \u06af\u0630\u0634\u062a\u0647\u200c\u0627\u06cc \u0648\u062c\u0648\u062f \u0646\u062f\u0627\u0631\u062f.</div>';
        }
    },

    // PERSON PROFILE
    openPersonProfile: async function(contactId) {
        const previousContactId = window.currentContactId;
        console.log('[CONTACT_SWITCH] Switching contact profile', {
            timestamp: new Date().toISOString(),
            previousContactId: previousContactId,
            newContactId: contactId,
            contactChanged: previousContactId !== contactId,
            url: window.location.href,
            userAgent: navigator.userAgent
        });

        window.currentContactId = contactId;
        document.getElementById('activity-contact-id').value = contactId;
        const dealContactId = document.getElementById('deal-contact-id');
        if (dealContactId) {
            dealContactId.value = contactId;
        }
        // Clear deal context when switching to a different contact to avoid cross-lead updates.
        if (previousContactId && previousContactId !== contactId) {
            const dealIdInput = document.getElementById('deal-id');
            const dealModal = document.getElementById('modalDeal');
            const isDealModalOpen = dealModal && !dealModal.classList.contains('hidden');
            
            if (dealIdInput) {
                // If deal modal is open and contact changed, clear deal-id to prevent errors
                if (isDealModalOpen && dealIdInput.value) {
                    dealIdInput.value = '';
                    const modalTitle = document.getElementById('deal-modal-title');
                    if (modalTitle) modalTitle.textContent = 'ثبت معامله جدید';
                    console.log('[CONTACT_SWITCH] Cleared deal-id because deal modal is open and contact changed');
                } else {
                    dealIdInput.value = '';
                }
            }
            const dealActDeal = document.getElementById('deal-activity-deal-id');
            if (dealActDeal) dealActDeal.value = '';
            const activityDealId = document.getElementById('activity-deal-id');
            if (activityDealId) activityDealId.value = '';
            this.state.currentDeal = null;
            console.log('[CONTACT_SWITCH] Cleared deal context to prevent cross-lead updates', {
                timestamp: new Date().toISOString(),
                previousContactId: previousContactId,
                newContactId: contactId,
                dealModalWasOpen: isDealModalOpen
            });
        }
        
        const res = await this.req('get_person_profile', {contact_id: contactId});
        if(res.status === 'success') {
            const person = res.person;
            const fullName = `${person.first_name || ''} ${person.last_name || ''}`.trim() || 'بدون نام';
            this.state.currentLead = person;
            
            // ذخیره activities برای بررسی بعدی
            this.state.currentLeadActivities = res.activities || [];
            const stageSelect = document.getElementById('activity-virtual-stage');
            if (stageSelect) {
                stageSelect.value = '';
            }
            if (Array.isArray(res.deals)) {
                this.state.currentDeals = res.deals;
                this.populateActivityDealSelect(res.deals);
            } else {
                this.state.currentDeals = [];
                const activityDealId = document.getElementById('activity-deal-id');
                if (activityDealId) activityDealId.value = '';
                const stageDisplay = document.getElementById('activity-current-stage-display');
                if (stageDisplay) stageDisplay.innerHTML = '<span class="text-sm text-gray-500">مرحله تعیین نشده</span>';
            }
            const referralNoteInput = document.querySelector('textarea[name="referral_note"]');
            if (referralNoteInput) referralNoteInput.value = '';
            
            const referralPhones = [person.mobile_phone, person.secondary_mobile_phone].filter(Boolean).join(' ، ');
            const referralPhonesEl = document.getElementById('referral-phones');
            if (referralPhonesEl) referralPhonesEl.textContent = referralPhones || '-';
            
            document.getElementById('prof-name').textContent = fullName;
            document.getElementById('prof-mobile').textContent = person.mobile_phone || '-';
            const profEmail = document.getElementById('prof-email');
            if (profEmail) profEmail.textContent = person.email || '-';
            
            // Info tab
            document.getElementById('prof-info-name').textContent = fullName;
            document.getElementById('prof-info-mobile').textContent = person.mobile_phone || '-';
            document.getElementById('prof-info-mobile2').textContent = person.secondary_mobile_phone || '-';
            const mobile3El = document.getElementById('prof-info-mobile3');
            const mobile4El = document.getElementById('prof-info-mobile4');
            if (mobile3El) mobile3El.textContent = person.mobile_phone_3 || '-';
            if (mobile4El) mobile4El.textContent = person.mobile_phone_4 || '-';
            const profInfoEmail = document.getElementById('prof-info-email');
            if (profInfoEmail) profInfoEmail.textContent = person.email || '-';
            document.getElementById('prof-info-stage').innerHTML = '';
            document.getElementById('prof-info-owner').textContent = person.owner_name || '---';
            const prevOwnerEl = document.getElementById('prof-info-prev-owner');
            if (prevOwnerEl) prevOwnerEl.textContent = person.previous_owner_name || '-';
            const regDateDisplay = person.register_time_jalali || (person.register_time ? toPersianDate(person.register_time) : '-');
            const regEl = document.getElementById('prof-info-register');
            if (regEl) regEl.textContent = regDateDisplay || '-';
            const saleEl = document.getElementById('prof-info-sale-status');
            if (saleEl) {
                saleEl.textContent = String(person.sale_status) === '1' ? 'موفق' : 'ناموفق';
            }
            const nationalEl = document.getElementById('prof-info-national-id');
            if (nationalEl) nationalEl.textContent = person.national_id || '-';
            document.getElementById('prof-info-city').textContent = person.city || '-';
            document.getElementById('prof-info-job').textContent = person.job_title || '-';
            const jobDescEl = document.getElementById('prof-info-job-desc');
            if (jobDescEl) jobDescEl.textContent = person.job_description || '-';
            document.getElementById('prof-info-source').textContent = this.getLeadSourceLabel(person.acquaintance_source);
            document.getElementById('prof-info-source-detail').textContent = this.getLeadDetailLabel(person.acquaintance_detail);
            document.getElementById('prof-info-topic').textContent = this.getContentTopicLabel(person.content_topic);
            document.getElementById('prof-info-products').textContent = person.customer_products || '-';
            const acqEl = document.getElementById('prof-info-acq-duration');
            if (acqEl) acqEl.textContent = person.acquaintance_duration || '-';
            const extraEl = document.getElementById('prof-info-extra-info');
            if (extraEl) extraEl.textContent = person.extra_info || '-';

            const financialLabel = (val) => {
                switch (val) {
                    case 'high': return 'بالا';
                    case 'medium_high': return 'متوسط رو به بالا';
                    case 'medium_low': return 'متوسط رو به پایین';
                    case 'low': return 'پایین';
                    default: return '-';
                }
            };
            const rangeLabel = (val, type) => {
                const map = type === 'asset' ? {
                    '0-50': '0 تا 50 میلیون تومان',
                    '50-200': '50 تا 200 میلیون تومان',
                    '200-500': '200 تا 500 میلیون تومان',
                    '500-1000': '500 تا 1000 میلیون تومان',
                    '1000-plus': 'بیش از 1 میلیارد تومان'
                } : {
                    '0-20': '0 تا 20 میلیون تومان',
                    '20-100': '20 تا 100 میلیون تومان',
                    '100-200': '100 تا 200 میلیون تومان',
                    '200-500': '200 تا 500 میلیون تومان',
                    '500-plus': 'بیش از 500 میلیون تومان'
                };
                return map[val] || '-';
            };

        // Customer level
        const customerLevelEl = document.getElementById('prof-info-customer-level');
        if (customerLevelEl) {
            const levelMap = {
                'طلایی': 'طلایی',
                'نقره‌ای': 'نقره‌ای',
                'برنز': 'برنز'
            };
            customerLevelEl.textContent = levelMap[person.customer_level] || '-';
        }
        const customerLevelNotesEl = document.getElementById('prof-info-customer-level-notes');
        if (customerLevelNotesEl) {
            customerLevelNotesEl.textContent = person.customer_level_notes || '';
        }

        const finEl = document.getElementById('prof-info-financial-level');
        if (finEl) finEl.textContent = financialLabel(person.financial_level);
        const assetEl = document.getElementById('prof-info-asset');
        if (assetEl) assetEl.textContent = rangeLabel(person.asset_estimation, 'asset');
        const incomeEl = document.getElementById('prof-info-income');
        if (incomeEl) incomeEl.textContent = rangeLabel(person.income_estimation, 'income');
            const purchaseBadge = document.getElementById('prof-info-purchase-status');
            if (purchaseBadge) {
                if (person.has_previous_purchase === 1 || person.has_previous_purchase === '1') {
                    purchaseBadge.textContent = 'خرید قبلی دارد';
                } else if (person.has_previous_purchase === 0 || person.has_previous_purchase === '0') {
                    purchaseBadge.textContent = 'خرید قبلی ندارد';
                } else {
                    purchaseBadge.textContent = '-';
                }
            }

            // Requested services: try person first, then latest deal
            let reqServices = person.requested_services || '';
            if (typeof reqServices === 'string') {
                try {
                    const parsed = JSON.parse(reqServices);
                    if (Array.isArray(parsed)) reqServices = parsed;
                } catch (e) {
                    // keep string
                }
            }
            if (!reqServices || (Array.isArray(reqServices) && reqServices.length === 0)) {
                const currentDeal = (this.state.currentDeals || [])[0] || this.state.currentDeal;
                if (currentDeal && currentDeal.requested_services) {
                    try {
                        const parsed = JSON.parse(currentDeal.requested_services);
                        if (Array.isArray(parsed)) reqServices = parsed;
                    } catch (e) {
                        reqServices = currentDeal.requested_services;
                    }
                }
            }
            const reqEl = document.getElementById('prof-info-requested-services');
            if (reqEl) {
                if (Array.isArray(reqServices)) {
                    reqEl.textContent = reqServices.length ? reqServices.join('، ') : '-';
                } else if (reqServices) {
                    reqEl.textContent = reqServices;
                } else {
                    reqEl.textContent = '-';
                }
            }
            
            // مخفی کردن فیلد شیوه آشنایی برای کاربران غیر ادمین و غیر CRM
            const sourceContainer = document.getElementById('prof-info-source-container');
            if (sourceContainer) {
                if (this.isAdminOrCrmSpecialist()) {
                    sourceContainer.classList.remove('hidden');
                } else {
                    sourceContainer.classList.add('hidden');
                }
            }
            document.getElementById('prof-info-background').textContent = person.background_info || '-';
            const editBtn = document.getElementById('lead-edit-btn');
            if (editBtn) {
                // CRM specialist is view-only, cannot edit lead info
                const isAdmin = window.APP_USER?.role === 'admin';
                const canEdit = isAdmin || this.isLeadOwner(person);
                editBtn.classList.toggle('hidden', !canEdit);
            }
            
            // Render deals
        const dealsDiv = document.getElementById('prof-deals');
        dealsDiv.innerHTML = '';
        const currentDeals = Array.isArray(res.deals) ? res.deals : [];
        const legacyDeals = Array.isArray(res.legacy_deals) ? res.legacy_deals : [];

        const renderDealCard = (d, isLegacy = false) => {
            const price = d.estimated_price ? parseInt(d.estimated_price).toLocaleString() : '0';
            const statusColor = d.status === 'Won' ? 'border-green-500' : (d.status === 'Lost' ? 'border-red-500' : 'border-blue-500');
            const statusText = d.status === 'Won' ? '\u0645\u0648\u0641\u0642' : (d.status === 'Lost' ? '\u0646\u0627\u0645\u0648\u0641\u0642' : '\u062f\u0631 \u062c\u0631\u06cc\u0627\u0646');
            const dealId = d.didar_deal_id || d.id;
            const legacyBadge = isLegacy
                ? '<span class="inline-block ml-2 px-2 py-0.5 rounded bg-purple-500/20 text-purple-300 text-[10px]">\u0645\u0639\u0627\u0645\u0644\u0647 \u06af\u0630\u0634\u062a\u0647 \u062f\u06cc\u062f\u0627\u0631</span>'
                : '';

            const deleteBtn = this.isAdmin() ? `
                    <button onclick="event.stopPropagation(); app.deleteDeal('${dealId}')"
                            class="text-red-500 hover:text-red-700 text-xs px-2 py-1 rounded transition"
                            title="\u062d\u0630\u0641 \u0645\u0639\u0627\u0645\u0644\u0647">
                        <i class="fa-solid fa-trash"></i>
                    </button>` : '';

            return `
                    <div class="bg-gray-700/30 p-3 rounded border-l-4 ${statusColor} mb-2 hover:bg-gray-700/50 transition relative">
                        ${deleteBtn ? `<div class="absolute top-2 left-2 z-10">${deleteBtn}</div>` : ''}
                        <div class="cursor-pointer" onclick="app.openDealModal('${dealId}')">
                            <div class="flex justify-between items-start">
                                <div class="flex-1">
                                    <span class="text-sm text-white font-bold">${d.title || '-'}</span>
                                    ${legacyBadge}
                                    <div class="text-[10px] text-gray-500 mt-1">
                                        <span class="inline-block px-2 py-0.5 rounded ${d.status === 'Won' ? 'bg-green-500/20 text-green-400' : (d.status === 'Lost' ? 'bg-red-500/20 text-red-400' : 'bg-blue-500/20 text-blue-400')}">${statusText}</span>
                                        ${d.register_time ? ' | ' + toPersianDate(d.register_time) : ''}
                                    </div>
                                </div>
                                <span class="text-xs text-yellow-500 font-mono font-bold ml-2">${price} T</span>
                            </div>
                        </div>
                    </div>`;
        };

        if (currentDeals.length) {
            currentDeals.forEach(d => {
                dealsDiv.innerHTML += renderDealCard(d, false);
            });
        }

        if (legacyDeals.length) {
            if (currentDeals.length) {
                dealsDiv.innerHTML += '<div class="text-xs text-gray-400 mt-4 mb-2">\u0645\u0639\u0627\u0645\u0644\u0627\u062a \u06af\u0630\u0634\u062a\u0647 \u062f\u06cc\u062f\u0627\u0631</div>';
            }
            legacyDeals.forEach(d => {
                dealsDiv.innerHTML += renderDealCard(d, true);
            });
        }

        if (!currentDeals.length && !legacyDeals.length) {
            dealsDiv.innerHTML = '<p class="text-sm text-gray-500">\u0647\u06cc\u0686 \u0645\u0639\u0627\u0645\u0644\u0647\u200c\u0627\u06cc \u0628\u0631\u0627\u06cc \u0646\u0645\u0627\u06cc\u0634 \u0648\u062c\u0648\u062f \u0646\u062f\u0627\u0631\u062f.</p>';
        }

        // Render activities
        const actDiv = document.getElementById('prof-activities');
        actDiv.innerHTML = '';
        if(res.activities && res.activities.length) {
            res.activities.forEach(a => {
                // Extract virtual stage and purchase info from result_note
                let displayNote = a.result_note || a.note || 'بدون توضیحات';
                let stageBadge = '';
                let purchaseBadge = '';
                
                // Check for virtual stage in stage field or result_note
                if (a.stage) {
                    const stageTitles = {
                        'contact_no_answer': 'تماس عدم پاسخ',
                        'contact_followup': 'تماس پیگیری',
                        'deal_registered': 'ثبت معامله و لینک پرداخت',
                        'payment': 'پرداخت',
                        'payment_followup': 'پیگیری مانده حساب',
                        'course_delivered': 'برگزاری دوره',
                        'support': 'پشتیبانی'
                    };
                    const stageTitle = stageTitles[a.stage];
                    if (stageTitle) {
                        stageBadge = `<span class="inline-block bg-purple-600 text-xs px-2 py-0.5 rounded mb-2">📍 ${stageTitle}</span>`;
                    }
                }
                
                // Check for purchase info in result_note
                if (displayNote.includes('🟢 این لید قبلاً خرید دیگری')) {
                    purchaseBadge = `<span class="inline-block bg-green-600 text-xs px-2 py-0.5 rounded mb-2 mr-2">🟢 خرید قبلی با شماره دیگر: بله</span>`;
                    // Remove from display note
                    displayNote = displayNote.replace(/🟢 این لید قبلاً خرید دیگری با شماره دیگری داشته است\./g, '').trim();
                    displayNote = displayNote.replace(/⚪ این لید خرید قبلی با شماره دیگر نداشته است\./g, '').trim();
                    displayNote = displayNote.replace(/⚪ خرید قبلی با شماره دیگر: نامشخص/g, '').trim();
                    displayNote = displayNote.replace(/📍 مرحله کاریز:.*/g, '').trim();
                } else if (displayNote.includes('⚪ این لید خرید قبلی') || displayNote.includes('⚪ خرید قبلی با شماره دیگر')) {
                    if (displayNote.includes('نامشخص')) {
                        purchaseBadge = `<span class="inline-block bg-yellow-600 text-xs px-2 py-0.5 rounded mb-2 mr-2">⚪ خرید قبلی با شماره دیگر: نامشخص</span>`;
                    } else {
                        purchaseBadge = `<span class="inline-block bg-gray-600 text-xs px-2 py-0.5 rounded mb-2 mr-2">⚪ خرید قبلی با شماره دیگر: خیر</span>`;
                    }
                    // Remove from display note
                    displayNote = displayNote.replace(/⚪ این لید خرید قبلی با شماره دیگر نداشته است\./g, '').trim();
                    displayNote = displayNote.replace(/⚪ خرید قبلی با شماره دیگر: نامشخص/g, '').trim();
                    displayNote = displayNote.replace(/📍 مرحله کاریز:.*/g, '').trim();
                }
                
                const activityId = a.didar_activity_id || a.id;
                const deleteBtn = this.isAdmin() ? `
                    <button onclick="event.stopPropagation(); app.deleteActivity('${activityId}')" 
                            class="text-red-500 hover:text-red-700 text-xs px-2 py-1 rounded transition" 
                            title="حذف فعالیت">
                        <i class="fa-solid fa-trash"></i>
                    </button>` : '';
                
                // Check if activity is scheduled - check multiple conditions
                const hasDueDate = a.due_date && a.due_date.trim() !== '';
                const isScheduledFlag = a.is_scheduled === 1 || a.is_scheduled === true || a.is_scheduled === '1';
                const isScheduled = isScheduledFlag || (!a.is_done && hasDueDate);
                const isEdited = a.last_update_time && a.last_update_time !== a.register_date && a.last_update_time !== a.created_at;
                
                // Parse and format dates
                let dueDate = '';
                if (hasDueDate) {
                    try {
                        dueDate = toPersianDateTime(a.due_date);
                    } catch (e) {
                        console.warn('Error parsing due_date:', a.due_date, e);
                        dueDate = a.due_date; // Fallback to raw value
                    }
                }
                
                const registerDate = a.register_date ? toPersianDateTime(a.register_date) : '';
                
                let statusBadge = '';
                if (a.is_done === 1 || a.is_done === true || a.is_done === '1') {
                    statusBadge = '<span class="text-[10px] bg-green-500/20 text-green-300 px-2 py-0.5 rounded">✓ انجام شده</span>';
                } else if (isScheduled) {
                    statusBadge = '<span class="text-[10px] bg-yellow-500/20 text-yellow-300 px-2 py-0.5 rounded">⏰ برنامه‌ریزی شده</span>';
                } else {
                    statusBadge = '<span class="text-[10px] bg-blue-500/20 text-blue-300 px-2 py-0.5 rounded">📝 ثبت شده</span>';
                }
                
                const editBtn = !a.is_done && isScheduled ? `
                    <button onclick="event.stopPropagation(); app.openEditScheduledActivityModal('${activityId}')" 
                            class="text-blue-500 hover:text-blue-700 text-xs px-2 py-1 rounded transition" 
                            title="ویرایش فعالیت">
                        <i class="fa-solid fa-edit"></i>
                    </button>` : '';
                
                const clickHandler = !a.is_done && isScheduled ? `onclick="app.openEditScheduledActivityModal('${activityId}')"` : '';
                const cursorClass = !a.is_done && isScheduled ? 'cursor-pointer hover:bg-gray-700/30 transition' : '';
                
                actDiv.innerHTML += `
                <div class="relative pr-8 pb-4">
                    <div class="absolute top-1 right-0 w-4 h-4 bg-slate-700 rounded-full border-2 ${a.is_done ? 'border-green-500' : isScheduled ? 'border-yellow-500' : 'border-blue-500'} z-10"></div>
                    ${editBtn || deleteBtn ? `<div class="absolute top-1 left-1 z-10 flex gap-1">${editBtn}${deleteBtn}</div>` : ''}
                    ${isEdited ? `<div class="absolute top-1 left-12 z-10"><span class="text-[9px] bg-purple-500/20 text-purple-300 px-1.5 py-0.5 rounded" title="ویرایش شده"><i class="fa-solid fa-pencil"></i></span></div>` : ''}
                    <div class="bg-gray-800 p-3 rounded-lg ${cursorClass}" ${clickHandler}>
                        <div class="flex justify-between mb-1">
                            <div class="flex items-center gap-2">
                                <span class="text-xs font-bold text-blue-300">
                                    <i class="fa-solid fa-${a.is_done ? 'check-circle' : isScheduled ? 'clock' : 'calendar-check'} ml-1"></i>
                                    ${a.activity_type_title || 'فعالیت'}
                                </span>
                                ${statusBadge}
                            </div>
                            <span class="text-[9px] text-gray-500">${registerDate}</span>
                        </div>
                        <div class="mb-2">
                            ${stageBadge || ''}
                            ${purchaseBadge || ''}
                        </div>
                        <div class="text-[10px] text-gray-400 space-y-0.5 mb-2">
                            ${registerDate ? `<div><i class="fa-solid fa-calendar-plus ml-1"></i> تاریخ ثبت: ${registerDate}</div>` : ''}
                            ${dueDate ? `<div><i class="fa-solid fa-calendar-clock ml-1"></i> تاریخ/ساعت برنامه‌ریزی: ${dueDate}</div>` : ''}
                            ${isScheduled && !dueDate ? `<div><i class="fa-solid fa-clock ml-1"></i> فعالیت برنامه‌ریزی شده (تاریخ مشخص نشده)</div>` : ''}
                        </div>
                        <p class="text-xs text-gray-300 leading-relaxed">${displayNote || '<span class="text-gray-500 italic">بدون توضیحات</span>'}</p>
                    </div>
                </div>`;
            });
            } else {
                actDiv.innerHTML = '<p class="text-sm text-gray-500">هیچ فعالیتی ثبت نشده است.</p>';
            }
            
            // Load activity types for modal
            this.loadActivityTypesForModal();
            
            // Render satisfactions if available
            if (res.satisfactions) {
                this.renderSatisfactions(res.satisfactions);
            }
            
            this.switchProfileTab('info');
            this.openModal('modalProfile');
            
            // بررسی نمایش فیلد خرید قبلی بعد از باز شدن مودال
            setTimeout(() => {
                this.handleActivityFormVisibility();
            }, 200);
        }
    },
    
    renderSatisfactions(satisfactions) {
        const container = document.getElementById('prof-satisfactions');
        if (!container) return;
        
        if (satisfactions && satisfactions.length > 0) {
            container.innerHTML = satisfactions.map(sat => {
                const stars = '⭐'.repeat(sat.satisfaction_score || 0);
                const date = sat.register_date ? toPersianDateTime(sat.register_date) : '';
                const registeredBy = sat.registered_by_name || 'نامشخص';
                const note = sat.satisfaction_note ? `<p class="text-sm text-gray-300 mt-2">${this.escapeHtml(sat.satisfaction_note)}</p>` : '';
                
                return `
                    <div class="glass p-4 rounded-lg">
                        <div class="flex justify-between items-start mb-2">
                            <div>
                                <div class="text-lg mb-1">${stars}</div>
                                <div class="text-xs text-gray-400">ثبت شده توسط: ${this.escapeHtml(registeredBy)}</div>
                            </div>
                            <div class="text-xs text-gray-500">${date}</div>
                        </div>
                        ${note}
                    </div>
                `;
            }).join('');
        } else {
            container.innerHTML = '<p class="text-sm text-gray-500 text-center py-4">هیچ رضایتی ثبت نشده است.</p>';
        }
    },
    
    openSatisfactionModal() {
        const contactId = window.currentContactId;
        if (!contactId) {
            Swal.fire('خطا', 'لید انتخاب نشده است', 'error');
            return;
        }
        document.getElementById('satisfaction-contact-id').value = contactId;
        document.querySelectorAll('input[name="satisfaction_score"]').forEach(r => r.checked = false);
        document.getElementById('satisfaction-score-text').textContent = 'امتیاز را انتخاب کنید';
        document.querySelector('textarea[name="satisfaction_note"]').value = '';
        this.openModal('modalSatisfaction');
    },
    
    selectSatisfactionScore(score) {
        document.getElementById(`score-${score}`).checked = true;
        const texts = {
            1: 'خیلی ضعیف',
            2: 'ضعیف',
            3: 'متوسط',
            4: 'خوب',
            5: 'عالی'
        };
        document.getElementById('satisfaction-score-text').textContent = texts[score] || '';
    },
    
    async saveSatisfaction(e) {
        e.preventDefault();
        const form = e.target;
        const data = Object.fromEntries(new FormData(form));
        
        if (!data.satisfaction_score) {
            Swal.fire('خطا', 'لطفاً امتیاز رضایت را انتخاب کنید', 'error');
            return;
        }
        
        Swal.fire({
            title: 'در حال ثبت...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('save_satisfaction', data);
            if (res.status === 'success') {
                Swal.fire({toast: true, position: 'bottom-end', icon: 'success', title: 'رضایت ثبت شد', showConfirmButton: false, timer: 2000});
                this.closeModal('modalSatisfaction');
                form.reset();
                
                // Reload person profile to show new satisfaction
                if (window.currentContactId) {
                    await this.openPersonProfile(window.currentContactId);
                    this.switchProfileTab('satisfactions');
                }
            } else {
                Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
            }
        } catch (error) {
            console.error('Error in saveSatisfaction:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در ثبت رضایت: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    openDidarSearchModal() {
        document.getElementById('didar-search-phone').value = '';
        document.getElementById('didar-search-keywords').value = '';
        document.getElementById('didar-search-results').innerHTML = '<p class="text-center text-gray-500 py-8">برای جستجو، شماره تلفن یا کلمات کلیدی را وارد کنید</p>';
        this.openModal('modalDidarSearch');
    },

    async openLocalMobileLookup() {
        const { value } = await Swal.fire({
            title: 'استعلام سریع شماره موبایل',
            input: 'text',
            inputPlaceholder: '۱۰ رقم موبایل بدون صفر اول (مثل: 9123456789)',
            inputAttributes: {
                maxlength: 10,
                oninput: function() {
                    // فقط اجازه تایپ اعداد بده و اعداد فارسی/عربی را به انگلیسی تبدیل کن
                    this.value = this.value.replace(/[^\d]/g, '').replace(/[۰-۹]/g, function(d) {
                        return String.fromCharCode(d.charCodeAt(0) - 1728);
                    }).replace(/[٠-٩]/g, function(d) {
                        return String.fromCharCode(d.charCodeAt(0) - 1584);
                    });
                }
            },
            showCancelButton: true,
            confirmButtonText: 'جستجو',
            cancelButtonText: 'لغو',
            inputValidator: (v) => {
                const raw = (v || '').trim();
                if (!raw) return 'شماره موبایل را وارد کنید';
                const digits = raw.replace(/[^\d]/g, '');
                if (digits.length !== 10) return 'شماره موبایل باید دقیقاً 10 رقم باشد (بدون صفر اول)';
                if (!digits.startsWith('9')) return 'شماره موبایل باید با رقم 9 شروع شود';
                return null;
            }
        });
        if (!value) return;

        const digits = toEnglishDigits(String(value).trim()).replace(/[^\d]/g, '');

        Swal.fire({
            title: 'در حال بررسی...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await this.req('lookup_mobile', { mobile_phone: digits });
            Swal.close();

            if (res.status !== 'success') {
                Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
                return;
            }

            const matches = Array.isArray(res.matches) ? res.matches : [];
            if (matches.length === 0) {
                Swal.fire('یافت نشد', 'این شماره در دیتابیس محلی موجود نیست.', 'info');
                return;
            }

            const html = matches.map(m => {
                const displayName = m.display_name || 'بدون نام';
                const ownerName = m.owner_name || 'نامشخص';
                const previousOwnerName = m.previous_owner_name || '-';
                const contactId = m.didar_contact_id || '-';

                return `
                    <div class="text-right p-3 bg-gray-800 rounded mb-2">
                        <div class="text-sm text-white font-bold">${this.escapeHtml(displayName)}</div>
                        <div class="text-xs text-gray-300 mt-1">شناسه: ${this.escapeHtml(String(contactId))}</div>
                        <div class="text-xs text-gray-300 mt-1">مسئول فعلی: ${this.escapeHtml(ownerName)}</div>
                        <div class="text-xs text-gray-400 mt-1">مسئول قبلی: ${this.escapeHtml(previousOwnerName)}</div>
                    </div>
                `;
            }).join('');

            Swal.fire({
                title: `یافت شد (${matches.length})`,
                html,
                confirmButtonText: 'بستن'
            });
        } catch (e) {
            console.error('Local mobile lookup failed:', e);
            Swal.close();
            Swal.fire('خطا', 'خطا در ارتباط با سرور', 'error');
        }
    },
    
    async searchDidar(e) {
        e.preventDefault();
        const form = e.target;
        const phone = form.phone.value.trim();
        const keywords = form.keywords.value.trim();
        
        if (!phone && !keywords) {
            Swal.fire('خطا', 'لطفاً شماره تلفن یا کلمات کلیدی را وارد کنید', 'error');
            return;
        }
        
        Swal.fire({
            title: 'در حال جستجو...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('search_didar', { phone, keywords });
            Swal.close();
            
            if (res.status === 'success') {
                this.renderDidarSearchResults(res.results || []);
            } else {
                Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
            }
        } catch (error) {
            console.error('Error in searchDidar:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در جستجو: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    renderDidarSearchResults(results) {
        const container = document.getElementById('didar-search-results');
        
        // Store results in state for later use
        this.state.didarSearchResults = results || [];
        
        if (!results || results.length === 0) {
            container.innerHTML = '<p class="text-center text-gray-500 py-8">نتیجه‌ای یافت نشد</p>';
            return;
        }
        
        container.innerHTML = results.map(person => {
            const name = person.DisplayName || (person.FirstName + ' ' + person.LastName) || 'بدون نام';
            const mobile = person.MobilePhone || '-';
            const email = person.Email || '-';
            const contactId = person.Id || '';
            
            return `
                <div class="glass p-4 rounded-lg hover:bg-gray-800/50 cursor-pointer" onclick="app.viewDidarPersonDetails('${contactId}')">
                    <div class="flex justify-between items-start">
                        <div class="flex-1">
                            <div class="font-bold text-white mb-1">${this.escapeHtml(name)}</div>
                            <div class="text-sm text-gray-400">موبایل: ${this.escapeHtml(mobile)}</div>
                            ${email !== '-' ? `<div class="text-sm text-gray-400">ایمیل: ${this.escapeHtml(email)}</div>` : ''}
                        </div>
                        <button class="btn-primary px-3 py-1 rounded text-xs">مشاهده جزئیات</button>
                    </div>
                </div>
            `;
        }).join('');
    },
    
    async viewDidarPersonDetails(contactId) {
        // First, try to get person from search results
        let person = null;
        if (this.state.didarSearchResults && Array.isArray(this.state.didarSearchResults)) {
            person = this.state.didarSearchResults.find(p => p.Id === contactId);
        }
        
        // If found in search results, use it directly and fetch deals/activities
        if (person) {
            Swal.fire({
                title: 'در حال بارگذاری معاملات و فعالیت‌ها...',
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading()
            });
            
            try {
                const res = await this.req('get_didar_person', { contact_id: contactId, person_from_search: 'true' });
                Swal.close();
                
                console.log('get_didar_person response:', res);
                console.log('Deals count:', res.deals?.length || 0);
                console.log('Activities count:', res.activities?.length || 0);
                
                if (res.status === 'success') {
                    // Use person from search results (more complete) but use deals/activities from API
                    console.log('Rendering with deals:', res.deals?.length || 0, 'activities:', res.activities?.length || 0);
                    this.renderDidarPersonDetails(person, res.deals || [], res.activities || []);
                    this.openModal('modalDidarPerson');
                    // No warning needed since person data is available from search results
                } else {
                    // If API fails, still show person data from search
                    this.renderDidarPersonDetails(person, [], []);
                    this.openModal('modalDidarPerson');
                    Swal.fire('هشدار', 'اطلاعات کامل دریافت نشد، اما اطلاعات پایه نمایش داده می‌شود', 'warning');
                }
            } catch (error) {
                console.error('Error in viewDidarPersonDetails:', error);
                Swal.close();
                // Still show person data from search even if API fails
                this.renderDidarPersonDetails(person, [], []);
                this.openModal('modalDidarPerson');
                Swal.fire('هشدار', 'اطلاعات کامل دریافت نشد، اما اطلاعات پایه نمایش داده می‌شود', 'warning');
            }
        } else {
            // If not in search results, try API (which will search for the person)
            Swal.fire({
                title: 'در حال بارگذاری...',
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading()
            });
            
            try {
                const res = await this.req('get_didar_person', { contact_id: contactId });
                Swal.close();
                
                console.log('get_didar_person response (no search):', res);
                console.log('Deals count:', res.deals?.length || 0);
                console.log('Activities count:', res.activities?.length || 0);
                
                if (res.status === 'success' && res.person) {
                    console.log('Rendering with deals:', res.deals?.length || 0, 'activities:', res.activities?.length || 0);
                    this.renderDidarPersonDetails(res.person, res.deals || [], res.activities || []);
                    this.openModal('modalDidarPerson');
                } else if (res.status === 'partial') {
                    // Person not found but deals/activities available
                    console.log('Partial response - showing deals/activities anyway');
                    if (res.deals && res.deals.length > 0 || res.activities && res.activities.length > 0) {
                        this.renderDidarPersonDetails(null, res.deals || [], res.activities || []);
                        this.openModal('modalDidarPerson');
                    } else {
                        Swal.fire('خطا', 'اطلاعات شخص یافت نشد. لطفاً دوباره جستجو کنید.', 'error');
                    }
                } else {
                    Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
                }
            } catch (error) {
                console.error('Error in viewDidarPersonDetails:', error);
                Swal.close();
                Swal.fire('خطا', 'خطا در دریافت اطلاعات: ' + (error.message || 'خطای نامشخص'), 'error');
            }
        }
    },
    
    renderDidarPersonDetails(person, deals, activities) {
        console.log('renderDidarPersonDetails called with:', {
            person: person ? 'exists' : 'null',
            dealsCount: deals?.length || 0,
            activitiesCount: activities?.length || 0
        });
        
        const container = document.getElementById('didar-person-details');
        const nameEl = document.getElementById('didar-person-name');
        
        if (!container) {
            console.error('didar-person-details container not found!');
            return;
        }
        
        if (!person) {
            console.warn('Person is null, using placeholder');
        }
        
        const name = person ? (person.DisplayName || (person.FirstName + ' ' + person.LastName) || 'بدون نام') : 'اطلاعات شخص در دسترس نیست';
        if (nameEl) nameEl.textContent = name;
        
        // Store activities in state for filtering
        this.state.didarActivities = activities || [];
        this.state.didarActivitiesFiltered = activities || [];
        
        console.log('About to render HTML with', deals?.length || 0, 'deals and', activities?.length || 0, 'activities');
        
        container.innerHTML = `
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                <div class="glass p-4">
                    <div class="text-xs text-gray-400 mb-1">نام</div>
                    <div class="text-white font-bold">${this.escapeHtml(name)}</div>
                </div>
                <div class="glass p-4">
                    <div class="text-xs text-gray-400 mb-1">موبایل</div>
                    <div class="text-white font-bold">${this.escapeHtml(person?.MobilePhone || '-')}</div>
                </div>
                <div class="glass p-4">
                    <div class="text-xs text-gray-400 mb-1">ایمیل</div>
                    <div class="text-white font-bold">${this.escapeHtml(person?.Email || '-')}</div>
                </div>
                <div class="glass p-4">
                    <div class="text-xs text-gray-400 mb-1">شهر</div>
                    <div class="text-white font-bold">${this.escapeHtml(person?.City || '-')}</div>
                </div>
                <div class="glass p-4">
                    <div class="text-xs text-gray-400 mb-1">شغل</div>
                    <div class="text-white font-bold">${this.escapeHtml(person?.JobTitle || '-')}</div>
                </div>
                <div class="glass p-4">
                    <div class="text-xs text-gray-400 mb-1">مسئول</div>
                    <div class="text-white font-bold">${this.escapeHtml(person?.Owner?.DisplayName || '-')}</div>
                </div>
            </div>
            
            <div class="mb-6">
                <h4 class="font-bold text-white mb-3">معاملات (${deals?.length || 0})</h4>
                <div class="space-y-2">
                    ${deals && deals.length > 0 ? deals.map(deal => {
                        const dealId = deal.Id || deal.id || deal.Code || '';
                        const status = deal.Status || 'Pending';
                        const isClosed = status === 'Won' || status === 'Lost';
                        const reactivateBtn = this.isAdmin() && isClosed ? `
                            <button onclick="event.stopPropagation(); app.reactivateDealFromProfile('${dealId}')" class="text-blue-500 hover:text-blue-700 text-xs px-2 py-1 rounded transition ml-2" title="فعال‌سازی مجدد">
                                <i class="fa-solid fa-undo"></i>
                            </button>` : '';
                        return `
                            <div class="glass p-3 rounded relative">
                                ${reactivateBtn ? `<div class="absolute top-2 left-2">${reactivateBtn}</div>` : ''}
                                <div class="font-bold text-white ${reactivateBtn ? 'mr-8' : ''}">${this.escapeHtml(deal.Title || 'بدون عنوان')}</div>
                                <div class="text-sm text-gray-400">کد: ${deal.Code || '-'} | وضعیت: ${status === 'Won' ? 'موفق' : (status === 'Lost' ? 'ناموفق' : 'جاری')}</div>
                            </div>
                        `;
                    }).join('') : '<p class="text-gray-500 text-sm">معامله‌ای ثبت نشده</p>'}
                </div>
            </div>
            
            <div>
                <div class="flex justify-between items-center mb-3">
                    <h4 class="font-bold text-white">تاریخچه فعالیت‌ها و یادداشت‌ها (${activities?.length || 0})</h4>
                    <div class="flex gap-2">
                        <input type="text" id="activity-search-input" placeholder="جستجو در فعالیت‌ها..." 
                            class="input text-sm w-48" onkeyup="app.filterActivities(this.value)">
                        <select id="activity-filter-type" class="input text-sm w-32" onchange="app.filterActivities()">
                            <option value="">همه انواع</option>
                        </select>
                        <select id="activity-filter-status" class="input text-sm w-32" onchange="app.filterActivities()">
                            <option value="">همه وضعیت‌ها</option>
                            <option value="done">انجام شده</option>
                            <option value="pending">در انتظار</option>
                        </select>
                    </div>
                </div>
                <div id="activities-list" class="space-y-3 max-h-96 overflow-y-auto">
                    ${this.renderActivitiesList(activities)}
                </div>
            </div>
        `;
        
        // Store activities in state for filtering
        this.state.didarActivities = activities || [];
        this.state.didarActivitiesFiltered = activities || [];
        
        // Load activity types for filter dropdown
        this.loadActivityTypesForDidarFilter();
    },
    
    async viewDidarDeals(contactId) {
        if (!contactId) {
            Swal.fire('خطا', 'شناسه شخص مشخص نشده است', 'error');
            return;
        }
        
        console.log('viewDidarDeals called with contactId:', contactId, 'type:', typeof contactId);
        
        Swal.fire({
            title: 'در حال بارگذاری معاملات...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('get_legacy_deals', { contact_id: contactId });
            Swal.close();
            
            console.log('get_legacy_deals response:', res);
            
            if (res.status === 'success') {
                // Store deals in state for later use (to avoid API inconsistency)
                this.state.didarDeals = res.deals || [];
                this.renderDidarDeals(this.state.didarDeals);
                this.openModal('modalDidarDeals');
            } else {
                Swal.fire('خطا', res.message || 'خطا در دریافت معاملات', 'error');
            }
        } catch (error) {
            console.error('Error in viewDidarDeals:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در دریافت معاملات: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    renderDidarDeals(deals) {
        const container = document.getElementById('didar-deals-list');

        if (!container) {
            console.error('didar-deals-list container not found!');
            return;
        }

        if (!deals || deals.length === 0) {
            container.innerHTML = '<p class="text-center text-gray-500 py-8">معامله‌ای برای نمایش وجود ندارد</p>';
            return;
        }

        // Sort by register time (newest first)
        const sorted = [...deals].sort((a, b) => {
            const dateA = new Date(a.RegisterTime || a.register_time || a.register_date || 0);
            const dateB = new Date(b.RegisterTime || b.register_time || b.register_date || 0);
            return dateB - dateA;
        });

        container.innerHTML = sorted.map(deal => {
            const status = deal.Status || deal.status || 'Pending';
            const statusText = status === 'Won' ? 'موفق' : (status === 'Lost' ? 'ناموفق' : 'در جریان');
            const statusColor = status === 'Won' ? 'border-green-500 bg-green-500/10' :
                              (status === 'Lost' ? 'border-red-500 bg-red-500/10' : 'border-blue-500 bg-blue-500/10');
            const priceValue = deal.Price || deal.estimated_price || deal.final_price || 0;
            const price = priceValue ? parseInt(priceValue).toLocaleString() : '0';
            const registerDateValue = deal.RegisterTime || deal.register_time || deal.register_date;
            const registerDate = registerDateValue ? toPersianDateTime(registerDateValue) : '';

            const dealId = deal.Id || deal.didar_deal_id || deal.id || '';
            const title = deal.Title || deal.title || 'عنوان معامله';
            const code = deal.Code || deal.code || deal.didar_deal_id || '-';
            const description = deal.Description || deal.description || '';

            if (!dealId) {
                return `
                <div class="glass p-4 rounded-lg border-l-4 ${statusColor}">
                    <div class="flex justify-between items-start">
                        <div class="flex-1">
                            <div class="font-bold text-white mb-1">${this.escapeHtml(title)}</div>
                            <div class="text-sm text-gray-400">
                                <span>کد: ${this.escapeHtml(code)}</span>
                                <span class="mx-2">|</span>
                                <span class="px-2 py-0.5 rounded ${status === 'Won' ? 'bg-green-500/20 text-green-400' : (status === 'Lost' ? 'bg-red-500/20 text-red-400' : 'bg-blue-500/20 text-blue-400')}">${statusText}</span>
                                ${registerDate ? `<span class=\"mx-2\">|</span><span>${registerDate}</span>` : ''}
                            </div>
                            ${description ? `<div class=\"text-sm text-gray-300 mt-2\">${this.escapeHtml(description)}</div>` : ''}
                        </div>
                        <div class="text-left ml-4">
                            <div class="text-lg font-bold text-yellow-500">${price}</div>
                            <div class="text-xs text-gray-500">ارزش معامله</div>
                        </div>
                    </div>
                </div>
            `;
            }

            return `
                <div class="glass p-4 rounded-lg border-l-4 ${statusColor} cursor-pointer hover:bg-gray-800/50 transition" onclick="app.openDealModal('${this.escapeHtml(dealId)}')">
                    <div class="flex justify-between items-start">
                        <div class="flex-1">
                            <div class="font-bold text-white mb-1">${this.escapeHtml(title)}</div>
                            <div class="text-sm text-gray-400">
                                <span>کد: ${this.escapeHtml(code)}</span>
                                <span class="mx-2">|</span>
                                <span class="px-2 py-0.5 rounded ${status === 'Won' ? 'bg-green-500/20 text-green-400' : (status === 'Lost' ? 'bg-red-500/20 text-red-400' : 'bg-blue-500/20 text-blue-400')}">${statusText}</span>
                                ${registerDate ? `<span class=\"mx-2\">|</span><span>${registerDate}</span>` : ''}
                            </div>
                            ${description ? `<div class=\"text-sm text-gray-300 mt-2\">${this.escapeHtml(description)}</div>` : ''}
                        </div>
                        <div class="text-left ml-4">
                            <div class="text-lg font-bold text-yellow-500">${price}</div>
                            <div class="text-xs text-gray-500">ارزش معامله</div>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
    },
    
    async viewDealDetails(dealId) {
        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله مشخص نشده است', 'error');
            return;
        }
        
        // First, try to find deal in cached state (from deals list)
        let deal = null;
        if (this.state.didarDeals && this.state.didarDeals.length > 0) {
            deal = this.state.didarDeals.find(d => (d.Id || d.id) === dealId);
            if (deal) {
                console.log('Found deal in cache, using cached data:', {
                    Id: deal.Id,
                    Code: deal.Code,
                    Title: deal.Title
                });
            }
        }
        
        // If deal found in cache, use it directly and only fetch activities
        if (deal) {
            Swal.fire({
                title: 'در حال بارگذاری فعالیت‌ها...',
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading()
            });
            
            try {
                // Only fetch activities for this deal
                const activitiesRes = await this.req('get_deal_activities', { deal_id: dealId });
                Swal.close();
                
                if (activitiesRes.status === 'success') {
                    this.renderDealDetails(deal, activitiesRes.activities || []);
                    this.openModal('modalDealDetails');
                } else {
                    // Even if activities fail, show deal details without activities
                    this.renderDealDetails(deal, []);
                    this.openModal('modalDealDetails');
                }
            } catch (error) {
                console.error('Error fetching activities:', error);
                Swal.close();
                // Show deal details even if activities fetch fails
                this.renderDealDetails(deal, []);
                this.openModal('modalDealDetails');
            }
        } else {
            // Deal not in cache, fetch full details from API
            Swal.fire({
                title: 'در حال بارگذاری جزئیات معامله...',
                allowOutsideClick: false,
                didOpen: () => Swal.showLoading()
            });
            
            try {
                const res = await this.req('get_deal_details', { deal_id: dealId });
                Swal.close();
                
                if (res.status === 'success') {
                    this.renderDealDetails(res.deal || {}, res.activities || []);
                    this.openModal('modalDealDetails');
                } else {
                    Swal.fire('خطا', res.message || 'خطا در دریافت جزئیات معامله', 'error');
                }
            } catch (error) {
                console.error('Error in viewDealDetails:', error);
                Swal.close();
                Swal.fire('خطا', 'خطا در دریافت جزئیات معامله: ' + (error.message || 'خطای نامشخص'), 'error');
            }
        }
    },
    
    renderDealDetails(deal, activities) {
        // Update title
        const titleEl = document.getElementById('deal-details-title');
        if (titleEl) {
            titleEl.textContent = `جزئیات معامله: ${this.escapeHtml(deal.Title || 'بدون عنوان')}`;
        }
        
        // Render deal info
        const infoContainer = document.getElementById('deal-details-content');
        if (!infoContainer) {
            console.error('deal-details-content container not found!');
            return;
        }
        
        const status = deal.Status || 'Pending';
        const statusText = status === 'Won' ? 'موفق' : (status === 'Lost' ? 'ناموفق' : 'جاری');
        const statusColor = status === 'Won' ? 'text-green-400' : (status === 'Lost' ? 'text-red-400' : 'text-blue-400');
        const price = deal.Price ? parseInt(deal.Price).toLocaleString() : '0';
        const finalPrice = deal.FinalAdjustmentPrice ? parseInt(deal.FinalAdjustmentPrice).toLocaleString() : price;
        const registerDate = deal.RegisterTime ? toPersianDateTime(deal.RegisterTime) : '';
        const wonDate = deal.ChangeToWonTime ? toPersianDateTime(deal.ChangeToWonTime) : '';
        const lostDate = deal.ChangeToLossTime ? toPersianDateTime(deal.ChangeToLossTime) : '';
        const lastUpdate = deal.LastUpdateTime ? toPersianDateTime(deal.LastUpdateTime) : '';
        
        const contact = deal.Contact || deal.Person || {};
        const owner = deal.Owner || {};
        const items = deal.Items || [];
        
        infoContainer.innerHTML = `
            <!-- Basic Info -->
            <div class="glass p-6 rounded-lg">
                <h4 class="font-bold text-white text-lg mb-4">اطلاعات پایه</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <div class="text-xs text-gray-400 mb-1">عنوان</div>
                        <div class="text-white font-bold">${this.escapeHtml(deal.Title || 'بدون عنوان')}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">کد معامله</div>
                        <div class="text-white">${deal.Code || '-'}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">وضعیت</div>
                        <div class="${statusColor} font-bold">${statusText}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">مبلغ</div>
                        <div class="text-yellow-500 font-bold">${price} تومان</div>
                    </div>
                    ${finalPrice !== price ? `
                    <div>
                        <div class="text-xs text-gray-400 mb-1">مبلغ نهایی (پس از تعدیل)</div>
                        <div class="text-yellow-500 font-bold">${finalPrice} تومان</div>
                    </div>
                    ` : ''}
                    <div>
                        <div class="text-xs text-gray-400 mb-1">احتمال موفقیت</div>
                        <div class="text-white">${deal.Probability || 0}%</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">تاریخ ثبت</div>
                        <div class="text-white">${registerDate || '-'}</div>
                    </div>
                    ${wonDate ? `
                    <div>
                        <div class="text-xs text-gray-400 mb-1">تاریخ موفقیت</div>
                        <div class="text-green-400">${wonDate}</div>
                    </div>
                    ` : ''}
                    ${lostDate ? `
                    <div>
                        <div class="text-xs text-gray-400 mb-1">تاریخ ناموفق</div>
                        <div class="text-red-400">${lostDate}</div>
                    </div>
                    ` : ''}
                    ${lastUpdate ? `
                    <div>
                        <div class="text-xs text-gray-400 mb-1">آخرین به‌روزرسانی</div>
                        <div class="text-white">${lastUpdate}</div>
                    </div>
                    ` : ''}
                </div>
                ${deal.Description ? `
                <div class="mt-4">
                    <div class="text-xs text-gray-400 mb-1">توضیحات</div>
                    <div class="text-white text-sm whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(deal.Description))}</div>
                </div>
                ` : ''}
            </div>
            
            <!-- Contact Info -->
            <div class="glass p-6 rounded-lg">
                <h4 class="font-bold text-white text-lg mb-4">اطلاعات تماس</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <div class="text-xs text-gray-400 mb-1">نام</div>
                        <div class="text-white">${this.escapeHtml(contact.DisplayName || '-')}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">کد تماس</div>
                        <div class="text-white">${contact.Code || '-'}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">موبایل</div>
                        <div class="text-white">${contact.MobilePhone || '-'}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">ایمیل</div>
                        <div class="text-white">${contact.Email || '-'}</div>
                    </div>
                </div>
            </div>
            
            <!-- Owner Info -->
            <div class="glass p-6 rounded-lg">
                <h4 class="font-bold text-white text-lg mb-4">مسئول معامله</h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <div class="text-xs text-gray-400 mb-1">نام</div>
                        <div class="text-white">${this.escapeHtml(owner.DisplayName || '-')}</div>
                    </div>
                    <div>
                        <div class="text-xs text-gray-400 mb-1">ایمیل</div>
                        <div class="text-white">${owner.Email || '-'}</div>
                    </div>
                </div>
            </div>
            
            <!-- Deal Items -->
            ${items.length > 0 ? `
            <div class="glass p-6 rounded-lg">
                <h4 class="font-bold text-white text-lg mb-4">آیتم‌های معامله</h4>
                <div class="space-y-3">
                    ${items.map(item => `
                        <div class="bg-gray-800/50 p-4 rounded-lg">
                            <div class="flex justify-between items-start">
                                <div class="flex-1">
                                    <div class="font-bold text-white mb-1">${this.escapeHtml(item.Title || item.TitleForInvoice || 'بدون عنوان')}</div>
                                    <div class="text-sm text-gray-400">
                                        <span>تعداد: ${item.Quantity || 1}</span>
                                        ${item.Unit ? `<span class="mx-2">|</span><span>واحد: ${item.Unit}</span>` : ''}
                                    </div>
                                    ${item.Description ? `<div class="text-sm text-gray-300 mt-2">${this.escapeHtml(this.stripHtml(item.Description))}</div>` : ''}
                                </div>
                                <div class="text-left ml-4">
                                    <div class="text-yellow-500 font-bold">${item.PriceAfterDiscount ? parseInt(item.PriceAfterDiscount).toLocaleString() : (item.Price ? parseInt(item.Price).toLocaleString() : '0')} تومان</div>
                                    ${item.Discount && item.Discount > 0 ? `
                                    <div class="text-xs text-gray-400 line-through">${parseInt(item.Price || 0).toLocaleString()} تومان</div>
                                    <div class="text-xs text-green-400">تخفیف: ${item.DiscountPercent ? item.DiscountPercent + '%' : parseInt(item.Discount).toLocaleString() + ' تومان'}</div>
                                    ` : ''}
                                </div>
                            </div>
                        </div>
                    `).join('')}
                </div>
            </div>
            ` : ''}
        `;
        
        // Render activities
        const activitiesContainer = document.getElementById('deal-activities-content');
        if (activitiesContainer) {
            if (activities && activities.length > 0) {
                this.renderDidarActivities(activities, activitiesContainer);
            } else {
                activitiesContainer.innerHTML = '<p class="text-center text-gray-500 py-8">فعالیتی برای این معامله ثبت نشده است</p>';
            }
        }
    },
    
    showDealDetailsTab(tab) {
        // Hide all tabs
        document.querySelectorAll('.deal-details-tab').forEach(el => el.classList.add('hidden'));
        
        // Remove active class from all tab buttons
        document.querySelectorAll('[id^="deal-tab-"]').forEach(el => {
            el.classList.remove('text-white', 'border-blue-500');
            el.classList.add('text-gray-400', 'border-transparent');
        });
        
        // Show selected tab
        const tabContent = document.getElementById(`deal-details-${tab}`);
        const tabButton = document.getElementById(`deal-tab-${tab}`);
        
        if (tabContent) {
            tabContent.classList.remove('hidden');
        }
        
        if (tabButton) {
            tabButton.classList.remove('text-gray-400', 'border-transparent');
            tabButton.classList.add('text-white', 'border-blue-500');
        }
    },
    
    async viewDidarActivities(contactId) {
        if (!contactId) {
            Swal.fire('خطا', 'شناسه شخص مشخص نشده است', 'error');
            return;
        }
        
        console.log('viewDidarActivities called with contactId:', contactId, 'type:', typeof contactId);
        
        Swal.fire({
            title: 'در حال بارگذاری فعالیت‌ها...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('get_lead_didar_data', { contact_id: contactId });
            Swal.close();
            
            console.log('get_lead_didar_data response:', res);
            
            if (res.status === 'success') {
                this.renderDidarActivities(res.activities || []);
                this.openModal('modalDidarActivities');
            } else {
                Swal.fire('خطا', res.message || 'خطا در دریافت فعالیت‌ها', 'error');
            }
        } catch (error) {
            console.error('Error in viewDidarActivities:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در دریافت فعالیت‌ها: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    renderDidarActivities(activities, container = null) {
        // If container not provided, use default
        if (!container) {
            container = document.getElementById('didar-activities-list');
        }
        
        if (!container) {
            console.error('Activities container not found!');
            return;
        }
        
        if (!activities || activities.length === 0) {
            container.innerHTML = '<p class="text-center text-gray-500 py-8">فعالیتی یافت نشد</p>';
            return;
        }
        
        const resolveUserName = (didarId) => {
            const users = this.state.didarUsers || [];
            if (!didarId || !users.length) return '';
            const user = users.find(u =>
                u.owner_id === didarId ||
                u.didar_user_id === didarId ||
                u.id === didarId
            );
            return user?.display_name || '';
        };

        // Sort by RegisterDate (newest first)
        const sorted = [...activities].sort((a, b) => {
            const dateA = a.RegisterDate || a.register_date ? new Date(a.RegisterDate || a.register_date) : new Date(0);
            const dateB = b.RegisterDate || b.register_date ? new Date(b.RegisterDate || b.register_date) : new Date(0);
            return dateB - dateA;
        });

        container.innerHTML = sorted.map(act => {
            const isDone = act.IsDone === true || act.IsDone === 1 || act.is_done === 1 || act.is_done === true;
            const isNote = (act.ActivityTypeId || act.activity_type_id) === '00000000-0000-0000-0000-000000000000';
            const activityType = act.ActivityType?.Title || act.ActivityTypeTitle || act.activity_type_title || (isNote ? 'یادداشت' : 'فعالیت');

            let raw = act.raw_json;
            if (typeof raw === 'string') {
                try {
                    raw = JSON.parse(raw);
                } catch (_) {
                    raw = null;
                }
            }

            const creator = act.Creator?.DisplayName
                || act.CreatorName
                || act.creator_name
                || raw?.creator_name
                || resolveUserName(act.creator_didar_id)
                || resolveUserName(act.owner_didar_id)
                || 'نامشخص';

            const owner = act.Owner?.DisplayName
                || act.OwnerName
                || act.owner_name
                || raw?.owner_name
                || resolveUserName(act.owner_didar_id)
                || 'نامشخص';

            const note = act.Note || act.note || '';
            const resultNote = act.ResultNote || act.result_note || '';
            const registerDateRaw = act.RegisterDate || act.register_date || act.registerDate || '';
            const doneDateRaw = act.DoneDate || act.done_date || act.doneDate || '';
            const dueDateRaw = act.DueDate || act.due_date || act.dueDate || '';
            const registerDate = registerDateRaw ? toPersianDateTime(registerDateRaw) : '';
            const doneDate = doneDateRaw ? toPersianDateTime(doneDateRaw) : '';
            const dueDate = dueDateRaw ? toPersianDateTime(dueDateRaw) : '';
            const isScheduled = act.is_scheduled === 1 || act.is_scheduled === true || (!isDone && dueDate);
            const registerDateCompare = act.register_date || act.RegisterDate || '';
            const isEdited = act.last_update_time && act.last_update_time !== registerDateCompare;
            
            // Determine status and date to show
            let statusBadge = '';
            let dateInfo = '';
            
            if (isDone) {
                statusBadge = '<span class="text-xs bg-green-500/20 text-green-300 px-2 py-0.5 rounded">✓ انجام شده</span>';
                if (doneDate) {
                    dateInfo = `<div class="text-xs text-green-400 mt-1"><i class="fa-solid fa-check-circle ml-1"></i> تاریخ انجام: ${doneDate}</div>`;
                }
            } else if (isScheduled) {
                statusBadge = '<span class="text-xs bg-yellow-500/20 text-yellow-300 px-2 py-0.5 rounded">⏰ برنامه‌ریزی شده</span>';
                if (dueDate) {
                    dateInfo = `<div class="text-xs text-yellow-400 mt-1"><i class="fa-solid fa-clock ml-1"></i> موعد: ${dueDate}</div>`;
                }
            } else {
                statusBadge = '<span class="text-xs bg-blue-500/20 text-blue-300 px-2 py-0.5 rounded">📝 ثبت شده</span>';
            }
            
            const activityId = act.Id || act.didar_activity_id || act.id;
            const deleteBtn = this.isAdmin() ? `
                <button onclick="event.stopPropagation(); app.deleteActivity('${activityId}')" 
                        class="text-red-500 hover:text-red-700 text-xs px-2 py-1 rounded transition" 
                        title="حذف فعالیت">
                    <i class="fa-solid fa-trash"></i>
                </button>` : '';
            
            const editBtn = !isDone && isScheduled ? `
                <button onclick="event.stopPropagation(); app.openEditScheduledActivityModal('${activityId}')" 
                        class="text-blue-500 hover:text-blue-700 text-xs px-2 py-1 rounded transition" 
                        title="ویرایش فعالیت">
                    <i class="fa-solid fa-edit"></i>
                </button>` : '';
            
            return `
                <div class="glass p-4 rounded-lg border-l-4 ${isDone ? 'border-green-500' : isScheduled ? 'border-yellow-500' : 'border-blue-500'} relative cursor-default">
                    ${deleteBtn || editBtn ? `<div class="absolute top-2 left-2 z-10 flex gap-1">${editBtn}${deleteBtn}</div>` : ''}
                    ${isEdited ? `<div class="absolute top-2 right-2 z-10"><span class="text-xs bg-purple-500/20 text-purple-300 px-2 py-0.5 rounded" title="ویرایش شده"><i class="fa-solid fa-pencil ml-1"></i> ویرایش شده</span></div>` : ''}
                    <div class="flex justify-between items-start mb-2">
                        <div class="flex-1">
                            <div class="flex items-center gap-2 mb-1">
                                <span class="text-sm font-bold ${isNote ? 'text-purple-300' : 'text-blue-300'}">
                                    <i class="fa-solid fa-${isNote ? 'sticky-note' : 'calendar-check'} ml-1"></i>
                                    ${this.escapeHtml(activityType)}
                                </span>
                                ${statusBadge}
                            </div>
                            <div class="text-xs text-gray-400 mb-2 space-y-1">
                                ${registerDate ? `<div><i class="fa-solid fa-calendar-plus ml-1"></i> تاریخ ثبت: ${registerDate}</div>` : ''}
                                ${dueDate ? `<div><i class="fa-solid fa-calendar-clock ml-1"></i> تاریخ/ساعت برنامه‌ریزی: ${dueDate}</div>` : ''}
                                ${isScheduled && !dueDate ? `<div><i class="fa-solid fa-clock ml-1"></i> فعالیت برنامه‌ریزی شده (تاریخ مشخص نشده)</div>` : ''}
                                ${doneDate && isDone ? `<div><i class="fa-solid fa-check-double ml-1"></i> تاریخ انجام: ${doneDate}</div>` : ''}
                            </div>
                            <div class="text-xs text-gray-500 mb-2">
                                <span>ایجاد کننده: ${this.escapeHtml(creator)}</span>
                                ${owner !== creator ? `<span class="mx-2">|</span><span>مسئول: ${this.escapeHtml(owner)}</span>` : ''}
                            </div>
                            ${dateInfo}
                        </div>
                    </div>
                    
                    ${note ? `
                        <div class="mb-2">
                            <div class="text-xs text-gray-400 mb-1"><i class="fa-solid fa-note-sticky ml-1"></i> یادداشت:</div>
                            <div class="text-sm text-gray-300 bg-gray-800/50 p-2 rounded whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(note))}</div>
                        </div>
                    ` : ''}
                    
                    ${resultNote ? `
                        <div class="mb-2">
                            <div class="text-xs text-gray-400 mb-1"><i class="fa-solid fa-clipboard-check ml-1"></i> نتیجه:</div>
                            <div class="text-sm text-gray-300 bg-gray-800/50 p-2 rounded whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(resultNote))}</div>
                        </div>
                    ` : ''}
                    
                    ${!note && !resultNote ? `
                        <div class="text-xs text-gray-500 italic">بدون توضیحات</div>
                    ` : ''}
                </div>
            `;
        }).join('');
    },
    
    renderActivitiesList(activities) {
        if (!activities || activities.length === 0) {
            return '<p class="text-gray-500 text-sm text-center py-4">فعالیتی ثبت نشده است</p>';
        }
        
        const resolveUserName = (didarId) => {
            const users = this.state.didarUsers || [];
            if (!didarId || !users.length) return '';
            const user = users.find(u =>
                u.owner_id === didarId ||
                u.didar_user_id === didarId ||
                u.id === didarId
            );
            return user?.display_name || '';
        };

        // Sort by RegisterDate (newest first)
        const sorted = [...activities].sort((a, b) => {
            const dateA = a.RegisterDate || a.register_date ? new Date(a.RegisterDate || a.register_date) : new Date(0);
            const dateB = b.RegisterDate || b.register_date ? new Date(b.RegisterDate || b.register_date) : new Date(0);
            return dateB - dateA;
        });

        return sorted.map(act => {
            const isDone = act.IsDone === true || act.IsDone === 1 || act.is_done === 1 || act.is_done === true;
            const isNote = (act.ActivityTypeId || act.activity_type_id) === '00000000-0000-0000-0000-000000000000';
            const activityType = act.ActivityType?.Title || act.ActivityTypeTitle || act.activity_type_title || (isNote ? 'یادداشت' : 'فعالیت');

            let raw = act.raw_json;
            if (typeof raw === 'string') {
                try {
                    raw = JSON.parse(raw);
                } catch (_) {
                    raw = null;
                }
            }

            const creator = act.Creator?.DisplayName
                || act.CreatorName
                || act.creator_name
                || raw?.creator_name
                || resolveUserName(act.creator_didar_id)
                || resolveUserName(act.owner_didar_id)
                || 'نامشخص';

            const owner = act.Owner?.DisplayName
                || act.OwnerName
                || act.owner_name
                || raw?.owner_name
                || resolveUserName(act.owner_didar_id)
                || 'نامشخص';

            const note = act.Note || act.note || '';
            const resultNote = act.ResultNote || act.result_note || '';
            const registerDateRaw = act.RegisterDate || act.register_date || act.registerDate || '';
            const doneDateRaw = act.DoneDate || act.done_date || act.doneDate || '';
            const registerDate = registerDateRaw ? toPersianDateTime(registerDateRaw) : '';
            const doneDate = doneDateRaw ? toPersianDateTime(doneDateRaw) : '';
            
            // Check for due date - try multiple possible field names
            let dueDate = '';
            if (act.DueDate || act.due_date) {
                try {
                    dueDate = toPersianDateTime(act.DueDate || act.due_date);
                } catch (e) {
                    dueDate = act.DueDate || act.due_date; // Fallback to raw value
                }
            }

            
            // Check if activity is scheduled
            const hasDueDate = dueDate && dueDate.trim() !== '';
            const isScheduledFlag = act.is_scheduled === 1 || act.is_scheduled === true || act.is_scheduled === '1';
            const isScheduled = isScheduledFlag || (!isDone && hasDueDate);
            
            // Determine status badge
            let statusBadge = '';
            if (isDone) {
                statusBadge = '<span class="text-xs bg-green-500/20 text-green-300 px-2 py-0.5 rounded">✓ انجام شده</span>';
            } else if (isScheduled) {
                statusBadge = '<span class="text-xs bg-yellow-500/20 text-yellow-300 px-2 py-0.5 rounded">⏰ برنامه‌ریزی شده</span>';
            } else {
                statusBadge = '<span class="text-xs bg-blue-500/20 text-blue-300 px-2 py-0.5 rounded">📝 ثبت شده</span>';
            }
            
            const activityId = act.Id || act.didar_activity_id || act.id;
            
            return `
                <div class="glass p-4 rounded-lg border-r-4 ${isDone ? 'border-green-500' : isScheduled ? 'border-yellow-500' : 'border-blue-500'} cursor-default">
                    <div class="flex justify-between items-start mb-2">
                        <div class="flex-1">
                            <div class="flex items-center gap-2 mb-1">
                                <span class="text-xs font-bold ${isNote ? 'text-purple-300' : 'text-blue-300'}">
                                    <i class="fa-solid fa-${isNote ? 'sticky-note' : (isDone ? 'check-circle' : isScheduled ? 'clock' : 'calendar-check')} ml-1"></i>
                                    ${this.escapeHtml(activityType)}
                                </span>
                                ${statusBadge}
                            </div>
                            <div class="font-bold text-white text-sm mb-1">${this.escapeHtml(act.Title || 'بدون عنوان')}</div>
                        </div>
                        <div class="text-left text-xs text-gray-400">
                            ${registerDate ? `<div><i class="fa-solid fa-calendar-plus ml-1"></i> ثبت: ${registerDate}</div>` : ''}
                            ${doneDate ? `<div class="text-green-400 mt-1"><i class="fa-solid fa-check-circle ml-1"></i> انجام: ${doneDate}</div>` : ''}
                            ${dueDate && isScheduled ? `<div class="text-yellow-400 mt-1"><i class="fa-solid fa-calendar-clock ml-1"></i> موعد: ${dueDate}</div>` : ''}
                            ${isScheduled && !dueDate ? `<div class="text-yellow-400 mt-1"><i class="fa-solid fa-clock ml-1"></i> برنامه‌ریزی شده (تاریخ مشخص نشده)</div>` : ''}
                        </div>
                    </div>
                    
                    ${note ? `
                        <div class="mb-2">
                            <div class="text-xs text-gray-400 mb-1">یادداشت:</div>
                            <div class="text-sm text-gray-300 bg-gray-800/50 p-2 rounded whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(note))}</div>
                        </div>
                    ` : ''}
                    
                    ${resultNote ? `
                        <div class="mb-2">
                            <div class="text-xs text-gray-400 mb-1">نتیجه:</div>
                            <div class="text-sm text-gray-300 bg-gray-800/50 p-2 rounded whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(resultNote))}</div>
                        </div>
                    ` : ''}
                    
                    <div class="flex justify-between items-center mt-2 pt-2 border-t border-gray-700">
                        <div class="text-xs text-gray-500">
                            <span>ثبت‌کننده: <span class="text-gray-400">${this.escapeHtml(creator)}</span></span>
                            ${owner !== creator ? `<span class="mr-3">مسئول: <span class="text-gray-400">${this.escapeHtml(owner)}</span></span>` : ''}
                        </div>
                        ${act.Duration ? `<div class="text-xs text-gray-500">مدت: ${act.Duration} دقیقه</div>` : ''}
                    </div>
                </div>
            `;
        }).join('');
    },
    
    filterActivities(searchText = null, filterType = null, filterStatus = null) {
        if (!this.state.didarActivities) return;
        
        const search = searchText !== null ? searchText : (document.getElementById('activity-search-input')?.value || '');
        const type = filterType !== null ? filterType : (document.getElementById('activity-filter-type')?.value || '');
        const status = filterStatus !== null ? filterStatus : (document.getElementById('activity-filter-status')?.value || '');
        
        let filtered = [...this.state.didarActivities];
        
        // Filter by search text
        if (search) {
            const searchLower = search.toLowerCase();
            filtered = filtered.filter(act => {
                const title = (act.Title || '').toLowerCase();
                const note = (act.Note || '').toLowerCase();
                const resultNote = (act.ResultNote || '').toLowerCase();
                const activityType = (act.ActivityType?.Title || act.ActivityTypeTitle || '').toLowerCase();
                const creator = (act.Creator?.DisplayName || act.CreatorName || '').toLowerCase();
                return title.includes(searchLower) || 
                       note.includes(searchLower) || 
                       resultNote.includes(searchLower) ||
                       activityType.includes(searchLower) ||
                       creator.includes(searchLower);
            });
        }
        
        // Filter by activity type
        if (type) {
            filtered = filtered.filter(act => {
                const actTypeId = act.ActivityTypeId || '';
                return actTypeId === type;
            });
        }
        
        // Filter by status
        if (status) {
            if (status === 'done') {
                filtered = filtered.filter(act => act.IsDone === true || act.IsDone === 1);
            } else if (status === 'pending') {
                filtered = filtered.filter(act => !act.IsDone || act.IsDone === 0 || act.IsDone === false);
            }
        }
        
        this.state.didarActivitiesFiltered = filtered;
        
        const listContainer = document.getElementById('activities-list');
        if (listContainer) {
            listContainer.innerHTML = this.renderActivitiesList(filtered);
        }
    },
    
    async loadActivityTypesForDidarFilter() {
        try {
            const res = await this.req('get_activity_types');
            if (res.types && Array.isArray(res.types)) {
                const select = document.getElementById('activity-filter-type');
                if (select) {
                    // Clear existing options except first one
                    select.innerHTML = '<option value="">همه انواع</option>';
                    res.types.forEach(type => {
                        const option = document.createElement('option');
                        option.value = type.Id;
                        option.textContent = type.Title;
                        select.appendChild(option);
                    });
                }
            }
        } catch (error) {
            console.error('Error loading activity types for filter:', error);
        }
    },
    
    switchProfileTab(tab) {
        document.querySelectorAll('.profile-tab').forEach(t => {
            t.classList.remove('active', 'text-yellow-500', 'border-b-2', 'border-yellow-500');
            t.classList.add('text-gray-400');
        });
        document.querySelectorAll('.profile-tab-content').forEach(c => c.classList.add('hidden'));
        
        const tabEl = document.querySelector(`.profile-tab[onclick*="${tab}"]`);
        if(tabEl) {
            tabEl.classList.add('active', 'text-yellow-500', 'border-b-2', 'border-yellow-500');
            tabEl.classList.remove('text-gray-400');
        }
        
        const contentEl = document.getElementById(`profile-tab-${tab}`);
        if(contentEl) contentEl.classList.remove('hidden');
        
        // Load support logs when switching to support tab
        if (tab === 'support' && window.currentContactId) {
            this.loadSupportLogs(window.currentContactId);
        }
    },
    
    async loadActivityTypesForModal() {
        const selects = [
            document.getElementById('activity-type-select'),
            document.getElementById('deal-activity-type-select')
        ].filter(Boolean);
        if (selects.length === 0) return;

        selects.forEach(sel => sel.innerHTML = '<option value="">-- انتخاب نوع فعالیت --</option>');

        // Load activity types if not already loaded
        if (!this.state.activityTypes || this.state.activityTypes.length === 0) {
            try {
                await this.loadActivityTypes();
            } catch (error) {
                console.error('Error loading activity types:', error);
                // Fallback to hardcoded types if API fails
                this.state.activityTypes = [
                    { Id: '2a335dc1-4834-484b-b6dc-9ddd97eef67e', Title: 'تماس ورودی', Direction: 'incoming', Category: 'call', IsDisabled: false },
                    { Id: '790079e8-74ac-4760-9527-097627d3e7ec', Title: 'تماس خروجی', Direction: 'outgoing', Category: 'call', IsDisabled: false },
                    { Id: '3b446dc2-5945-595c-c7ed-0eee08ff78fd', Title: 'چت ورودی', Direction: 'incoming', Category: 'chat', IsDisabled: false },
                    { Id: '4c557ed3-6a56-606d-d8fe-1fff19gg89ge', Title: 'چت خروجی', Direction: 'outgoing', Category: 'chat', IsDisabled: false }
                ];
            }
        }

        if (!this.state.activityTypes || this.state.activityTypes.length === 0) {
            const fallback = this.getFallbackActivityTypes();
            if (fallback.length) {
                this.state.activityTypes = fallback;
            }
        }

        const types = (this.state.activityTypes && this.state.activityTypes.length > 0)
            ? this.state.activityTypes.filter(t => !t.IsDisabled)
            : [
                { Id: '2a335dc1-4834-484b-b6dc-9ddd97eef67e', Title: 'تماس ورودی', Direction: 'incoming', Category: 'call' },
                { Id: '790079e8-74ac-4760-9527-097627d3e7ec', Title: 'تماس خروجی', Direction: 'outgoing', Category: 'call' },
                { Id: '3b446dc2-5945-595c-c7ed-0eee08ff78fd', Title: 'چت ورودی', Direction: 'incoming', Category: 'chat' },
                { Id: '4c557ed3-6a56-606d-d8fe-1fff19gg89ge', Title: 'چت خروجی', Direction: 'outgoing', Category: 'chat' }
            ];

        const buildOptions = (sel) => {
            if (!sel) return;
            sel.innerHTML = '<option value="">-- انتخاب نوع فعالیت --</option>';
            types.forEach(type => {
                const opt = document.createElement('option');
                opt.value = type.Id;
                opt.textContent = type.Title || type.Id;
                opt.dataset.direction = type.Direction || 'outgoing';
                sel.appendChild(opt);
            });
        };

        selects.forEach(buildOptions);

        const handleChange = (sel) => {
            sel.addEventListener('change', () => {
                const selectedOption = sel.options[sel.selectedIndex];
                const directionInput = sel.id === 'deal-activity-type-select'
                    ? document.getElementById('deal-activity-direction')
                    : document.getElementById('activity-direction');
                if (selectedOption && directionInput) {
                    directionInput.value = selectedOption.dataset.direction || 'outgoing';
                }
            });
        };

        selects.forEach(handleChange);

        // Reset purchase question visibility
        const purchaseSelect = document.getElementById('activity-has-purchase');
        if (purchaseSelect) {
            purchaseSelect.value = '';
        }
        const dealPurchaseSelect = document.getElementById('deal-activity-has-purchase');
        if (dealPurchaseSelect) {
            dealPurchaseSelect.value = '';
        }
        this.handlePurchaseOtherNumberChange();
    },
    
    // SYNC
    async sync() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر می‌تواند همگام‌سازی را انجام دهد', 'error');
            return;
        }
        
        Swal.fire({
            title: 'همگام‌سازی',
            text: 'در حال همگام‌سازی با دیدار...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        const res = await this.req('sync', {entity_type: 'all'});
        Swal.close();
        
        if(res.status === 'success') {
            Swal.fire({
                icon: 'success',
                title: 'موفق',
                text: res.message || 'همگام‌سازی با موفقیت انجام شد',
                timer: 2000
            });
            this.refresh();
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },

    // SAVE USER
    async saveUser(e) {
        e.preventDefault();
        const data = Object.fromEntries(new FormData(e.target));
        const res = await this.req('save_user', data);
        if(res.status === 'success') {
            this.closeModal('modalUser');
            this.loadUsers();
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'ذخیره شد', showConfirmButton:false, timer:2000});
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    openUserModal(userId = null) {
        const form = document.querySelector('#modalUser form');
        form.reset();
        document.getElementById('user-form-id').value = userId || '';
        
        // Load Didar users
        const select = document.getElementById('user-didar-select');
        select.innerHTML = '<option value="">-- انتخاب کاربر از دیدار --</option>';
        if(window.didarUsers) {
            window.didarUsers.forEach(u => {
                const opt = document.createElement('option');
                opt.value = u.Id || u.UserId;
                opt.textContent = u.DisplayName + ' (' + u.UserName + ')';
                select.appendChild(opt);
            });
        }
        
        if(userId) {
            // Load user data for edit
            // This would require a get_user endpoint
        }
        
        this.openModal('modalUser');
    },
    
    openEditUserRoleModal(userId, currentRole, userName) {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر می‌تواند نقش کاربران را تغییر دهد', 'error');
            return;
        }
        const form = document.querySelector('#modalEditUserRole form');
        form.reset();
        document.getElementById('edit-role-user-id').value = userId;
        document.getElementById('edit-role-user-name').textContent = userName;
        document.getElementById('edit-role-select').value = currentRole;
        this.openModal('modalEditUserRole');
    },
    
    async saveUserRole(e) {
        e.preventDefault();
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر می‌تواند نقش کاربران را تغییر دهد', 'error');
            return;
        }
        const form = e.target;
        const data = Object.fromEntries(new FormData(form));
        
        Swal.fire({
            title: 'در حال ذخیره...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('save_user', data);
            Swal.close();
            if (res.status === 'success') {
                Swal.fire({toast: true, position: 'bottom-end', icon: 'success', title: 'نقش کاربر تغییر یافت', showConfirmButton: false, timer: 2000});
                this.closeModal('modalEditUserRole');
                this.loadUsers();
            } else {
                Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
            }
        } catch (error) {
            console.error('Error in saveUserRole:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در تغییر نقش: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    openChangePasswordModal(userId = null) {
        const form = document.querySelector('#modalChangePassword form');
        form.reset();
        const targetUserId = userId || window.currentUserId || window.CURRENT_USER_ID || 0;
        document.getElementById('change-password-user-id').value = targetUserId;
        
        // If admin changing someone else's password, hide current password field
        const isAdmin = (window.APP_USER?.role === 'admin') ? 'true' : 'false';
        const isOwnPassword = targetUserId == window.CURRENT_USER_ID || 0;
        
        const currentSection = document.getElementById('change-password-current-section');
        const currentInput = document.getElementById('change-password-current');
        
        if (isAdmin && !isOwnPassword) {
            currentSection.style.display = 'none';
            currentInput.removeAttribute('required');
        } else {
            currentSection.style.display = 'block';
            currentInput.setAttribute('required', 'required');
        }
        
        this.openModal('modalChangePassword');
    },
    
    async changePassword(e) {
        e.preventDefault();
        const formData = new FormData(e.target);
        const data = Object.fromEntries(formData);
        
        const res = await this.req('change_password', data);
        if(res.status === 'success') {
            this.closeModal('modalChangePassword');
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:res.message || 'رمز عبور تغییر یافت', showConfirmButton:false, timer:2000});
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    openChangePasswordModal(userId = null) {
        const form = document.querySelector('#modalChangePassword form');
        form.reset();
        const targetUserId = userId || window.CURRENT_USER_ID || 0;
        document.getElementById('change-password-user-id').value = targetUserId;
        
        // If admin changing someone else's password, hide current password field
        const isAdmin = (window.APP_USER?.role === 'admin') ? 'true' : 'false';
        const isOwnPassword = targetUserId == window.CURRENT_USER_ID || 0;
        
        const currentSection = document.getElementById('change-password-current-section');
        const currentInput = document.getElementById('change-password-current');
        
        if (isAdmin && !isOwnPassword) {
            currentSection.style.display = 'none';
            currentInput.removeAttribute('required');
        } else {
            currentSection.style.display = 'block';
            currentInput.setAttribute('required', 'required');
        }
        
        this.openModal('modalChangePassword');
    },
    
    async changePassword(e) {
        e.preventDefault();
        const formData = new FormData(e.target);
        const data = Object.fromEntries(formData);
        
        const res = await this.req('change_password', data);
        if(res.status === 'success') {
            this.closeModal('modalChangePassword');
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:res.message || 'رمز عبور تغییر یافت', showConfirmButton:false, timer:2000});
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    async changePasswordFromSettings(e) {
        e.preventDefault();
        const formData = new FormData(e.target);
        const data = Object.fromEntries(formData);
        
        const res = await this.req('change_password', data);
        if(res.status === 'success') {
            // Reset form
            e.target.reset();
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:res.message || 'رمز عبور تغییر یافت', showConfirmButton:false, timer:2000});
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    // SAVE ACTIVITY
    async saveActivity(e) {
        e.preventDefault();
        
        try {
            console.log('saveActivity called');
            const form = e.target;
            const contactId = window.currentContactId || form.contact_id.value;
            const dealSelectVal = document.getElementById('activity-deal-select')?.value || '';
            const dealHiddenVal = document.getElementById('activity-deal-id')?.value || '';
            const dealId = dealSelectVal || dealHiddenVal;
            
            // Check if selected deal is closed
            if (this.state.currentDeals && dealId) {
                const currentDeal = this.state.currentDeals.find(d => (d.didar_deal_id || d.id || d.deal_id) === dealId);
                if (currentDeal && (currentDeal.status === 'Won' || currentDeal.status === 'Lost')) {
                    Swal.fire('خطا', 'این معامله بسته شده است و امکان ثبت فعالیت ندارد.', 'error');
                    return;
                }
            }
            
            if(!contactId) {
                Swal.fire('خطا', 'لطفاً شخص را انتخاب کنید', 'error');
                return;
            }
            
            console.log('Contact ID:', contactId);
            
            // بررسی و تنظیم وضعیت required فیلد خرید قبلی قبل از submit
            try {
                this.handleActivityFormVisibility();
            } catch(err) {
                console.error('Error in handleActivityFormVisibility:', err);
            }
            
                const pipelineStage = form.pipeline_stage?.value || '';
                const isScheduled = form.is_scheduled?.checked || false;

                // pipelineStage is only required when a deal is selected
                if (dealId && !pipelineStage) {
                    Swal.fire('خطا', 'لطفاً مرحله کاریز معامله را انتخاب کنید', 'error');
                return;
            }

            if (!form.checkValidity()) {
                form.reportValidity();
                return;
            }
            
            // Get due date if scheduled
            let dueDate = '';
            // isScheduled already declared above
            if (isScheduled) {
                const persianDateValue = form.due_date?.value || '';
                const hourSelect = document.getElementById('activity-due-hour');
                const minuteSelect = document.getElementById('activity-due-minute');
                const hour = hourSelect?.value || '00';
                const minute = minuteSelect?.value || '00';
                
                if (!persianDateValue || !persianDateValue.trim()) {
                    Swal.fire('خطا', 'لطفاً تاریخ را انتخاب کنید', 'error');
                    return;
                }
                
                if (!hour || hour === '') {
                    Swal.fire('خطا', 'لطفاً ساعت را انتخاب کنید', 'error');
                    return;
                }
                
                if (!minute || minute === '') {
                    Swal.fire('خطا', 'لطفاً دقیقه را انتخاب کنید', 'error');
                    return;
                }
                
                try {
                    // Convert Persian/Farsi digits to English digits
                    const persianToEnglish = (str) => {
                        const persianDigits = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
                        const englishDigits = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
                        let result = str;
                        for (let i = 0; i < persianDigits.length; i++) {
                            result = result.replace(new RegExp(persianDigits[i], 'g'), englishDigits[i]);
                        }
                        return result;
                    };
                    
                    // Convert Persian digits to English
                    const englishDateValue = persianToEnglish(persianDateValue.trim());
                    console.log('Original date:', persianDateValue);
                    console.log('Converted date:', englishDateValue);
                    
                    // Parse Persian date string (format: YYYY/MM/DD)
                    const dateParts = englishDateValue.split('/');
                    if (dateParts.length !== 3) {
                        throw new Error('فرمت تاریخ نامعتبر است');
                    }
                    
                    const year = parseInt(dateParts[0]);
                    const month = parseInt(dateParts[1]);
                    const day = parseInt(dateParts[2]);
                    const hours = parseInt(hour);
                    const minutes = parseInt(minute);
                    
                    console.log('Parsed values:', { year, month, day, hours, minutes });
                    
                    // Validate
                    if (isNaN(year) || isNaN(month) || isNaN(day) || isNaN(hours) || isNaN(minutes)) {
                        throw new Error('مقادیر تاریخ یا زمان نامعتبر است');
                    }
                    
                    // Additional validation
                    if (year < 1300 || year > 1500) {
                        throw new Error('سال نامعتبر است');
                    }
                    if (month < 1 || month > 12) {
                        throw new Error('ماه نامعتبر است');
                    }
                    if (day < 1 || day > 31) {
                        throw new Error('روز نامعتبر است');
                    }
                    
                    // Use the helper function that already works
                    // Format: HH:mm
                    const timeString = `${hour}:${minute}`;
                    dueDate = persianToGregorian(englishDateValue, timeString);
                    
                    if (!dueDate) {
                        throw new Error('خطا در تبدیل تاریخ. لطفاً تاریخ معتبری انتخاب کنید.');
                    }
                    
                    // persianToGregorian now returns format: YYYY-MM-DD HH:mm:ss (local timezone)
                    console.log('Converted date (local timezone):', dueDate);
                    
                    // Validate the final date
                    if (!dueDate || dueDate.length < 19) {
                        throw new Error('فرمت تاریخ نهایی نامعتبر است');
                    }
                    
                    console.log('Sending due date to server:', dueDate);
                    
                } catch (e) {
                    console.error('Error converting date:', e, 'Input:', persianDateValue);
                    Swal.fire('خطا', 'تاریخ وارد شده معتبر نیست. لطفاً تاریخ و زمان را به درستی وارد کنید. (' + (e.message || 'خطای نامشخص') + ')', 'error');
                    return;
                }
            }
            
                const data = {
                    contact_id: contactId,
                    deal_id: dealId,
                    activity_type_id: form.activity_type_id?.value || '',
                    note: form.note?.value || '',
                    result_note: form.result_note?.value || '',
                    direction: form.direction?.value || 'outgoing',
                    current_stage: form.current_stage?.value || 'keep_current',
                    pipeline_stage: form.pipeline_stage?.value || '',
                    has_previous_purchase_other_number: form.has_previous_purchase_other_number?.value || '',
                    previous_purchase_number: form.previous_purchase_number?.value || '',
                    referral_note: form.referral_note?.value || '',
                    failure_reason: form.failure_reason?.value || '',
                    failure_other_description: form.failure_other_description?.value || '',
                    is_done: form.is_done?.checked ? '1' : '0',
                    is_scheduled: isScheduled ? '1' : '0',
                    due_date: dueDate,
                    reminder_enabled: form.reminder_enabled?.checked ? '1' : '0',
                    reminder_offset_minutes: this.calculateReminderOffset(form)
                };
            
            console.log('Sending data:', data);
            
            Swal.showLoading();
            const res = await this.req('save_activity', data);
            Swal.close();
            
            console.log('Response received:', res);
            
            if(res.status === 'success') {
                Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'فعالیت ثبت شد', showConfirmButton:false, timer:2000});
                this.closeModal('modalActivity');
                form.reset();
                if(window.currentContactId) {
                    // Reload person profile to get updated virtual stage
                    await this.openPersonProfile(window.currentContactId);
                    // Reload deals to منعکس کردن تغییر مرحله کاریز
                    await this.loadDeals();
                    // اگر مودال معامله باز است، آن را بدون بستن رفرش کنیم
                    const dealModal = document.getElementById('modalDeal');
                    if (dealModal && !dealModal.classList.contains('hidden')) {
                        const currentDealId = this.state.currentDeal?.didar_deal_id || document.getElementById('deal-id')?.value || '';
                        if (currentDealId) {
                            await this.openDealModal(currentDealId);
                            this.showDealFormTab('activities');
                        }
                    }
                    // Also reload leads list if we're on leads view
                    if (document.getElementById('view-leads') && !document.getElementById('view-leads').classList.contains('hidden')) {
                        await this.loadLeads();
                    }
                }
            } else {
                Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
            }
        } catch(error) {
            console.error('Error in saveActivity:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در ثبت فعالیت: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    // HANDLE DEAL STATUS CHANGE
    handleDealStatusChange() {
        const status = document.getElementById('deal-status').value;
        const form = document.getElementById('deal-form');
        const paymentDetails = document.getElementById('deal-payment-details');
        const failureReasonSection = document.getElementById('deal-failure-reason-section');
        const failureReason = document.getElementById('deal-failure-reason');
        const failureOtherReason = document.getElementById('deal-failure-other-reason');
        const failureOtherDesc = document.getElementById('deal-failure-other-description');
        const pipelineStageSelect = document.getElementById('deal-pipeline-stage');
        const pipelineStageIndicator = document.getElementById('pipeline-stage-required-indicator');
        
        const isLost = status === 'lost';
        const shouldRequirePayments = status === 'won';

        if (paymentDetails) paymentDetails.style.display = 'block';

        if (failureReasonSection) failureReasonSection.classList.toggle('hidden', !isLost);
        if (failureReason) {
            failureReason.required = isLost;
            if (!isLost) failureReason.value = '';
        }
        if (failureOtherReason) failureOtherReason.classList.toggle('hidden', !isLost);
        if (failureOtherDesc) {
            failureOtherDesc.required = isLost;
            if (!isLost) {
                failureOtherDesc.value = '';
            } else {
                failureOtherDesc.placeholder = 'لطفاً توضیحات دلیل ناموفق بودن را وارد کنید';
            }
        }

        // Make pipeline stage optional when status is lost
        if (pipelineStageSelect) {
            pipelineStageSelect.required = !isLost;
            if (isLost && !pipelineStageSelect.value) {
                // Optionally set a default value for lost deals
                pipelineStageSelect.value = 'معامله ناموفق';
            }
        }
        // Update the required indicator
        if (pipelineStageIndicator) {
            pipelineStageIndicator.style.display = isLost ? 'none' : 'inline';
        }

        const payableAmount = document.getElementById('deal-payable-amount');
        const paymentConditions = document.getElementById('deal-payment-conditions');
        if (payableAmount) payableAmount.required = true;
        if (paymentConditions) paymentConditions.required = true;

        if (form) {
            const paymentDates = form.querySelectorAll('input[name="payment_date[]"]');
            const paymentAmounts = form.querySelectorAll('input[name="payment_amount[]"]');
            const paymentMethods = form.querySelectorAll('select[name="payment_method[]"]');
            const paymentDescriptions = form.querySelectorAll('textarea[name="payment_description[]"]');
        const paymentCardLast4s = form.querySelectorAll('input[name="payment_card_last4[]"]');
            paymentDates.forEach(input => input.required = shouldRequirePayments);
            paymentAmounts.forEach(input => input.required = shouldRequirePayments);
            paymentMethods.forEach(select => select.required = shouldRequirePayments);
            paymentDescriptions.forEach(textarea => textarea.required = shouldRequirePayments);
        }

        this.updateDealCurrentFields(null, status);
    },

    handleDealPipelineStageChange() {
        const stageSelect = document.getElementById('deal-pipeline-stage');
        const statusSelect = document.getElementById('deal-status');
        const stageVal = stageSelect?.value || '';
        const isFailed = stageVal === 'معامله ناموفق' || stageVal === 'deal_failed';
        const isSuccess = stageVal === 'اتمام' || stageVal === 'deal_success';
        if (statusSelect) {
            if (isFailed) {
                statusSelect.value = 'lost';
            } else if (isSuccess) {
                statusSelect.value = 'won';
            }
        }
        this.handleDealStatusChange();
        this.updateDealCurrentFields(stageVal, statusSelect?.value || '');
    },

    handleActivityStageChange(stageVal) {
        const statusSelect = document.getElementById('deal-status');
        const contactId = document.getElementById('deal-activity-contact-id')?.value || window.currentContactId || '';
        const dealId = document.getElementById('deal-activity-deal-id')?.value || document.getElementById('deal-id')?.value || '';
        const paymentsOk = this.hasCompletePaymentsForSuccess();
        const failureBox = document.getElementById('failure-reason-extra');
        const stageSelect = document.getElementById('activity-virtual-stage') || document.getElementById('deal-activity-stage');
        console.log('handleActivityStageChange', { stageVal, paymentsOk, contactId, dealId });

        if (stageVal === 'deal_failed' || stageVal === 'معامله ناموفق') {
            if (statusSelect) {
                statusSelect.value = 'lost';
                this.handleDealStatusChange();
            }
            const pipelineStageSelect = document.getElementById('deal-pipeline-stage');
            if (pipelineStageSelect) {
                pipelineStageSelect.value = 'معامله ناموفق';
            }
            this.toggleDealActivityFailure();
            this.toggleActivityFailureOtherReason();
            return;
        }

        if (stageVal === 'deal_success') {
            if (!paymentsOk) {
                Swal.fire('خطا', 'برای ثبت معامله موفق، جزئیات پرداخت باید کامل باشد.', 'error');
                if (stageSelect) stageSelect.value = this.state.activityStagePrev || '';
                console.warn('Blocked deal_success due to missing payments');
                return;
            }
            Swal.fire({
                icon: 'info',
                title: 'ثبت معامله موفق',
                text: 'معامله به عنوان فروش موفق ثبت و بایگانی می‌شود.',
                confirmButtonText: 'متوجه شدم'
            });
            if (statusSelect) {
                statusSelect.value = 'won';
                this.handleDealStatusChange();
            }
            const pipelineStageSelect = document.getElementById('deal-pipeline-stage');
            if (pipelineStageSelect) {
                pipelineStageSelect.value = 'اتمام';
            }
        }

        this.toggleDealActivityFailure();
        this.toggleActivityFailureOtherReason();
        this.updateDealCurrentFields(stageVal, statusSelect?.value || '');
        this.state.activityStagePrev = stageVal || '';
    },
    
	    // TOGGLE FAILURE OTHER REASON (deal form + deal-failed modal)
	    toggleFailureOtherReason() {
	        // Deal form in modalDeal (inline reason select)
	        const dealReasonSelect = document.getElementById('deal-failure-reason');
	        const dealOtherDiv = document.getElementById('deal-failure-other-reason');
	        const dealOtherDesc = document.getElementById('deal-failure-other-description');
	        const dealStatusSelect = document.getElementById('deal-status');

	        if (dealReasonSelect && dealOtherDiv && dealOtherDesc && dealStatusSelect) {
	            const reason = dealReasonSelect.value;
	            const dealStatus = dealStatusSelect.value;

	            // For lost deals, description is always shown + required
	            if (dealStatus === 'lost') {
	                dealOtherDiv.classList.remove('hidden');
	                dealOtherDesc.required = true;
	                dealOtherDesc.placeholder = 'لطفاً توضیحات دلیل ناموفق بودن را وارد کنید';
	            } else {
	                // For other statuses, only show description when "other" is selected
	                if (reason === 'other') {
	                    dealOtherDiv.classList.remove('hidden');
	                    dealOtherDesc.required = true;
	                } else {
	                    dealOtherDiv.classList.add('hidden');
	                    dealOtherDesc.required = false;
	                    dealOtherDesc.value = '';
	                }
	            }
	        }

	        // ModalDealFailed (quick mark as failed)
	        const modalReasonSelect = document.getElementById('failure-reason');
	        const modalOtherDiv = document.getElementById('failure-other-reason');
	        const modalOtherDesc = document.getElementById('failure-other-description');
	        if (modalReasonSelect && modalOtherDiv && modalOtherDesc) {
	            const reason = modalReasonSelect.value;
	            const shouldShow = !!reason;
	            modalOtherDiv.classList.toggle('hidden', !shouldShow);
	            modalOtherDesc.required = shouldShow;
	            if (!shouldShow) {
	                modalOtherDesc.value = '';
	            }
	        }
	    },

    populateActivityDealSelect(deals = [], selectedDealId = '', fallbackStage = '') {
        const select = document.getElementById('activity-deal-select');
        const hiddenDealId = document.getElementById('activity-deal-id');
        const stageDisplay = document.getElementById('activity-current-stage-display');
        const currentStageInput = document.getElementById('activity-current-stage');
        const pipelineSelect = document.getElementById('activity-virtual-stage');
        if (!select) return;

        const normalizedDeals = Array.isArray(deals) ? deals : [];
        // Cache for reuse when modal is opened بدون رفرش
        this.state.currentDeals = normalizedDeals;

        select.innerHTML = '<option value=\"\">-- انتخاب معامله مرتبط --</option>';
        normalizedDeals.forEach(d => {
            const option = document.createElement('option');
            const id = d.didar_deal_id || d.id || d.deal_id || '';
            option.value = id;
            option.dataset.stage = d.pipeline_stage || '';
            
            const isClosed = d.status === 'Won' || d.status === 'Lost';
            if (isClosed) {
                option.disabled = true;
                const statusText = d.status === 'Won' ? 'موفق' : 'ناموفق';
                option.textContent = `${d.title || 'معامله'} (${statusText}) - بسته شده`;
            } else {
                const stageLabel = option.dataset.stage ? ` | ${option.dataset.stage}` : '';
                option.textContent = `${d.title || 'معامله'}${stageLabel}`;
            }
            select.appendChild(option);
        });

        const updateStageDisplay = () => {
            const opt = select.selectedOptions[0];
            const stage = opt?.dataset.stage || fallbackStage || '';
            if (hiddenDealId) hiddenDealId.value = select.value || '';
            if (currentStageInput) currentStageInput.value = stage || 'keep_current';
            if (pipelineSelect) {
                pipelineSelect.value = stage || '';
            }
            if (stageDisplay) {
                stageDisplay.innerHTML = stage
                    ? `<span class=\"text-sm text-gray-200\">${stage}</span>`
                    : '<span class=\"text-sm text-gray-500\">مرحله تعیین نشده</span>';
            }
        };

        select.onchange = () => {
            updateStageDisplay();
            this.handleActivityDealSelection();
        };

        let presetId = selectedDealId || select.dataset.defaultSelected || '';
        if (!presetId && normalizedDeals.length === 1) {
            const d = normalizedDeals[0];
            const isClosed = d.status === 'Won' || d.status === 'Lost';
            if (!isClosed) {
                presetId = d.didar_deal_id || d.id || d.deal_id || '';
            }
        }
        if (presetId) {
            select.value = presetId;
        } else {
            select.value = '';
        }

        updateStageDisplay();
        this.handleActivityDealSelection();
    },
    
    // SAVE DEAL
    async saveDeal(e) {
        e.preventDefault();
        const form = e.target;
        const contactId = this.state.currentLead?.didar_contact_id || window.currentContactId || form.contact_id.value;
        const dealId = form.deal_id.value;
        let dealStatus = form.deal_status.value;
        let pipelineStage = form.pipeline_stage?.value || '';

        // COMPREHENSIVE LOGGING FOR TROUBLESHOOTING DATA FRAGMENTATION
        console.log('[DEAL_SAVE] Starting deal save process', {
            timestamp: new Date().toISOString(),
            contactId: contactId,
            dealId: dealId,
            windowCurrentContactId: window.currentContactId,
            formContactId: form.contact_id?.value,
            formTitle: form.title?.value,
            dealStatus: dealStatus,
            pipelineStage: pipelineStage,
            isNewDeal: !dealId,
            formDataKeys: Array.from(new FormData(form).keys()),
            userAgent: navigator.userAgent,
            url: window.location.href,
            sessionStorage: Object.keys(sessionStorage),
            localStorage: Object.keys(localStorage)
        });

        // For existing deals, status and stage are ALWAYS required
        // No fallback to existing values - user must explicitly select new values

        const isNewDeal = !dealId;
        const productCheckboxes = form.querySelectorAll('input[name="product_ids[]"]:checked');
        let productIds = Array.from(productCheckboxes).map(cb => cb.value);
        
        if(!contactId) {
            Swal.fire('خطا', 'لطفاً شخص را انتخاب کنید', 'error');
            return;
        }
        
        if(!dealStatus) {
            Swal.fire('خطا', 'لطفاً وضعیت معامله را انتخاب کنید', 'error');
            return;
        }
        // Pipeline stage is only required if status is NOT lost
        if(!pipelineStage && dealStatus !== 'lost') {
            Swal.fire('خطا', 'لطفاً مرحله کاریز معامله را انتخاب کنید', 'error');
            return;
        }

        const stageMeansSuccess = (pipelineStage === 'deal_success' || pipelineStage === 'اتمام');
        const stageMeansFailed = (pipelineStage === 'deal_failed' || pipelineStage === 'معامله ناموفق');
        if (stageMeansSuccess) {
            form.deal_status.value = 'won';
            dealStatus = 'won';
        } else if (stageMeansFailed) {
            form.deal_status.value = 'lost';
            dealStatus = 'lost';
        }
        
        // If status is 'lost' and pipeline_stage is empty, automatically set it to 'معامله ناموفق'
        if (dealStatus === 'lost' && !pipelineStage) {
            pipelineStage = 'معامله ناموفق';
            if (form.pipeline_stage) {
                form.pipeline_stage.value = 'معامله ناموفق';
            }
        }

        // Enforce required deal info fields
        const requiredInfoFields = [
            { field: form.title, label: 'خدمات قابل ارائه' },
            { field: form.service_cost, label: 'هزینه خدمات' },
            { field: form.price_list_code, label: 'کد لیست قیمت' },
            { field: form.payable_amount, label: 'مبلغ قابل پرداخت' },
            { field: form.payment_conditions, label: 'شرایط پرداخت' }
        ];
        for (const item of requiredInfoFields) {
            const val = item.field?.value ?? '';
            if (String(val).trim() === '') {
                Swal.fire('خطا', `${item.label} الزامی است`, 'error');
                if (item.field?.focus) item.field.focus();
                return;
            }
        }

        const selectedTitle = form.title?.value?.trim() || '';
        const selectedTitleOption = form.title?.selectedOptions?.[0];
        const isGeneratedTitle =
            (selectedTitleOption && selectedTitleOption.dataset && selectedTitleOption.dataset.generated === '1') ||
            /^Ù…Ø¹Ø§Ù…Ù„Ù‡\s+\S+\s+\d{10,12}$/u.test(selectedTitle);
        if (isNewDeal && isGeneratedTitle) {
            Swal.fire('خطا', '\u0644\u0637\u0641\u0627\u064b \u062e\u062f\u0645\u0627\u062a \u0642\u0627\u0628\u0644 \u0627\u0631\u0627\u0626\u0647 \u0631\u0627 \u0627\u0646\u062a\u062e\u0627\u0628 \u06a9\u0646\u06cc\u062f', 'error');
            if (form.title?.focus) form.title.focus();
            return;
        }

        if (productIds.length === 0 && selectedTitleOption?.dataset?.productId) {
            productIds = [selectedTitleOption.dataset.productId];
        }

        const paymentDates = form.querySelectorAll('input[name="payment_date[]"]');
        const paymentAmounts = form.querySelectorAll('input[name="payment_amount[]"]');
        const paymentMethods = form.querySelectorAll('select[name="payment_method[]"]');
        const paymentDescriptions = form.querySelectorAll('textarea[name="payment_description[]"]');
        const paymentCardLast4s = form.querySelectorAll('input[name="payment_card_last4[]"]');
        const paymentBanks = form.querySelectorAll('select[name="destination_bank[]"]');
        const mustHavePayments = dealStatus === 'won' || stageMeansSuccess;
        const anyPaymentFilled = [
            ...Array.from(paymentDates).map(i => i.value),
            ...Array.from(paymentAmounts).map(i => i.value),
            ...Array.from(paymentMethods).map(i => i.value),
            ...Array.from(paymentDescriptions).map(i => i.value)
        ].some(v => String(v || '').trim() !== '');
        
        // Validate failure reason for lost deals
        if (dealStatus === 'lost') {
            const failureReason = form.failure_reason.value;
            if (!failureReason) {
                Swal.fire('خطا', 'لطفاً دلیل ناموفق بودن معامله را انتخاب کنید', 'error');
                return;
            }
            // برای همه دلایل، توضیحات اجباری است
            const otherDesc = form.failure_other_description.value;
            if (!otherDesc || !otherDesc.trim()) {
                Swal.fire('خطا', 'لطفاً توضیحات دلیل ناموفق بودن را وارد کنید', 'error');
                const descField = document.getElementById('deal-failure-other-description');
                if (descField) descField.focus();
                return;
            }
        }
        
	        const validatePaymentRow = (i) => {
	            const dateInput = paymentDates[i];
	            
	            // Skip validation for readonly rows (already saved payments from before the update)
	            if (dateInput?.readOnly) {
	                return true;
	            }
	            
	            const dateVal = paymentDates[i]?.value?.trim() || '';
	            const amountValRaw = paymentAmounts[i]?.value || '';
	            const amountVal = parseMoneyValue(amountValRaw);
	            const methodVal = paymentMethods[i]?.value || '';
	            const descVal = paymentDescriptions[i]?.value?.trim() || '';
            const cardLast4 = paymentCardLast4s[i]?.value?.trim() || '';
            const bankVal = paymentBanks[i]?.value || '';
	            const hasAnyValue = dateVal || amountValRaw || methodVal || descVal;

	            if (!hasAnyValue && !mustHavePayments) {
	                return true;
	            }
	            if (!dateVal) {
                Swal.fire('خطا', `تاریخ پرداخت شماره ${i + 1} الزامی است`, 'error');
                paymentDates[i]?.focus();
	                return false;
	            }
	            if (!amountValRaw || amountVal <= 0) {
	                Swal.fire('خطا', `مبلغ پرداخت شماره ${i + 1} الزامی است`, 'error');
	                paymentAmounts[i]?.focus();
	                return false;
	            }
            if (!methodVal) {
                Swal.fire('خطا', `نحوه پرداخت شماره ${i + 1} الزامی است`, 'error');
                paymentMethods[i]?.focus();
                return false;
            }
            const meta = this.getLookupMeta('payment_methods', methodVal);
            // Explicitly check for methods that DON'T require card
            const methodsWithoutCard = ['cash', 'check', 'نقد', 'چک'];
            const isMethodWithoutCard = methodsWithoutCard.includes(methodVal.toLowerCase());
            
            const requiresCard = isMethodWithoutCard 
                ? false 
                : (typeof meta.requires_card_last4 === 'boolean'
                    ? meta.requires_card_last4
                    : (methodVal === 'card_to_card' || methodVal === 'card_reader' || methodVal === 'card_to_card_lookup'));
            const requiresBank = typeof meta.requires_bank_destination === 'boolean'
                ? meta.requires_bank_destination
                : (methodVal === 'card_to_card' || methodVal === 'card_to_card_lookup');

            if (requiresCard) {
                if (!/^\d{4}$/.test(cardLast4)) {
                    Swal.fire('خطا', `4 رقم آخر کارت باید وارد شود در قسمت ${i + 1} پرداخت`, 'error');
                    paymentCardLast4s[i]?.focus();
                    return false;
                }
            }
            if (requiresBank) {
                if (!bankVal) {
                    Swal.fire('خطا', `بانک مقصد باید انتخاب شود در قسمت ${i + 1} پرداخت`, 'error');
                    paymentBanks[i]?.focus();
                    return false;
                }
            }
if (!descVal) {
                Swal.fire('خطا', `توضیح پرداخت شماره ${i + 1} الزامی است`, 'error');
                paymentDescriptions[i]?.focus();
                return false;
            }
            return true;
        };

        if (mustHavePayments) {
            if (paymentDates.length === 0) {
                Swal.fire('خطا', 'لطفاً حداقل یک پرداخت را تکمیل کنید', 'error');
                return;
            }
            for (let i = 0; i < paymentDates.length; i++) {
                if (!validatePaymentRow(i)) return;
            }
        } else if (anyPaymentFilled) {
            for (let i = 0; i < paymentDates.length; i++) {
                if (!validatePaymentRow(i)) return;
            }
        } else if (dealStatus === 'won' && !this.hasCompletePaymentsForSuccess()) {
            Swal.fire('خطا', 'برای ثبت معامله موفق، جزئیات پرداخت باید کامل باشد.', 'error');
            return;
        }
        
	        // Get discount occasion (use custom if other)
	        const hasDiscount = String(form.has_discount?.value || '0') === '1';
	        const discountType = form.discount_type?.value || '';
	        let discountOccasion = form.discount_occasion?.value || '';
	        if (discountOccasion === 'other') {
	            discountOccasion = form.discount_occasion_custom?.value || '';
	        }
	        let discountAmount = parseMoneyValue(form.discount_amount?.value || '');
	        const refundAmount = parseMoneyValue(form.refund_amount?.value || '');
	        const refundDescription = (form.refund_description?.value || '').trim();

	        if (hasDiscount && discountType === 'refund') {
	            if (refundAmount <= 0) {
	                Swal.fire('خطا', 'مبلغ مرجوعی الزامی است', 'error');
	                const el = document.getElementById('deal-refund-amount');
	                if (el) el.focus();
	                return;
	            }
	            if (!refundDescription) {
	                Swal.fire('خطا', 'توضیحات مرجوعی الزامی است', 'error');
	                const el = document.getElementById('deal-refund-description');
	                if (el) el.focus();
	                return;
	            }
	            // Refund is not a discount amount/occasion
	            discountOccasion = '';
	            discountAmount = 0;
	        } else if (!hasDiscount) {
	            discountOccasion = '';
	            discountAmount = 0;
	        }
        
	        // Collect multiple payments (only if status is not lost)
	        const payments = [];
	        
	        for (let i = 0; i < paymentDates.length; i++) {
	            const persianDate = paymentDates[i]?.value?.trim() || '';
	            const amountRaw = paymentAmounts[i]?.value?.trim() || '';
	            const method = paymentMethods[i]?.value || '';
	            const description = paymentDescriptions[i]?.value?.trim() || '';
            const cardLast4 = paymentCardLast4s[i]?.value?.trim() || '';
	            const bankVal = paymentBanks[i]?.value?.trim() || '';
	            const amount = parseMoneyValue(amountRaw);
	            
	            // Only add payment if date and amount are provided
	            if (persianDate && amount > 0) {
	                let gregorianDate = '';
	                // Convert Persian date to Gregorian
	                if (persianDate) {
	                    try {
	                        const gDateStr = persianToGregorian(persianDate);
	                        if (gDateStr) {
	                            // Extract just the date part (YYYY-MM-DD)
	                            gregorianDate = gDateStr.split('T')[0];
	                        }
	                    } catch(e) {
	                        console.error('Error converting date:', e, 'Date:', persianDate);
	                    }
	                }
	                
	                payments.push({
	                    date: gregorianDate || '',
	                    date_persian: persianDate || '',
	                    amount,
	                    method: method || '',
	                    description: description || '',
	                    card_last4: cardLast4 || '',
	                    destination_bank: bankVal || ''
	                });
	            }
	        }
        
        // Calculate total payment amount
        const totalPaymentAmount = payments.reduce((sum, p) => sum + (p.amount || 0), 0);
        
        // Prepare failure reason fields
        let failureReasonCode = '';
        let failureReasonText = '';
        let failureReasonDescription = '';
        if (dealStatus === 'lost') {
            const selectedReason = form.failure_reason.value;
            failureReasonCode = selectedReason || '';
            failureReasonDescription = (form.failure_other_description.value || '').trim();
            const reasonText = {
                'no_financial_ability': 'عدم توان مالی',
                'no_purchase_intent': 'فعلا قصد خرید ندارد',
                'competitor_purchase': 'خرید از رقبا',
                'wrong_number': 'شماره اشتباه',
                'irrelevant_lead': 'لید نامرتبط',
                'other': 'سایر'
            };
            failureReasonText = reasonText[selectedReason] || '';
            if (!failureReasonCode) {
                Swal.fire('خطا', 'لطفاً دلیل ناموفق شدن را انتخاب کنید', 'error');
                return;
            }
            if (!failureReasonDescription) {
                Swal.fire('خطا', 'توضیحات دلیل ناموفق شدن الزامی است', 'error');
                return;
            }
        }
        
	        const data = {
	            deal_id: dealId || '',
	            contact_id: contactId,
	            deal_status: dealStatus, // pending, won, lost
	            pipeline_stage: pipelineStage,
	            product_ids: productIds.join(','),
	            title: form.title.value, // عنوان معامله = خدمات قابل ارائه
	            service_cost: parseMoneyValue(form.service_cost.value) || 0,
	            price_list_code: form.price_list_code.value,
	            has_discount: form.has_discount.value,
	            discount_type: discountType,
	            discount_occasion: discountOccasion,
	            discount_amount: discountAmount,
	            refund_amount: refundAmount,
	            refund_description: refundDescription,
	            payable_amount: parseMoneyValue(form.payable_amount.value) || 0,
	            payment_conditions: form.payment_conditions.value,
	            payments: JSON.stringify(payments), // Convert to JSON string for FormData
	            payment_amount: totalPaymentAmount, // Total for backward compatibility
	            failure_reason: failureReasonCode,
	            failure_reason_text: failureReasonText,
	            failure_other_description: failureReasonDescription
	        };

        // Include requested services from activities tab (hidden field JSON)
        const reqServicesHidden = document.getElementById('deal-activity-requested-services');
        if (reqServicesHidden && reqServicesHidden.value) {
            data.requested_services = reqServicesHidden.value;
        }

        // Include مالی fields (financial_level, asset_estimation, income_estimation, has_previous_purchase)
        const finLevel = document.getElementById('deal-activity-financial-level')?.value || '';
        const assetEst = document.getElementById('deal-activity-asset')?.value || '';
        const incomeEst = document.getElementById('deal-activity-income')?.value || '';
        const hasPurchase = document.getElementById('deal-activity-has-purchase')?.value || '';
        if (finLevel) data.financial_level = finLevel;
        if (assetEst) data.asset_estimation = assetEst;
        if (incomeEst) data.income_estimation = incomeEst;
        if (hasPurchase !== '') data.has_previous_purchase = hasPurchase;

        // اگر هیچ خدمات درخواستی از تب فعالیت نیامده، عنوان انتخاب‌شده را به عنوان یک خدمت بفرست
        if (!data.requested_services || data.requested_services === '[]') {
            const titleVal = form.title.value?.trim();
            if (titleVal) {
                data.requested_services = JSON.stringify([titleVal]);
            }
        }
        
        console.log('Sending payments to backend:', payments);

        // COMPREHENSIVE LOGGING BEFORE SENDING DEAL SAVE REQUEST
        console.log('[DEAL_SAVE_REQUEST] Sending deal save request to backend', {
            timestamp: new Date().toISOString(),
            data: data,
            contactId: data.contact_id,
            dealId: data.deal_id,
            title: data.title,
            windowCurrentContactId: window.currentContactId,
            paymentsCount: Array.isArray(data.payments) ? data.payments.length : 0,
            formTitle: form.title?.value,
            formContactId: form.contact_id?.value,
            url: window.location.href,
            sessionId: this.getSessionId ? this.getSessionId() : 'unknown'
        });

        Swal.showLoading();
        const res = await this.req('save_deal', data);
        Swal.close();

        // LOGGING DEAL SAVE RESPONSE
        console.log('[DEAL_SAVE_RESPONSE] Deal save response received', {
            timestamp: new Date().toISOString(),
            status: res.status,
            message: res.message,
            deal_id: res.deal_id,
            contact_id: data.contact_id,
            original_deal_id: data.deal_id,
            success: res.status === 'success'
        });

        if(res.status === 'success') {
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title: dealId ? 'معامله به‌روزرسانی شد' : 'معامله ثبت شد', showConfirmButton:false, timer:2000});
            
            // Update deal-id if it was a new deal
            const currentDealId = document.getElementById('deal-id').value;
            if (!currentDealId && res.deal_id) {
                document.getElementById('deal-id').value = res.deal_id;
            }
            
            // Close modal and reload
            this.closeModal('modalDeal');
            if (window.currentContactId) {
                this.openPersonProfile(window.currentContactId);
            }
            if (document.getElementById('deals-container')) {
                this.loadDeals();
            }
        } else {
            const mismatchMsg = 'شناسه معامله با لید انتخاب‌شده مطابقت ندارد';
            if ((res.message || '').includes(mismatchMsg)) {
                console.warn('[DEAL_SAVE_BLOCKED] Deal/contact mismatch detected; clearing deal context', {
                    timestamp: new Date().toISOString(),
                    contact_id: data.contact_id,
                    deal_id: data.deal_id,
                    message: res.message
                });
                const dealIdInput = document.getElementById('deal-id');
                if (dealIdInput) dealIdInput.value = '';
                const dealActDeal = document.getElementById('deal-activity-deal-id');
                if (dealActDeal) dealActDeal.value = '';
                const activityDealId = document.getElementById('activity-deal-id');
                if (activityDealId) activityDealId.value = '';
                this.state.currentDeal = null;
            }
            console.error('[DEAL_SAVE_ERROR] Deal save failed', {
                timestamp: new Date().toISOString(),
                error: res.message,
                status: res.status,
                contact_id: data.contact_id,
                deal_id: data.deal_id,
                title: data.title,
                fullResponse: res
            });
            Swal.fire('خطا', res.message, 'error');
        }
    },

    saveDealFromTab(tab) {
        const form = document.getElementById('deal-form');
        if (!form) return;
        this.state.activeDealTab = tab;
        return this.saveDeal({ preventDefault: () => {}, target: form });
    },
    
    // MARK DEAL AS SUCCESS
    async markDealAsSuccess() {
        const dealId = document.getElementById('deal-id').value;
        const contactId = document.getElementById('deal-contact-id').value;
        
        if (!dealId) {
            Swal.fire('خطا', 'لطفاً ابتدا معامله را ذخیره کنید', 'error');
            return;
        }
        
        Swal.fire({
            title: 'تایید معامله موفق',
            text: 'آیا از موفق بودن این معامله اطمینان دارید؟',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'بله، معامله موفق است',
            cancelButtonText: 'انصراف',
            confirmButtonColor: '#10b981'
        }).then(async (result) => {
            if (result.isConfirmed) {
                Swal.showLoading();
                const res = await this.req('set_deal_status', {
                    deal_id: dealId,
                    status: '1', // Won
                    contact_id: contactId
                });
                Swal.close();
                
                if (res.status === 'success') {
                    Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'معامله به عنوان موفق ثبت شد', showConfirmButton:false, timer:2000});
                    this.closeModal('modalDeal');
                    if (window.currentContactId) {
                        this.openPersonProfile(window.currentContactId);
                    }
                    if (document.getElementById('deals-container')) {
                        this.loadDeals();
                    }
                } else {
                    Swal.fire('خطا', res.message, 'error');
                }
            }
        });
    },
    
    // MARK DEAL AS FAILED
    markDealAsFailed() {
        const dealId = document.getElementById('deal-id').value;
        const contactId = document.getElementById('deal-contact-id').value;
        
        if (!dealId) {
            Swal.fire('خطا', 'لطفاً ابتدا معامله را ذخیره کنید', 'error');
            return;
        }
        
        document.getElementById('failed-deal-id').value = dealId;
        document.getElementById('failed-contact-id').value = contactId;
        document.getElementById('failure-reason').value = '';
        document.getElementById('failure-other-description').value = '';
        document.getElementById('failure-other-reason').classList.add('hidden');
        this.openModal('modalDealFailed');
    },
    
	    
    // CONFIRM DEAL FAILED
    async confirmDealFailed(e) {
        e.preventDefault();
        
        const dealId = document.getElementById('failed-deal-id').value;
        const contactId = document.getElementById('failed-contact-id').value;
        const reason = document.getElementById('failure-reason').value;
        const otherDesc = document.getElementById('failure-other-description').value;
        
	        // Map reason to Persian text
	        const reasonText = {
            'no_financial_ability': 'عدم توان مالی',
            'no_purchase_intent': 'فعلا قصد خرید ندارد',
            'competitor_purchase': 'خرید از رقبا',
            'wrong_number': 'شماره اشتباه',
            'irrelevant_lead': 'لید نامرتبط',
            'other': 'سایر'
        };
        
        Swal.showLoading();
        if (!reason) {
            Swal.fire('خطا', 'لطفاً دلیل ناموفق شدن را انتخاب کنید', 'error');
            return;
        }
        // For consistency with deal form, always require description
        if (!otherDesc.trim()) {
            Swal.fire('خطا', 'لطفاً توضیحات را وارد کنید', 'error');
            return;
        }

        const res = await this.req('set_deal_status', {
            deal_id: dealId,
            status: '2', // Lost
            contact_id: contactId,
            failure_reason: reason,
            failure_reason_text: reasonText[reason] || '',
            failure_other_description: otherDesc
        });
        Swal.close();
        
        if (res.status === 'success') {
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'معامله به عنوان ناموفق ثبت شد', showConfirmButton:false, timer:2000});
            this.closeModal('modalDealFailed');
            this.closeModal('modalDeal');
            if (window.currentContactId) {
                this.openPersonProfile(window.currentContactId);
            }
            if (document.getElementById('deals-container')) {
                this.loadDeals();
            }
        } else {
            Swal.fire('خطا', res.message, 'error');
        }
    },
    
    // DELETE DEAL (admin only)
    async deleteDeal(dealId) {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند معاملات را حذف کند', 'error');
            return;
        }
        
        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله مشخص نشده است', 'error');
            return;
        }
        
        const result = await Swal.fire({
            title: 'آیا مطمئن هستید؟',
            text: 'این معامله به طور دائم حذف خواهد شد',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'بله، حذف کن',
            cancelButtonText: 'لغو',
            reverseButtons: true
        });
        
        if (!result.isConfirmed) {
            return;
        }
        
        Swal.fire({
            title: 'در حال حذف...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('delete_deal', { deal_id: dealId });
            Swal.close();
            
            if (res.status === 'success') {
                Swal.fire('موفق', res.message || 'معامله با موفقیت حذف شد', 'success');
                
                // Refresh deals list
                if (document.getElementById('deals-container')) {
                    this.loadDeals();
                }
                
                // Refresh person profile if open
                if (window.currentContactId) {
                    this.openPersonProfile(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در حذف معامله', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در حذف معامله: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    // REACTIVATE DEAL (admin only)
    async reactivateDeal(dealId) {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند معاملات بسته شده را فعال کند', 'error');
            return;
        }

        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله مشخص نشده است', 'error');
            return;
        }

        // نمایش modal برای انتخاب مرحله کاریز جدید
        const { value: formData } = await Swal.fire({
            title: 'فعال‌سازی مجدد معامله',
            html: `
                <div class="text-right mb-4">
                    <p class="text-sm text-gray-600 mb-3">مرحله کاریز جدید را انتخاب کنید:</p>
                    <select id="reactivate-stage" class="swal2-input w-full text-right">
                        <option value="">-- انتخاب مرحله کاریز --</option>
                        <option value="تماس پیگیری">تماس پیگیری</option>
                        <option value="ارسال لینک پرداخت">ارسال لینک پرداخت</option>
                        <option value="پرداخت">پرداخت</option>
                        <option value="پیگیری مانده حساب">پیگیری مانده حساب</option>
                        <option value="برگزاری دوره">برگزاری دوره</option>
                        <option value="پشتیبانی">پشتیبانی</option>
                        <option value="ارجاع به CRM">ارجاع به CRM</option>
                    </select>
                </div>
                <div class="text-right">
                    <label class="block text-sm text-gray-600 mb-2">دلیل فعال‌سازی مجدد (اختیاری):</label>
                    <textarea id="reactivate-reason" class="swal2-textarea w-full text-right" placeholder="توضیح دهید چرا این معامله را فعال می‌کنید..." rows="3"></textarea>
                </div>
            `,
            showCancelButton: true,
            confirmButtonText: 'فعال‌سازی',
            cancelButtonText: 'لغو',
            confirmButtonColor: '#10b981',
            cancelButtonColor: '#6b7280',
            preConfirm: () => {
                const stage = document.getElementById('reactivate-stage').value;
                const reason = document.getElementById('reactivate-reason').value;

                if (!stage) {
                    Swal.showValidationMessage('مرحله کاریز جدید را انتخاب کنید');
                    return false;
                }

                return { stage, reason };
            }
        });

        if (!formData) return;

        // نمایش loading
        Swal.fire({
            title: 'در حال فعال‌سازی...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await this.req('deal/reactivate', {
                deal_id: dealId,
                new_pipeline_stage: formData.stage,
                reactivation_reason: formData.reason
            });
            Swal.close();

            if (res.status === 'success') {
                Swal.fire({
                    title: 'موفق',
                    text: res.message || 'معامله با موفقیت فعال شد',
                    icon: 'success',
                    confirmButtonText: 'باشه'
                });

                // بروزرسانی لیست معاملات
                if (document.getElementById('deals-container')) {
                    this.loadDeals();
                }

                // بروزرسانی پروفایل شخص اگر باز است
                if (window.currentContactId) {
                    this.openPersonProfile(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در فعال‌سازی معامله', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در فعال‌سازی معامله: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    // DELETE DEAL FROM MODAL (admin only)
    async deleteDealFromModal() {
        const dealId = document.getElementById('deal-id').value;
        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله مشخص نشده است', 'error');
            return;
        }

        const result = await Swal.fire({
            title: 'آیا مطمئن هستید؟',
            text: 'این معامله به طور دائم حذف خواهد شد و قابل بازیابی نیست',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#6b7280',
            confirmButtonText: 'بله، حذف کن',
            cancelButtonText: 'لغو',
            reverseButtons: true
        });

        if (!result.isConfirmed) {
            return;
        }

        Swal.fire({
            title: 'در حال حذف...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await this.req('delete_deal', { deal_id: dealId });
            Swal.close();

            if (res.status === 'success') {
                Swal.fire({
                    title: 'موفق',
                    text: res.message || 'معامله با موفقیت حذف شد',
                    icon: 'success',
                    confirmButtonText: 'باشه'
                });

                // بستن modal و بروزرسانی لیست‌ها
                this.closeModal('modalDeal');
                if (document.getElementById('deals-container')) {
                    this.loadDeals();
                }
                if (window.currentContactId) {
                    this.openPersonProfile(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در حذف معامله', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در حذف معامله: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    // REACTIVATE DEAL FROM MODAL (admin only)
    async reactivateDealFromModal() {
        const dealId = document.getElementById('deal-id').value;
        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله مشخص نشده است', 'error');
            return;
        }

        // نمایش modal انتخاب مرحله کاریز
        const { value: formData } = await Swal.fire({
            title: 'فعال‌سازی مجدد معامله',
            html: `
                <div class="text-right mb-4">
                    <p class="text-sm text-gray-600 mb-3">مرحله کاریز جدید را انتخاب کنید:</p>
                    <select id="reactivate-stage" class="swal2-input w-full text-right">
                        <option value="">-- انتخاب مرحله کاریز --</option>
                        <option value="تماس پیگیری">تماس پیگیری</option>
                        <option value="ارسال لینک پرداخت">ارسال لینک پرداخت</option>
                        <option value="پرداخت">پرداخت</option>
                        <option value="پیگیری مانده حساب">پیگیری مانده حساب</option>
                        <option value="برگزاری دوره">برگزاری دوره</option>
                        <option value="پشتیبانی">پشتیبانی</option>
                        <option value="ارجاع به CRM">ارجاع به CRM</option>
                    </select>
                </div>
                <div class="text-right">
                    <label class="block text-sm text-gray-600 mb-2">دلیل فعال‌سازی مجدد (اختیاری):</label>
                    <textarea id="reactivate-reason" class="swal2-textarea w-full text-right" placeholder="توضیح دهید چرا این معامله را فعال می‌کنید..." rows="3"></textarea>
                </div>
            `,
            showCancelButton: true,
            confirmButtonText: 'فعال‌سازی',
            cancelButtonText: 'لغو',
            confirmButtonColor: '#10b981',
            cancelButtonColor: '#6b7280',
            preConfirm: () => {
                const stage = document.getElementById('reactivate-stage').value;
                const reason = document.getElementById('reactivate-reason').value;

                if (!stage) {
                    Swal.showValidationMessage('مرحله کاریز جدید را انتخاب کنید');
                    return false;
                }

                return { stage, reason };
            }
        });

        if (!formData) return;

        // نمایش loading
        Swal.fire({
            title: 'در حال فعال‌سازی...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await this.req('deal/reactivate', {
                deal_id: dealId,
                new_pipeline_stage: formData.stage,
                reactivation_reason: formData.reason
            });
            Swal.close();

            if (res.status === 'success') {
                Swal.fire({
                    title: 'موفق',
                    text: res.message || 'معامله با موفقیت فعال شد',
                    icon: 'success',
                    confirmButtonText: 'باشه'
                });

                // بروزرسانی وضعیت در modal و لیست‌ها
                this.closeModal('modalDeal');
                if (document.getElementById('deals-container')) {
                    this.loadDeals();
                }
                if (window.currentContactId) {
                    this.openPersonProfile(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در فعال‌سازی معامله', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در فعال‌سازی معامله: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    // REACTIVATE DEAL FROM PROFILE (admin only)
    async reactivateDealFromProfile(dealId) {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند معاملات بسته شده را فعال کند', 'error');
            return;
        }

        if (!dealId) {
            Swal.fire('خطا', 'شناسه معامله مشخص نشده است', 'error');
            return;
        }

        // نمایش modal انتخاب مرحله کاریز
        const { value: formData } = await Swal.fire({
            title: 'فعال‌سازی مجدد معامله',
            html: `
                <div class="text-right mb-4">
                    <p class="text-sm text-gray-600 mb-3">مرحله کاریز جدید را انتخاب کنید:</p>
                    <select id="reactivate-stage" class="swal2-input w-full text-right">
                        <option value="">-- انتخاب مرحله کاریز --</option>
                        <option value="تماس پیگیری">تماس پیگیری</option>
                        <option value="ارسال لینک پرداخت">ارسال لینک پرداخت</option>
                        <option value="پرداخت">پرداخت</option>
                        <option value="پیگیری مانده حساب">پیگیری مانده حساب</option>
                        <option value="برگزاری دوره">برگزاری دوره</option>
                        <option value="پشتیبانی">پشتیبانی</option>
                        <option value="ارجاع به CRM">ارجاع به CRM</option>
                    </select>
                </div>
                <div class="text-right">
                    <label class="block text-sm text-gray-600 mb-2">دلیل فعال‌سازی مجدد (اختیاری):</label>
                    <textarea id="reactivate-reason" class="swal2-textarea w-full text-right" placeholder="توضیح دهید چرا این معامله را فعال می‌کنید..." rows="3"></textarea>
                </div>
            `,
            showCancelButton: true,
            confirmButtonText: 'فعال‌سازی',
            cancelButtonText: 'لغو',
            confirmButtonColor: '#10b981',
            cancelButtonColor: '#6b7280',
            preConfirm: () => {
                const stage = document.getElementById('reactivate-stage').value;
                const reason = document.getElementById('reactivate-reason').value;

                if (!stage) {
                    Swal.showValidationMessage('مرحله کاریز جدید را انتخاب کنید');
                    return false;
                }

                return { stage, reason };
            }
        });

        if (!formData) return;

        // نمایش loading
        Swal.fire({
            title: 'در حال فعال‌سازی...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const res = await this.req('deal/reactivate', {
                deal_id: dealId,
                new_pipeline_stage: formData.stage,
                reactivation_reason: formData.reason
            });
            Swal.close();

            if (res.status === 'success') {
                Swal.fire({
                    title: 'موفق',
                    text: res.message || 'معامله با موفقیت فعال شد',
                    icon: 'success',
                    confirmButtonText: 'باشه'
                });

                // بروزرسانی پروفایل شخص
                if (window.currentContactId) {
                    this.openPersonProfile(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در فعال‌سازی معامله', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در فعال‌سازی معامله: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    // DELETE ACTIVITY (admin only)
    async deleteActivity(activityId) {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند فعالیت‌ها را حذف کند', 'error');
            return;
        }
        
        if (!activityId) {
            Swal.fire('خطا', 'شناسه فعالیت مشخص نشده است', 'error');
            return;
        }
        
        const result = await Swal.fire({
            title: 'آیا مطمئن هستید؟',
            text: 'این فعالیت به طور دائم حذف خواهد شد',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'بله، حذف کن',
            cancelButtonText: 'لغو',
            reverseButtons: true
        });
        
        if (!result.isConfirmed) {
            return;
        }
        
        Swal.fire({
            title: 'در حال حذف...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('delete_activity', { activity_id: activityId });
            Swal.close();
            
            if (res.status === 'success') {
                Swal.fire('موفق', res.message || 'فعالیت با موفقیت حذف شد', 'success');
                
                // Refresh person profile if open
                if (window.currentContactId) {
                    this.openPersonProfile(window.currentContactId);
                }
                
                // Refresh activities modal if open
                const activitiesModal = document.getElementById('modalDidarActivities');
                if (activitiesModal && !activitiesModal.classList.contains('hidden')) {
                    this.viewDidarActivities(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در حذف فعالیت', 'error');
            }
        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در حذف فعالیت: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    updateDealModalSummary(person = null, deal = null) {
        const fill = (id, value) => {
            const el = document.getElementById(id);
            if (el) el.textContent = value || '-';
        };
        const stage = deal?.pipeline_stage || document.getElementById('deal-pipeline-stage')?.value || '';
        const statusLabel = (person && typeof person.sale_status !== 'undefined')
            ? (String(person.sale_status) === '1' ? 'موفق' : 'ناموفق')
            : '-';

        fill('deal-summary-name', person ? `${person.first_name || ''} ${person.last_name || ''}`.trim() || (deal?.contact_name || 'بدون نام') : (deal?.contact_name || 'بدون نام'));
        fill('deal-summary-owner', person?.owner_name || deal?.owner_name || this.state.currentLead?.owner_name || APP_USER.name || '---');
        fill('deal-summary-prev-owner', person?.previous_owner_name || '-');
        fill('deal-summary-sale-status', statusLabel);
        fill('deal-summary-mobile', person?.mobile_phone || deal?.contact_mobile || '-');
        fill('deal-summary-mobile2', person?.secondary_mobile_phone || '-');
        fill('deal-summary-mobile3', person?.mobile_phone_3 || '-');
        fill('deal-summary-mobile4', person?.mobile_phone_4 || '-');
        fill('deal-summary-city', person?.city || '-');
        fill('deal-summary-job', person?.job_title || '-');
        fill('deal-summary-didar-id', deal?.didar_deal_id || deal?.id || person?.didar_contact_id || '-');
        fill('deal-summary-didar-id-box', deal?.didar_deal_id || deal?.id || person?.didar_contact_id || '-');

        const stageBadge = document.getElementById('deal-summary-stage');
        if (stageBadge) {
            stageBadge.textContent = stage || 'مرحله تعیین نشده';
        }

        this.updateDealCurrentFields(stage, deal?.status);
    },

    updateDealCurrentFields(stageValue = null, statusValue = null) {
        const statusMap = {
            pending: 'جاری',
            won: 'موفق',
            lost: 'ناموفق',
            Pending: 'جاری',
            Won: 'موفق',
            Lost: 'ناموفق'
        };
        const statusSelect = document.getElementById('deal-status');
        const stageSelect = document.getElementById('deal-pipeline-stage');
        const statusText = statusValue || statusSelect?.value || '';
        const stageText = stageValue || stageSelect?.value || '';
        const statusBox = document.getElementById('deal-current-status');
        const stageBox = document.getElementById('deal-current-stage');
        if (statusBox) {
            statusBox.textContent = statusMap[statusText] || '-';
        }
        if (stageBox) {
            stageBox.textContent = stageText || '-';
        }
    },

    showDealFormTab(tab = 'info') {
        const tabs = ['info', 'activities', 'payments'];
        tabs.forEach(t => {
            const section = document.getElementById(`deal-form-tab-${t}`);
            const btn = document.getElementById(`deal-tab-btn-${t}`);
            if (section) section.classList.toggle('hidden', t !== tab);
            if (btn) {
                if (t === tab) {
                    btn.classList.remove('text-gray-300', 'bg-transparent');
                    btn.classList.add('bg-gray-800', 'text-white');
                } else {
                    btn.classList.add('text-gray-300');
                    btn.classList.remove('bg-gray-800', 'text-white');
                }
            }
        });

        // Buttons are inside each tab; no global save toggle needed
    },

    updateDealActivitiesBadge(count = 0) {
        const badge = document.getElementById('deal-activities-badge');
        if (badge) badge.textContent = count > 99 ? '99+' : count;
    },

	    hasCompletePaymentsForSuccess() {
	        const collected = this.collectCurrentPayments();
	        const cached = this.state.lastPaymentsCache || [];
	        const stored = Array.isArray(this.state.currentDeal?.payments) ? this.state.currentDeal.payments : [];
	        const list = [...collected, ...cached, ...stored];
	        const valid = list.filter(p => {
	            const amountVal = parseMoneyValue(p.amount || 0);
	            const hasDate = !!(p.date_persian || p.date);
	            const hasMethod = !!(p.method && String(p.method).trim() !== '');
	            return hasDate && hasMethod && amountVal > 0;
	        });
        const result = valid.length > 0;
        console.log('hasCompletePaymentsForSuccess', { collected, cached, stored, validCount: valid.length, result });
        return result;
    },

        populateDealActivityServices(selected = []) {
        // Keep selected services in state
        this.state.dealActivityRequestedServices = Array.isArray(selected) ? [...new Set(selected.map(String))] : [];

        const select = document.getElementById('deal-activity-service-select');
        if (select) {
            select.innerHTML = '<option value="">-- انتخاب خدمت --</option>';

            const lookupOptions = this.getLookupOptions('allowed_services');
            if (lookupOptions.length) {
                lookupOptions.forEach(service => {
                    const o = document.createElement('option');
                    o.value = service.value;
                    o.textContent = service.label || service.value;
                    select.appendChild(o);
                });
            } else {
                // fallback: read from deal-title select (static services)
                const titleSelect = document.getElementById('deal-title');
                if (titleSelect) {
                    Array.from(titleSelect.options || []).forEach(opt => {
                        if (!opt.value) return;
                        const o = document.createElement('option');
                        o.value = opt.value;
                        o.textContent = opt.textContent;
                        select.appendChild(o);
                    });
                }
            }

            // Try to refresh from API in background (optional)
            this.req('get_allowed_services').then(res => {
                if (res.status === 'success' && res.services && res.services.length > 0) {
                    select.innerHTML = '<option value="">-- انتخاب خدمت --</option>';
                    res.services.forEach(service => {
                        const o = document.createElement('option');
                        o.value = service;
                        o.textContent = service;
                        select.appendChild(o);
                    });
                }
            }).catch(() => {});
        }
        this.renderDealActivityServices();
    },

    renderDealActivityServices() {
        const chips = document.getElementById('deal-activity-service-chips');
        const hidden = document.getElementById('deal-activity-requested-services');
        if (!chips || !hidden) return;
        const items = Array.isArray(this.state.dealActivityRequestedServices) ? this.state.dealActivityRequestedServices : [];
        chips.innerHTML = '';
        items.forEach(service => {
            const chip = document.createElement('span');
            chip.className = 'inline-flex items-center gap-1 px-2 py-1 rounded-full bg-gray-800 text-gray-100 text-xs border border-gray-700';
            chip.textContent = service;
            const btn = document.createElement('button');
            btn.type = 'button';
            btn.className = 'text-red-400 hover:text-red-200 ml-1';
            btn.innerHTML = '&times;';
            btn.onclick = () => this.removeDealActivityService(service);
            chip.appendChild(btn);
            chips.appendChild(chip);
        });
        hidden.value = JSON.stringify(items);
    },

    addDealActivityService() {
        if (this.state.dealHasActivity) {
            Swal.fire('خطا', 'پس از ثبت اولین فعالیت امکان ویرایش خدمات نیست', 'error');
            return;
        }
        const select = document.getElementById('deal-activity-service-select');
        if (!select) return;
        const value = select.value;
        if (!value) return;
        this.state.dealActivityRequestedServices = Array.isArray(this.state.dealActivityRequestedServices)
            ? this.state.dealActivityRequestedServices
            : [];
        if (!this.state.dealActivityRequestedServices.includes(value)) {
            this.state.dealActivityRequestedServices.push(value);
            this.renderDealActivityServices();
        }
        select.value = '';
    },

    removeDealActivityService(service) {
        if (this.state.dealHasActivity) {
            Swal.fire('خطا', 'پس از ثبت اولین فعالیت امکان حذف خدمات نیست', 'error');
            return;
        }
        if (!Array.isArray(this.state.dealActivityRequestedServices)) return;
        this.state.dealActivityRequestedServices = this.state.dealActivityRequestedServices.filter(s => s !== service);
        this.renderDealActivityServices();
    },

    toggleAcqLessThanMonth() {
        const less = document.getElementById('deal-activity-acq-less');
        const years = document.getElementById('deal-activity-acq-years');
        const months = document.getElementById('deal-activity-acq-months');
        if (!less || !years || !months) return;
        if (less.checked) {
            years.disabled = true;
            months.disabled = true;
            years.value = '0';
            months.value = '0';
            years.required = false;
            months.required = false;
        } else {
            years.disabled = false;
            months.disabled = false;
            years.required = true;
            months.required = true;
        }
    },

    // Show/hide initial info fields (city/job/financial/etc.) to avoid re-entry in deals
    toggleInitialInfoVisibility(show = true) {
        // initial info fields removed from deal modal; keep no-op for compatibility
        return;
    },

    lockDealActivityExtraFields(values = {}) {
        const setReadonly = (el, condition) => {
            if (!el) return;
            if (condition && String(condition).trim() !== '') {
                if ('readOnly' in el) el.readOnly = true;
                if ('disabled' in el && el.tagName === 'SELECT') el.disabled = true;
                el.classList.add('bg-gray-900', 'cursor-not-allowed');
                el.dataset.locked = 'true';
            } else {
                if ('readOnly' in el) el.readOnly = false;
                if ('disabled' in el && el.tagName === 'SELECT') el.disabled = false;
                el.classList.remove('bg-gray-900', 'cursor-not-allowed');
                el.dataset.locked = '';
            }
        };

        setReadonly(document.getElementById('deal-activity-city'), values.city);
        setReadonly(document.getElementById('deal-activity-job'), values.job_title);
        setReadonly(document.getElementById('deal-activity-job-desc'), values.job_description);
        setReadonly(document.getElementById('deal-activity-extra-info'), values.extra_info);

        const fin = document.getElementById('deal-activity-financial-level');
        const asset = document.getElementById('deal-activity-asset');
        const income = document.getElementById('deal-activity-income');
        setReadonly(fin, values.financial_level);
        setReadonly(asset, values.asset_estimation);
        setReadonly(income, values.income_estimation);

        // Requested services should stay editable
        const svcSelect = document.getElementById('deal-activity-service-select');
        const svcAdd = document.getElementById('deal-activity-service-add');
        if (svcSelect) svcSelect.disabled = false;
        if (svcAdd) svcAdd.disabled = false;

        // خرید قبلی اگر تعیین شده باشد قفل شود
        const purchaseSelect = document.getElementById('deal-activity-has-purchase');
        const purchaseLocked = values.has_previous_purchase === 0 || values.has_previous_purchase === 1 || values.has_previous_purchase === '0' || values.has_previous_purchase === '1';
        if (purchaseSelect) {
            if (purchaseLocked) {
                purchaseSelect.value = String(values.has_previous_purchase);
                purchaseSelect.disabled = true;
                purchaseSelect.dataset.locked = 'true';
                purchaseSelect.classList.add('bg-gray-900', 'cursor-not-allowed');
            } else {
                // For new deals or when no previous purchase value exists, keep field enabled
                purchaseSelect.disabled = false;
                purchaseSelect.dataset.locked = '';
                purchaseSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
            }
        }

        // Acquaintance duration
        const acqLess = document.getElementById('deal-activity-acq-less');
        const acqYears = document.getElementById('deal-activity-acq-years');
        const acqMonths = document.getElementById('deal-activity-acq-months');
        if (acqLess && acqYears && acqMonths) {
            const locked = values.acquaintance_duration && String(values.acquaintance_duration).trim() !== '';
            acqLess.disabled = locked;
            acqYears.disabled = locked || acqLess.checked;
            acqMonths.disabled = locked || acqLess.checked;
            acqLess.dataset.locked = locked ? 'true' : '';
            acqYears.dataset.locked = locked ? 'true' : '';
            acqMonths.dataset.locked = locked ? 'true' : '';
            if (locked) {
                acqYears.classList.add('cursor-not-allowed', 'bg-gray-900');
                acqMonths.classList.add('cursor-not-allowed', 'bg-gray-900');
            } else {
                acqYears.classList.remove('cursor-not-allowed', 'bg-gray-900');
                acqMonths.classList.remove('cursor-not-allowed', 'bg-gray-900');
            }
        }
    },

    populateDealActivityStages() {
        const src = document.getElementById('deal-pipeline-stage');
        const dest = document.getElementById('deal-activity-stage');
        if (!src || !dest) return;
        const current = dest.value;
        dest.innerHTML = '';
        const addOpt = (val, text) => {
            const o = document.createElement('option');
            o.value = val;
            o.textContent = text;
            dest.appendChild(o);
        };
        addOpt('', '-- انتخاب مرحله کاریز فروش --');
        Array.from(src.options || []).forEach(opt => {
            if (!opt.value) return;
            if (opt.value === 'بلاتکلیف' || opt.value === 'معامله جدید') return;
            addOpt(opt.value, opt.textContent);
        });
        if (!Array.from(dest.options).some(o => o.value === 'deal_failed')) {
            addOpt('deal_failed', 'معامله ناموفق');
        }
        if (!Array.from(dest.options).some(o => o.value === 'deal_success')) {
            addOpt('deal_success', 'معامله موفق');
        }

        dest.onchange = () => {
            this.state.activityStagePrev = dest.value;
            this.toggleDealActivityFailure();
            this.handleActivityStageChange(dest.value);
        };
        this.state.activityStagePrev = current || '';
        dest.value = current;
        this.toggleDealActivityFailure();
    },

    toggleDealActivityFailure() {
        const stage = document.getElementById('activity-virtual-stage')?.value || document.getElementById('deal-activity-stage')?.value || '';
        const failureBox = document.getElementById('failure-reason-extra');
        const reason = document.getElementById('activity-failure-reason');
        const otherBox = document.getElementById('activity-failure-other-reason');
        const otherDesc = document.getElementById('activity-failure-other-description');
        const stageMeta = this.getLookupMeta('activity_stages', stage);
        const showFailureFromMeta = stageMeta.show_failure_reason === true || stageMeta.showFailureReason === true;
        const required = stage === 'deal_failed' || stage === 'معامله ناموفق' || showFailureFromMeta;

        if (failureBox) failureBox.classList.toggle('hidden', !required);
        if (reason) {
            reason.required = required;
            if (!required) reason.value = '';
        }
        if (otherBox) otherBox.classList.toggle('hidden', !required);
        if (otherDesc) {
            otherDesc.required = required;
            if (!required) {
                otherDesc.value = '';
            } else {
                otherDesc.placeholder = 'لطفاً توضیحات دلیل عدم موفقیت را وارد کنید';
            }
        }

        this.toggleActivityFailureOtherReason();
    },

    toggleDealActivityFailureOther() {
        const reason = document.getElementById('deal-activity-failure-reason')?.value || '';
        const otherBox = document.getElementById('deal-activity-failure-other');
        const otherDesc = document.getElementById('deal-activity-failure-other-description');
        const show = reason === 'other';
        if (otherBox) otherBox.classList.toggle('hidden', !show);
        if (otherDesc) {
            otherDesc.required = show;
            if (!show) otherDesc.value = '';
        }
    },

    async saveDealActivity(e) {
        e.preventDefault();
        if (this.state.dealHasActivity) {
            Swal.fire('خطا', 'برای این معامله قبلاً فعالیت ثبت شده است', 'error');
            return;
        }
        const contactId = document.getElementById('deal-activity-contact-id')?.value || window.currentContactId || '';
        const dealId = document.getElementById('deal-activity-deal-id')?.value || document.getElementById('deal-id')?.value || '';
        const stage = document.getElementById('activity-virtual-stage')?.value || document.getElementById('deal-activity-stage')?.value || '';
        const note = document.getElementById('deal-activity-result-note')?.value || '';
        // Requested services from state/hidden input
        let services = Array.isArray(this.state.dealActivityRequestedServices) ? this.state.dealActivityRequestedServices : [];
        if (!services.length) {
            const hiddenServices = document.getElementById('deal-activity-requested-services')?.value;
            try {
                const parsed = JSON.parse(hiddenServices || '[]');
                if (Array.isArray(parsed)) services = parsed;
            } catch (err) {
                console.warn('Could not parse deal activity services', err);
            }
        }
        const financialLevel = document.getElementById('deal-activity-financial-level')?.value || '';
        const asset = document.getElementById('deal-activity-asset')?.value || '';
        const income = document.getElementById('deal-activity-income')?.value || '';
        const isDone = document.getElementById('deal-activity-is-done')?.checked ? '1' : '0';
        const failureReason = document.getElementById('deal-activity-failure-reason')?.value || '';
        const failureOther = document.getElementById('deal-activity-failure-other-description')?.value || '';
        const activityType = document.getElementById('deal-activity-type-select')?.value || '';
        const direction = document.getElementById('deal-activity-direction')?.value || 'outgoing';
        const hasPurchaseOther = document.getElementById('deal-activity-has-purchase')?.value || '';
        const prevPurchaseNumber = document.getElementById('deal-activity-previous-number')?.value || '';
        const city = document.getElementById('deal-activity-city')?.value || '';
        const jobTitle = document.getElementById('deal-activity-job')?.value || '';
        const jobDescription = document.getElementById('deal-activity-job-desc')?.value || '';
        const extraInfo = document.getElementById('deal-activity-extra-info')?.value || '';
        const acqLess = document.getElementById('deal-activity-acq-less')?.checked;
        const acqYears = document.getElementById('deal-activity-acq-years')?.value || '0';
        const acqMonths = document.getElementById('deal-activity-acq-months')?.value || '0';
        const acquaintanceDuration = acqLess ? 'کمتر از 1 ماه' : `${acqYears} سال ${acqMonths} ماه`;
        const hasPreviousPurchaseVal = document.getElementById('deal-activity-has-purchase')?.value ?? '';

        if (!contactId) {
            Swal.fire('خطا', 'شناسه لید خالی است', 'error');
            return;
        }
        if (!dealId) {
            console.warn('saveDealActivity blocked: empty dealId', {contactId});
            Swal.fire('خطا', 'ابتدا معامله را ذخیره کنید، سپس فعالیت ثبت کنید', 'error');
            this.showDealFormTab('info');
            return;
        }
        if (!stage) {
            Swal.fire('خطا', 'مرحله کاریز را انتخاب کنید', 'error');
            return;
        }
        if (!note.trim()) {
            Swal.fire('خطا', 'نتیجه فعالیت را وارد کنید', 'error');
            return;
        }
        if (!services.length) {
            Swal.fire('خطا', 'خدمات درخواستی را انتخاب کنید', 'error');
            return;
        }
        const isStageFailed = stage === 'deal_failed' || stage === 'معامله ناموفق';
        const isStageSuccess = stage === 'deal_success';

        if (isStageFailed && !failureReason) {
            Swal.fire('خطا', 'دلیل ناموفق را انتخاب کنید', 'error');
            return;
        }
        if (isStageFailed && failureReason === 'other' && !failureOther.trim()) {
            Swal.fire('خطا', 'توضیح دلیل ناموفق را وارد کنید', 'error');
            return;
        }
        if (isStageSuccess && !this.hasCompletePaymentsForSuccess()) {
            Swal.fire('خطا', 'برای ثبت معامله موفق، جزئیات پرداخت باید کامل باشد.', 'error');
            console.warn('saveDealActivity blocked: no complete payments for success', { stage, dealId });
            return;
        }
        if (!activityType) {
            Swal.fire('خطا', 'نوع فعالیت را انتخاب کنید', 'error');
            return;
        }

        const isLocked = (el) => el?.dataset?.locked === 'true';
        const cityEl = document.getElementById('deal-activity-city');
        const jobEl = document.getElementById('deal-activity-job');
        const prevPurchaseEl = document.getElementById('deal-activity-previous-number');
        const acqYearsEl = document.getElementById('deal-activity-acq-years');
        const acqMonthsEl = document.getElementById('deal-activity-acq-months');

        if (!city.trim() && !isLocked(cityEl)) {
            Swal.fire('خطا', 'شهر را وارد کنید', 'error');
            return;
        }
        if (!jobTitle.trim() && !isLocked(jobEl)) {
            Swal.fire('خطا', 'شغل را وارد کنید', 'error');
            return;
        }

        const acqLocked = isLocked(acqYearsEl) || isLocked(acqMonthsEl);
        if (!acqLocked && !acqLess && acqYears === '0' && acqMonths === '0') {
            Swal.fire('خطا', 'مدت زمان آشنایی را وارد کنید', 'error');
            return;
        }

        let prevPurchaseClean = (prevPurchaseNumber || '').trim();
        if (prevPurchaseClean) {
            prevPurchaseClean = prevPurchaseClean.replace(/\D/g, '');
            if (!/^[1-9]\d{9}$/.test(prevPurchaseClean)) {
                Swal.fire('خطا', 'شماره خرید قبلی باید ۱۰ رقم و بدون صفر شروع باشد', 'error');
                return;
            }
        }
        const cityValue = city.trim();
        const jobValue = jobTitle.trim();
        const jobDescValue = jobDescription.trim();
        const extraInfoValue = extraInfo.trim();

        const payload = {
            contact_id: contactId,
            deal_id: dealId,
            pipeline_stage: isStageFailed ? '' : (isStageSuccess ? 'اتمام' : stage),
            result_note: note,
            requested_services: JSON.stringify(services),
            financial_level: financialLevel,
            asset_estimation: asset,
            income_estimation: income,
            is_done: isDone,
            activity_type_id: activityType,
            direction,
            has_previous_purchase_other_number: hasPurchaseOther,
            previous_purchase_number: prevPurchaseClean,
            city: cityValue,
            job_title: jobValue,
            job_description: jobDescValue,
            acquaintance_duration: acquaintanceDuration,
            extra_info: extraInfoValue,
            failure_reason: failureReason,
            failure_other_description: failureOther,
            has_previous_purchase: hasPreviousPurchaseVal === '' ? null : hasPreviousPurchaseVal
        };

        try {
            Swal.showLoading();
            const res = await this.req('save_activity', payload);
            if (isStageFailed) {
                const reasonText = {
                    'no_financial_ability': 'عدم توان مالی',
                    'no_purchase_intent': 'فعلا قصد خرید ندارد',
                    'competitor_purchase': 'خرید از رقبا',
                    'wrong_number': 'شماره اشتباه',
                    'irrelevant_lead': 'لید نامرتبط',
                    'other': 'سایر'
                };
                const statusRes = await this.req('set_deal_status', {
                    deal_id: dealId,
                    status: '0',
                    contact_id: contactId,
                    pipeline_stage: 'معامله ناموفق',
                    failure_reason: failureReason,
                    failure_reason_text: reasonText[failureReason] || '',
                    failure_other_description: failureOther
                });
                if (statusRes?.status !== 'success') {
                    console.warn('set_deal_status failed for lost', statusRes);
                    Swal.fire('خطا', statusRes?.message || 'وضعیت معامله به ناموفق تغییر نکرد', 'error');
                    return;
                }
            } else if (isStageSuccess) {
                const statusRes = await this.req('set_deal_status', { deal_id: dealId, status: '1', contact_id: contactId, pipeline_stage: 'اتمام' });
                if (statusRes?.status !== 'success') {
                    console.warn('set_deal_status failed for won', statusRes);
                    Swal.fire('خطا', statusRes?.message || 'وضعیت معامله به موفق تغییر نکرد', 'error');
                    return;
                }
            }
            Swal.close();
            if (res.status === 'success') {
                Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'فعالیت ثبت شد', showConfirmButton:false, timer:2000});
                this.state.dealHasActivity = true;
                this.toggleDealActivityForm(true);
                // refresh profile/deal list
                if (window.currentContactId) {
                    await this.openPersonProfile(window.currentContactId);
                }
                if (document.getElementById('deals-container')) {
                    await this.loadDeals();
                }
                await this.openDealModal(dealId);
                this.showDealFormTab('activities');
            } else {
                Swal.fire('خطا', res.message || 'خطای نامشخص در ثبت فعالیت', 'error');
            }
        } catch (err) {
            console.error('saveDealActivity error', err);
            Swal.close();
            Swal.fire('خطا', 'ثبت فعالیت انجام نشد', 'error');
        }
    },

    // Save only extra info (without creating activity) for missing fields
    async saveDealInfoOnly() {
        const contactId = document.getElementById('deal-activity-contact-id')?.value || window.currentContactId || '';
        const dealId = document.getElementById('deal-activity-deal-id')?.value || document.getElementById('deal-id')?.value || '';
        if (!contactId) {
            Swal.fire('خطا', 'شناسه لید خالی است', 'error');
            return;
        }
        const payload = {
            contact_id: contactId,
            deal_id: dealId
        };

        const addIfEditable = (id, key) => {
            const el = document.getElementById(id);
            if (!el || el.dataset.locked === 'true') return;
            const val = (el.value || '').trim();
            if (val) payload[key] = val;
        };

        addIfEditable('deal-activity-city', 'city');
        addIfEditable('deal-activity-job', 'job_title');
        addIfEditable('deal-activity-job-desc', 'job_description');
        addIfEditable('deal-activity-extra-info', 'extra_info');
        addIfEditable('deal-activity-financial-level', 'financial_level');
        addIfEditable('deal-activity-asset', 'asset_estimation');
        addIfEditable('deal-activity-income', 'income_estimation');

        // Acquaintance duration
        const acqLess = document.getElementById('deal-activity-acq-less');
        const acqYears = document.getElementById('deal-activity-acq-years');
        const acqMonths = document.getElementById('deal-activity-acq-months');
        if (acqYears && acqMonths && !(acqYears.dataset.locked === 'true' || acqMonths.dataset.locked === 'true')) {
            let acqVal = '';
            if (acqLess?.checked) {
                acqVal = 'کمتر از 1 ماه';
            } else {
                acqVal = `${acqYears.value || '0'} سال ${acqMonths.value || '0'} ماه`;
            }
            if (acqVal.trim()) payload.acquaintance_duration = acqVal;
        }

        // Requested services
        if (Array.isArray(this.state.dealActivityRequestedServices) && this.state.dealActivityRequestedServices.length > 0) {
            payload.requested_services = JSON.stringify(this.state.dealActivityRequestedServices);
        }

        // Purchase status if not locked
        const purchaseSelect = document.getElementById('deal-activity-has-purchase');
        if (purchaseSelect && purchaseSelect.dataset.locked !== 'true') {
            const val = purchaseSelect.value;
            if (val !== '') {
                payload.has_previous_purchase = val;
            }
        }

        if (Object.keys(payload).length <= 2) {
            Swal.fire('خطا', 'فیلد قابل ویرایش یا مقداری برای ذخیره وجود ندارد', 'error');
            return;
        }

        try {
            Swal.showLoading();
            const res = await this.req('save_deal_info', payload);
            Swal.close();
            if (res.status === 'success') {
                Swal.fire({toast:true, position:'bottom-end', icon:'success', title:res.message || 'ذخیره شد', showConfirmButton:false, timer:2000});
                if (contactId) await this.openPersonProfile(contactId);
                if (dealId) await this.openDealModal(dealId);
            } else {
                Swal.fire('خطا', res.message || 'ذخیره انجام نشد', 'error');
            }
        } catch (err) {
            console.error('saveDealInfoOnly error', err);
            Swal.close();
            Swal.fire('خطا', 'ذخیره اطلاعات انجام نشد', 'error');
        }
    },

    startActivityForDeal(presetStage = '') {
        // محاسبه contactId از چند منبع تا همیشه مقدار داشته باشد
        const contactId =
            document.getElementById('deal-contact-id')?.value ||
            this.state.currentDeal?.local_person_id ||
            this.state.currentDeal?.contact_didar_id ||
            this.state.currentLead?.id ||
            this.state.currentLead?.didar_contact_id ||
            '';
        if (!contactId) {
            Swal.fire('خطا', 'شناسه شخص برای این معامله مشخص نیست', 'error');
            return;
        }
        const dealId = document.getElementById('deal-id')?.value || this.state.currentDeal?.didar_deal_id || this.state.currentDeal?.id || '';
        const activityContactInput = document.getElementById('activity-contact-id');
        if (activityContactInput) {
            activityContactInput.value = contactId;
        }
        window.currentContactId = contactId; // برای openModal که از window.currentContactId استفاده می‌کند

        // اطمینان از اینکه لیست معاملات در اکتیویتی پر است
        let deals = Array.isArray(this.state.currentDeals) ? [...this.state.currentDeals] : [];
        if (dealId && !deals.find(d => String(d.didar_deal_id || d.id || d.deal_id) === String(dealId))) {
            if (this.state.currentDeal) {
                deals.push(this.state.currentDeal);
            }
        }
        // اگر هیچ معامله‌ای در لیست نبود، یک معامله موقت برای انتخاب فعالیت بسازیم
        if (!dealId && deals.length === 0 && (this.state.currentDeal || contactId)) {
            const pipelineStageFallback = document.getElementById('deal-pipeline-stage')?.value || '';
            const tempDeal = {
                didar_deal_id: '',
                id: '',
                deal_id: '',
                contact_didar_id: contactId,
                title: document.getElementById('deal-title')?.value || 'معامله',
                pipeline_stage: pipelineStageFallback
            };
            deals.push(tempDeal);
            this.state.currentDeal = tempDeal;
        }

        const pipelineStage = presetStage || document.getElementById('deal-pipeline-stage')?.value || this.state.currentDeal?.pipeline_stage || '';
        this.populateActivityDealSelect(deals, dealId, pipelineStage);

        const stageSelect = document.getElementById('activity-virtual-stage');
        if (stageSelect && pipelineStage) {
            stageSelect.value = pipelineStage;
        }
        if (pipelineStage) {
            const stageDisplay = document.getElementById('activity-current-stage-display');
            const currentStageInput = document.getElementById('activity-current-stage');
            if (stageDisplay) stageDisplay.innerHTML = `<span class="text-sm text-gray-200">${pipelineStage}</span>`;
            if (currentStageInput) currentStageInput.value = pipelineStage;
        }

        // ابتدا مودال را باز کن تا عناصر فرم حتماً در دسترس باشند، سپس استیج را دوباره ست کن
        this.openModal('modalActivity');
        const activityDealIdInput = document.getElementById('activity-deal-id');
        if (activityDealIdInput && dealId) {
            activityDealIdInput.value = dealId;
        }
        if (stageSelect && pipelineStage) {
            stageSelect.value = pipelineStage;
        }
        const stageDisplay = document.getElementById('activity-current-stage-display');
        const currentStageInput = document.getElementById('activity-current-stage');
        if (stageDisplay) stageDisplay.innerHTML = pipelineStage
            ? `<span class="text-sm text-gray-200">${pipelineStage}</span>`
            : '<span class="text-sm text-gray-500">مرحله تعیین نشده</span>';
        if (currentStageInput) currentStageInput.value = pipelineStage || 'keep_current';

        // برای جلوگیری از پوشانده شدن مودال فعالیت، ابتدا مودال معامله را ببند
        this.closeModal('modalDeal');
    },

    scrollToDealPayments() {
        this.showDealFormTab('payments');
        const section = document.getElementById('deal-payment-details');
        if (section) {
            section.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    },

    openSupportFromDeal() {
        if (!this.state.currentDeal) {
            Swal.fire('خطا', 'ابتدا معامله را انتخاب یا ذخیره کنید', 'error');
            return;
        }
        this.openCourseModal(this.state.currentDeal, 'پشتیبانی');
    },

    openCourseFromDeal() {
        if (!this.state.currentDeal) {
            Swal.fire('خطا', 'ابتدا معامله را انتخاب یا ذخیره کنید', 'error');
            return;
        }
        this.openCourseModal(this.state.currentDeal, 'برگزاری دوره');
    },
    
    // OPEN DEAL MODAL
    async openDealModal(dealId) {
        let personForSummary = null;
        
        // Populate lead info display in deal modal
        const dealInfoDisplay = document.getElementById('deal-lead-info-display');
        const dealNameText = document.getElementById('deal-lead-name-text');
        if (this.state.currentLead && dealInfoDisplay && dealNameText) {
            const fullName = `${this.state.currentLead.first_name || ''} ${this.state.currentLead.last_name || ''}`.trim() || 'بدون نام';
            dealNameText.textContent = `${fullName} (${this.state.currentLead.mobile_phone || '-'})`;
            dealInfoDisplay.classList.remove('hidden');
        } else if (dealInfoDisplay) {
            dealInfoDisplay.classList.add('hidden');
        }

        // ensure products list
        if (!this.state.products || this.state.products.length === 0) {
            await this.loadProducts();
        }
        const stageSelectEl = document.getElementById('deal-pipeline-stage');
        if (stageSelectEl) {
            stageSelectEl.onchange = () => {
                const badge = document.getElementById('deal-summary-stage');
                if (badge) badge.textContent = stageSelectEl.value || 'مرحله تعیین نشده';
                this.updateDealCurrentFields();
            };
        }
        // برای معامله جدید، تب اطلاعات معامله نمایش داده شود تا ابتدا معامله ذخیره گردد
        this.showDealFormTab('info');
        
        // ✅ Ensure configs loaded then apply (Tooltips, Logic, Defaults, Roles)
        await this.ensureFieldConfigsLoaded();
        this.applyFieldConfigurations('deal', '#deal-form');

        // اگر اطلاعات اولیه در پروفایل لید موجود است، آن بخش را مخفی کن
        const person = personForSummary || this.state.currentLead;
        const infoHasValues =
            !!(person?.city || person?.job_title || person?.job_description || person?.extra_info ||
                person?.financial_level || person?.asset_estimation || person?.income_estimation ||
                person?.acquaintance_duration || (person?.has_previous_purchase !== undefined && person?.has_previous_purchase !== null));
        this.toggleInitialInfoVisibility(!infoHasValues);
        await this.loadActivityTypesForModal();

	        if (!dealId) {
	            // New deal - Reset form first
	            this.state.dealHasActivity = false;
	            this.setPaymentsLockState(false);
	            document.getElementById('deal-modal-title').textContent = 'ثبت معامله جدید';
	            document.getElementById('deal-form').reset();
            document.getElementById('deal-id').value = '';
            document.getElementById('deal-contact-id').value = '';
            document.getElementById('deal-user-name').value = '';
            document.getElementById('deal-user-mobile').value = '';
	            this.state.dealActivityRequestedServices = [];
	            this.populateDealActivityServices();
	            this.renderDealActivityServices();

	            // Ensure action buttons are visible (previous deal might have hidden them)
	            const saveInfoBtn = document.getElementById('deal-info-submit-btn');
		            const savePaymentsBtn = document.querySelector("button[onclick*=\"saveDealFromTab('payments')\"]");
	            const addPaymentBtn = document.getElementById('deal-add-payment-btn');
	            const activityBtns = document.querySelectorAll('button[onclick*="startActivityForDeal"]');
	            if (saveInfoBtn) saveInfoBtn.classList.remove('hidden');
	            if (savePaymentsBtn) savePaymentsBtn.classList.remove('hidden');
	            if (addPaymentBtn) addPaymentBtn.classList.remove('hidden');
	            activityBtns.forEach(btn => btn.classList.remove('hidden'));
	            this.enableDealFormFields();
	            this.dealIsWon = false;

            // Ensure purchase field is enabled after form reset
            const purchaseSelectAfterReset = document.getElementById('deal-activity-has-purchase');
            if (purchaseSelectAfterReset) {
                purchaseSelectAfterReset.disabled = false;
                purchaseSelectAfterReset.dataset.locked = '';
                purchaseSelectAfterReset.classList.remove('bg-gray-900', 'cursor-not-allowed');
            }
	            this.state.currentDeal = null;
	            const pipelineStageSelect = document.getElementById('deal-pipeline-stage');
	            if (pipelineStageSelect) {
	                pipelineStageSelect.value = '';
	                pipelineStageSelect.disabled = false;
	                pipelineStageSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
	            }
            this.renderProductOptions();
            
		            // Enable deal status select for new deals
		            let dealStatusSelect = document.getElementById('deal-status');
		        if (dealStatusSelect) {
		            dealStatusSelect.disabled = false;
		            dealStatusSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
		        }
            
            // Load contact info from currentContactId or currentLead
            const contactId = window.currentContactId;
            if (!contactId) {
                // No contact selected - show error
                Swal.fire('خطا', 'لطفاً ابتدا یک شخص را انتخاب کنید', 'error');
                return;
            }

            if (!this.state.currentLead || this.state.currentLead.didar_contact_id !== contactId) {
                this.state.currentDeals = [];
            }
            
            document.getElementById('deal-contact-id').value = contactId;
            const dealActContact = document.getElementById('deal-activity-contact-id');
            if (dealActContact) dealActContact.value = contactId;
            const dealActDeal = document.getElementById('deal-activity-deal-id');
            if (dealActDeal) dealActDeal.value = '';
            
            // Try to get from currentLead first (if profile is already loaded)
            if (this.state.currentLead && this.state.currentLead.didar_contact_id === contactId) {
                const person = this.state.currentLead;
                personForSummary = person;
                const fullName = `${person.first_name || ''} ${person.last_name || ''}`.trim() || 'بدون نام';
                document.getElementById('deal-user-name').value = fullName;
                document.getElementById('deal-user-mobile').value = person.mobile_phone || '';
            } else {
                // Load person info from API
                try {
                    const res = await this.req('get_person_profile', {contact_id: contactId});
                    if (res.status === 'success' && res.person) {
                        const person = res.person;
                        personForSummary = person;
                        const fullName = `${person.first_name || ''} ${person.last_name || ''}`.trim() || 'بدون نام';
                        document.getElementById('deal-user-name').value = fullName;
                        document.getElementById('deal-user-mobile').value = person.mobile_phone || '';
                        // Update currentLead for future use
                        this.state.currentLead = person;
                        this.state.currentDeals = res.deals || [];
                    } else {
                        console.error('Failed to load person profile:', res);
                        Swal.fire('خطا', res.message || 'خطا در بارگذاری اطلاعات شخص', 'error');
                        return;
                    }
                } catch (error) {
                    console.error('Error loading person profile:', error);
                    Swal.fire('خطا', 'خطا در بارگذاری اطلاعات شخص', 'error');
                    return;
                }
            }

            // Generate deal title for new deals
            try {
                const titleRes = await this.req('generate_deal_title', {contact_id: contactId});
                if (titleRes.status === 'success' && titleRes.title) {
                    const titleSelect = document.getElementById('deal-title');
                    if (titleSelect) {
                        // Add the generated title as an option if it doesn't exist
                        const existingOption = Array.from(titleSelect.options).find(option => option.value === titleRes.title);
                        if (!existingOption) {
                            const newOption = document.createElement('option');
                            newOption.value = titleRes.title;
                            newOption.textContent = titleRes.title;
                            newOption.dataset.generated = '1';
                            titleSelect.appendChild(newOption);
                        } else {
                            existingOption.dataset.generated = '1';
                        }
                        titleSelect.value = titleRes.title;
                        const modalTitleEl = document.getElementById('deal-modal-title');
                        if (modalTitleEl) {
                            modalTitleEl.textContent = `ثبت معامله جدید: ${titleRes.title}`;
                        }
                    }
                }
            } catch (error) {
                console.error('Failed to generate deal title:', error);
                // Continue without setting title - user can select manually
            }

            this.paymentRowCounter = 0;
            // برای معامله جدید، پرداختی پیش‌فرض اضافه نکن تا کاربر در صورت نیاز خودش اضافه کند
            this.loadPayments([], false, false);
            this.toggleDiscountFields();
            // Set default status placeholders for new deals
            dealStatusSelect = document.getElementById('deal-status');
            if (dealStatusSelect) {
                dealStatusSelect.disabled = false; // Enable for new deals
                dealStatusSelect.value = '';
                this.handleDealStatusChange();
            }
            // Reset dealIsWon for new deals
            this.dealIsWon = false;
            
            // Ensure all fields are enabled for new deals
            this.enableDealFormFields();

            // Summary + activity dropdown for new deal
            const stageValue = pipelineStageSelect?.value || '';
            // Prefill extra info fields from person (if available)
            const cityInput = document.getElementById('deal-activity-city');
            const jobInput = document.getElementById('deal-activity-job');
            const jobDescInput = document.getElementById('deal-activity-job-desc');
            const extraInfoInput = document.getElementById('deal-activity-extra-info');
            const acqYears = document.getElementById('deal-activity-acq-years');
            const acqMonths = document.getElementById('deal-activity-acq-months');
            const acqLess = document.getElementById('deal-activity-acq-less');
            const acqVal = personForSummary?.acquaintance_duration || '';
            if (cityInput) cityInput.value = personForSummary?.city || '';
            if (jobInput) jobInput.value = personForSummary?.job_title || '';
            if (jobDescInput) jobDescInput.value = personForSummary?.job_description || '';
            if (extraInfoInput) extraInfoInput.value = personForSummary?.extra_info || '';
            const purchaseSelect = document.getElementById('deal-activity-has-purchase');
            if (purchaseSelect) {
                // Always reset lock/disabled state for a brand-new deal
                purchaseSelect.disabled = false;
                purchaseSelect.dataset.locked = '';
                purchaseSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');

                // For new deals, always start with empty value (no default selection)
                purchaseSelect.value = '';
            }
            // Hide previous-number field by default for new deals
            const prevPurchaseField = document.getElementById('deal-previous-purchase-number-field');
            if (prevPurchaseField) prevPurchaseField.classList.add('hidden');
            this.handlePurchaseOtherNumberChange();
            if (acqYears && acqMonths && acqLess) {
                if (acqVal && acqVal.includes('کمتر')) {
                    acqLess.checked = true;
                    this.toggleAcqLessThanMonth();
                } else {
                    const match = acqVal.match(/(\d+)\s*سال.*?(\d+)\s*ماه/);
                    acqYears.value = match ? match[1] : '0';
                    acqMonths.value = match ? match[2] : '0';
                    acqLess.checked = false;
                    this.toggleAcqLessThanMonth();
                }
            }
            this.lockDealActivityExtraFields({
                city: cityInput?.value || '',
                job_title: jobInput?.value || '',
                job_description: jobDescInput?.value || '',
                extra_info: extraInfoInput?.value || '',
                financial_level: document.getElementById('deal-activity-financial-level')?.value || '',
                asset_estimation: document.getElementById('deal-activity-asset')?.value || '',
                income_estimation: document.getElementById('deal-activity-income')?.value || '',
                requested_services: this.state.dealActivityRequestedServices,
                acquaintance_duration: acqVal,
                has_previous_purchase: undefined // Always keep purchase field unlocked for new deals
            });
            this.updateDealModalSummary(personForSummary || this.state.currentLead, {
                pipeline_stage: stageValue,
                contact_name: document.getElementById('deal-user-name').value || '',
                contact_mobile: document.getElementById('deal-user-mobile').value || '',
                owner_name: (personForSummary || this.state.currentLead)?.owner_name || ''
            });
            const dealActStage = document.getElementById('deal-activity-stage');
            const dealActCurrent = document.getElementById('deal-activity-current-stage');
            if (dealActStage) dealActStage.value = stageValue;
            if (dealActCurrent) dealActCurrent.textContent = stageValue || '-';
            this.populateDealActivityStages();
            this.populateActivityDealSelect(this.state.currentDeals || []);
            this.populateDealActivityServices();
            this.showDealFormTab('activities');
            // فعالیت‌ها در تب فعالیت
            this.renderDealActivitiesInline(contactId);
            this.toggleDealActivityForm(false);
            
            this.openModal('modalDeal');
            return;
        }
        
        // Load deal
        const res = await this.req('get_deal', {deal_id: dealId});
	        if (res.status === 'success' && res.deal) {
            this.setPaymentsLockState(true);
            const deal = res.deal;
            const dealTitleText = deal.title || deal.pipeline_stage_title || deal.pipeline_stage || 'معامله';
            document.getElementById('deal-modal-title').textContent = `ویرایش معامله: ${dealTitleText}`;
            
            // Set deal status in form
            const dealStatusSelect = document.getElementById('deal-status');
            const isWonDeal = deal.status === 'Won';
            const pipelineStageSelect = document.getElementById('deal-pipeline-stage');
            const productContainer = document.getElementById('deal-products-list');
            
            // For editing, keep selects empty to force user selection if they want to change
            if (dealStatusSelect) {
                dealStatusSelect.value = '';
                this.handleDealStatusChange();
            }
            if (pipelineStageSelect) {
                pipelineStageSelect.value = '';
            }
            
            // Display current values in read-only fields
            const statusText = deal.status === 'Won' ? 'موفق' : (deal.status === 'Lost' ? 'ناموفق' : 'جاری');
            const currentStatusDisplay = document.getElementById('deal-current-status');
            const currentStageDisplay = document.getElementById('deal-current-stage');
            
            if (currentStatusDisplay) currentStatusDisplay.textContent = statusText;
            if (currentStageDisplay) currentStageDisplay.textContent = deal.pipeline_stage || '-';

            this.updateDealCurrentFields(deal.pipeline_stage || '', deal.status);

            // Mark products
            if (Array.isArray(deal.products)) {
                const selectedIds = deal.products.map(p => p.product_id);
                this.renderProductOptions(selectedIds);
            } else {
                this.renderProductOptions();
            }
            
            // Save deal info for course modal buttons
            this.state.currentDeal = deal;
            this.state.currentDeals = [deal];
            this.populateDealActivityStages();
            let requestedServices = [];
            try {
                if (deal.requested_services) {
                    const parsed = JSON.parse(deal.requested_services);
                    if (Array.isArray(parsed)) requestedServices = parsed;
                }
            } catch (e) {
                // ignore parse errors
            }
            // Fallback: if عنوان حاوی | خدمات است، از آن استخراج کن
            if ((!requestedServices || requestedServices.length === 0) && deal.title && deal.title.includes('|')) {
                const parts = deal.title.split('|');
                const svc = (parts[parts.length - 1] || '').trim();
                if (svc) requestedServices = [svc];
            }
            this.populateDealActivityServices(requestedServices);

            const dealContactIdInput = document.getElementById('deal-contact-id');
            const dealActContact = document.getElementById('deal-activity-contact-id');
            const dealActDeal = document.getElementById('deal-activity-deal-id');
            const dealActStage = document.getElementById('deal-activity-stage');
            const dealActCurrent = document.getElementById('deal-activity-current-stage');
            const normalizedContactId = deal.contact_didar_id || deal.contact_id || deal.contactId || deal.local_person_id || '';
            if (dealContactIdInput) dealContactIdInput.value = normalizedContactId;
            if (dealActContact) dealActContact.value = normalizedContactId;
            if (dealActDeal) dealActDeal.value = deal.didar_deal_id || deal.id || '';
            if (dealActStage) dealActStage.value = deal.pipeline_stage || '';
            if (dealActCurrent) dealActCurrent.textContent = deal.pipeline_stage || '-';

            const setVal = (id, val) => {
                const el = document.getElementById(id);
                if (el) el.value = val ?? '';
            };
            const extra = deal.extra_fields || {};
            setVal('deal-activity-financial-level', deal.financial_level || extra.financial_level || '');
            setVal('deal-activity-asset', deal.asset_estimation || extra.asset_estimation || '');
            setVal('deal-activity-income', deal.income_estimation || extra.income_estimation || '');
            const purchaseSelect = document.getElementById('deal-activity-has-purchase');
            // Capture purchase value (from deal -> extra fields -> person) but don't lock yet
            const purchaseVal = (deal.has_previous_purchase !== undefined && deal.has_previous_purchase !== null)
                ? deal.has_previous_purchase
                : (extra.has_previous_purchase !== undefined && extra.has_previous_purchase !== null
                    ? extra.has_previous_purchase
                    : personForSummary?.has_previous_purchase);
            const hasPurchaseVal = purchaseVal === 0 || purchaseVal === 1 || purchaseVal === '0' || purchaseVal === '1';
            if (purchaseSelect) {
                purchaseSelect.value = hasPurchaseVal ? String(purchaseVal) : '';
                purchaseSelect.disabled = false;
                purchaseSelect.dataset.locked = '';
                purchaseSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
            }

            this.showDealFormTab('activities');
            this.renderDealActivitiesInline(normalizedContactId);
            const hasAct = await this.checkDealHasActivity(dealId);
            this.state.dealHasActivity = hasAct;
            this.toggleDealActivityForm(hasAct);
            if (purchaseSelect) {
                const shouldLockPurchase =
                    (purchaseVal === 1 || purchaseVal === '1') ||
                    (hasPurchaseVal && hasAct);
                if (shouldLockPurchase) {
                    purchaseSelect.disabled = true;
                    purchaseSelect.dataset.locked = 'true';
                    purchaseSelect.classList.add('bg-gray-900', 'cursor-not-allowed');
                } else {
                    purchaseSelect.disabled = false;
                    purchaseSelect.dataset.locked = '';
                    purchaseSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
                    purchaseSelect.value = '';
                }
            }
            
            // Check if deal is Won or Lost
            const isWonOrLost = deal.status === 'Won' || deal.status === 'Lost';
            
            // For won or lost deals, disable form fields and actions
            const saveInfoBtn = document.getElementById('deal-info-submit-btn');
            const savePaymentsBtn = document.querySelector('button[onclick*="saveDealFromTab(\'payments\')"]');
            const addPaymentBtn = document.getElementById('deal-add-payment-btn');
            const activityBtns = document.querySelectorAll('button[onclick*="startActivityForDeal"]');
            
            if (isWonOrLost) {
                // Disable deal status select
                if (dealStatusSelect) {
                    dealStatusSelect.disabled = true;
                    dealStatusSelect.classList.add('bg-gray-900', 'cursor-not-allowed');
                }
                if (pipelineStageSelect) {
                    pipelineStageSelect.disabled = true;
                    pipelineStageSelect.classList.add('bg-gray-900', 'cursor-not-allowed');
                }
                
                // Hide action buttons
                if (saveInfoBtn) saveInfoBtn.classList.add('hidden');
                if (savePaymentsBtn) savePaymentsBtn.classList.add('hidden');
                if (addPaymentBtn) addPaymentBtn.classList.add('hidden');
                activityBtns.forEach(btn => btn.classList.add('hidden'));
                if (pipelineStageSelect) {
                    pipelineStageSelect.disabled = true;
                }
                
                // Disable service and price fields
                const titleField = document.getElementById('deal-title');
                const serviceCostField = document.getElementById('deal-service-cost');
                const priceListCodeField = document.getElementById('deal-price-list-code');
                const payableAmountField = document.getElementById('deal-payable-amount');
                const hasDiscountField = document.getElementById('deal-has-discount');
                const discountTypeField = document.getElementById('deal-discount-type');
                const discountOccasionField = document.getElementById('deal-discount-occasion');
                const discountOccasionCustomField = document.getElementById('deal-discount-occasion-custom');
                const discountAmountField = document.getElementById('deal-discount-amount');
                const paymentConditionsField = document.getElementById('deal-payment-conditions');
                
                // Apply readonly/disabled to all fields
                const readonlyClasses = ['bg-gray-900', 'cursor-not-allowed'];
                if (titleField) {
                    titleField.disabled = true;
                    readonlyClasses.forEach(className => titleField.classList.add(className));
                }
                if (serviceCostField) {
                    serviceCostField.readOnly = true;
                    readonlyClasses.forEach(className => serviceCostField.classList.add(className));
                }
                if (priceListCodeField) {
                    priceListCodeField.readOnly = true;
                    readonlyClasses.forEach(className => priceListCodeField.classList.add(className));
                }
                if (payableAmountField) {
                    payableAmountField.readOnly = true;
                    readonlyClasses.forEach(className => payableAmountField.classList.add(className));
                }
                if (hasDiscountField) {
                    hasDiscountField.disabled = true;
                    readonlyClasses.forEach(className => hasDiscountField.classList.add(className));
                }
                if (discountTypeField) {
                    discountTypeField.disabled = true;
                    readonlyClasses.forEach(className => discountTypeField.classList.add(className));
                }
                if (discountOccasionField) {
                    discountOccasionField.disabled = true;
                    readonlyClasses.forEach(className => discountOccasionField.classList.add(className));
                }
                if (discountOccasionCustomField) {
                    discountOccasionCustomField.readOnly = true;
                    readonlyClasses.forEach(className => discountOccasionCustomField.classList.add(className));
                }
                if (discountAmountField) {
                    discountAmountField.readOnly = true;
                    readonlyClasses.forEach(className => discountAmountField.classList.add(className));
                }
                if (paymentConditionsField) {
                    paymentConditionsField.readOnly = true;
                    readonlyClasses.forEach(className => paymentConditionsField.classList.add(className));
                }
                
                // Store readonly state for existing payments (only for won deals)
                if (isWonDeal) {
                    this.dealIsWon = true;
                } else {
                    this.dealIsWon = false;
                }
            } else {
                // Enable deal status select for pending deals
                if (dealStatusSelect) {
                    dealStatusSelect.disabled = false;
                    dealStatusSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
                }
                if (pipelineStageSelect) {
                    pipelineStageSelect.disabled = false;
                    pipelineStageSelect.classList.remove('bg-gray-900', 'cursor-not-allowed');
                }

                // Show action buttons
                if (saveInfoBtn) saveInfoBtn.classList.remove('hidden');
                if (savePaymentsBtn) savePaymentsBtn.classList.remove('hidden');
                if (addPaymentBtn) addPaymentBtn.classList.remove('hidden');
                activityBtns.forEach(btn => btn.classList.remove('hidden'));
                
                // Re-enable all other fields
                this.enableDealFormFields();

                this.dealIsWon = false;
            }
            
	            // Load failure reason if deal is lost
	            if (deal.status === 'Lost') {
	                const failureSection = document.getElementById('deal-failure-reason-section');
	                const failureReasonSelect = document.getElementById('deal-failure-reason');
	                const failureOtherDiv = document.getElementById('deal-failure-other-reason');
	                const failureOtherDesc = document.getElementById('deal-failure-other-description');

	                if (failureSection) failureSection.classList.remove('hidden');
	                if (failureReasonSelect) {
	                    failureReasonSelect.value = deal.failure_reason_code || '';
	                    failureReasonSelect.disabled = true; // Closed deal => read-only
	                }
	                if (failureOtherDiv) failureOtherDiv.classList.remove('hidden');
	                if (failureOtherDesc) {
	                    failureOtherDesc.value = deal.failure_reason_description || '';
	                    failureOtherDesc.readOnly = true;
	                }
	            } else {
	                const failureSection = document.getElementById('deal-failure-reason-section');
	                const failureReasonSelect = document.getElementById('deal-failure-reason');
	                const failureOtherDiv = document.getElementById('deal-failure-other-reason');
	                const failureOtherDesc = document.getElementById('deal-failure-other-description');
	                if (failureSection) failureSection.classList.add('hidden');
	                if (failureReasonSelect) {
	                    failureReasonSelect.value = '';
	                    failureReasonSelect.disabled = false;
	                }
	                if (failureOtherDiv) failureOtherDiv.classList.add('hidden');
	                if (failureOtherDesc) {
	                    failureOtherDesc.value = '';
	                    failureOtherDesc.readOnly = false;
	                }
	            }
            
            document.getElementById('deal-id').value = deal.didar_deal_id || deal.id;
            document.getElementById('deal-contact-id').value = normalizedContactId;
            
            // Load contact info - try from deal first, then from API if missing
            if (deal.contact_name && deal.contact_mobile) {
                document.getElementById('deal-user-name').value = deal.contact_name;
                document.getElementById('deal-user-mobile').value = deal.contact_mobile;
                personForSummary = personForSummary || {
                    first_name: deal.contact_name,
                    mobile_phone: deal.contact_mobile,
                    owner_name: deal.owner_name
                };
            }
            
            if (!personForSummary && (deal.local_person_id || deal.contact_didar_id || deal.contact_id || deal.contactId)) {
                // Load person info from API if not in deal data
                try {
                    const profileContactId = normalizedContactId;
                    const personRes = await this.req('get_person_profile', {contact_id: profileContactId});
                    if (personRes.status === 'success' && personRes.person) {
                        const person = personRes.person;
                        personForSummary = person;
                        this.state.currentLead = person;
                        this.state.currentDeals = personRes.deals || [deal];
                        const fullName = `${person.first_name || ''} ${person.last_name || ''}`.trim() || 'بدون نام';
                        document.getElementById('deal-user-name').value = fullName;
                        document.getElementById('deal-user-mobile').value = person.mobile_phone || '';
                    } else {
                        document.getElementById('deal-user-name').value = deal.contact_name || '';
                        document.getElementById('deal-user-mobile').value = deal.contact_mobile || '';
                    }
                } catch (error) {
                    console.error('Error loading person profile for deal:', error);
                    document.getElementById('deal-user-name').value = deal.contact_name || '';
                    document.getElementById('deal-user-mobile').value = deal.contact_mobile || '';
                }
            } else if (!personForSummary) {
                document.getElementById('deal-user-name').value = deal.contact_name || '';
                document.getElementById('deal-user-mobile').value = deal.contact_mobile || '';
            }

            // Update summary & activity dropdown with latest data
            const cityInput = document.getElementById('deal-activity-city');
            const jobInput = document.getElementById('deal-activity-job');
            const jobDescInput = document.getElementById('deal-activity-job-desc');
            const extraInfoInput = document.getElementById('deal-activity-extra-info');
            const acqYears = document.getElementById('deal-activity-acq-years');
            const acqMonths = document.getElementById('deal-activity-acq-months');
            const acqLess = document.getElementById('deal-activity-acq-less');
            
            // Prioritize currentLead (latest data) over deal snapshot
            const currentPerson = (this.state.currentLead && (
                                    this.state.currentLead.id == deal.local_person_id || 
                                    this.state.currentLead.didar_contact_id === deal.contact_didar_id || 
                                    this.state.currentLead.didar_contact_id === deal.contact_id
                                )) 
                                ? this.state.currentLead 
                                : null;
            const getVal = (prop) => currentPerson?.[prop] || deal[prop] || personForSummary?.[prop] || '';

            const acqVal = getVal('acquaintance_duration');
            if (cityInput) cityInput.value = getVal('city');
            if (jobInput) jobInput.value = getVal('job_title');
            if (jobDescInput) jobDescInput.value = getVal('job_description');
            if (extraInfoInput) extraInfoInput.value = getVal('extra_info');
            if (purchaseSelect) {
                purchaseSelect.value = hasPurchaseVal ? String(purchaseVal) : '';
            }
            if (acqYears && acqMonths && acqLess) {
                if (acqVal && acqVal.includes('کمتر')) {
                    acqLess.checked = true;
                    this.toggleAcqLessThanMonth();
                } else {
                    const match = acqVal.match(/(\d+)\s*سال.*?(\d+)\s*ماه/);
                    acqYears.value = match ? match[1] : '0';
                    acqMonths.value = match ? match[2] : '0';
                    acqLess.checked = false;
                    this.toggleAcqLessThanMonth();
                }
            }
            this.lockDealActivityExtraFields({
                city: cityInput?.value || '',
                job_title: jobInput?.value || '',
                job_description: jobDescInput?.value || '',
                extra_info: extraInfoInput?.value || '',
                financial_level: deal.financial_level || personForSummary?.financial_level || '',
                asset_estimation: deal.asset_estimation || personForSummary?.asset_estimation || '',
                income_estimation: deal.income_estimation || personForSummary?.income_estimation || '',
                requested_services: requestedServices,
                acquaintance_duration: acqVal,
                has_previous_purchase: (
                    (purchaseVal === 1 || purchaseVal === '1') ||
                    (hasPurchaseVal && hasAct)
                ) ? purchaseVal : undefined
            });
            this.updateDealModalSummary(personForSummary || this.state.currentLead, deal);
            
            // Update deal lead info display when opening existing deal
            const dealInfoDisplay = document.getElementById('deal-lead-info-display');
            const dealNameText = document.getElementById('deal-lead-name-text');
            if (dealInfoDisplay && dealNameText) {
                // Use person from deal or current lead - prioritize personForSummary
                const person = personForSummary || this.state.currentLead;
                const dealContactId = deal.contact_didar_id || deal.contact_id || '';
                const currentContactId = window.currentContactId || '';
                
                // If deal belongs to different contact than current, show warning
                if (dealContactId && currentContactId && dealContactId !== currentContactId) {
                    // This deal belongs to a different contact
                    const contactName = deal.contact_name || 'بدون نام';
                    dealNameText.textContent = `${contactName} (${deal.contact_mobile || '-'}) - ⚠️ این معامله متعلق به لید دیگری است`;
                    dealInfoDisplay.classList.remove('hidden');
                    dealInfoDisplay.classList.add('border-red-500'); // Add red border to indicate warning
                } else if (person && (
                    person.didar_contact_id === dealContactId || 
                    person.id == deal.local_person_id ||
                    (!dealContactId && person.didar_contact_id === currentContactId)
                )) {
                    const fullName = `${person.first_name || ''} ${person.last_name || ''}`.trim() || 'بدون نام';
                    dealNameText.textContent = `${fullName} (${person.mobile_phone || '-'})`;
                    dealInfoDisplay.classList.remove('hidden');
                    dealInfoDisplay.classList.remove('border-red-500');
                } else if (deal.contact_name) {
                    dealNameText.textContent = `${deal.contact_name} (${deal.contact_mobile || '-'})`;
                    dealInfoDisplay.classList.remove('hidden');
                    dealInfoDisplay.classList.remove('border-red-500');
                } else {
                    dealInfoDisplay.classList.add('hidden');
                }
            }
            
            this.populateActivityDealSelect(this.state.currentDeals || [deal], deal.didar_deal_id || deal.id);

            // ریست گزینه‌های خدمات بر اساس محصولات ثبت‌شده و سپس انتخاب مقدار معتبر
            const titleSelect = document.getElementById('deal-title');
            if (titleSelect) {
                this.renderDealTitleOptions(titleSelect.value || '');
            }
            let options = titleSelect ? Array.from(titleSelect.options || []) : [];

            // اگر requested_services موجود است، اولین مقدار را برای انتخاب استفاده کن
            let candidateService = (requestedServices && requestedServices[0]) || '';

            // اگر خالی بود ولی عنوان الگوی "| خدمت" داشت، بخش آخر را به‌عنوان خدمت بگیر
            if (!candidateService && deal.title && deal.title.includes('|')) {
                const parts = deal.title.split('|');
                candidateService = (parts[parts.length - 1] || '').trim();
            }

            if (titleSelect) {
                let hasOption = options.some(o => o.value === candidateService);
                if (!hasOption && candidateService) {
                    const legacyOption = document.createElement('option');
                    legacyOption.value = candidateService;
                    legacyOption.textContent = candidateService;
                    legacyOption.dataset.generated = '1';
                    titleSelect.appendChild(legacyOption);
                    hasOption = true;
                    options = Array.from(titleSelect.options || []);
                }
                titleSelect.value = hasOption ? candidateService : '';
            }
        const infoHasValues =
            !!(deal.city || deal.job_title || deal.job_description || deal.extra_info ||
                deal.financial_level || deal.asset_estimation || deal.income_estimation ||
                deal.acquaintance_duration || deal.has_previous_purchase !== undefined);
        this.toggleInitialInfoVisibility(!infoHasValues);
	            document.getElementById('deal-service-cost').value = formatMoneyString(deal.service_cost || '');
	            document.getElementById('deal-price-list-code').value = deal.price_list_code || '';
	            document.getElementById('deal-has-discount').value = deal.has_discount || 0;
	            document.getElementById('deal-discount-type').value = deal.discount_type || '';
	            document.getElementById('deal-discount-amount').value = formatMoneyString(deal.discount_amount || '');
	            const refundAmountEl = document.getElementById('deal-refund-amount');
	            const refundDescEl = document.getElementById('deal-refund-description');
	            if (refundAmountEl) refundAmountEl.value = formatMoneyString(deal.refund_amount || '');
	            if (refundDescEl) refundDescEl.value = deal.refund_description || '';
	            document.getElementById('deal-payable-amount').value = formatMoneyString(deal.payable_amount || '');
	            document.getElementById('deal-payment-conditions').value = deal.payment_conditions || '';
            
            // Handle discount occasion
            if (deal.discount_occasion) {
                const occasionMap = {
                    'black_friday': 'black_friday',
                    'teacher_birthday': 'teacher_birthday',
                    'nowruz': 'nowruz'
                };
                if (occasionMap[deal.discount_occasion]) {
                    document.getElementById('deal-discount-occasion').value = deal.discount_occasion;
                } else {
                    document.getElementById('deal-discount-occasion').value = 'other';
                    document.getElementById('deal-discount-occasion-custom').value = deal.discount_occasion;
                }
            }
            
            // Load payments
            this.paymentRowCounter = 0;
            let payments = [];
            console.log('Loading payments for deal:', deal);
            
            if (deal.payments && Array.isArray(deal.payments) && deal.payments.length > 0) {
                console.log('Using payments array:', deal.payments);
                payments = deal.payments.map(p => {
                    // Ensure date_persian exists
                    let datePersian = p.date_persian || '';
                    if (!datePersian && p.date) {
                        // Try to convert Gregorian to Persian
                        try {
                            const gDate = new Date(p.date);
                            if (!isNaN(gDate.getTime())) {
                                const pDate = new persianDate(gDate);
                                datePersian = pDate.format('YYYY/MM/DD');
                            } else {
                                // If date is already in Persian format, use it as is
                                datePersian = p.date;
                            }
                        } catch(e) {
                            datePersian = p.date || '';
                        }
                    }
                    
                    // Normalize payment keys to standard format
                    // Support both card_last4 and card_last_4, but use card_last4 as standard
                    const cardLast4 = p.card_last4 || p.card_last_4 || '';
                    const destinationBank = p.destination_bank || p.bank_destination || '';
                    const method = p.method || p.payment_method || '';
                    
                    const paymentObj = {
                        ...p,
                        date_persian: datePersian,
                        date: p.date || '',
                        method: method,
                        card_last4: cardLast4,  // Always use card_last4 (standard)
                        destination_bank: destinationBank  // Always use destination_bank (standard)
                    };
                    console.log('Processed payment:', paymentObj);
                    return paymentObj;
                });
            } else if (deal.payment_amount || deal.payment_method) {
                console.log('Using legacy payment fields');
                // Legacy single payment - convert date if needed
                let persianDate = '';
                if (deal.payment_date) {
                    try {
                        const gDate = new Date(deal.payment_date);
                        if (!isNaN(gDate.getTime())) {
                            const pDate = new persianDate(gDate);
                            persianDate = pDate.format('YYYY/MM/DD');
                        } else {
                            // If date is already in Persian format, use it as is
                            persianDate = deal.payment_date;
                        }
                    } catch(e) {
                        persianDate = deal.payment_date || '';
                    }
                }
                payments = [{
                    date: deal.payment_date || '',
                    date_persian: persianDate,
                    amount: deal.payment_amount || '',
                    method: deal.payment_method || '',
                    description: deal.payment_description || ''
                }];
                console.log('Legacy payment:', payments[0]);
            }
            
            console.log('Final payments array:', payments);
            // Pass null to loadPayments so it can decide readonly based on payment data
            this.loadPayments(payments, null, false);

            this.toggleDiscountFields();
            this.toggleDiscountOccasion();
            this.toggleCustomOccasion();

            // نمایش/پنهان کردن دکمه فعال‌سازی بر اساس وضعیت معامله
            const reactivateBtn = document.getElementById('reactivate-deal-btn');
            if (reactivateBtn) {
                const isClosed = deal.status === 'Won' || deal.status === 'Lost';
                reactivateBtn.style.display = this.isAdmin() && isClosed ? 'inline-block' : 'none';
            }

            this.openModal('modalDeal');
        } else {
            Swal.fire('خطا', res.message || 'خطا در بارگذاری معامله', 'error');
        }
    },
    
    // PAYMENT MANAGEMENT
    paymentRowCounter: 0,

    setPaymentsLockState(isLocked) {
        this.state.paymentsLocked = !!isLocked;
        const addBtn = document.getElementById('deal-add-payment-btn');
        if (addBtn) {
            addBtn.disabled = false;
            addBtn.title = isLocked ? 'پرداخت‌های ثبت‌شده قابل ویرایش نیستند؛ برای ثبت پرداخت جدید کلیک کنید' : '';
            addBtn.classList.remove('opacity-50', 'pointer-events-none');
        }
    },

    collectCurrentPayments() {
        const dates = document.querySelectorAll('input[name="payment_date[]"]');
        const amounts = document.querySelectorAll('input[name="payment_amount[]"]');
        const methods = document.querySelectorAll('select[name="payment_method[]"]');
        const descriptions = document.querySelectorAll('textarea[name="payment_description[]"]');
        const cardLast4s = document.querySelectorAll('input[name="payment_card_last4[]"]');
        const len = Math.max(dates.length, amounts.length, methods.length, descriptions.length, cardLast4s.length);
        const list = [];
        for (let i = 0; i < len; i++) {
            list.push({
                date_persian: dates[i]?.value || '',
                amount: amounts[i]?.value || '',
                method: methods[i]?.value || '',
                description: descriptions[i]?.value || '',
                card_last4: cardLast4s[i]?.value || ''
            });
        }
        return list.filter(p => p.date_persian || p.amount || p.method || p.description);
    },
    
    populatePaymentMethods(selectElement, selectedValue = '') {
        const options = this.getLookupOptions('payment_methods');
        selectElement.innerHTML = '<option value="">-- انتخاب نحوه پرداخت --</option>';
        options.forEach(opt => {
            const optElem = document.createElement('option');
            optElem.value = opt.value;
            optElem.textContent = opt.label;
            if (opt.value === selectedValue) optElem.selected = true;
            selectElement.appendChild(optElem);
        });
    },

    populateDestinationBanks(selectElement, selectedValue = '') {
        const options = this.getLookupOptions('payment_destination_banks');
        selectElement.innerHTML = '<option value="">-- انتخاب بانک --</option>';
        options.forEach(opt => {
            const optElem = document.createElement('option');
            optElem.value = opt.value;
            optElem.textContent = opt.label;
            if (opt.value === selectedValue) optElem.selected = true;
            selectElement.appendChild(optElem);
        });
    },

    addPaymentRow(paymentData = null, readonly = null) {
        // default readonly comes from paymentsLocked unless explicitly passed
        if (readonly === null) {
            const hasData = paymentData !== null && (
                paymentData.date || paymentData.date_persian || paymentData.amount || paymentData.method || paymentData.description
            );
            readonly = (this.state.paymentsLocked && hasData) || (this.dealIsWon && hasData);
        }
        const container = document.getElementById('payment-rows');
        if (!container) return;
        
        const rowId = `payment-row-${this.paymentRowCounter++}`;
        const row = document.createElement('div');
        row.id = rowId;
        row.className = 'bg-gray-800/50 p-4 rounded border border-gray-700 relative';
        
        // Use date_persian if available, otherwise try to convert date
        let dateValue = paymentData?.date_persian || '';
        if (!dateValue && paymentData?.date) {
            // Try to convert Gregorian to Persian
            try {
                const gDate = new Date(paymentData.date);
                if (!isNaN(gDate.getTime())) {
                    const pDate = new persianDate(gDate);
                    dateValue = pDate.format('YYYY/MM/DD');
                } else {
                    // If date is already in Persian format, use it as is
                    dateValue = paymentData.date;
                }
            } catch(e) {
                dateValue = paymentData.date || '';
            }
        }
        
        // Do not auto-fill date for empty rows; user should set it manually
        const amountValue = paymentData?.amount || '';
        const amountDisplay = formatMoneyString(amountValue);
        // Normalize method key (support both method and payment_method)
        const methodValue = paymentData?.method || paymentData?.payment_method || '';
        const descriptionValue = paymentData?.description || '';
        // Normalize card_last4 key (support both card_last4 and card_last_4, but use card_last4 as standard)
        const cardLast4Value = paymentData?.card_last4 || paymentData?.card_last_4 || '';
        // Normalize destination_bank key
        const bankValue = paymentData?.destination_bank || paymentData?.bank_destination || '';
        
        // Debug logging
        if (paymentData) {
            console.log('addPaymentRow - paymentData:', paymentData);
            console.log('addPaymentRow - dateValue:', dateValue, 'methodValue:', methodValue, 'amountValue:', amountValue);
        }
        
        // Readonly attributes
        const readonlyAttr = readonly ? 'readonly' : '';
        const readonlyClass = readonly ? 'bg-gray-900 cursor-not-allowed pointer-events-none opacity-70' : '';
        const isExistingPayment = paymentData !== null;
        const canDeletePayment = this.canUserDeleteFieldClient('payment_delete');
        const removeButton = (readonly || !canDeletePayment) ? '' : `<button type="button" onclick="app.removePaymentRow('${rowId}')" class="absolute top-2 left-2 text-red-500 hover:text-red-400 text-sm">
                <i class="fa-solid fa-times"></i>
            </button>`;
        
	        row.innerHTML = `
            ${removeButton}
            <div class="space-y-3 ${readonly ? 'pr-0' : 'pr-6'}">
                <div>
                    <label class="text-xs text-gray-400 mb-1 block">تاریخ پرداخت (شمسی)</label>
                    <input type="text" name="payment_date[]" class="input payment-date-jalali ${readonlyClass}" placeholder="1403/01/01" value="${dateValue}" ${readonlyAttr} ${readonly ? '' : 'required'}>
                </div>
                <div class="grid grid-cols-2 gap-3">
	                    <div>
	                        <label class="text-xs text-gray-400 mb-1 block">مبلغ پرداخت (تومان)</label>
	                        <input type="text" inputmode="numeric" name="payment_amount[]" class="input money-input ${readonlyClass}" placeholder="0" value="${amountDisplay}" ${readonlyAttr} ${readonly ? '' : 'required'} autocomplete="off">
	                    </div>
                    <div>
                        <label class="text-xs text-gray-400 mb-1 block">نحوه پرداخت</label>
                        <select name="payment_method[]" class="input bg-gray-800 payment-method-select ${readonlyClass}" ${readonly ? '' : 'required'}>
                            <option value="">-- انتخاب نحوه پرداخت --</option>
                            <!-- Options will be populated via app.populatePaymentMethods() -->
                        </select>
                    </div>
                </div>
                <div class="grid grid-cols-2 gap-3">
                    <div class="payment-card-last4 ${(methodValue === 'card_to_card' || methodValue === 'card_reader' || methodValue === 'card_to_card_lookup') ? '' : 'hidden'}">
                        <label class="text-xs text-gray-400 mb-1 block">4 رقم آخر کارت پرداخت کننده</label>
                        <input type="text" name="payment_card_last4[]" inputmode="numeric" maxlength="4" pattern="\\d{4}" class="input ${readonly ? '' : readonlyClass}" placeholder="1234" value="${cardLast4Value}" ${((methodValue === 'card_to_card' || methodValue === 'card_reader' || methodValue === 'card_to_card_lookup') && !readonly) ? 'required' : ''}>
                    </div>
                    <div class="payment-bank ${(methodValue === 'card_to_card' || methodValue === 'card_to_card_lookup') ? '' : 'hidden'}">
                        <label class="text-xs text-gray-400 mb-1 block">بانک مقصد پرداخت</label>
                        <select name="destination_bank[]" class="input bg-gray-800 destination-bank-select ${readonlyClass}">
                            <option value="">-- انتخاب بانک --</option>
                            <!-- Options will be populated via app.populateDestinationBanks() -->
                        </select>
                    </div>
                </div>
                <div>
                    <label class="text-xs text-gray-400 mb-1 block">توضیح پرداخت</label>
                    <textarea name="payment_description[]" class="input h-20 resize-none ${readonlyClass}" placeholder="توضیحات این پرداخت" ${readonlyAttr}>${descriptionValue}</textarea>
                </div>
            </div>
        `;
        
        container.appendChild(row);

        // Populate dynamic selects from lookups
        const methodSelect = row.querySelector('.payment-method-select');
        const bankSelect = row.querySelector('.destination-bank-select');
        const cardInput = row.querySelector('input[name="payment_card_last4[]"]');
        if (methodSelect) {
            this.populatePaymentMethods(methodSelect, methodValue);
        }
        if (bankSelect) {
            this.populateDestinationBanks(bankSelect, bankValue || '');
        }

        // Apply field-level editability per config and role (considering if existing payment)
        const canEditMethod = this.canUserEditFieldClient('payment_method', isExistingPayment);
        const canEditCard = this.canUserEditFieldClient('payment_card_last4', isExistingPayment);
        const canEditBank = this.canUserEditFieldClient('destination_bank', isExistingPayment);
        const disableSelect = (el) => { if (el) { el.disabled = true; el.classList.add('bg-gray-900', 'cursor-not-allowed', 'opacity-70'); } };
        const disableInput = (el) => { if (el) { el.readOnly = true; el.classList.add('bg-gray-900', 'cursor-not-allowed', 'opacity-70'); } };

        if (!canEditMethod && methodSelect) disableSelect(methodSelect);
        if (!canEditCard && cardInput) disableInput(cardInput);
        if (!canEditBank && bankSelect) disableSelect(bankSelect);

        if (!readonly) {
            const methodSelect = row.querySelector('select[name="payment_method[]"]');
            const cardWrap = row.querySelector('.payment-card-last4');
            const bankWrap = row.querySelector('.payment-bank');
            const bankSelect = row.querySelector('select[name="destination_bank[]"]');
            
            const togglePaymentFields = () => {
                const method = methodSelect?.value || '';
                const meta = this.getLookupMeta('payment_methods', method);
                const isCardMethod = typeof meta.requires_card_last4 === 'boolean'
                    ? meta.requires_card_last4
                    : ['card_to_card', 'card_reader', 'card_to_card_lookup'].includes(method);
                
                // Toggle card last4 field
                if (cardWrap) {
                    cardWrap.classList.toggle('hidden', !isCardMethod);
                }
                if (cardInput) {
                    cardInput.required = isCardMethod;
                    if (!isCardMethod) {
                        cardInput.value = '';
                    }
                }
                
                // Toggle bank field (show for card_to_card)
                const isBankRequired = typeof meta.requires_bank_destination === 'boolean'
                    ? meta.requires_bank_destination
                    : ['card_to_card', 'card_to_card_lookup'].includes(method);
                if (bankWrap) {
                    bankWrap.classList.toggle('hidden', !isBankRequired);
                }
                if (bankSelect) {
                    bankSelect.required = isBankRequired;
                    if (!isBankRequired) {
                        bankSelect.value = '';
                    }
                }
            };
            
            if (methodSelect) {
                methodSelect.addEventListener('change', togglePaymentFields);
                togglePaymentFields();
            }
        }
        
        // Initialize Persian date picker for this row (only if not readonly)
        if (!readonly) {
            setTimeout(() => {
                const dateInput = row.querySelector('.payment-date-jalali');
                if (dateInput) {
                    const initDatePicker = () => {
                        // Check if jQuery and persianDatepicker are loaded
                        if (typeof window.$ !== 'undefined' && typeof window.$.fn !== 'undefined' && typeof window.$.fn.persianDatepicker !== 'undefined') {
                            try {
                                // Destroy existing datepicker if any
                                if (window.$(dateInput).data('persianDatepicker')) {
                                    window.$(dateInput).persianDatepicker('destroy');
                                }
                                
                                // Initialize persian datepicker (matching index.php configuration)
                                // Using initialValue: false and observer: true so it reads from input value
                                window.$(dateInput).persianDatepicker({
                                    observer: true,
                                    format: 'YYYY/MM/DD',
                                    calendarType: 'persian',
                                    initialValue: false,
                                    autoClose: true
                                });
                                
                                console.log('Persian datepicker initialized for:', dateInput);
                            } catch(e) {
                                console.error('Error initializing date picker:', e);
                            }
                        } else {
                            // Retry after a short delay if libraries not loaded yet
                            console.log('Waiting for jQuery/persianDatepicker to load...');
                            setTimeout(initDatePicker, 200);
                        }
                    };
                    initDatePicker();
                } else {
                    console.error('Date input not found in row');
                }
            }, 200);
        }
    },
    
    removePaymentRow(rowId) {
        const row = document.getElementById(rowId);
        if (row) {
            row.remove();
        }
    },
    
    enableDealFormFields() {
        // Remove readonly/disabled state from all form fields
        const readonlyClasses = ['bg-gray-900', 'cursor-not-allowed'];
        const fields = [
            'deal-title',
            'deal-service-cost',
            'deal-price-list-code',
            'deal-payable-amount',
            'deal-has-discount',
            'deal-discount-type',
            'deal-discount-occasion',
            'deal-discount-occasion-custom',
            'deal-discount-amount',
            'deal-payment-conditions'
        ];
        
        fields.forEach(fieldId => {
            const field = document.getElementById(fieldId);
            if (field) {
                if (field.tagName === 'SELECT') {
                    field.disabled = false;
                } else {
                    field.readOnly = false;
                }
                // Remove each class separately
                readonlyClasses.forEach(className => {
                    field.classList.remove(className);
                });
            }
        });
    },
    
    loadPayments(payments, readonly = null, addDefault = false) {
        const container = document.getElementById('payment-rows');
        if (!container) return;
        container.innerHTML = '';
        const hasData = (p) => {
            if (!p) return false;
            const amountVal = parseMoneyValue(p.amount || 0);
            const hasMethod = !!(p.method && String(p.method).trim() !== '');
            const hasDesc = !!(p.description && String(p.description).trim() !== '');
            const hasDate = !!(p.date && String(p.date).trim() !== '') || !!(p.date_persian && String(p.date_persian).trim() !== '');
            return amountVal > 0 || hasMethod || hasDesc || (hasDate && amountVal > 0);
        };
        const normalized = Array.isArray(payments) ? payments.filter(hasData) : [];
        this.state.lastPaymentsCache = JSON.parse(JSON.stringify(normalized));
        
        if (normalized && normalized.length > 0) {
            normalized.forEach(payment => {
                // Pass null to addPaymentRow so it can decide readonly based on payment data, dealIsWon, and paymentsLocked
                this.addPaymentRow(payment, readonly);
            });
        } else if (addDefault) {
            // Add one empty row by default (always editable) when explicitly allowed
            this.addPaymentRow(null, false);
        }
    },
    
    // TOGGLE DISCOUNT FIELDS
    toggleDiscountFields() {
        const hasDiscount = document.getElementById('deal-has-discount').value === '1';
        const discountFields = document.getElementById('discount-fields');
        if (hasDiscount) {
            discountFields.classList.remove('hidden');
        } else {
            discountFields.classList.add('hidden');
            const type = document.getElementById('deal-discount-type');
            const occasion = document.getElementById('deal-discount-occasion');
            const custom = document.getElementById('deal-discount-occasion-custom');
            const amount = document.getElementById('deal-discount-amount');
            const refundAmount = document.getElementById('deal-refund-amount');
            const refundDesc = document.getElementById('deal-refund-description');
            if (type) type.value = '';
            if (occasion) occasion.value = '';
            if (custom) {
                custom.value = '';
                custom.classList.add('hidden');
                custom.required = false;
            }
            if (amount) amount.value = '';
            if (refundAmount) refundAmount.value = '';
            if (refundDesc) refundDesc.value = '';
        }
        this.toggleDiscountOccasion();
    },
    
    // TOGGLE DISCOUNT OCCASION
    toggleDiscountOccasion() {
        const discountType = document.getElementById('deal-discount-type').value;
        const occasionField = document.getElementById('discount-occasion-field');
        const refundFields = document.getElementById('refund-fields');
        const refundAmount = document.getElementById('deal-refund-amount');
        const refundDesc = document.getElementById('deal-refund-description');
        const discountAmount = document.getElementById('deal-discount-amount');
        if (discountType === 'occasion') {
            occasionField.classList.remove('hidden');
        } else {
            occasionField.classList.add('hidden');
        }
        const isRefund = discountType === 'refund';
        if (refundFields) refundFields.classList.toggle('hidden', !isRefund);
        if (refundAmount) refundAmount.required = isRefund;
        if (refundDesc) refundDesc.required = isRefund;
        if (discountAmount) discountAmount.classList.toggle('hidden', isRefund);
        if (!isRefund) {
            if (refundAmount) refundAmount.value = '';
            if (refundDesc) refundDesc.value = '';
        }
        this.toggleCustomOccasion();
    },
    
    // TOGGLE CUSTOM OCCASION
    toggleCustomOccasion() {
        const discountType = document.getElementById('deal-discount-type')?.value || '';
        if (discountType === 'refund') {
            const customField = document.getElementById('deal-discount-occasion-custom');
            if (customField) {
                customField.classList.add('hidden');
                customField.required = false;
                customField.value = '';
            }
            const occasionSelect = document.getElementById('deal-discount-occasion');
            if (occasionSelect) {
                occasionSelect.value = '';
            }
            return;
        }
        const occasion = document.getElementById('deal-discount-occasion').value;
        const customField = document.getElementById('deal-discount-occasion-custom');
        if (occasion === 'other') {
            customField.classList.remove('hidden');
            customField.required = true;
        } else {
            customField.classList.add('hidden');
            customField.required = false;
        }
    },
    
    // USERS
    async loadUsers() {
        const res = await this.req('get_users_list');
        if(res.status === 'success') {
            window.didarUsers = res.didar_users || [];
            this.renderUsers(res.users || [], res.crm_owner || {});
        }
    },
    
    async saveCrmOwner() {
        if (!this.isAdmin()) return;
        const select = document.getElementById('crm-owner-select');
        if (!select) return;
        const crmOwnerId = select.value;
        if (!crmOwnerId) {
            Swal.fire('خطا', 'لطفاً کارشناس CRM را انتخاب کنید', 'error');
            return;
        }
        Swal.showLoading();
        const res = await this.req('save_crm_settings', {crm_owner_id: crmOwnerId});
        Swal.close();
        if (res.status === 'success') {
            Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'ذخیره شد', showConfirmButton:false, timer:2000});
            this.loadUsers();
        } else {
            Swal.fire('خطا', res.message || 'خطا در ذخیره تنظیمات', 'error');
        }
    },

    async runMigration() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر می‌تواند migration را اجرا کند', 'error');
            return;
        }
        
        Swal.fire({
            title: 'در حال اجرای Migration...',
            text: 'لطفاً صبر کنید',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('run_migration', {});
            Swal.close();
            if (res.status === 'success') {
                Swal.fire({
                    icon: 'success',
                    title: 'موفق',
                    text: res.message || 'Migration با موفقیت اجرا شد',
                    timer: 3000
                });
                // Reload users list
                this.loadUsers();
            } else {
                Swal.fire('خطا', res.message || 'خطا در اجرای migration', 'error');
            }
        } catch (error) {
            console.error('Error in runMigration:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در اجرای migration: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    renderUsers(users, crmOwner = {}) {
        const tbody = document.getElementById('users-table');
        if(!tbody) return;
        tbody.innerHTML = '';
        const isAdmin = this.isAdmin();
        users.forEach(u => {
            tbody.innerHTML += `<tr>
                <td class="p-3">${u.display_name || (u.first_name + ' ' + u.last_name)}</td>
                <td class="p-3 font-mono text-xs">${u.username || u.display_name || '-'}</td>
                <td class="p-3 text-xs"><span class="px-2 py-0.5 bg-slate-800 rounded">${u.role === 'admin' ? 'مدیر' : (u.role === 'crm_specialist' ? 'کارشناس CRM' : 'کارشناس')}</span></td>
                <td class="p-3 font-mono text-xs text-gray-500">${u.didar_user_id || '-'}</td>
                <td class="p-3">
                    <div class="flex gap-2">
                        ${isAdmin ? `<button onclick="app.openEditUserRoleModal(${u.id}, '${u.role}', '${this.escapeHtml(u.display_name || (u.first_name + ' ' + u.last_name))}')" class="text-xs text-blue-500 hover:text-blue-400 px-2 py-1 rounded bg-blue-500/10">
                            <i class="fa-solid fa-user-gear ml-1"></i> ویرایش نقش
                        </button>` : ''}
                        <button onclick="app.openChangePasswordModal(${u.id})" class="text-xs text-yellow-500 hover:text-yellow-400 px-2 py-1 rounded bg-yellow-500/10">
                            <i class="fa-solid fa-key ml-1"></i> تغییر رمز
                        </button>
                    </div>
                </td>
            </tr>`;
        });
        
        const crmSelect = document.getElementById('crm-owner-select');
        const crmLabel = document.getElementById('crm-owner-current');
        if (crmSelect) {
            crmSelect.innerHTML = '<option value="">-- انتخاب کاربر از دیدار --</option>';
            (window.didarUsers || []).forEach(u => {
                const opt = document.createElement('option');
                const value = u.Id || u.UserId;
                opt.value = value || '';
                opt.textContent = (u.DisplayName || ((u.FirstName || '') + ' ' + (u.LastName || ''))) + (u.UserName ? ` (${u.UserName})` : '');
                crmSelect.appendChild(opt);
            });
            if (crmOwner?.id) {
                crmSelect.value = crmOwner.id;
            }
        }
        if (crmLabel) {
            crmLabel.textContent = crmOwner?.name ? `کاربر فعلی: ${crmOwner.name}` : 'کاربر فعلی: انتخاب نشده';
        }
        
        this.state.crmOwnerId = crmOwner?.id || null;
        this.state.crmOwnerName = crmOwner?.name || '';
    },

    async loadLookupAdmin(force = false) {
        if (!this.isAdmin()) return;
        const select = document.getElementById('lookup-group-select');
        if (!select) return;

        if (!force && this.state.lookupGroups && this.state.lookupGroups.length) {
            if (this.state.lookupSelectedGroup) {
                select.value = this.state.lookupSelectedGroup;
                await this.onLookupGroupChange();
            }
            return;
        }

        const res = await this.req('get_lookup_groups', { include_inactive: 1 });
        if (res.status !== 'success') {
            Swal.fire('خطا', res.message || 'خطا در بارگذاری گروه‌ها', 'error');
            return;
        }
        this.state.lookupGroups = res.groups || [];
        
        select.innerHTML = '<option value="">-- انتخاب کنید --</option>';
        
        // Add Lookup Groups
        const groupGroup = document.createElement('optgroup');
        groupGroup.label = 'گروه‌های لوکاپ (دارای لیست)';
        this.state.lookupGroups.forEach(group => {
            const opt = document.createElement('option');
            opt.value = group.code;
            opt.textContent = group.title || group.code;
            groupGroup.appendChild(opt);
        });
        select.appendChild(groupGroup);

        // Add System Fields
        const systemGroup = document.createElement('optgroup');
        systemGroup.label = 'فیلدهای سیستمی';
        [
            { code: 'deal_title', title: 'عنوان معامله' },
            { code: 'payment_method', title: 'نحوه پرداخت' },
            { code: 'payment_amount', title: 'مبلغ پرداخت' },
            { code: 'payment_card_last4', title: '۴ رقم آخر کارت' },
            { code: 'destination_bank', title: 'بانک مقصد پرداخت' },
            { code: 'refund_amount', title: 'مبلغ مرجوعی' },
            { code: 'service_cost', title: 'مبلغ خدمات' },
            { code: 'payable_amount', title: 'مبلغ قابل پرداخت' },
            { code: 'person_name', title: 'نام لید' },
            { code: 'person_mobile', title: 'شماره موبایل لید' }
        ].forEach(field => {
            const opt = document.createElement('option');
            opt.value = field.code;
            opt.textContent = field.title;
            systemGroup.appendChild(opt);
        });
        select.appendChild(systemGroup);

        if (!this.state.lookupSelectedGroup && this.state.lookupGroups.length) {
            this.state.lookupSelectedGroup = this.state.lookupGroups[0].code;
        }
        if (this.state.lookupSelectedGroup) {
            select.value = this.state.lookupSelectedGroup;
            await this.onLookupGroupChange();
        }
    },

    async ensureFieldConfigsLoaded() {
        if (!this.state.fieldConfigurations || !this.state.fieldConfigurations.length) {
            await this.loadFieldConfigurationsPublic('deal');
        }
    },

    async loadFieldConfigurationsPublic(entityType = 'deal') {
        try {
            const res = await this.req('get_field_configurations_public', { entity_type: entityType });
            if (res.status === 'success') {
                this.state.fieldConfigurations = res.configurations || [];
            }
        } catch (error) {
            console.error('Error loading public field configurations:', error);
        }
    },

    async onLookupGroupChange() {
        const select = document.getElementById('lookup-group-select');
        const groupCode = select?.value || '';
        this.state.lookupSelectedGroup = groupCode;
        
        const addBtn = document.getElementById('lookup-add-item-btn');
        const itemsSection = document.getElementById('lookup-items-section');
        const quickSettings = document.getElementById('field-quick-settings');
        
        if (!groupCode) {
            document.getElementById('lookup-items-table').innerHTML = '';
            if (addBtn) addBtn.disabled = true;
            if (itemsSection) itemsSection.classList.add('hidden');
            if (quickSettings) quickSettings.classList.add('hidden');
            return;
        }

        // Show items section if it's a real lookup group
        const isLookupGroup = this.state.lookupGroups.some(g => g.code === groupCode);
        
        // Items section visibility
        if (itemsSection) itemsSection.classList.remove('hidden');
        
        if (isLookupGroup) {
            if (addBtn) addBtn.disabled = false;
            await this.loadLookupItems(groupCode);
        } else {
            // For system fields, clear items but keep settings
            document.getElementById('lookup-items-table').innerHTML = '<tr><td colspan="8" class="p-8 text-center text-gray-500 italic">این یک فیلد سیستمی است و گزینه‌های لیست ندارد. فقط تنظیمات بالا اعمال می‌شود.</td></tr>';
            if (addBtn) addBtn.disabled = true;
        }

        // Load settings for this groupCode/fieldName
        await this.loadFieldConfigurationForAdmin(groupCode);
    },

    async loadFieldConfigurationForAdmin(fieldName) {
        const advSettings = document.getElementById('field-advanced-settings');
        
        const setVal = (id, val, isCheck = false) => {
            const el = document.getElementById(id);
            if (!el) return;
            if (isCheck) el.checked = !!val;
            else el.value = val || '';
        };
        
        try {
            const res = await this.req('get_field_configuration', { field_name: fieldName });
            const config = (res.status === 'success' && res.configuration) ? res.configuration : null;
            
            // Populate all fields safely
            setVal('fc-field-label', config?.field_label);
            setVal('fc-is-required', config?.is_required === 1, true);
            setVal('fc-is-editable', config?.is_editable !== 0, true); // Default to true
            setVal('fc-is-editable-after-create', config?.is_editable_after_create !== 0, true); // Default to true
            setVal('fc-is-active', config?.is_active !== 0, true);     // Default to true
            setVal('fc-help-text', config?.help_text);
            setVal('fc-default-value', config?.default_value);
            setVal('fc-group-name', config?.group_name);
            setVal('fc-validation-pattern', config?.validation_pattern);
            
            // Parse and populate conditional logic UI
            const conditionalLogic = config?.conditional_logic;
            if (conditionalLogic && typeof conditionalLogic === 'object') {
                setVal('fc-conditional-field', conditionalLogic.field);
                setVal('fc-conditional-operator', conditionalLogic.operator || '=');
                setVal('fc-conditional-value', conditionalLogic.value);
                setVal('fc-conditional-logic', JSON.stringify(conditionalLogic));
            } else if (conditionalLogic && typeof conditionalLogic === 'string') {
                try {
                    const parsed = JSON.parse(conditionalLogic);
                    setVal('fc-conditional-field', parsed.field);
                    setVal('fc-conditional-operator', parsed.operator || '=');
                    setVal('fc-conditional-value', parsed.value);
                } catch (e) {}
                setVal('fc-conditional-logic', conditionalLogic);
            } else {
                setVal('fc-conditional-field', '');
                setVal('fc-conditional-operator', '=');
                setVal('fc-conditional-value', '');
                setVal('fc-conditional-logic', '');
            }

            // Set edit roles checkboxes
            const editRoles = config?.edit_allow_roles || [];
            document.querySelectorAll('.role-checkbox').forEach(cb => {
                cb.checked = editRoles.includes(cb.value);
            });

            // Set delete roles checkboxes
            const deleteRoles = config?.delete_allow_roles || [];
            document.querySelectorAll('.delete-role-checkbox').forEach(cb => {
                cb.checked = deleteRoles.includes(cb.value);
            });

            // Set view roles checkboxes
            const viewRoles = config?.view_allow_roles || [];
            document.querySelectorAll('.view-role-checkbox').forEach(cb => {
                cb.checked = viewRoles.includes(cb.value);
            });

            // Show the advanced settings panel
            if (advSettings) advSettings.classList.remove('hidden');
        } catch (error) {
            console.error('Error loading field config:', error);
        }
    },

    async saveFieldConfigurationMerged() {
        const fieldName = document.getElementById('lookup-group-select').value;
        if (!fieldName) return;

        // Collect edit roles
        const selectedRoles = Array.from(
            document.querySelectorAll('.role-checkbox:checked')
        ).map(cb => cb.value);

        // Collect delete roles
        const selectedDeleteRoles = Array.from(
            document.querySelectorAll('.delete-role-checkbox:checked')
        ).map(cb => cb.value);

        // Collect view roles
        const selectedViewRoles = Array.from(
            document.querySelectorAll('.view-role-checkbox:checked')
        ).map(cb => cb.value);

        const isLookupGroup = (this.state.lookupGroups || []).some(g => g.code === fieldName);

        const data = {
            field_name: fieldName,
            field_label: document.getElementById('fc-field-label')?.value || '',
            field_type: isLookupGroup ? 'select' : 'text',
            lookup_group_code: isLookupGroup ? fieldName : null,
            is_required: document.getElementById('fc-is-required')?.checked ? 1 : 0,
            is_editable: document.getElementById('fc-is-editable')?.checked ? 1 : 0,
            is_editable_after_create: document.getElementById('fc-is-editable-after-create')?.checked ? 1 : 0,
            is_active: document.getElementById('fc-is-active')?.checked ? 1 : 0,
            edit_allow_roles: JSON.stringify(selectedRoles),
            delete_allow_roles: JSON.stringify(selectedDeleteRoles),
            view_allow_roles: JSON.stringify(selectedViewRoles),
            help_text: document.getElementById('fc-help-text')?.value || '',
            default_value: document.getElementById('fc-default-value')?.value || '',
            group_name: document.getElementById('fc-group-name')?.value || '',
            validation_pattern: document.getElementById('fc-validation-pattern')?.value || '',
            conditional_logic: document.getElementById('fc-conditional-logic')?.value || ''
        };

        try {
            const res = await this.req('save_field_configuration', data);
            if (res.status === 'success') {
                // Toast success
                const Toast = Swal.mixin({
                    toast: true,
                    position: 'bottom-end',
                    showConfirmButton: false,
                    timer: 2000,
                    timerProgressBar: true
                });
                Toast.fire({
                    icon: 'success',
                    title: 'تنظیمات ذخیره شد'
                });
                
                // Update current label in UI
                const currentLabel = document.getElementById('fc-current-label');
                if (currentLabel && data.field_label) {
                    currentLabel.textContent = data.field_label;
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در ذخیره‌سازی', 'error');
            }
        } catch (error) {
            Swal.fire('خطا', 'خطا در ارتباط با سرور', 'error');
        }
    },

    // Apply conditional logic from UI fields
    applyConditionalLogicFromUI() {
        const field = document.getElementById('fc-conditional-field')?.value?.trim();
        const operator = document.getElementById('fc-conditional-operator')?.value || '=';
        const value = document.getElementById('fc-conditional-value')?.value?.trim();

        if (!field) {
            // Clear conditional logic
            document.getElementById('fc-conditional-logic').value = '';
            this.saveFieldConfigurationMerged();
            return;
        }

        const logic = JSON.stringify({
            field: field,
            operator: operator,
            value: value
        });

        document.getElementById('fc-conditional-logic').value = logic;
        this.saveFieldConfigurationMerged();

        Swal.fire({
            toast: true,
            position: 'bottom-end',
            icon: 'success',
            title: 'منطق شرطی ذخیره شد',
            showConfirmButton: false,
            timer: 2000
        });
    },

    async applyFieldConfigurations(entityType, containerSelector) {
        try {
            const res = await this.req('get_field_configurations', { entity_type: entityType });
            if (res.status !== 'success') return;
            
            const configs = res.configurations || [];
            const container = document.querySelector(containerSelector);
            if (!container) return;

            configs.forEach(config => {
                const fieldName = config.field_name;
                const input = container.querySelector(`[name="${fieldName}"], [name="${fieldName}[]"]`);
                if (!input) return;

                // 1. Tooltips (Help Text)
                if (config.help_text) {
                    const label = input.closest('div')?.querySelector('label');
                    if (label) {
                        label.title = config.help_text;
                        label.classList.add('cursor-help');
                        if (!label.querySelector('.fa-circle-info')) {
                            label.innerHTML += ` <i class="fa-solid fa-circle-info text-[9px] text-gray-500"></i>`;
                        }
                    }
                }

                // 2. Default Values (for new records)
                const dealIdInput = document.getElementById('deal-id');
                const isNew = !dealIdInput || !dealIdInput.value;
                if (isNew && config.default_value && (!input.value || input.value === '0')) {
                    input.value = config.default_value;
                    if (input.tagName === 'SELECT') input.dispatchEvent(new Event('change'));
                }

                // 3. Conditional Logic (support both legacy and new schema)
                if (config.conditional_logic) {
                    try {
                        const logic = typeof config.conditional_logic === 'string' ? JSON.parse(config.conditional_logic) : config.conditional_logic;
                        const wrapper = input.closest('.grid > div') || input.closest('div');
                        const evaluate = (actualValue, operator, expected) => {
                            switch (operator) {
                                case '=':
                                case '==':
                                    return actualValue == expected;
                                case '!=':
                                case '<>':
                                    return actualValue != expected;
                                case 'in':
                                    return Array.isArray(expected) && expected.includes(actualValue);
                                case 'not_in':
                                    return Array.isArray(expected) && !expected.includes(actualValue);
                                case 'empty':
                                    return !actualValue;
                                case 'not_empty':
                                    return !!actualValue;
                                case 'contains':
                                    return typeof actualValue === 'string' && actualValue.includes(expected);
                                default:
                                    return true;
                            }
                        };

                        // New schema: {field, operator, value}
                        if (logic.field) {
                            const dependInput = container.querySelector(`[name="${logic.field}"], [name="${logic.field}[]"]`);
                            if (dependInput) {
                                const checkVisibility = () => {
                                    const val = dependInput.value;
                                    const shouldShow = evaluate(val, logic.operator || '=', logic.value);
                                    if (wrapper) wrapper.classList.toggle('hidden', !shouldShow);
                                    if (input && config.is_required === 1) input.required = shouldShow;
                                };
                                dependInput.addEventListener('change', checkVisibility);
                                checkVisibility();
                            }
                        }

                        // Legacy schema: {depends_on, show_if}
                        if (logic.depends_on && logic.show_if) {
                            const dependInput = container.querySelector(`[name="${logic.depends_on}"], [name="${logic.depends_on}[]"]`);
                            if (dependInput) {
                                const checkVisibility = () => {
                                    const val = dependInput.value;
                                    const shouldShow = Array.isArray(logic.show_if) ? logic.show_if.includes(val) : val === logic.show_if;
                                    if (wrapper) wrapper.classList.toggle('hidden', !shouldShow);
                                    if (input && config.is_required === 1) input.required = shouldShow;
                                };
                                dependInput.addEventListener('change', checkVisibility);
                                checkVisibility();
                            }
                        }
                    } catch (e) {}
                }

                // 4. View permissions
                const userRole = (typeof window.APP_USER !== 'undefined' ? window.APP_USER.role : 'agent');
                const viewRoles = config.view_allow_roles || [];
                if (viewRoles.length && !viewRoles.includes(userRole)) {
                    const wrapper = input.closest('.grid > div') || input.closest('div');
                    if (wrapper) wrapper.classList.add('hidden');
                    input.readOnly = true;
                    if (input.tagName === 'SELECT') input.disabled = true;
                    input.required = false;
                }

                // 5. Role-based Access (Frontend UI)
                const allowedRoles = config.edit_allow_roles || [];
                const editable = config.is_editable !== 0;
                if (allowedRoles.length && !allowedRoles.includes(userRole) || !editable) {
                    input.readOnly = true;
                    if (input.tagName === 'SELECT') input.disabled = true;
                    input.classList.add('bg-gray-900/40', 'cursor-not-allowed', 'opacity-70');
                }
            });
        } catch (e) { console.error('Error applying field configs:', e); }
    },

    /**
     * Check if the current user can edit a field based on configuration
     * @param {string} fieldName - The field name to check
     * @param {boolean} isExisting - Whether the entity already exists (for editable_after_create check)
     * @returns {boolean} - Whether the user can edit this field
     */
    canUserEditFieldClient(fieldName, isExisting = false) {
        const role = (typeof window.APP_USER !== 'undefined' ? window.APP_USER.role : 'agent') || 'agent';
        
        // Define aliases for client-side check
        const aliases = {
            'payment_method': ['payment_methods'],
            'destination_bank': ['payment_destination_banks', 'bank_destination'],
            'payment_card_last4': ['card_last4', 'card_last_4', 'payment_card_last_4'],
            'deal_title': ['title']
        };

        const namesToCheck = [fieldName, ...(aliases[fieldName] || [])];
        
        // Find config among aliases (prioritize alias/group code first)
        let config = null;
        for (const name of namesToCheck) {
            config = (this.state.fieldConfigurations || []).find(c => c.field_name === name);
            if (config) break;
        }
        
        // No config = allow by default
        if (!config) return true;
        
        // Check if active
        if (config.is_active === 0) return false;
        
        // Check if editable at all
        if (config.is_editable === 0) return false;
        
        // Check if editable after create (for existing records)
        if (isExisting && config.is_editable_after_create === 0) {
            return false;
        }

        // Check view role permissions
        const viewRoles = config.view_allow_roles || [];
        if (viewRoles.length && !viewRoles.includes(role)) {
            return false;
        }
        
        // Check role permissions
        const roles = config.edit_allow_roles || [];
        if (!roles.length) return true;
        return roles.includes(role);
    },

    /**
     * Check if the current user can delete based on field configuration
     * @param {string} fieldName - The field name (usually a delete permission key like 'payment_delete')
     * @returns {boolean} - Whether the user can delete
     */
    canUserDeleteFieldClient(fieldName) {
        const role = (typeof window.APP_USER !== 'undefined' ? window.APP_USER.role : 'agent') || 'agent';
        
        // Define aliases for client-side check
        const aliases = {
            'payment_delete': ['payment_methods', 'payment_method'],
        };

        const namesToCheck = [fieldName, ...(aliases[fieldName] || [])];
        
        let config = null;
        for (const name of namesToCheck) {
            config = (this.state.fieldConfigurations || []).find(c => c.field_name === name);
            if (config) break;
        }
        
        // No config = allow by default
        if (!config) return true;
        
        // Check if active
        if (config.is_active === 0) return false;
        
        // Check delete role permissions
        const deleteRoles = config.delete_allow_roles || [];
        if (!deleteRoles.length) return true;
        return deleteRoles.includes(role);
    },

    async loadLookupItems(groupCode) {
        if (!groupCode) return;
        const res = await this.req('get_lookup_items', { group: groupCode });
        if (res.status !== 'success') {
            Swal.fire('خطا', res.message || 'خطا در بارگذاری گزینه‌ها', 'error');
            return;
        }
        this.state.lookupItems = res.items || [];
        this.renderLookupItemsTable(groupCode, this.state.lookupItems);
    },

    renderLookupItemsTable(groupCode, items) {
        const tbody = document.getElementById('lookup-items-table');
        if (!tbody) return;
        const parentOptions = groupCode === 'lead_source_details' ? this.getLookupOptions('lead_sources') : [];
        tbody.innerHTML = (items || []).map(item => {
            const id = item.id || '';
            const title = this.escapeHtml(item.title || '');
            const value = this.escapeHtml(item.value || '');
            const code = this.escapeHtml(item.code || '');
            const metaJson = this.escapeHtml(item.meta_json || '');
            const sortOrder = item.sort_order ?? 0;
            const isActive = String(item.is_active ?? 1) === '1' ? 'checked' : '';
            const parentValue = item.parent_value || '';
            const parentSelect = parentOptions.length ? `
                <select class="input bg-gray-800 text-xs" name="parent_value">
                    <option value="">-</option>
                    ${parentOptions.map(opt => `<option value="${this.escapeHtml(opt.value)}" ${opt.value === parentValue ? 'selected' : ''}>${this.escapeHtml(opt.label || opt.value)}</option>`).join('')}
                </select>
            ` : '<span class="text-xs text-gray-500">-</span>';

            return `
                <tr data-id="${id}" data-group="${this.escapeHtml(groupCode)}">
                    <td class="p-3"><input class="input bg-gray-800 text-xs" name="title" value="${title}"></td>
                    <td class="p-3"><input class="input bg-gray-800 text-xs" name="value" value="${value}"></td>
                    <td class="p-3"><input class="input bg-gray-800 text-xs" name="code" value="${code}"></td>
                    <td class="p-3"><input class="input bg-gray-800 text-[10px] font-mono" name="meta_json" value="${metaJson}"></td>
                    <td class="p-3">${parentSelect}</td>
                    <td class="p-3"><input class="input bg-gray-800 text-xs" name="sort_order" value="${sortOrder}" type="number"></td>
                    <td class="p-3 text-center"><input type="checkbox" name="is_active" class="accent-yellow-500" ${isActive}></td>
                    <td class="p-3">
                        <div class="flex gap-2">
                            <button type="button" class="px-3 py-1 rounded bg-blue-600 hover:bg-blue-500 text-xs" onclick="app.saveLookupRow(this)">Save</button>
                            <button type="button" class="px-3 py-1 rounded bg-red-600 hover:bg-red-500 text-xs" onclick="app.deleteLookupRow(this)">Delete</button>
                        </div>
                    </td>
                </tr>
            `;
        }).join('');
    },

    addLookupRow() {
        const groupCode = this.state.lookupSelectedGroup;
        if (!groupCode) {
            Swal.fire('خطا', 'ابتدا یک گروه را انتخاب کنید', 'error');
            return;
        }
        const tbody = document.getElementById('lookup-items-table');
        if (!tbody) return;
        const parentOptions = groupCode === 'lead_source_details' ? this.getLookupOptions('lead_sources') : [];
        const parentSelect = parentOptions.length ? `
            <select class="input bg-gray-800 text-xs" name="parent_value">
                <option value="">-</option>
                ${parentOptions.map(opt => `<option value="${this.escapeHtml(opt.value)}">${this.escapeHtml(opt.label || opt.value)}</option>`).join('')}
            </select>
        ` : '<span class="text-xs text-gray-500">-</span>';
        const row = document.createElement('tr');
        row.dataset.id = '';
        row.dataset.group = groupCode;
        row.innerHTML = `
            <td class="p-3"><input class="input bg-gray-800 text-xs" name="title" value=""></td>
            <td class="p-3"><input class="input bg-gray-800 text-xs" name="value" value=""></td>
            <td class="p-3"><input class="input bg-gray-800 text-xs" name="code" value=""></td>
            <td class="p-3"><input class="input bg-gray-800 text-[10px] font-mono" name="meta_json" value=""></td>
            <td class="p-3">${parentSelect}</td>
            <td class="p-3"><input class="input bg-gray-800 text-xs" name="sort_order" value="0" type="number"></td>
            <td class="p-3 text-center"><input type="checkbox" name="is_active" class="accent-yellow-500" checked></td>
            <td class="p-3">
                <div class="flex gap-2">
                    <button type="button" class="px-3 py-1 rounded bg-blue-600 hover:bg-blue-500 text-xs" onclick="app.saveLookupRow(this)">Save</button>
                    <button type="button" class="px-3 py-1 rounded bg-red-600 hover:bg-red-500 text-xs" onclick="app.deleteLookupRow(this)">Delete</button>
                </div>
            </td>
        `;
        tbody.appendChild(row);
    },

    async saveLookupRow(button) {
        const row = button.closest('tr');
        if (!row) return;
        const groupCode = row.dataset.group;
        const id = row.dataset.id || '';
        const title = row.querySelector('[name="title"]')?.value?.trim() || '';
        const value = row.querySelector('[name="value"]')?.value?.trim() || '';
        const code = row.querySelector('[name="code"]')?.value?.trim() || '';
        const metaJson = row.querySelector('[name="meta_json"]')?.value?.trim() || '';
        const sortOrder = row.querySelector('[name="sort_order"]')?.value || '0';
        const isActive = row.querySelector('[name="is_active"]')?.checked ? '1' : '0';
        const parentValue = row.querySelector('[name="parent_value"]')?.value || '';

        if (!title) {
            Swal.fire('خطا', 'عنوان باید وارد شود', 'error');
            return;
        }
        if (metaJson) {
            try {
                JSON.parse(metaJson);
            } catch (e) {
                Swal.fire('خطا', 'فرمت JSON معتبر نیست', 'error');
                return;
            }
        }

        const res = await this.req('save_lookup_item', {
            id,
            group_code: groupCode,
            title,
            value,
            code,
            meta_json: metaJson,
            sort_order: sortOrder,
            is_active: isActive,
            parent_value: parentValue
        });
        if (res.status === 'success') {
            if (res.id) {
                row.dataset.id = res.id;
            }
            Swal.fire('انجام شد', 'ذخیره شد', 'success');
            await this.loadLookups();
            await this.loadLookupItems(this.state.lookupSelectedGroup);
        } else {
            Swal.fire('خطا', res.message || 'خطا در ذخیره', 'error');
        }
    },

    async deleteLookupRow(button) {
        const row = button.closest('tr');
        if (!row) return;
        const id = row.dataset.id || '';
        if (!id) {
            row.remove();
            return;
        }
        const result = await Swal.fire({
            title: 'حذف گزینه؟',
            text: 'این گزینه حذف خواهد شد',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'حذف',
            cancelButtonText: 'انصراف'
        });
        if (!result.isConfirmed) return;
        const res = await this.req('delete_lookup_item', { id });
        if (res.status === 'success') {
            row.remove();
            await this.loadLookups();
            await this.loadLookupItems(this.state.lookupSelectedGroup);
        } else {
            Swal.fire('خطا', res.message || 'خطا در حذف', 'error');
        }
    },
    // UI Helpers
	    setView(id) {
	        document.querySelectorAll('.view-section').forEach(e => e.classList.add('hidden'));
	        const viewEl = document.getElementById('view-'+id);
	        if(viewEl) viewEl.classList.remove('hidden');
        
        document.querySelectorAll('.sidebar-item').forEach(e => e.classList.remove('active', 'text-yellow-500'));
        const menuEl = document.getElementById('menu-'+id);
        if(menuEl) menuEl.classList.add('active', 'text-yellow-500');
        
	        const titles = {
	                    'dashboard': 'داشبورد',
	                    'leads': 'لیدها',
	                    'blataklif-leads': 'لیدهای بلاتکلیف',
	                    'deals': 'معاملات',
	                    'legacy-deals': 'معاملات گذشته دیدار',
	                    'referrals': 'ارجاعات',
	                    'persons': 'اشخاص',
	                    'settings': 'تنظیمات',
	                    'exports': 'خروجی‌ها',
	                    'audit-logs': 'لاگ‌ها',
	                    'phone-logs': 'لاگ تماس‌ها',
	                    'users': 'کاربران',
	                    'lookups': 'Lookup Manager',
	                    'reminders': 'یادآوری‌ها'
	                };
	        const titleEl = document.getElementById('page-title');
	        if(titleEl) titleEl.textContent = titles[id] || id;
	        
	        if(id === 'dashboard') this.loadDashboard();
            if(id === 'reminders') this.loadReminders();
	        if(id === 'leads') {
	            this.loadProducts();
	            this.loadLeads();
	            this.initLeadFiltersDatePickers();
	        }
	        if (id === 'blataklif-leads') {
	            this.loadBlataklifLeads(1);
	        }
	        if(id === 'deals') {
	            this.loadProducts(); // ensure products loaded for deal modal
	            this.loadDeals();
	        }
        if (id === 'legacy-deals') {
            this.loadLegacyDeals();
        }
	        if(id === 'referrals' && this.isAdmin()) this.loadReferrals();
	        if(id === 'persons') this.loadPersons();
	        if(id === 'settings') this.loadSettings();
	        if(id === 'users') this.loadUsers();
            if(id === 'lookups' && this.isAdmin()) this.loadLookupAdmin(true);
	        if(id === 'exports' && this.isAdmin()) this.loadExportCatalog();
	        if(id === 'audit-logs' && this.isAdmin()) this.loadAuditLogs(true);
	        if(id === 'phone-logs' && this.isAdmin()) this.loadPhoneLogs();
	    },

	    deleteLeadFromLeadModal() {
	        const form = document.getElementById('lead-form');
	        const contactId = form?.dataset?.contactId || '';
	        return this.deleteLead(contactId);
	    },

	    auditLogs: { offset: 0, limit: 100, total: 0, rows: [] },

	    async loadAuditLogs(reset = false) {
	        if (!this.isAdmin()) return;
	        if (reset) this.auditLogs.offset = 0;

	        const action = document.getElementById('audit-filter-action')?.value || '';
	        const entityType = document.getElementById('audit-filter-entity')?.value || '';
            const personSearch = document.getElementById('audit-filter-person')?.value || '';
	        const dateFromJ = (document.getElementById('audit-filter-from')?.value || '').trim();
	        const dateToJ = (document.getElementById('audit-filter-to')?.value || '').trim();
	        const dateFromG = dateFromJ ? (persianToGregorian(this.toEnglishDigits(dateFromJ), '00:00') || '') : '';
	        const dateToG = dateToJ ? (persianToGregorian(this.toEnglishDigits(dateToJ), '23:59') || '') : '';
	        const dateFrom = dateFromG ? String(dateFromG).split(' ')[0] : '';
	        const dateTo = dateToG ? String(dateToG).split(' ')[0] : '';

	        try {
	            this.initJalaliDatePickers([
	                document.getElementById('audit-filter-from'),
	                document.getElementById('audit-filter-to')
	            ]);
	            const res = await this.req('get_audit_logs', {
	                action,
	                entity_type: entityType,
                    person_search: personSearch,
	                date_from: dateFrom,
	                date_to: dateTo,
	                limit: this.auditLogs.limit,
	                offset: this.auditLogs.offset
	            });
	            if (res.status !== 'success') {
	                Swal.fire('خطا', res.message || 'خطا در دریافت لاگ‌ها', 'error');
	                return;
	            }
	            this.auditLogs.total = res.logs?.total || 0;
	            this.auditLogs.rows = res.logs?.rows || [];
	            this.renderAuditLogs();
	        } catch (e) {
	            Swal.fire('خطا', 'خطا در دریافت لاگ‌ها', 'error');
	        }
	    },

	    async loadPhoneLogs() {
	        if (!this.isAdmin()) return;
	        
	        // Load phone logs view
	        const viewEl = document.getElementById('view-phone-logs');
	        if (!viewEl) {
	            const content = document.getElementById('content');
	            const section = document.createElement('div');
	            section.id = 'view-phone-logs';
	            section.className = 'view-section hidden flex-1 flex flex-col overflow-hidden';
	            
	            // Load HTML using fetch
	            try {
	                const response = await fetch('?view=phone-logs');
	                const html = await response.text();
	                section.innerHTML = html;
	                content.appendChild(section);
	            } catch (e) {
	                console.error('Failed to load phone logs view:', e);
	                Swal.fire('خطا', 'خطا در بارگذاری صفحه', 'error');
	            }
	        }
	    },

	    auditLogsPrev() {
	        if (this.auditLogs.offset <= 0) return;
	        this.auditLogs.offset = Math.max(0, this.auditLogs.offset - this.auditLogs.limit);
	        this.loadAuditLogs(false);
	    },

	    auditLogsNext() {
	        if (this.auditLogs.offset + this.auditLogs.limit >= this.auditLogs.total) return;
	        this.auditLogs.offset += this.auditLogs.limit;
	        this.loadAuditLogs(false);
	    },

	    renderAuditLogs() {
	        const tbody = document.getElementById('audit-logs-table');
	        const info = document.getElementById('audit-logs-info');
	        if (!tbody) return;

	        const rows = Array.isArray(this.auditLogs.rows) ? this.auditLogs.rows : [];
	        if (!rows.length) {
	            tbody.innerHTML = '<tr><td class="p-3 text-gray-500" colspan="7">لاگی یافت نشد</td></tr>';
	        } else {
	            tbody.innerHTML = rows.map(r => {
	                let snapshot = null;
	                try { snapshot = r.snapshot_json ? JSON.parse(r.snapshot_json) : null; } catch (e) {}

	                const actor = r.actor_name_resolved || r.actor_name || (r.actor_didar_id || '') || (r.actor_user_id ? `user#${r.actor_user_id}` : '');
	                const actorIp = r.actor_ip || '';
	                const action = r.action === 'delete' ? 'حذف' : (r.action === 'owner_change' ? 'تغییر مسئول' : (r.action || ''));
	                const entity = r.entity_type === 'person' ? 'لید' : (r.entity_type === 'deal' ? 'معامله' : (r.entity_type || ''));
                    
                    let relatedPerson = r.related_person_name || '';
                    if (r.related_person_mobile) relatedPerson += ` (${r.related_person_mobile})`;
                    if (!relatedPerson && r.related_person_didar_id) relatedPerson = r.related_person_didar_id;

	                let detail = '';
	                if (r.action === 'delete' && r.entity_type === 'person') {
	                    const p = snapshot?.person;
	                    if (p) detail = `${(p.first_name || '')} ${(p.last_name || '')}`.trim() || (p.mobile_phone || '');
	                } else if (r.action === 'delete' && r.entity_type === 'deal') {
	                    const d = snapshot?.deal;
	                    if (d) detail = (d.title || '') || '';
	                } else if (r.action === 'owner_change') {
	                    const personName = snapshot?.person_name || '';
	                    const oldName = snapshot?.old_owner_name || (snapshot?.old_owner_id || '');
	                    const newName = snapshot?.new_owner_name || (snapshot?.new_owner_id || '');
	                    detail = `${personName ? (personName + ' | ') : ''}از ${oldName || '-'} به ${newName || '-'}`;
	                } else if (snapshot) {
                        // Generic JSON handler
                        if (snapshot.details) {
                             const d = snapshot.details;
                             if (d.text) detail += `Text: ${d.text} `;
                             if (d.element) detail += `[${d.element}] `;
                             if (d.path) detail += `Path: ${d.path} `;
                             if (d.uri) detail += `URI: ${d.uri} `;
                             if (d.input) detail += `Input: ${JSON.stringify(d.input).substring(0, 50)}... `;
                        } else if (snapshot.uri) {
                             detail += `URI: ${snapshot.uri} `;
                             if (snapshot.input) detail += `Input: ${JSON.stringify(snapshot.input).substring(0, 50)}... `;
                        } else {
                             // Clean up keys
                             const keys = Object.keys(snapshot).filter(k => k!=='details');
                             if (keys.length > 0) {
                                 detail = keys.map(k => `${k}: ${typeof snapshot[k] === 'object' ? '...' : snapshot[k]}`).join(', ');
                             } else {
                                 detail = JSON.stringify(snapshot).substring(0, 100);
                             }
                        }
                    }

	                return `
	                    <tr>
	                        <td class="p-3 whitespace-nowrap">${this.escapeHtml(r.created_at || '')}</td>
	                        <td class="p-3 whitespace-nowrap">${this.escapeHtml(actor || '')}</td>
                            <td class="p-3 whitespace-nowrap">${this.escapeHtml(relatedPerson)}</td>
	                        <td class="p-3 whitespace-nowrap">${this.escapeHtml(actorIp || '')}</td>
	                        <td class="p-3 whitespace-nowrap">${this.escapeHtml(action)}</td>
	                        <td class="p-3 whitespace-nowrap">${this.escapeHtml(entity)}</td>
	                        <td class="p-3">${this.escapeHtml(detail || '')}</td>
	                    </tr>
	                `;
	            }).join('');
	        }

	        const from = this.auditLogs.total ? (this.auditLogs.offset + 1) : 0;
	        const to = Math.min(this.auditLogs.offset + this.auditLogs.limit, this.auditLogs.total);
	        if (info) info.textContent = `نمایش ${from} تا ${to} از ${this.auditLogs.total}`;
	    },

	    async loadExportCatalog(force = false) {
	        if (!this.isAdmin()) return;

	        if (this.state.exportCatalog && !force) {
	            this.renderExportUi();
	            return;
	        }

	        const datasetSelect = document.getElementById('export-dataset');
	        if (datasetSelect) datasetSelect.innerHTML = '<option value="">در حال بارگذاری...</option>';

	        try {
	            const res = await this.req('export_catalog', {});
	            if (res.status !== 'success') {
	                Swal.fire('خطا', res.message || 'خطا در دریافت لیست خروجی‌ها', 'error');
	                return;
	            }
	            this.state.exportCatalog = res;
	            this.renderExportUi();
	        } catch (e) {
	            Swal.fire('خطا', 'خطا در دریافت لیست خروجی‌ها', 'error');
	        }
	    },

	    renderExportUi() {
	        const datasetSelect = document.getElementById('export-dataset');
	        const catalog = this.state.exportCatalog;
	        if (!datasetSelect || !catalog || !Array.isArray(catalog.datasets)) return;

	        datasetSelect.innerHTML = '<option value="">-- انتخاب کنید --</option>' + catalog.datasets.map(d => {
	            return `<option value="${this.escapeHtml(d.id)}">${this.escapeHtml(d.title)}</option>`;
	        }).join('');

	        if (!this.state.exportCurrentDatasetId && catalog.datasets.length) {
	            this.state.exportCurrentDatasetId = catalog.datasets[0].id;
	        }
	        if (this.state.exportCurrentDatasetId) {
	            datasetSelect.value = this.state.exportCurrentDatasetId;
	        }

	        this.onExportDatasetChange();
	    },

	    onExportDatasetChange() {
	        const datasetSelect = document.getElementById('export-dataset');
	        const formatSelect = document.getElementById('export-format');
	        const catalog = this.state.exportCatalog;
	        if (!datasetSelect || !catalog || !Array.isArray(catalog.datasets)) return;

	        const datasetId = datasetSelect.value;
	        this.state.exportCurrentDatasetId = datasetId || null;
	        const dataset = catalog.datasets.find(d => d.id === datasetId);

	        if (formatSelect) {
	            const allowed = dataset?.formats || ['csv', 'xlsx'];
	            formatSelect.innerHTML = allowed.map(f => `<option value="${f}">${f.toUpperCase()}</option>`).join('');
	            if (!allowed.includes(formatSelect.value)) {
	                formatSelect.value = allowed[0] || 'csv';
	            }
	        }

	        this.renderExportFilters(dataset);
	        this.renderExportColumns(dataset);
	    },

	    renderExportFilters(dataset) {
	        const container = document.getElementById('export-filters');
	        if (!container) return;
	        if (!dataset) {
	            container.innerHTML = '';
	            return;
	        }

	        const filters = Array.isArray(dataset.filters) ? dataset.filters : [];
	        container.innerHTML = filters.map(f => {
	            const key = this.escapeHtml(f.key || '');
	            const label = this.escapeHtml(f.label || key);
	            const type = f.type || 'text';
	            const placeholder = this.escapeHtml(f.placeholder || '');

	            if (type === 'checkbox') {
	                return `
	                    <label class="flex items-center gap-2 text-sm text-gray-300 bg-gray-900/40 border border-gray-700 rounded-lg px-3 py-2">
	                        <input type="checkbox" class="accent-yellow-500" data-export-filter="${key}" value="1" onchange="app.onExportFilterChange()">
	                        <span>${label}</span>
	                    </label>
	                `;
	            }

	            if (type === 'select') {
	                const opts = Array.isArray(f.options) ? f.options : [];
	                const optionsHtml = opts.map(o => `<option value="${this.escapeHtml(o.value ?? '')}">${this.escapeHtml(o.label ?? '')}</option>`).join('');
	                return `
	                    <div>
	                        <label class="block text-xs text-gray-400 mb-1">${label}</label>
	                        <select class="input bg-gray-800 w-full" data-export-filter="${key}">${optionsHtml}</select>
	                    </div>
	                `;
	            }

	            if (type === 'date_jalali') {
	                return `
	                    <div>
	                        <label class="block text-xs text-gray-400 mb-1">${label}</label>
	                        <input type="text" class="input bg-gray-800 w-full export-date-jalali" data-export-filter="${key}" data-jalali-picker="1" placeholder="${placeholder}">
	                    </div>
	                `;
	            }

	            return `
	                <div>
	                    <label class="block text-xs text-gray-400 mb-1">${label}</label>
	                    <input type="text" class="input bg-gray-800 w-full" data-export-filter="${key}" placeholder="${placeholder}">
	                </div>
	            `;
	        }).join('');

	        this.initJalaliDatePickers(container.querySelectorAll('input[data-jalali-picker="1"]'));
	    },

	    onExportFilterChange() {
	        // Used to re-render columns when include_raw_json toggles
	        const datasetSelect = document.getElementById('export-dataset');
	        const catalog = this.state.exportCatalog;
	        if (!datasetSelect || !catalog || !Array.isArray(catalog.datasets)) return;
	        const dataset = catalog.datasets.find(d => d.id === datasetSelect.value);
	        this.renderExportColumns(dataset);
	    },

	    renderExportColumns(dataset) {
	        const container = document.getElementById('export-columns');
	        if (!container) return;
	        if (!dataset) {
	            container.innerHTML = '';
	            return;
	        }

	        const includeRaw = !!document.querySelector('[data-export-filter="include_raw_json"]')?.checked;
	        const cols = Array.isArray(dataset.columns) ? dataset.columns : [];

	        container.innerHTML = cols.map(c => {
	            const key = c.key || '';
	            const label = c.label || key;
	            const disabled = (key === 'raw_json' && !includeRaw);
	            const checked = !disabled && (key !== 'raw_json'); // default: raw_json unchecked unless enabled by user
	            return `
	                <label class="flex items-center gap-2 text-sm text-gray-300 bg-gray-900/40 border border-gray-700 rounded-lg px-3 py-2 export-col-item" data-export-col-item="1">
	                    <input type="checkbox" class="accent-yellow-500" data-export-column="${this.escapeHtml(key)}" ${checked ? 'checked' : ''} ${disabled ? 'disabled' : ''}>
	                    <span class="${disabled ? 'text-gray-600' : ''}">${this.escapeHtml(label)}</span>
	                </label>
	            `;
	        }).join('');

	        this.filterExportColumns();
	    },

	    filterExportColumns() {
	        const q = (document.getElementById('export-columns-search')?.value || '').trim().toLowerCase();
	        document.querySelectorAll('#export-columns .export-col-item').forEach(el => {
	            const text = (el.textContent || '').toLowerCase();
	            if (!q || text.includes(q)) {
	                el.classList.remove('hidden');
	            } else {
	                el.classList.add('hidden');
	            }
	        });
	    },

	    selectAllExportColumns(on) {
	        document.querySelectorAll('#export-columns input[type="checkbox"][data-export-column]').forEach(cb => {
	            if (cb.disabled) return;
	            cb.checked = !!on;
	        });
	    },

	    collectExportFilters() {
	        const out = {};
	        document.querySelectorAll('[data-export-filter]').forEach(el => {
	            const key = el.getAttribute('data-export-filter');
	            if (!key) return;
	            if (el.type === 'checkbox') {
	                out[key] = el.checked ? '1' : '0';
	            } else {
	                out[key] = (el.value || '').trim();
	            }
	        });

	        // Convert Jalali date filters to Gregorian where backend needs it.
	        // - *_from_jalali -> *_from (00:00)
	        // - *_to_jalali -> *_to (23:59)
	        Object.keys(out).forEach((k) => {
	            if (!k.endsWith('_jalali')) return;
	            const val = (out[k] || '').trim();
	            if (!val) return;

	            const base = k.replace(/_jalali$/, '');
	            if (base.endsWith('_from')) {
	                const g = persianToGregorian(val, '00:00');
	                if (g) out[base] = g;
	            } else if (base.endsWith('_to')) {
	                const g = persianToGregorian(val, '23:59');
	                if (g) out[base] = g;
	            } else {
	                // fallback: store as-is (not converted)
	            }
	        });

	        return out;
	    },

	    initJalaliDatePickers(nodeList) {
	        const inputs = Array.from(nodeList || []).filter(Boolean);
	        if (!inputs.length) return;

	        const initInput = (input) => {
	            if (!input) return;
	            if (typeof $ === 'undefined' || typeof $.fn.persianDatepicker === 'undefined') {
	                setTimeout(() => initInput(input), 200);
	                return;
	            }
	            try {
	                if ($(input).data('persianDatepicker')) {
	                    $(input).persianDatepicker('destroy');
	                }
	            } catch (_) {}
	            $(input).persianDatepicker({
	                format: 'YYYY/MM/DD',
	                initialValue: false,
	                autoClose: true,
	                observer: true
	            });
	        };

	        inputs.forEach(initInput);
	    },

	    collectExportColumns() {
	        const cols = [];
	        document.querySelectorAll('#export-columns input[type="checkbox"][data-export-column]').forEach(cb => {
	            if (cb.checked && !cb.disabled) {
	                cols.push(cb.getAttribute('data-export-column'));
	            }
	        });
	        return cols.filter(Boolean);
	    },

	    submitExport() {
	        if (!this.isAdmin()) {
	            Swal.fire('خطا', 'فقط مدیر می‌تواند خروجی بگیرد', 'error');
	            return;
	        }

	        const dataset = document.getElementById('export-dataset')?.value || '';
	        const format = document.getElementById('export-format')?.value || 'csv';
	        if (!dataset) {
	            Swal.fire('خطا', 'نوع خروجی را انتخاب کنید', 'error');
	            return;
	        }

	        const columns = this.collectExportColumns();
	        const filters = this.collectExportFilters();

	        document.getElementById('export-form-dataset').value = dataset;
	        document.getElementById('export-form-format').value = format;
	        document.getElementById('export-form-columns').value = JSON.stringify(columns);
	        document.getElementById('export-form-filters').value = JSON.stringify(filters);

	        document.getElementById('export-form').submit();
	    },
    
    // رفتن به بخش لیدها با فیلتر مشخص
    goToLeadsWithFilter(stage = 'all') {
        this.setView('leads');
        // تنظیم فیلترها
        const stageFilter = document.getElementById('leads-stage-filter');
        const registerFrom = document.getElementById('leads-register-from');
        const registerTo = document.getElementById('leads-register-to');
        const activityFrom = document.getElementById('leads-activity-from');
        const activityTo = document.getElementById('leads-activity-to');
        const productFilter = document.getElementById('leads-product-filter');
        
        if(stageFilter) stageFilter.value = stage;
        [registerFrom, registerTo, activityFrom, activityTo].forEach(inp => { if (inp) inp.value = ''; });
        if (productFilter) productFilter.value = '';
        
        // بارگذاری لیدها با فیلتر
        this.loadLeads();
    },

    goToDealsWithFilter(pipelineStage = 'all') {
        this.setView('deals');
        const stageFilter = document.getElementById('deals-pipeline-filter');
        if (stageFilter) stageFilter.value = pipelineStage;
        this.loadDeals();
    },
    
    openModal(id) { 
        const el = document.getElementById(id);
        if(el) el.classList.remove('hidden');
        // Prevent background focus when modal open
        this.setPageInert(true);
        
        // Initialize Persian date picker when activity modal opens
        if(id === 'modalActivity') {
            // Populate lead info display
            const infoDisplay = document.getElementById('activity-lead-info-display');
            const nameText = document.getElementById('activity-lead-name-text');
            if (this.state.currentLead && infoDisplay && nameText) {
                const fullName = `${this.state.currentLead.first_name || ''} ${this.state.currentLead.last_name || ''}`.trim() || 'بدون نام';
                nameText.textContent = `${fullName} (${this.state.currentLead.mobile_phone || '-'})`;
                infoDisplay.classList.remove('hidden');
            } else if (infoDisplay) {
                infoDisplay.classList.add('hidden');
            }

            // تنظیم contactId از window.currentContactId
            if (window.currentContactId) {
                const activityContactId = document.getElementById('activity-contact-id');
                if (activityContactId) {
                    activityContactId.value = window.currentContactId;
                }
            }

            // Populate deals dropdown with cached deals for this شخص
            if (this.state.currentDeals && this.state.currentDeals.length) {
                this.populateActivityDealSelect(this.state.currentDeals);
            } else {
                this.populateActivityDealSelect([]);
            }
            this.toggleDealActivityFailure();
            
            // Reset schedule fields
            const isScheduledCheckbox = document.getElementById('activity-is-scheduled');
            const scheduleFields = document.getElementById('activity-schedule-fields');
            if (isScheduledCheckbox) isScheduledCheckbox.checked = false;
            if (scheduleFields) scheduleFields.classList.add('hidden');
            
            // Load activity types for modal
            this.loadActivityTypesForModal();

            // Initialize deal selection handling
            this.handleActivityDealSelection();

            const activityStageSelect = document.getElementById('activity-virtual-stage');
            if (activityStageSelect) {
                activityStageSelect.onchange = (e) => {
                    this.state.activityStagePrev = e.target.value;
                    this.toggleDealActivityFailure();
                    this.handleActivityStageChange(e.target.value);
                };
            }
            
            setTimeout(() => {
                // Initialize activity date picker when modal opens (update every time)
                this.initActivityDatePicker(); // This will update the datepicker with today's date
                this.initPersianDatePicker();
                this.syncActivityDatePickerToTehran();
                
                // افزودن event listener به select های مرحله
                // فقط برای تنظیم فرم، بدون وابستگی به virtual_stage
            }, 100);
        }
        
        // Set contact ID for deal modal
        if(id === 'modalDeal' && window.currentContactId) {
            const dealContactId = document.getElementById('deal-contact-id');
            if (dealContactId) {
                dealContactId.value = window.currentContactId;
            }
        }
    },
    
    initPersianDatePicker() {
        // Initialize Persian date picker for activity modal
        const dateInput = document.getElementById('due-date-jalali');
        if (!dateInput) return;
        
        // Wait for jQuery and persianDatepicker to be available
        if (typeof $ === 'undefined' || typeof $.fn.persianDatepicker === 'undefined') {
            // Retry after a short delay
            setTimeout(() => this.initPersianDatePicker(), 200);
            return;
        }
        
        // Destroy existing picker if any
        try {
            if ($(dateInput).data('persianDatepicker')) {
                $(dateInput).persianDatepicker('destroy');
            }
        } catch(e) {
            console.log('No existing picker to destroy');
        }
        
        // Initialize Persian date picker
        try {
            $(dateInput).persianDatepicker({
                observer: true,
                format: 'YYYY/MM/DD',
                altField: '#due-date-hidden',
                altFormat: 'YYYY-MM-DD',
                calendarType: 'persian',
                timePicker: {
                    enabled: false
                },
                initialValue: false,
                autoClose: true,
                calendar: {
                    persian: {
                        locale: 'fa'
                    }
                }
            });
        } catch(e) {
            console.error('Error initializing Persian date picker:', e);
            // Fallback: make it a simple text input
            dateInput.removeAttribute('readonly');
        }
    },
    
    closeModal(id) { 
        const el = document.getElementById(id);
        if(el) el.classList.add('hidden');
        // Restore background focusability if no other modals are open
        const anyOpen = document.querySelectorAll('.fixed.inset-0:not(.hidden)').length > 0;
        if (!anyOpen) {
            this.setPageInert(false);
        }
    },
    
    setPageInert(isInert) {
        // فقط روی بخش‌های محتوایی صفحه (view-section ها و سایدبار) اعمال می‌کنیم، نه روی مودال‌ها
        const elements = document.querySelectorAll('.view-section, aside');
        elements.forEach(el => {
            if (isInert) {
                el.setAttribute('aria-hidden', 'true');
                el.classList.add('bg-inert-active');
            } else {
                el.removeAttribute('aria-hidden');
                el.classList.remove('bg-inert-active');
            }
        });
        console.log('setPageInert', isInert, 'applied on', elements.length, 'elements');
    },
    
    // NOTIFICATION FUNCTIONS
    async loadNotifications() {
        try {
            const res = await this.req('get_notifications', { unread_only: '1', limit: 20 });
            if (res.status === 'success') {
                this.renderNotifications(res.notifications || []);
                this.updateNotificationBadge(res.unread_count || 0);
            }
        } catch (error) {
            console.error('Error loading notifications:', error);
        }
    },
    
    renderNotifications(notifications) {
        const container = document.getElementById('notifications-list');
        if (!container) return;
        
        if (notifications.length === 0) {
            container.innerHTML = '<div class="text-center text-gray-500 text-sm py-8">اعلانی وجود ندارد</div>';
            return;
        }
        
        container.innerHTML = notifications.map(notif => {
            const isRead = notif.is_read == 1;
            const timeAgo = this.getTimeAgo(notif.created_at);
            const icon = this.getNotificationIcon(notif.type);
            
            return `
                <div class="p-3 border-b border-gray-700 hover:bg-gray-700/50 cursor-pointer ${!isRead ? 'bg-blue-500/10' : ''}" 
                     onclick="app.handleNotificationClick('${notif.id}', '${notif.entity_type || ''}', '${notif.entity_id || ''}')">
                    <div class="flex items-start gap-3">
                        <div class="text-2xl">${icon}</div>
                        <div class="flex-1">
                            <div class="flex items-center justify-between mb-1">
                                <h4 class="text-sm font-bold text-white ${!isRead ? '' : 'text-gray-400'}">${this.escapeHtml(notif.title)}</h4>
                                ${!isRead ? '<span class="w-2 h-2 bg-blue-500 rounded-full"></span>' : ''}
                            </div>
                            <p class="text-xs text-gray-400 mb-1">${this.escapeHtml(notif.message || '')}</p>
                            <div class="flex items-center justify-between">
                                <span class="text-[10px] text-gray-500">${timeAgo}</span>
                                <button onclick="event.stopPropagation(); app.deleteNotification('${notif.id}')" 
                                        class="text-red-400 hover:text-red-300 text-xs">
                                    <i class="fa-solid fa-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }).join('');
    },
    
    getNotificationIcon(type) {
        const icons = {
            'lead_assigned': '👤',
            'activity_new': '📝',
            'activity_reminder': '⏰',
            'deal_created': '💼',
            'deal_status_changed': '🔄',
            'satisfaction_new': '⭐'
        };
        return icons[type] || '🔔';
    },
    
    getTimeAgo(dateString) {
        if (!dateString) return '';
        try {
            const date = new Date(dateString);
            const now = new Date();
            const diff = now - date;
            const minutes = Math.floor(diff / 60000);
            const hours = Math.floor(minutes / 60);
            const days = Math.floor(hours / 24);
            
            if (minutes < 1) return 'همین الان';
            if (minutes < 60) return `${minutes} دقیقه پیش`;
            if (hours < 24) return `${hours} ساعت پیش`;
            if (days < 7) return `${days} روز پیش`;
            
            return toPersianDateTime(dateString);
        } catch (e) {
            return dateString;
        }
    },
    
    updateNotificationBadge(count) {
        const badge = document.getElementById('notification-badge');
        if (badge) {
            if (count > 0) {
                badge.textContent = count > 99 ? '99+' : count;
                badge.classList.remove('hidden');
            } else {
                badge.classList.add('hidden');
            }
        }
    },
    
    toggleNotifications() {
        const dropdown = document.getElementById('notifications-dropdown');
        if (!dropdown) return;
        
        if (dropdown.classList.contains('hidden')) {
            dropdown.classList.remove('hidden');
            this.loadNotifications();
        } else {
            dropdown.classList.add('hidden');
        }
    },
    
    async handleNotificationClick(notificationId, entityType, entityId) {
        // Mark as read
        await this.markNotificationRead(notificationId);
        
        // Navigate to entity if exists
        if (entityType && entityId) {
            if (entityType === 'lead' || entityType === 'person') {
                await this.openPersonProfile(entityId);
            } else if (entityType === 'activity') {
                // Could open activity details if needed
            } else if (entityType === 'deal') {
                // Could open deal details if needed
            }
        }
        
        // Reload notifications
        await this.loadNotifications();
    },
    
    async markNotificationRead(notificationId) {
        try {
            await this.req('mark_notification_read', { notification_id: notificationId });
        } catch (error) {
            console.error('Error marking notification as read:', error);
        }
    },
    
    async markAllNotificationsRead() {
        try {
            const res = await this.req('mark_all_notifications_read', {});
            if (res.status === 'success') {
                await this.loadNotifications();
            }
        } catch (error) {
            console.error('Error marking all notifications as read:', error);
        }
    },
    
    async deleteNotification(notificationId) {
        try {
            const res = await this.req('delete_notification', { notification_id: notificationId });
            if (res.status === 'success') {
                await this.loadNotifications();
            }
        } catch (error) {
            console.error('Error deleting notification:', error);
        }
    },
    
    // POLLING SYSTEM
    startNotificationPolling() {
        // Check reminders every 30 seconds
        setInterval(async () => {
            try {
                const res = await this.req('check_reminders', {});
                if (res.status === 'success') {
                    // Update notifications
                    if (res.notifications && res.notifications.length > 0) {
                        await this.loadNotifications();
                    }

                    // Show browser notifications for new notifications
                    if (res.new_notifications && res.new_notifications.length > 0) {
                        res.new_notifications.forEach(notif => {
                            this.showBrowserNotification(notif);
                        });
                    }

                    // Show browser notifications for popup notifications
                    if (res.popup_notifications && res.popup_notifications.length > 0) {
                        res.popup_notifications.forEach(notif => {
                            this.showBrowserNotification(notif);
                        });
                    }

                    // Show popup notifications
                    if (res.popup_notifications && res.popup_notifications.length > 0) {
                        console.log('Showing popup notifications:', res.popup_notifications);
                        res.popup_notifications.forEach(notif => {
                            if (this.notificationPopupEnabled) {
                                this.showNotificationPopup(notif);
                            }
                            if (this.notificationBrowserEnabled) {
                                this.showBrowserNotification(notif);
                            }
                        });
                    }
                }
            } catch (error) {
                console.error('Error checking reminders:', error);
            }
        }, 30000); // 30 seconds
    },
    
    // BROWSER NOTIFICATIONS
    requestNotificationPermission() {
        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission().then(permission => {
                if (permission === 'granted') {
                    console.log('Notification permission granted');
                }
            });
        }
    },

    // NOTIFICATION SOUND FUNCTIONS
    notificationSoundEnabled: true,

    playNotificationSound() {
        if (!this.notificationSoundEnabled) return;

        try {
            // Create audio context if not exists
            if (!this.audioContext) {
                this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
            }

            // Create a simple beep sound
            const oscillator = this.audioContext.createOscillator();
            const gainNode = this.audioContext.createGain();

            oscillator.connect(gainNode);
            gainNode.connect(this.audioContext.destination);

            oscillator.frequency.setValueAtTime(800, this.audioContext.currentTime); // Frequency in Hz
            oscillator.frequency.setValueAtTime(600, this.audioContext.currentTime + 0.1);

            gainNode.gain.setValueAtTime(0.1, this.audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, this.audioContext.currentTime + 0.5);

            oscillator.start(this.audioContext.currentTime);
            oscillator.stop(this.audioContext.currentTime + 0.5);
        } catch (error) {
            console.log('Sound not supported, trying HTML5 audio');
            // Fallback to HTML5 audio
            try {
                const audio = new Audio();
                audio.src = 'data:audio/wav;base64,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';
                audio.play().catch(e => console.log('Audio play failed:', e));
            } catch (e) {
                console.log('Audio not supported');
            }
        }
    },

    toggleNotificationSound() {
        this.notificationSoundEnabled = !this.notificationSoundEnabled;
        this.updateSoundButton();
        localStorage.setItem('notificationSoundEnabled', this.notificationSoundEnabled);

        // Show feedback
        const message = this.notificationSoundEnabled ? 'صدا نوتیفیکیشن فعال شد' : 'صدا نوتیفیکیشن غیرفعال شد';
        Swal.fire({
            toast: true,
            position: 'bottom-end',
            icon: 'info',
            title: message,
            timer: 2000,
            showConfirmButton: false
        });
    },

    // NOTIFICATION SETTINGS
    showNotificationSettings() {
        Swal.fire({
            title: 'تنظیمات نوتیفیکیشن',
            html: `
                <div class="text-right space-y-4">
                    <div class="flex items-center justify-between">
                        <label class="flex items-center">
                            <input type="checkbox" id="sound-enabled" class="ml-2">
                            <span>فعال کردن صدا</span>
                        </label>
                    </div>
                    <div class="flex items-center justify-between">
                        <label class="flex items-center">
                            <input type="checkbox" id="popup-enabled" class="ml-2" checked>
                            <span>نمایش پاپ‌آپ</span>
                        </label>
                    </div>
                    <div class="flex items-center justify-between">
                        <label class="flex items-center">
                            <input type="checkbox" id="browser-notif-enabled" class="ml-2" checked>
                            <span>نوتیفیکیشن مرورگر (دسکتاپ)</span>
                        </label>
                        <div id="browser-notif-status" class="text-sm"></div>
                    </div>
                    <div class="text-center mt-3">
                        <button type="button" onclick="app.requestNotificationPermission()" class="px-3 py-1 bg-blue-600 text-white text-sm rounded hover:bg-blue-700 transition">
                            درخواست دسترسی نوتیفیکیشن
                        </button>
                    </div>
                </div>
            `,
            confirmButtonText: 'ذخیره',
            cancelButtonText: 'لغو',
            showCancelButton: true,
            preConfirm: () => {
                const soundEnabled = document.getElementById('sound-enabled').checked;
                const popupEnabled = document.getElementById('popup-enabled').checked;
                const browserNotifEnabled = document.getElementById('browser-notif-enabled').checked;

                // Save settings
                localStorage.setItem('notificationSoundEnabled', soundEnabled);
                localStorage.setItem('notificationPopupEnabled', popupEnabled);
                localStorage.setItem('notificationBrowserEnabled', browserNotifEnabled);

                // Update current settings
                this.notificationSoundEnabled = soundEnabled;
                this.notificationPopupEnabled = popupEnabled;
                this.notificationBrowserEnabled = browserNotifEnabled;

                // Update status display
                this._updateNotificationStatus();

                this.updateSoundButton();

                return {
                    soundEnabled,
                    popupEnabled,
                    browserNotifEnabled
                };
            }
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    toast: true,
                    position: 'bottom-end',
                    icon: 'success',
                    title: 'تنظیمات ذخیره شد',
                    timer: 2000,
                    showConfirmButton: false
                });
            }
        });

        // Load current settings
        setTimeout(() => {
            document.getElementById('sound-enabled').checked = this.notificationSoundEnabled;
            document.getElementById('popup-enabled').checked = localStorage.getItem('notificationPopupEnabled') !== 'false';
            document.getElementById('browser-notif-enabled').checked = localStorage.getItem('notificationBrowserEnabled') !== 'false';

            // Show current browser permission status
            this._updateNotificationStatus();
        }, 100);
    },

    updateSoundButton() {
        const button = document.getElementById('sound-toggle-btn');
        if (button) {
            const icon = button.querySelector('i');
            if (icon) {
                icon.className = this.notificationSoundEnabled ?
                    'fa-solid fa-volume-up text-xl' :
                    'fa-solid fa-volume-mute text-xl';
            }
            button.className = this.notificationSoundEnabled ?
                'text-green-400 hover:text-green-300 transition p-2 rounded-full hover:bg-gray-800 focus:outline-none focus:ring-2 focus:ring-gray-700' :
                'text-gray-400 hover:text-gray-300 transition p-2 rounded-full hover:bg-gray-800 focus:outline-none focus:ring-2 focus:ring-gray-700';
            button.title = this.notificationSoundEnabled ? 'صدا فعال است - کلیک برای غیرفعال کردن' : 'صدا غیرفعال است - کلیک برای فعال کردن';
        }
    },

    // NOTIFICATION POPUP FUNCTIONS
    notificationPopups: [],

    showNotificationPopup(notification) {
        // Check if popup should be shown
        console.log('Show notification popup called:', notification);

        // Check user settings
        if (!this.notificationPopupEnabled) {
            console.log('Popup notifications disabled by user settings');
            return;
        }

        // Default to show popup if not specified
        if (notification.show_popup !== undefined && !notification.show_popup) {
            console.log('Popup disabled for this notification');
            return;
        }

        // Play sound if enabled
        if (this.notificationSoundEnabled) {
            this.playNotificationSound();
        }

        const popupId = 'popup-' + notification.id + '-' + Date.now();

        // Create popup element
        const popup = document.createElement('div');
        popup.className = 'notification-popup';
        popup.id = popupId;

        const icon = this.getNotificationIcon(notification.type);
        const timeAgo = this.getTimeAgo(notification.created_at);

        popup.innerHTML = `
            <div class="notification-popup-content">
                <div class="notification-popup-header">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <span style="font-size: 18px;">${icon}</span>
                        <h4 class="notification-popup-title">${this.escapeHtml(notification.title)}</h4>
                    </div>
                        <div class="flex items-center gap-2">
                            <button class="notification-popup-close" onclick="app.closeNotificationPopup('${popupId}')" title="بستن این اعلان">
                                <i class="fa-solid fa-times"></i>
                            </button>
                            ${this.notificationPopups.length > 1 ? `<button class="notification-popup-close-all" onclick="app.closeAllNotificationPopups()" title="بستن همه اعلان‌ها">
                                <i class="fa-solid fa-times-circle"></i>
                            </button>` : ''}
                        </div>
                </div>
                <p class="notification-popup-message">${this.escapeHtml(notification.message || '')}</p>
                <div class="notification-popup-footer">
                    <span class="notification-popup-time">${timeAgo}</span>
                    <div class="notification-popup-actions">
                        <button class="notification-popup-action primary" onclick="app.handleNotificationClick('${notification.id}', '${notification.entity_type || ''}', '${notification.entity_id || ''}'); app.closeNotificationPopup('${popupId}')">
                            مشاهده
                        </button>
                        <button class="notification-popup-action secondary" onclick="app.closeNotificationPopup('${popupId}')">
                            رد
                        </button>
                    </div>
                </div>
            </div>
        `;

        // Add to page
        document.body.appendChild(popup);

        // Add to tracking array
        this.notificationPopups.push({
            id: popupId,
            element: popup,
            notification: notification
        });

        // Show with animation
        console.log('Adding popup to page:', popupId);
        setTimeout(() => {
            console.log('Adding show class to popup:', popupId);
            popup.classList.add('show');
        }, 100);

        // Remove auto-close timer - popups stay until user interaction

        // No limit on popup count - all notifications stay visible until user interaction
    },

    closeNotificationPopup(popupId) {
        const popup = document.getElementById(popupId);
        if (!popup) return;

        popup.classList.remove('show');
        popup.classList.add('hide');

        setTimeout(() => {
            if (popup.parentNode) {
                popup.parentNode.removeChild(popup);
            }
        }, 300);

        // Remove from tracking array
        this.notificationPopups = this.notificationPopups.filter(p => p.id !== popupId);
    },

    closeAllNotificationPopups() {
        const currentPopups = [...this.notificationPopups]; // Copy array to avoid modification during iteration
        currentPopups.forEach(popup => {
            this.closeNotificationPopup(popup.id);
        });
    },

    // Get count of active popups
    getActivePopupCount() {
        return this.notificationPopups.length;
    },

    showBrowserNotification(notification) {
        console.log('Attempting to show browser notification:', notification);

        // Check user settings
        if (!this.notificationBrowserEnabled) {
            console.log('Browser notifications disabled by user settings');
            return;
        }

        if (!('Notification' in window)) {
            console.log('This browser does not support notifications');
            return;
        }

        // Check permission and show notification
        if (Notification.permission === 'granted') {
            this._createBrowserNotification(notification);
        } else if (Notification.permission === 'default') {
            console.log('Requesting notification permission...');
            this.requestNotificationPermission().then((permission) => {
                if (permission === 'granted') {
                    console.log('Permission granted, creating notification');
                    this._createBrowserNotification(notification);
                } else {
                    console.log('Permission denied or failed');
                    this._showPermissionDeniedMessage();
                }
            });
        } else {
            console.log('Notifications are blocked by user');
            this._showPermissionDeniedMessage();
        }
    },

    _createBrowserNotification(notification) {
        try {
            const browserNotif = new Notification(notification.title, {
                body: notification.message,
                icon: '/logo.png',
                badge: '/logo.png',
                tag: 'saleos-' + notification.id,
                requireInteraction: true, // Keep notification until user interacts
                silent: false,
                dir: 'rtl',
                lang: 'fa',
                silent: false,
                timestamp: Date.now()
            });

            console.log('Browser notification created:', browserNotif);

            // No auto-close - notification stays until user interacts with it
            let autoCloseTimeout;

            // Handle click - focus window and navigate
            browserNotif.onclick = () => {
                console.log('Notification clicked');
                if (autoCloseTimeout) {
                    clearTimeout(autoCloseTimeout);
                }
                window.focus();

                // Try to bring window to front
                if (window.focus) {
                    window.focus();
                }

                // Navigate to relevant page if possible
                if (notification.entity_type && notification.entity_id) {
                    console.log('Should navigate to:', notification.entity_type, notification.entity_id);
                    // You can add navigation logic here
                    // For example: window.location.href = `/activities/${notification.entity_id}`;
                }

                browserNotif.close();
            };

            // Handle close
            browserNotif.onclose = () => {
                console.log('Notification closed');
                clearTimeout(autoCloseTimeout);
            };

            // Handle show
            browserNotif.onshow = () => {
                console.log('Notification shown successfully');
            };

            // Handle error
            browserNotif.onerror = (error) => {
                console.error('Notification error:', error);
            };

        } catch (error) {
            console.error('Failed to create browser notification:', error);
        }
    },

    _showPermissionDeniedMessage() {
        const browser = this._getBrowserName();
        const instructions = this._getBrowserInstructions(browser);

        Swal.fire({
            title: 'دسترسی نوتیفیکیشن مرورگر',
            text: 'برای دریافت نوتیفیکیشن‌های دسکتاپ، لطفاً دسترسی نوتیفیکیشن مرورگر را فعال کنید.',
            icon: 'warning',
            confirmButtonText: 'راهنمایی',
            showCancelButton: true,
            cancelButtonText: 'بعداً'
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: `فعال کردن نوتیفیکیشن در ${browser}`,
                    html: instructions,
                    confirmButtonText: 'باشه',
                    width: 600
                });
            }
        });
    },

    _getBrowserName() {
        const userAgent = navigator.userAgent;
        if (userAgent.includes('Chrome') && !userAgent.includes('Edg')) {
            return 'Chrome';
        } else if (userAgent.includes('Firefox')) {
            return 'Firefox';
        } else if (userAgent.includes('Safari') && !userAgent.includes('Chrome')) {
            return 'Safari';
        } else if (userAgent.includes('Edg')) {
            return 'Edge';
        }
        return 'مرورگر';
    },

    _getBrowserInstructions(browser) {
        const instructions = {
            'Chrome': `
                <div class="text-right">
                    <p class="mb-2">در مرورگر Chrome:</p>
                    <ol class="text-sm text-gray-600 mb-3" style="direction: rtl; text-align: right;">
                        <li>1. روی آیکون قفل (🔒) در نوار آدرس کلیک کنید</li>
                        <li>2. گزینه "Site settings" را انتخاب کنید</li>
                        <li>3. بخش "Notifications" را پیدا کنید</li>
                        <li>4. روی "Allow" کلیک کنید</li>
                    </ol>
                    <p class="text-xs text-gray-500">یا در تنظیمات مرورگر: chrome://settings/content/notifications</p>
                </div>
            `,
            'Firefox': `
                <div class="text-right">
                    <p class="mb-2">در مرورگر Firefox:</p>
                    <ol class="text-sm text-gray-600 mb-3" style="direction: rtl; text-align: right;">
                        <li>1. روی آیکون قفل (🔒) در نوار آدرس کلیک کنید</li>
                        <li>2. گزینه "Allow notifications" را فعال کنید</li>
                    </ol>
                    <p class="text-xs text-gray-500">یا در تنظیمات: about:preferences#privacy</p>
                </div>
            `,
            'Edge': `
                <div class="text-right">
                    <p class="mb-2">در مرورگر Edge:</p>
                    <ol class="text-sm text-gray-600 mb-3" style="direction: rtl; text-align: right;">
                        <li>1. روی آیکون قفل (🔒) در نوار آدرس کلیک کنید</li>
                        <li>2. گزینه "Site permissions" را انتخاب کنید</li>
                        <li>3. بخش "Notifications" را پیدا کنید</li>
                        <li>4. روی "Allow" کلیک کنید</li>
                    </ol>
                </div>
            `,
            'Safari': `
                <div class="text-right">
                    <p class="mb-2">در مرورگر Safari:</p>
                    <ol class="text-sm text-gray-600 mb-3" style="direction: rtl; text-align: right;">
                        <li>1. Safari → Preferences را انتخاب کنید</li>
                        <li>2. تب Websites را انتخاب کنید</li>
                        <li>3. از منوی سمت چپ "Notifications" را انتخاب کنید</li>
                        <li>4. این سایت را پیدا کرده و روی "Allow" کلیک کنید</li>
                    </ol>
                </div>
            `
        };

        return instructions[browser] || `
            <div class="text-right">
                <p class="mb-2">در مرورگر شما:</p>
                <ol class="text-sm text-gray-600" style="direction: rtl; text-align: right;">
                    <li>1. روی آیکون قفل در نوار آدرس کلیک کنید</li>
                    <li>2. تنظیمات نوتیفیکیشن را پیدا کنید</li>
                    <li>3. دسترسی را فعال کنید</li>
                </ol>
            </div>
        `;
    },

    _updateNotificationStatus() {
        const statusEl = document.getElementById('browser-notif-status');
        if (!statusEl) return;

        let status = 'نامشخص';
        let color = 'text-gray-500';

        if (!('Notification' in window)) {
            status = 'مرورگر پشتیبانی نمی‌کند';
            color = 'text-red-500';
        } else {
            switch (Notification.permission) {
                case 'granted':
                    status = '✅ فعال';
                    color = 'text-green-500';
                    break;
                case 'denied':
                    status = '❌ بلاک شده';
                    color = 'text-red-500';
                    break;
                case 'default':
                    status = '⏳ نیاز به اجازه';
                    color = 'text-yellow-500';
                    break;
            }
        }

        statusEl.innerHTML = `<span class="${color} text-sm">${status}</span>`;
    },

    async requestNotificationPermission() {
        if (!('Notification' in window)) {
            console.log('Notifications not supported');
            return 'denied';
        }

        if (Notification.permission === 'granted') {
            return 'granted';
        }

        if (Notification.permission === 'denied') {
            return 'denied';
        }

            try {
                console.log('Requesting permission...');
                const permission = await Notification.requestPermission();
                console.log('Permission result:', permission);

                if (permission === 'granted') {
                    console.log('✅ Browser notification permission granted');
                    Swal.fire({
                        toast: true,
                        position: 'bottom-end',
                        icon: 'success',
                        title: 'دسترسی نوتیفیکیشن فعال شد',
                        timer: 2000,
                        showConfirmButton: false
                    });
                } else if (permission === 'denied') {
                    console.log('❌ Browser notification permission denied');
                    Swal.fire({
                        toast: true,
                        position: 'bottom-end',
                        icon: 'error',
                        title: 'دسترسی نوتیفیکیشن رد شد',
                        timer: 3000,
                        showConfirmButton: false
                    });
                } else {
                    console.log('⚠️ Browser notification permission default');
                }

                // Update status display if settings modal is open
                this._updateNotificationStatus();

                return permission;
            } catch (error) {
                console.error('❌ Error requesting notification permission:', error);
                return 'denied';
            }
    },

    // REMINDER CALCULATION FUNCTIONS
    calculateReminderOffset(form) {
        const hours = parseInt(form.reminder_offset_hours?.value || '24') || 0;
        const minutes = parseInt(form.reminder_offset_minutes?.value || '0') || 0;
        const totalMinutes = (hours * 60) + minutes;
        return totalMinutes > 0 ? totalMinutes : 1440; // Default to 24 hours if invalid
    },

    updateReminderOffsetTotal(formPrefix = '') {
        const hoursSelect = document.getElementById(`${formPrefix}activity-reminder-hours`);
        const minutesSelect = document.getElementById(`${formPrefix}activity-reminder-minutes`);
        const totalInput = document.getElementById(`${formPrefix}reminder-offset-total`);

        if (hoursSelect && minutesSelect && totalInput) {
            const hours = parseInt(hoursSelect.value) || 0;
            const minutes = parseInt(minutesSelect.value) || 0;
            const totalMinutes = (hours * 60) + minutes;
            totalInput.value = totalMinutes > 0 ? totalMinutes : 1440;
        }
    },

    initReminderOffsetListeners() {
        // Create activity reminder listeners
        const createHours = document.getElementById('activity-reminder-hours');
        const createMinutes = document.getElementById('activity-reminder-minutes');

        if (createHours) {
            createHours.addEventListener('change', () => this.updateReminderOffsetTotal(''));
        }
        if (createMinutes) {
            createMinutes.addEventListener('change', () => this.updateReminderOffsetTotal(''));
        }

        // Edit activity reminder listeners
        const editHours = document.getElementById('edit-activity-reminder-hours');
        const editMinutes = document.getElementById('edit-activity-reminder-minutes');

        if (editHours) {
            editHours.addEventListener('change', () => this.updateReminderOffsetTotal('edit-'));
        }
        if (editMinutes) {
            editMinutes.addEventListener('change', () => this.updateReminderOffsetTotal('edit-'));
        }
    },

    // ACTIVITY SCHEDULE FUNCTIONS
    toggleActivityScheduleFields() {
        const checkbox = document.getElementById('activity-is-scheduled');
        const fields = document.getElementById('activity-schedule-fields');
        const virtualStageSelect = document.getElementById('activity-virtual-stage');
        const resultNoteTextarea = document.getElementById('activity-result-note');
        const virtualStageRequired = document.getElementById('virtual-stage-required');
        
        if (!checkbox || !fields) return;
        
        if (checkbox.checked) {
            fields.classList.remove('hidden');
            // Initialize date picker
            this.initActivityDatePicker();
            
            // Make result_note optional for scheduled activities
            if (virtualStageSelect) {
                virtualStageSelect.removeAttribute('required');
            }
            if (resultNoteTextarea) {
                resultNoteTextarea.removeAttribute('required');
            }
            if (virtualStageRequired) {
                virtualStageRequired.classList.add('hidden');
            }
        } else {
            fields.classList.add('hidden');
            
            // Make result_note required again
            if (virtualStageSelect) {
                virtualStageSelect.setAttribute('required', 'required');
            }
            if (resultNoteTextarea) {
                resultNoteTextarea.setAttribute('required', 'required');
            }
            if (virtualStageRequired) {
                virtualStageRequired.classList.remove('hidden');
            }
        }
    },
    
    toggleActivityReminderFields() {
        const checkbox = document.getElementById('activity-reminder-enabled');
        const fields = document.getElementById('activity-reminder-fields');
        if (!checkbox || !fields) return;

        if (checkbox.checked) {
            fields.classList.remove('hidden');
            // Set default values if not already set
            const hoursSelect = document.getElementById('activity-reminder-hours');
            const minutesSelect = document.getElementById('activity-reminder-minutes');
            if (hoursSelect && hoursSelect.value === '') {
                hoursSelect.value = '24'; // 1 day default
            }
            if (minutesSelect && minutesSelect.value === '') {
                minutesSelect.value = '0';
            }
            // Update total
            this.updateReminderOffsetTotal('');
        } else {
            fields.classList.add('hidden');
        }
    },
    
    initActivityDatePicker() {
        const dateInput = document.getElementById('activity-due-date');
        const hourSelect = document.getElementById('activity-due-hour');
        const minuteSelect = document.getElementById('activity-due-minute');
        
        if (!dateInput) return;
        
        // Initialize hour and minute selects
        if (hourSelect) {
            hourSelect.innerHTML = '<option value="">ساعت</option>';
            for (let h = 0; h < 24; h++) {
                const hour = String(h).padStart(2, '0');
                const hourLabel = h < 12 ? `${hour} صبح` : `${hour} بعدازظهر`;
                hourSelect.innerHTML += `<option value="${hour}">${hourLabel}</option>`;
            }
            // Set default to current hour + 1 (next hour)
            const now = getTehranNow();
            const nextHour = (now.getHours() + 1) % 24;
            hourSelect.value = String(nextHour).padStart(2, '0');
        }
        
        if (minuteSelect) {
            minuteSelect.innerHTML = '<option value="">دقیقه</option>';
            for (let m = 0; m < 60; m += 5) {
                const minute = String(m).padStart(2, '0');
                minuteSelect.innerHTML += `<option value="${minute}">${minute}</option>`;
            }
            // Set default to 00
            minuteSelect.value = '00';
        }
        
        // Wait for jQuery and persianDatepicker
        if (typeof $ === 'undefined' || typeof $.fn.persianDatepicker === 'undefined') {
            setTimeout(() => this.initActivityDatePicker(), 200);
            return;
        }
        
        // Destroy existing picker
        try {
            if ($(dateInput).data('persianDatepicker')) {
                $(dateInput).persianDatepicker('destroy');
            }
        } catch(e) {
            console.log('No existing picker to destroy');
        }
        
        // Get today's date in Tehran timezone using moment-jalaali (accurate conversion)
        let todayFormatted = '';
        
        if (typeof moment !== 'undefined') {
            const testMoment = moment();
            if (typeof testMoment.jYear === 'function') {
                // Use moment-jalaali for accurate date conversion (fixes leap year bug)
                const now = new Date();
                const formatter = new Intl.DateTimeFormat('en-CA', {
                    timeZone: 'Asia/Tehran',
                    year: 'numeric',
                    month: '2-digit',
                    day: '2-digit'
                });
                const parts = formatter.formatToParts(now);
                const gYear = parseInt(parts.find(p => p.type === 'year').value);
                const gMonth = parseInt(parts.find(p => p.type === 'month').value);
                const gDay = parseInt(parts.find(p => p.type === 'day').value);
                
                const gregorianMoment = moment(`${gYear}-${String(gMonth).padStart(2, '0')}-${String(gDay).padStart(2, '0')}`, 'YYYY-MM-DD');
                todayFormatted = gregorianMoment.format('jYYYY/jMM/jDD').replace(/\//g, '/');
            } else {
                // Fallback to persianDate
                const todayTehran = getTehranPersianDate();
                if (todayTehran) {
                    todayFormatted = todayTehran.format('YYYY/MM/DD');
                }
            }
        }
        
        // Set default value to today if input is empty
        if (!dateInput.value || dateInput.value.trim() === '') {
            dateInput.value = todayFormatted;
        }
        
        // Initialize Persian date picker with better styling
        try {
            $(dateInput).persianDatepicker({
                observer: true,
                format: 'YYYY/MM/DD',
                altField: '#activity-due-date-gregorian',
                altFormat: 'YYYY-MM-DD',
                calendarType: 'persian',
                timePicker: {
                    enabled: false  // We'll use separate hour/minute selects
                },
                initialValue: dateInput.value || todayFormatted, // Use current value or today
                autoClose: true,
                toolbox: {
                    calendarSwitch: {
                        enabled: false
                    }
                },
                onlyTimePicker: false,
                onlySelectOnDate: true,
                calendar: {
                    persian: {
                        locale: 'fa',
                        dow: [0, 1, 2, 3, 4, 5, 6]  // Start week from Saturday (0=Saturday in persianDate)
                    }
                },
                onSelect: function(unixDate) {
                    // Update the input with formatted date
                    const pDate = new persianDate(unixDate);
                    dateInput.value = pDate.format('YYYY/MM/DD');
                }
            });
            this.syncActivityDatePickerToTehran();
            
            // Apply custom styles to the date picker
            setTimeout(() => {
                const pickerContainer = $('.pdp-container');
                if (pickerContainer.length) {
                    pickerContainer.css({
                        'background': 'linear-gradient(135deg, #1e293b 0%, #0f172a 100%)',
                        'border': '1px solid rgba(59, 130, 246, 0.3)',
                        'border-radius': '12px',
                        'box-shadow': '0 10px 40px rgba(0, 0, 0, 0.5)',
                        'padding': '15px',
                        'z-index': '10000'
                    });
                    
                    // Style calendar cells
                    pickerContainer.find('.pdp-day').css({
                        'color': '#e2e8f0',
                        'border-radius': '6px',
                        'transition': 'all 0.2s'
                    });
                    
                    pickerContainer.find('.pdp-day-selected').css({
                        'background': 'linear-gradient(135deg, #3b82f6, #8b5cf6)',
                        'color': '#fff',
                        'font-weight': 'bold',
                        'box-shadow': '0 2px 8px rgba(59, 130, 246, 0.4)'
                    });
                    
                    pickerContainer.find('.pdp-day-today').css({
                        'border': '2px solid #3b82f6',
                        'font-weight': 'bold'
                    });
                    
                    pickerContainer.find('.pdp-day:hover').css({
                        'background': 'rgba(59, 130, 246, 0.2)',
                        'transform': 'scale(1.05)'
                    });
                }
            }, 100);
            
        } catch(e) {
            console.error('Error initializing date picker:', e);
        }
    },

    syncActivityDatePickerToTehran() {
        const dateInput = document.getElementById('activity-due-date');
        if (!dateInput) return;
        if (typeof $ === 'undefined' || typeof $.fn.persianDatepicker === 'undefined') return;

        try {
            const tehranTime = getTehranNow().valueOf();
            $(dateInput).persianDatepicker('setDate', tehranTime);
        } catch (error) {
            console.error('Error syncing activity date picker to Tehran time:', error);
        }
    },
    
    // EDIT SCHEDULED ACTIVITY FUNCTIONS
    async openEditScheduledActivityModal(activityId) {
        try {
            // Get activity details from Didar API
            const res = await this.req('get_activity_details', { activity_id: activityId });
            if (res.status !== 'success' || !res.activity) {
                Swal.fire('خطا', 'فعالیت یافت نشد', 'error');
                return;
            }
            
            const activity = res.activity;
            
            // Fill form
            document.getElementById('edit-activity-id').value = activityId;
            document.getElementById('edit-activity-contact-id').value = activity.ContactIds?.[0] || activity.contact_didar_id || '';
            
            // Load activity types
            await this.loadActivityTypesForEditModal();
            
            // Set activity type
            const typeSelect = document.getElementById('edit-activity-type-id');
            if (typeSelect && activity.ActivityTypeId) {
                typeSelect.value = activity.ActivityTypeId;
            }
            
            // Set due date if exists
            if (activity.DueDate) {
                const dueDate = new Date(activity.DueDate);
                const pDate = new persianDate(dueDate);
                const dateStr = pDate.format('YYYY/MM/DD');
                const hour = String(dueDate.getHours()).padStart(2, '0');
                const minute = String(dueDate.getMinutes()).padStart(2, '0');
                
                document.getElementById('edit-activity-due-date').value = dateStr;
                document.getElementById('edit-activity-due-hour').value = hour;
                document.getElementById('edit-activity-due-minute').value = minute;
                
                // Initialize date picker (will update with current date)
                setTimeout(() => {
                    this.initEditActivityDatePicker();
                }, 100);
            } else {
                // If no due date, initialize with today's date
                setTimeout(() => {
                    this.initEditActivityDatePicker();
                }, 100);
            }
            
            // Set reminder
            if (activity.Notifies && activity.Notifies.length > 0) {
                document.getElementById('edit-activity-reminder-enabled').checked = true;
                this.toggleEditActivityReminderFields();
                const offset = activity.Notifies[0].NotifyOffset || 1440;

                // Convert minutes to hours and minutes
                const hours = Math.floor(offset / 60);
                const minutes = offset % 60;

                document.getElementById('edit-activity-reminder-hours').value = hours;
                document.getElementById('edit-activity-reminder-minutes').value = minutes;
                document.getElementById('edit-reminder-offset-total').value = offset;
            }
            
            this.openModal('modalEditScheduledActivity');
        } catch (error) {
            console.error('Error opening edit modal:', error);
            Swal.fire('خطا', 'خطا در بارگذاری اطلاعات فعالیت', 'error');
        }
    },
    
    async loadActivityTypesForEditModal() {
        try {
            const res = await this.req('get_activity_types', {});
            if (res.status === 'success' && res.types) {
                const select = document.getElementById('edit-activity-type-id');
                if (select) {
                    select.innerHTML = '<option value="">-- انتخاب نوع فعالیت --</option>';
                    res.types.forEach(type => {
                        const option = document.createElement('option');
                        option.value = type.Id || type.ActivityTypeId;
                        option.textContent = type.Title || 'فعالیت';
                        select.appendChild(option);
                    });
                }
            }
        } catch (error) {
            console.error('Error loading activity types:', error);
        }
    },
    
    toggleEditActivityReminderFields() {
        const checkbox = document.getElementById('edit-activity-reminder-enabled');
        const fields = document.getElementById('edit-activity-reminder-fields');
        if (!checkbox || !fields) return;

        if (checkbox.checked) {
            fields.classList.remove('hidden');
            // Set default values if not already set
            const hoursSelect = document.getElementById('edit-activity-reminder-hours');
            const minutesSelect = document.getElementById('edit-activity-reminder-minutes');
            if (hoursSelect && hoursSelect.value === '') {
                hoursSelect.value = '24'; // 1 day default
            }
            if (minutesSelect && minutesSelect.value === '') {
                minutesSelect.value = '0';
            }
            // Update total
            this.updateReminderOffsetTotal('edit-');
        } else {
            fields.classList.add('hidden');
        }
    },
    
    initEditActivityDatePicker() {
        const dateInput = document.getElementById('edit-activity-due-date');
        const hourSelect = document.getElementById('edit-activity-due-hour');
        const minuteSelect = document.getElementById('edit-activity-due-minute');
        
        if (!dateInput) return;
        
        // Initialize hour and minute selects
        if (hourSelect) {
            hourSelect.innerHTML = '<option value="">ساعت</option>';
            for (let h = 0; h < 24; h++) {
                const hour = String(h).padStart(2, '0');
                const hourLabel = h < 12 ? `${hour} صبح` : `${hour} بعدازظهر`;
                hourSelect.innerHTML += `<option value="${hour}">${hourLabel}</option>`;
            }
        }
        
        if (minuteSelect) {
            minuteSelect.innerHTML = '<option value="">دقیقه</option>';
            for (let m = 0; m < 60; m += 5) {
                const minute = String(m).padStart(2, '0');
                minuteSelect.innerHTML += `<option value="${minute}">${minute}</option>`;
            }
        }
        
        // Wait for jQuery and persianDatepicker
        if (typeof $ === 'undefined' || typeof $.fn.persianDatepicker === 'undefined') {
            setTimeout(() => this.initEditActivityDatePicker(), 200);
            return;
        }
        
        // Destroy existing picker
        try {
            if ($(dateInput).data('persianDatepicker')) {
                $(dateInput).persianDatepicker('destroy');
            }
        } catch(e) {
            console.log('No existing picker to destroy');
        }
        
        // Get today's date in Tehran timezone (as fallback if no value set)
        const todayTehran = getTehranPersianDate();
        const todayFormatted = todayTehran.format('YYYY/MM/DD');
        
        // Initialize Persian date picker
        try {
            $(dateInput).persianDatepicker({
                observer: true,
                format: 'YYYY/MM/DD',
                altField: '#edit-activity-due-date-gregorian',
                altFormat: 'YYYY-MM-DD',
                calendarType: 'persian',
                timePicker: {
                    enabled: false
                },
                initialValue: dateInput.value || todayFormatted, // Use current value or today
                autoClose: true,
                calendar: {
                    persian: {
                        locale: 'fa',
                        dow: [0, 1, 2, 3, 4, 5, 6]  // Start week from Saturday (0=Saturday in persianDate)
                    }
                },
                onSelect: function(unixDate) {
                    const pDate = new persianDate(unixDate);
                    dateInput.value = pDate.format('YYYY/MM/DD');
                }
            });
        } catch(e) {
            console.error('Error initializing date picker:', e);
        }
    },
    
    async updateScheduledActivity(e) {
        e.preventDefault();
        
        try {
            const form = e.target;
            const activityId = form.activity_id.value;
            const activityTypeId = form.activity_type_id.value;
            const dueDateInput = form.due_date.value;
            const hour = document.getElementById('edit-activity-due-hour').value;
            const minute = document.getElementById('edit-activity-due-minute').value;
            
            if (!activityId || !activityTypeId) {
                Swal.fire('خطا', 'نوع فعالیت الزامی است', 'error');
                return;
            }
            
            if (!dueDateInput || !hour || !minute) {
                Swal.fire('خطا', 'لطفاً تاریخ و زمان را کامل وارد کنید', 'error');
                return;
            }
            
            // Convert Persian date to Gregorian
            const persianToEnglish = (str) => {
                const persianDigits = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
                const englishDigits = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
                let result = str;
                for (let i = 0; i < persianDigits.length; i++) {
                    result = result.replace(new RegExp(persianDigits[i], 'g'), englishDigits[i]);
                }
                return result;
            };
            
            const englishDateValue = persianToEnglish(dueDateInput.trim());
            const timeString = `${hour}:${minute}`;
            let dueDate = persianToGregorian(englishDateValue, timeString);
            
            if (!dueDate) {
                Swal.fire('خطا', 'تاریخ وارد شده معتبر نیست', 'error');
                return;
            }
            
            dueDate = dueDate.replace('T', ' ') + ':00';
            
            const data = {
                activity_id: activityId,
                activity_type_id: activityTypeId,
                due_date: dueDate,
                reminder_enabled: form.reminder_enabled.checked ? '1' : '0',
                reminder_offset_minutes: this.calculateReminderOffset(form)
            };
            
            Swal.showLoading();
            const res = await this.req('update_activity', data);
            Swal.close();
            
            if (res.status === 'success') {
                Swal.fire({toast:true, position:'bottom-end', icon:'success', title:'فعالیت با موفقیت به‌روزرسانی شد', showConfirmButton:false, timer:2000});
                this.closeModal('modalEditScheduledActivity');
                
                // Reload activities if we're viewing a person profile
                if (window.currentContactId) {
                    await this.openPersonProfile(window.currentContactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطای نامشخص', 'error');
            }
        } catch (error) {
            console.error('Error updating activity:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در به‌روزرسانی فعالیت: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    // OPEN ACTIVITY DETAILS MODAL
    async openActivityDetailsModal(activityId) {
        if (!activityId) {
            console.error('Activity ID is required');
            return;
        }
        
        try {
            Swal.fire({
                title: 'در حال بارگذاری...',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            const res = await this.req('get_activity_details', {activity_id: activityId});
            
            if (res.status === 'success' && res.activity) {
                const act = res.activity;
                const isDone = act.IsDone === true || act.IsDone === 1;
                const isNote = act.ActivityTypeId === '00000000-0000-0000-0000-000000000000';
                const activityType = act.ActivityType?.Title || act.ActivityTypeTitle || (isNote ? 'یادداشت' : 'فعالیت');
                const creator = act.Creator?.DisplayName || act.CreatorName || 'نامشخص';
                const owner = act.Owner?.DisplayName || act.OwnerName || 'نامشخص';
                const note = act.Note || '';
                const resultNote = act.ResultNote || '';
                const registerDate = act.RegisterDate ? toPersianDateTime(act.RegisterDate) : '';
                const doneDate = act.DoneDate ? toPersianDateTime(act.DoneDate) : '';
                
                // Check for due date
                let dueDate = '';
                if (act.DueDate) {
                    try {
                        dueDate = toPersianDateTime(act.DueDate);
                    } catch (e) {
                        dueDate = act.DueDate;
                    }
                } else if (act.due_date) {
                    try {
                        dueDate = toPersianDateTime(act.due_date);
                    } catch (e) {
                        dueDate = act.due_date;
                    }
                }
                
                const hasDueDate = dueDate && dueDate.trim() !== '';
                const isScheduledFlag = act.is_scheduled === 1 || act.is_scheduled === true || act.is_scheduled === '1';
                const isScheduled = isScheduledFlag || (!isDone && hasDueDate);
                
                let statusText = '';
                if (isDone) {
                    statusText = '<span class="text-green-300">✓ انجام شده</span>';
                } else if (isScheduled) {
                    statusText = '<span class="text-yellow-300">⏰ برنامه‌ریزی شده</span>';
                } else {
                    statusText = '<span class="text-blue-300">📝 ثبت شده</span>';
                }
                
                Swal.fire({
                    title: 'جزئیات فعالیت',
                    html: `
                        <div class="text-right space-y-3">
                            <div class="border-b border-gray-700 pb-2">
                                <div class="flex items-center justify-between mb-2">
                                    <span class="text-lg font-bold text-blue-300">
                                        <i class="fa-solid fa-${isNote ? 'sticky-note' : (isDone ? 'check-circle' : isScheduled ? 'clock' : 'calendar-check')} ml-1"></i>
                                        ${this.escapeHtml(activityType)}
                                    </span>
                                    ${statusText}
                                </div>
                                <div class="text-sm text-gray-400">${this.escapeHtml(act.Title || 'بدون عنوان')}</div>
                            </div>
                            
                            <div class="space-y-2 text-sm">
                                ${registerDate ? `<div><span class="text-gray-400">تاریخ ثبت:</span> <span class="text-gray-300">${registerDate}</span></div>` : ''}
                                ${dueDate && isScheduled ? `<div><span class="text-gray-400">تاریخ/ساعت برنامه‌ریزی:</span> <span class="text-yellow-300">${dueDate}</span></div>` : ''}
                                ${doneDate && isDone ? `<div><span class="text-gray-400">تاریخ انجام:</span> <span class="text-green-300">${doneDate}</span></div>` : ''}
                                <div><span class="text-gray-400">ایجاد کننده:</span> <span class="text-gray-300">${this.escapeHtml(creator)}</span></div>
                                ${owner !== creator ? `<div><span class="text-gray-400">مسئول:</span> <span class="text-gray-300">${this.escapeHtml(owner)}</span></div>` : ''}
                            </div>
                            
                            ${note ? `
                                <div class="border-t border-gray-700 pt-2">
                                    <div class="text-sm text-gray-400 mb-1">یادداشت:</div>
                                    <div class="text-sm text-gray-300 bg-gray-800/50 p-2 rounded whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(note))}</div>
                                </div>
                            ` : ''}
                            
                            ${resultNote ? `
                                <div class="border-t border-gray-700 pt-2">
                                    <div class="text-sm text-gray-400 mb-1">نتیجه:</div>
                                    <div class="text-sm text-gray-300 bg-gray-800/50 p-2 rounded whitespace-pre-wrap">${this.escapeHtml(this.stripHtml(resultNote))}</div>
                                </div>
                            ` : ''}
                            
                            ${!note && !resultNote ? `
                                <div class="text-sm text-gray-500 italic text-center py-2">بدون توضیحات</div>
                            ` : ''}
                        </div>
                    `,
                    width: '600px',
                    confirmButtonText: 'بستن',
                    customClass: {
                        popup: 'swal2-popup-dark',
                        htmlContainer: 'text-right'
                    }
                });
            } else {
                Swal.fire('خطا', res.message || 'فعالیت یافت نشد', 'error');
            }
        } catch (error) {
            console.error('Error loading activity details:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در دریافت جزئیات فعالیت: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    // MANUAL POLLING TEST (Admin only)
    async testPolling() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند polling را تست کند', 'error');
            return;
        }

        try {
            Swal.fire({
                title: 'در حال چک کردن یادآوری‌ها...',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            const res = await this.req('check_reminders', {});

            Swal.close();

            if (res.status === 'success') {
                let message = 'سیستم polling چک شد\n';
                message += `🔔 نوتیفیکیشن‌های جدید: ${res.new_notifications ? res.new_notifications.length : 0}\n`;
                message += `🗯️ پاپ‌آپ نوتیفیکیشن‌ها: ${res.popup_notifications ? res.popup_notifications.length : 0}\n`;
                message += `📊 تعداد نوتیفیکیشن‌های خوانده نشده: ${res.unread_count || 0}`;

                Swal.fire({
                    icon: 'info',
                    title: 'نتیجه تست polling',
                    text: message,
                    confirmButtonText: 'باشه'
                });

                // Show any popup notifications found
                if (res.popup_notifications && res.popup_notifications.length > 0) {
                    res.popup_notifications.forEach(notif => {
                        this.showNotificationPopup(notif);
                    });
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در چک کردن یادآوری‌ها', 'error');
            }
        } catch (error) {
            console.error('Error testing polling:', error);
            Swal.fire('خطا', 'خطا در تست polling', 'error');
        }
    },

    // TEST POPUP DIRECTLY (Admin only)
    testPopupDirect() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند پاپ‌آپ را تست کند', 'error');
            return;
        }

        console.log('Testing popup directly');
        const testNotif = {
            id: 'test-' + Date.now(),
            title: 'تست کامل نوتیفیکیشن',
            message: 'این تست شامل پاپ‌آپ، صدا و نوتیفیکیشن مرورگر است',
            type: 'test',
            show_popup: 1,
            created_at: new Date().toISOString()
        };
        console.log('Test notification object:', testNotif);

        // Test all notification methods
        this.showNotificationPopup(testNotif);
        this.showBrowserNotification(testNotif);
        this.playNotificationSound();
    },

    // TEST BROWSER NOTIFICATION ONLY (Admin only)
    testBrowserNotification() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند نوتیفیکیشن مرورگر را تست کند', 'error');
            return;
        }

        console.log('Testing browser notification only');
        const testNotif = {
            id: 'browser-test-' + Date.now(),
            title: 'تست نوتیفیکیشن مرورگر',
            message: 'این نوتیفیکیشن باید روی دسکتاپ نمایش داده شود',
            type: 'browser_test',
            created_at: new Date().toISOString()
        };

        this.showBrowserNotification(testNotif);
    },

    // TEST NOTIFICATION (Admin only)
    async testNotification() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند اعلان را تست کند', 'error');
            return;
        }
        
        try {
            Swal.fire({
                title: 'در حال ایجاد اعلان تستی...',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            const res = await this.req('test_notification', {});
            
            if (res.status === 'success') {
                Swal.fire({
                    icon: 'success',
                    title: 'موفق',
                    text: 'اعلان تستی با موفقیت ایجاد شد. پاپ‌آپ را بررسی کنید.',
                    confirmButtonText: 'باشه'
                });

                // Refresh notifications and show popup
                await this.loadNotifications();

                // Show popup notification immediately for testing
                if (res.notification) {
                    const testNotif = {
                        id: res.notification.id,
                        title: res.notification.title,
                        message: res.notification.message,
                        type: res.notification.type,
                        entity_type: res.notification.entity_type,
                        entity_id: res.notification.entity_id,
                        created_at: res.notification.created_at,
                        show_popup: 1
                    };

                    // Show popup
                    this.showNotificationPopup(testNotif);

                    // Show browser notification
                    this.showBrowserNotification(testNotif);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در ایجاد اعلان تستی', 'error');
            }
        } catch (error) {
            console.error('Error testing notification:', error);
            Swal.fire('خطا', 'خطا در ایجاد اعلان تستی: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },
    
    // SYNC DIDAR USERS (Admin only)
    async syncDidarUsers() {
        if (!this.isAdmin()) {
            Swal.fire('خطا', 'فقط مدیر سیستم می‌تواند کاربران را همگام‌سازی کند', 'error');
            return;
        }
        
        try {
            const result = await Swal.fire({
                title: 'همگام‌سازی کاربران',
                text: 'آیا می‌خواهید کاربران از Didar همگام‌سازی کنید؟ این عملیات باعث بروزرسانی شناسه‌های کاربران می‌شود.',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'بله، همگام‌سازی کن',
                cancelButtonText: 'انصراف'
            });
            
            if (!result.isConfirmed) return;
            
            Swal.fire({
                title: 'در حال همگام‌سازی...',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            
            const res = await this.req('sync_didar_users', {});
            
            if (res.status === 'success') {
                Swal.fire({
                    icon: 'success',
                    title: 'موفق',
                    html: `
                        <div class="text-right">
                            <p>کاربران با موفقیت همگام‌سازی شدند</p>
                            <div class="mt-3 space-y-1 text-sm">
                                <p>جمع کل: <strong>${res.total_didar_users}</strong></p>
                                <p>به‌روزرسانی شده: <strong class="text-green-600">${res.updated}</strong></p>
                                <p>یافت نشده: <strong class="text-yellow-600">${res.not_found}</strong></p>
                            </div>
                        </div>
                    `,
                    confirmButtonText: 'باشه'
                });
            } else {
                Swal.fire('خطا', res.message || 'خطا در همگام‌سازی کاربران', 'error');
            }
        } catch (error) {
            console.error('Error syncing Didar users:', error);
            Swal.fire('خطا', 'خطا در همگام‌سازی کاربران: ' + (error.message || 'خطای نامشخص'), 'error');
        }
    },

    // IMPORT LEADS FUNCTIONS
    importLeadsState: {
        file: null,
        preview: null,
        columnMapping: null
    },

    openImportLeadsModal() {
        if (!this.isAdminOrCrmSpecialist()) {
            Swal.fire('خطا', 'فقط مدیر و کارشناس CRM می‌توانند لیدها را وارد کنند', 'error');
            return;
        }
        this.openModal('importLeadsModal');
        document.getElementById('import-step-1').classList.remove('hidden');
        document.getElementById('import-step-2').classList.add('hidden');
        document.getElementById('import-step-3').classList.add('hidden');
        document.getElementById('import-btn-preview').classList.add('hidden');
        document.getElementById('import-btn-execute').classList.add('hidden');
        document.getElementById('import-btn-done').classList.add('hidden');
        document.getElementById('import-file-selected').classList.add('hidden');
        document.getElementById('import-file-input').value = '';
        this.importLeadsState = { file: null, preview: null, columnMapping: null };
    },

    handleImportFileSelect(event) {
        const file = event.target.files?.[0];
        if (!file) return;

        const allowedTypes = [
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-excel',
            'text/csv'
        ];

        if (!allowedTypes.includes(file.type)) {
            Swal.fire('خطا', 'فقط فایل‌های Excel یا CSV قابل قبول هستند', 'error');
            return;
        }

        if (file.size > 5 * 1024 * 1024) {
            Swal.fire('خطا', 'حجم فایل نباید بیشتر از 5 مگابایت باشد', 'error');
            return;
        }

        this.importLeadsState.file = file;
        document.getElementById('import-file-name').textContent = file.name;
        document.getElementById('import-file-selected').classList.remove('hidden');
        document.getElementById('import-btn-preview').classList.remove('hidden');

        Swal.fire({
            toast: true,
            position: 'bottom-end',
            icon: 'success',
            title: 'فایل انتخاب شد',
            timer: 2000,
            showConfirmButton: false
        });
    },

    async previewImportLeads() {
        if (!this.importLeadsState.file) {
            Swal.fire('خطا', 'فایلی انتخاب نشده است', 'error');
            return;
        }

        const formData = new FormData();
        formData.append('file', this.importLeadsState.file);

        Swal.fire({
            title: 'در حال بررسی فایل...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            // Add action parameter
            const url = new URL(window.location.href);
            url.searchParams.set('action', 'import_leads_preview');

            const res = await fetch(url.toString(), {
                method: 'POST',
                body: formData
            });

            const data = await res.json();
            Swal.close();

            if (data.status !== 'success') {
                Swal.fire('خطا', data.message || 'خطای نامشخص', 'error');
                return;
            }

            this.importLeadsState.preview = data;
            this.renderImportPreview(data);

            document.getElementById('import-step-1').classList.add('hidden');
            document.getElementById('import-step-2').classList.remove('hidden');
            document.getElementById('import-btn-preview').classList.add('hidden');
            document.getElementById('import-btn-execute').classList.remove('hidden');

        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در بررسی فایل: ' + error.message, 'error');
        }
    },

    renderImportPreview(data) {
        document.getElementById('import-preview-success').textContent = data.valid_count || 0;
        document.getElementById('import-preview-error').textContent = data.error_count || 0;

        const tbody = document.getElementById('import-preview-table');
        tbody.innerHTML = (data.preview || []).map((row) => {
            const hasError = row.errors && row.errors.length > 0;
            return `
                <tr class="${hasError ? 'bg-red-900/20' : 'bg-green-900/20'}">
                    <td class="border border-gray-600 p-2">${row.row_number}</td>
                    <td class="border border-gray-600 p-2">${this.escapeHtml(row.lead_id)}</td>
                    <td class="border border-gray-600 p-2 text-xs">${this.escapeHtml(row.owner_name.substring(0, 15))}</td>
                    <td class="border border-gray-600 p-2 text-xs">
                        ${hasError ? `<span class="text-red-400">${this.escapeHtml(row.errors[0])}</span>` : '<span class="text-green-400">✓</span>'}
                    </td>
                </tr>
            `;
        }).join('');
    },

    async executeImportLeads() {
        if (!this.importLeadsState.file) {
            Swal.fire('خطا', 'فایلی انتخاب نشده است', 'error');
            return;
        }

        const result = await Swal.fire({
            title: 'آیا مطمئن هستید؟',
            text: `${this.importLeadsState.preview?.valid_count || 0} لید وارد خواهد شد`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'بله، ادامه بده',
            cancelButtonText: 'لغو'
        });

        if (!result.isConfirmed) return;

        const formData = new FormData();
        formData.append('file', this.importLeadsState.file);

        Swal.fire({
            title: 'در حال وارد کردن لیدها...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });

        try {
            const url = new URL(window.location.href);
            url.searchParams.set('action', 'import_leads_execute');

            const res = await fetch(url.toString(), {
                method: 'POST',
                body: formData
            });

            const data = await res.json();
            Swal.close();

            if (data.status !== 'success') {
                Swal.fire('خطا', data.message || 'خطای نامشخص', 'error');
                return;
            }

            this.renderImportResults(data);

            document.getElementById('import-step-2').classList.add('hidden');
            document.getElementById('import-step-3').classList.remove('hidden');
            document.getElementById('import-btn-execute').classList.add('hidden');
            document.getElementById('import-btn-done').classList.remove('hidden');

        } catch (error) {
            Swal.close();
            Swal.fire('خطا', 'خطا در وارد کردن لیدها: ' + error.message, 'error');
        }
    },

    renderImportResults(data) {
        document.getElementById('import-result-success').textContent = data.success_count || 0;
        document.getElementById('import-result-error').textContent = data.failure_count || 0;

        const resultsDiv = document.getElementById('import-result-table');
        resultsDiv.innerHTML = (data.results || []).map(r => {
            const isSuccess = r.status === 'success';
            return `
                <div class="p-2 rounded ${isSuccess ? 'bg-green-900/20 border-l-2 border-green-600' : 'bg-red-900/20 border-l-2 border-red-600'}">
                    <p class="text-xs font-mono text-gray-300">ردیف ${r.row}</p>
                    <p class="text-xs ${isSuccess ? 'text-green-300' : 'text-red-300'}">${this.escapeHtml(r.message)}</p>
                </div>
            `;
        }).join('');
    },

    // FIELD CONFIGURATION MANAGER (Admin)
    switchLookupTab(tab) {
        // Hide all tabs
        document.querySelectorAll('.lookup-tab-content').forEach(el => el.classList.add('hidden'));
        document.querySelectorAll('.lookup-tab').forEach(el => el.classList.remove('active', 'border-blue-500'));

        // Show selected tab
        const tabEl = document.getElementById(`lookup-${tab}-tab`);
        if (tabEl) {
            tabEl.classList.remove('hidden');
        }

        // Mark button as active
        const btn = document.querySelector(`.lookup-tab[data-tab="${tab}"]`);
        if (btn) {
            btn.classList.add('active', 'border-blue-500');
            btn.classList.remove('border-transparent');
        }

        // Load data if needed
        if (tab === 'fields') {
            this.loadFieldConfigurations();
        }
    },

    async loadFieldConfigurations(forceRefresh = false) {
        try {
            const res = await this.req('get_field_configurations');
            if (res.status === 'success') {
                this.state.fieldConfigurations = res.configurations || [];
                this.renderFieldConfigurationList();
                this.populateFieldConfigSelect();
            }
        } catch (error) {
            console.error('Error loading field configurations:', error);
            Swal.fire('خطا', 'خطا در بارگذاری تنظیمات فیلدها', 'error');
        }
    },

    renderFieldConfigurationList() {
        const container = document.getElementById('field-config-list');
        if (!container) return;

        container.innerHTML = (this.state.fieldConfigurations || []).map(config => `
            <div class="bg-gray-800/30 p-3 rounded border border-gray-700/50 hover:border-gray-600 cursor-pointer transition" onclick="app.editFieldConfiguration('${config.field_name}')">
                <div class="flex justify-between items-start">
                    <div class="flex-1">
                        <h4 class="font-medium text-white text-sm">${this.escapeHtml(config.field_label)}</h4>
                        <p class="text-xs text-gray-400 mt-1">
                            <span class="bg-gray-700/50 px-2 py-1 rounded">${config.field_name}</span>
                            <span class="ml-2 text-gray-500">نوع: ${config.field_type}</span>
                        </p>
                        <p class="text-xs text-gray-500 mt-1">
                            ${config.is_required ? '✓ الزامی' : '○ اختیاری'}
                            ${config.is_editable ? ' | ✓ قابل ویرایش' : ' | ✗ فقط خواندنی'}
                            ${config.is_active ? ' | ✓ فعال' : ' | ✗ غیرفعال'}
                        </p>
                    </div>
                    <button onclick="event.stopPropagation(); app.deleteFieldConfiguration('${config.field_name}')" class="text-red-400 hover:text-red-300 text-xs p-2">
                        <i class="fa-solid fa-trash"></i>
                    </button>
                </div>
            </div>
        `).join('');
    },

    populateFieldConfigSelect() {
        const select = document.getElementById('field-config-select');
        if (!select) return;

        select.innerHTML = '<option value="">-- انتخاب فیلد --</option>' +
            (this.state.fieldConfigurations || []).map(config =>
                `<option value="${config.field_name}">${this.escapeHtml(config.field_label)}</option>`
            ).join('');
    },

    onFieldConfigSelect() {
        const fieldName = document.getElementById('field-config-select')?.value;
        if (!fieldName) {
            document.getElementById('field-config-editor').classList.add('hidden');
            return;
        }

        this.editFieldConfiguration(fieldName);
    },

    editFieldConfiguration(fieldName) {
        const config = (this.state.fieldConfigurations || []).find(c => c.field_name === fieldName);
        if (!config) return;

        // Populate form
        document.getElementById('fc-field-name').value = config.field_name;
        document.getElementById('fc-field-label').value = config.field_label || '';
        document.getElementById('fc-field-type').value = config.field_type || 'text';
        document.getElementById('fc-entity-type').value = config.entity_type || 'deal';
        document.getElementById('fc-lookup-group').value = config.lookup_group_code || '';
        document.getElementById('fc-sort-order').value = config.sort_order || 0;
        document.getElementById('fc-is-required').checked = config.is_required === 1;
        document.getElementById('fc-is-editable').checked = config.is_editable === 1;
        document.getElementById('fc-is-active').checked = config.is_active === 1;

        // Set roles
        const roles = config.edit_allow_roles || [];
        document.querySelectorAll('#fc-edit-allow-roles option').forEach(opt => {
            opt.selected = roles.includes(opt.value);
        });

        // Show editor
        document.getElementById('field-config-editor').classList.remove('hidden');
        document.getElementById('field-config-select').value = fieldName;
    },

    resetFieldConfigEditor() {
        document.getElementById('field-config-editor').classList.add('hidden');
        document.getElementById('field-config-select').value = '';
        document.getElementById('fc-field-name').value = '';
        document.getElementById('fc-field-label').value = '';
        document.getElementById('fc-is-required').checked = false;
        document.getElementById('fc-is-editable').checked = true;
        document.getElementById('fc-is-active').checked = true;
    },

    async saveFieldConfiguration() {
        const fieldName = document.getElementById('fc-field-name').value;
        if (!fieldName) {
            Swal.fire('خطا', 'نام فیلد الزامی است', 'error');
            return;
        }

        // Get selected roles
        const selectedRoles = Array.from(
            document.querySelectorAll('#fc-edit-allow-roles option:checked')
        ).map(opt => opt.value);

        const data = {
            field_name: fieldName,
            field_label: document.getElementById('fc-field-label').value,
            field_type: document.getElementById('fc-field-type').value,
            entity_type: document.getElementById('fc-entity-type').value,
            is_required: document.getElementById('fc-is-required').checked ? 1 : 0,
            is_editable: document.getElementById('fc-is-editable').checked ? 1 : 0,
            is_active: document.getElementById('fc-is-active').checked ? 1 : 0,
            edit_allow_roles: JSON.stringify(selectedRoles),
            lookup_group_code: document.getElementById('fc-lookup-group').value || null,
            sort_order: parseInt(document.getElementById('fc-sort-order').value) || 0
        };

        try {
            const res = await this.req('save_field_configuration', data);
            if (res.status === 'success') {
                Swal.fire('موفق', 'تنظیمات فیلد ذخیره شد', 'success');
                this.loadFieldConfigurations(true);
                this.resetFieldConfigEditor();
            } else {
                Swal.fire('خطا', res.message || 'خطا در ذخیره‌سازی', 'error');
            }
        } catch (error) {
            console.error('Error saving field configuration:', error);
            Swal.fire('خطا', 'خطا در ذخیره‌سازی تنظیمات: ' + error.message, 'error');
        }
    },

    async deleteFieldConfiguration(fieldName) {
        const confirm = await Swal.fire({
            title: 'حذف تنظیمات فیلد',
            text: `آیا مطمئن هستید که می‌خواهید ${fieldName} را حذف کنید؟`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'بله، حذف کن',
            cancelButtonText: 'خیر'
        });

        if (!confirm.isConfirmed) return;

        try {
            const res = await this.req('delete_field_configuration', { field_name: fieldName });
            if (res.status === 'success') {
                Swal.fire('موفق', 'تنظیمات فیلد حذف شد', 'success');
                this.loadFieldConfigurations(true);
                this.resetFieldConfigEditor();
            } else {
                Swal.fire('خطا', res.message || 'خطا در حذف', 'error');
            }
        } catch (error) {
            console.error('Error deleting field configuration:', error);
            Swal.fire('خطا', 'خطا در حذف: ' + error.message, 'error');
        }
    },

    // ========== KPI DASHBOARD ==========
    async showKPIDashboard() {
        this.openModal('modalKPI');
        if (!this.state.kpiFilters.date_from) {
            const range = this.getJalaliMonthRange();
            this.state.kpiFilters.date_from = range.date_from;
        }
        if (!this.state.kpiFilters.date_to) {
            const range = this.getJalaliMonthRange();
            this.state.kpiFilters.date_to = range.date_to;
        }
        await this.loadKpiFilterData();
        await this.loadKPIDashboard({ useState: true });
    },

    initKpiDatePickers() {
        console.log('[DEBUG] Initializing KPI datepickers...');
        console.log('[DEBUG] jQuery available:', typeof $ !== 'undefined');
        console.log('[DEBUG] $.fn available:', typeof $?.fn !== 'undefined');
        console.log('[DEBUG] $.fn.persianDatepicker available:', typeof $.fn?.persianDatepicker === 'function');

        // Wait for elements to be rendered
        const self = this;
        setTimeout(function() {
            if (typeof $ === 'undefined') {
                console.warn('[DEBUG] jQuery not available for KPI datepicker');
                return;
            }

            const $dateFrom = $('#kpi-date-from');
            const $dateTo = $('#kpi-date-to');

            console.log('[DEBUG] $dateFrom element:', $dateFrom.length > 0 ? 'found' : 'not found');
            console.log('[DEBUG] $dateTo element:', $dateTo.length > 0 ? 'found' : 'not found');

            if ($dateFrom.length > 0 && typeof $.fn?.persianDatepicker === 'function') {
                try {
                    $dateFrom.persianDatepicker('destroy');
                } catch (error) {
                    // Ignore if datepicker was not initialized yet.
                }
                $dateFrom.persianDatepicker({
                    format: 'YYYY/MM/DD',
                    initialValue: false,
                    autoClose: true,
                    observer: true
                });
                console.log('[DEBUG] KPI Date From datepicker initialized');
            } else {
                console.warn('[DEBUG] KPI Date From datepicker initialization failed', {
                    exists: $dateFrom.length > 0,
                    hasPersianDatepicker: typeof $.fn?.persianDatepicker === 'function'
                });
            }

            if ($dateTo.length > 0 && typeof $.fn?.persianDatepicker === 'function') {
                try {
                    $dateTo.persianDatepicker('destroy');
                } catch (error) {
                    // Ignore if datepicker was not initialized yet.
                }
                $dateTo.persianDatepicker({
                    format: 'YYYY/MM/DD',
                    initialValue: false,
                    autoClose: true,
                    observer: true
                });
                console.log('[DEBUG] KPI Date To datepicker initialized');
            } else {
                console.warn('[DEBUG] KPI Date To datepicker initialization failed', {
                    exists: $dateTo.length > 0,
                    hasPersianDatepicker: typeof $.fn?.persianDatepicker === 'function'
                });
            }

            const filters = self.state.kpiFilters || {};
            if (filters.date_from) {
                $dateFrom.val(toPersianDate(filters.date_from) || filters.date_from);
            }
            if (filters.date_to) {
                $dateTo.val(toPersianDate(filters.date_to) || filters.date_to);
            }
        }, 1000);
    },

    async loadKpiFilterData() {
        // Load users for owner filter
        if (this.isAdmin()) {
            try {
                const usersRes = await this.req('get_didar_users');
                if (usersRes.status === 'success') {
                    this.state.kpiUsers = (usersRes.users || []).map(user => ({
                        didar_user_id: user.owner_id || user.didar_user_id || '',
                        display_name: user.display_name || user.email || '',
                        email: user.email || '',
                        is_active: user.is_active ?? 1
                    }));
                } else {
                    console.warn('Failed to load users:', usersRes.message);
                    this.state.kpiUsers = [];
                }
            } catch (error) {
                console.error('Error loading users:', error);
                this.state.kpiUsers = [];
            }

            if (this.state.kpiUsers.length === 0) {
                try {
                    const usersRes = await this.req('get_users_list');
                    if (usersRes.status === 'success') {
                        this.state.kpiUsers = usersRes.users || [];
                    } else {
                        console.warn('Failed to load users:', usersRes.message);
                        this.state.kpiUsers = [];
                    }
                } catch (error) {
                    console.error('Error loading users:', error);
                    this.state.kpiUsers = [];
                }
            }
        } else {
            this.state.kpiUsers = [{
                didar_user_id: this.state.user?.didar_id || '',
                display_name: this.state.user?.name || '',
                is_active: 1
            }];
        }
        this.populateKpiOwnerFilter();

        // Load products for product filter
        try {
            const productsRes = await this.req('get_kpi_products');
            if (productsRes.status === 'success') {
                this.state.kpiProducts = productsRes.products || [];
                this.populateKpiProductFilter();
            } else {
                console.warn('Failed to load products:', productsRes.message);
                this.state.kpiProducts = [];
                this.populateKpiProductFilter();
            }
        } catch (error) {
            console.error('Error loading products:', error);
            this.state.kpiProducts = [];
            this.populateKpiProductFilter();
        }
    },

    populateKpiOwnerFilter() {
        const select = document.getElementById('kpi-owner-filter');
        if (!select) return;

        const currentValue = this.state.kpiFilters?.owner_didar_id || select.value;
        select.innerHTML = '<option value="">همه</option>';

        // فقط کاربران active را فیلتر کن
        const activeUsers = this.state.kpiUsers.filter(user => Number(user.is_active) === 1);

        activeUsers.forEach(user => {
            const opt = document.createElement('option');
            opt.value = user.didar_user_id || user.owner_id || '';
            const name = user.display_name || user.email || (user.first_name + ' ' + user.last_name);
            opt.textContent = name;
            select.appendChild(opt);
        });

        select.value = currentValue;
    },

    populateKpiProductFilter() {
        const select = document.getElementById('kpi-product-filter');
        if (!select) return;

        const currentValue = this.state.kpiFilters?.product_id || select.value;
        select.innerHTML = '<option value="">همه</option>';

        this.state.kpiProducts.forEach(product => {
            const opt = document.createElement('option');
            opt.value = product.id;
            opt.textContent = product.name;
            select.appendChild(opt);
        });

        select.value = currentValue;
    },

    async loadKPIDashboard(options = {}) {
        const useState = options.useState === true;
        const stateFilters = this.state.kpiFilters || {};
        const defaultRange = this.getJalaliMonthRange();
        let dateFrom = stateFilters.date_from || defaultRange.date_from;
        let dateTo = stateFilters.date_to || defaultRange.date_to;
        let ownerId = stateFilters.owner_didar_id || '';
        let productId = stateFilters.product_id || '';

        if (!useState) {
            const dateFromJ = (document.getElementById('kpi-date-from')?.value || '').trim();
            const dateToJ = (document.getElementById('kpi-date-to')?.value || '').trim();
            const ownerRaw = document.getElementById('kpi-owner-filter')?.value;
            const productRaw = document.getElementById('kpi-product-filter')?.value;

            if (dateFromJ) {
                const dateFromG = persianToGregorian(this.toEnglishDigits(dateFromJ), '00:00');
                if (dateFromG) {
                    dateFrom = String(dateFromG).split(' ')[0];
                } else if (/^\d{4}-\d{2}-\d{2}$/.test(dateFromJ)) {
                    dateFrom = dateFromJ;
                }
            }

            if (dateToJ) {
                const dateToG = persianToGregorian(this.toEnglishDigits(dateToJ), '23:59');
                if (dateToG) {
                    dateTo = String(dateToG).split(' ')[0];
                } else if (/^\d{4}-\d{2}-\d{2}$/.test(dateToJ)) {
                    dateTo = dateToJ;
                }
            }

            if (typeof ownerRaw === 'string') {
                ownerId = ownerRaw.trim();
            }
            if (typeof productRaw === 'string') {
                productId = productRaw.trim();
            }
        }

        this.state.kpiFilters = {
            date_from: dateFrom,
            date_to: dateTo,
            owner_didar_id: ownerId,
            product_id: productId
        };

        const res = await this.req('get_kpi', {
            date_from: dateFrom,
            date_to: dateTo,
            owner_didar_id: ownerId,
            product_id: productId
        });

        if (res.status === 'success') {
            this.renderKPIDashboard(res.kpi);
        } else {
            Swal.fire('خطا', res.message || 'خطا در بارگذاری KPI', 'error');
        }
    },

    renderKPIDashboard(kpi) {
        const content = document.getElementById('kpi-dashboard-content');
        if (!content) return;

        const filters = this.state.kpiFilters || {};
        const sales = kpi.sales || {};
        const settled = kpi.settled || {};
        const outstanding = kpi.outstanding || {};
        const lost = kpi.lost_sales || [];
        const openPipeline = kpi.open_pipeline || {};
        const dealsWithoutPayments = kpi.deals_without_payments || {};
        const productSales = kpi.product_sales || [];
        const dealConversion = kpi.deal_conversion || {};
        const leadConversion = kpi.lead_conversion || {};
        const timeToConversion = kpi.time_to_conversion || {};
        const target = kpi.target_achievement || {};

        // Calculate percentages for progress bar
        const achievementPercent = target.achievement_percent || 0;
        const progressColor = achievementPercent >= 100 ? 'bg-green-500' :
                             achievementPercent >= 75 ? 'bg-blue-500' :
                             achievementPercent >= 50 ? 'bg-yellow-500' : 'bg-red-500';

        // Format Toman amount (already in Toman, no conversion needed)
        const formatToman = (amount) => {
            const toman = parseFloat(amount) || 0;
            return this.formatNumber(toman);
        };

        const dateFromJalali = filters.date_from ? (toPersianDate(filters.date_from) || filters.date_from) : '';
        const dateToJalali = filters.date_to ? (toPersianDate(filters.date_to) || filters.date_to) : '';

        content.innerHTML = `
            <!-- Header Filters -->
            <div class="glass p-6 mb-6 sticky top-0 z-10">
                <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">از تاریخ</label>
                        <input type="text" id="kpi-date-from" class="w-full bg-gray-800 border border-gray-700 rounded p-2 text-white"
                               value="${dateFromJalali}">
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">تا تاریخ</label>
                        <input type="text" id="kpi-date-to" class="w-full bg-gray-800 border border-gray-700 rounded p-2 text-white"
                               value="${dateToJalali}">
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">مسئول</label>
                        <select id="kpi-owner-filter" class="w-full bg-gray-800 border border-gray-700 rounded p-2 text-white">
                            <option value="">همه</option>
                            ${this.getKpiOwnerOptions(filters.owner_didar_id)}
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">محصول</label>
                        <select id="kpi-product-filter" class="w-full bg-gray-800 border border-gray-700 rounded p-2 text-white">
                            <option value="">همه</option>
                            ${this.getKpiProductOptions(filters.product_id)}
                        </select>
                    </div>
                </div>
                <div class="mt-4 flex justify-end">
                    <button onclick="app.loadKPIDashboard()" class="bg-blue-600 hover:bg-blue-500 text-white px-6 py-2 rounded-lg font-bold transition">
                        <i class="fa-solid fa-search ml-2"></i> اعمال فیلتر
                    </button>
                </div>
            </div>

            <!-- KPI Cards Row 1 -->
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                <!-- Sales Card -->
                <div class="glass p-6 border-r-4 border-blue-500">
                    <div class="text-sm text-gray-400 mb-2">فروش کل</div>
                    <div class="text-3xl font-bold text-white mb-2">${formatToman(sales.total_collected_amount || 0)} تومان</div>
                    <div class="text-sm text-gray-400">تعداد معامله: ${sales.deal_count || 0}</div>
                    <div class="text-xs text-gray-500 mt-2">مبلغ قرارداد: ${formatToman(sales.total_contract_amount || 0)} تومان</div>
                </div>

                <!-- Target Achievement Card -->
                <div class="glass p-6 border-r-4 ${progressColor}">
                    <div class="text-sm text-gray-400 mb-2">درصد تحقق تارگت</div>
                    <div class="text-3xl font-bold text-white mb-2">${achievementPercent}%</div>
                    ${target.has_target ? `
                        <div class="w-full bg-gray-700 rounded-full h-2.5 mb-2">
                            <div class="${progressColor} h-2.5 rounded-full" style="width: ${Math.min(achievementPercent, 100)}%"></div>
                        </div>
                        <div class="text-xs text-gray-400">
                            هدف: ${formatToman(target.target_amount)} تومان | واقعی: ${formatToman(target.actual_amount)} تومان
                        </div>
                    ` : `
                        <div class="text-xs text-gray-400">هدف تعریف نشده</div>
                    `}
                </div>

                <!-- Outstanding Card -->
                <div class="glass p-6 border-r-4 border-red-500">
                    <div class="text-sm text-gray-400 mb-2">مانده حساب</div>
                    <div class="text-3xl font-bold text-red-400 mb-2">${formatToman(outstanding.outstanding_amount || 0)} تومان</div>
                    <div class="text-sm text-gray-400">تعداد معامله بدهکار: ${outstanding.deal_count || 0}</div>
                    <div class="text-xs text-gray-500 mt-2">قرارداد: ${formatToman(outstanding.total_contract_amount || 0)} تومان | وصول شده: ${formatToman(outstanding.total_collected || 0)} تومان</div>
                </div>
            </div>

            <!-- KPI Cards Row 2 -->
            <div class="grid grid-cols-1 md:grid-cols-5 gap-4 mb-6">
                <div class="glass p-4 border-r-4 border-green-500">
                    <div class="text-sm text-gray-400">تسویه شده کامل</div>
                    <div class="text-xl font-bold text-white">${settled.deal_count || 0}</div>
                    <div class="text-xs text-gray-500 mt-1">مبلغ کل: ${formatToman(settled.total_amount || 0)} تومان</div>
                    <div class="text-xs text-gray-500">وصول شده: ${formatToman(settled.total_collected_amount || 0)} تومان</div>
                </div>
                <div class="glass p-4 border-r-4 border-yellow-500">
                    <div class="text-sm text-gray-400">فروش ناموفق</div>
                    <div class="text-xl font-bold text-white">${this.sumLostSales(lost)}</div>
                    <div class="text-xs text-gray-500 mt-1">مبلغ کل: ${formatToman(this.sumLostSalesAmount(lost))} تومان</div>
                </div>
                <div class="glass p-4 border-r-4 border-purple-500">
                    <div class="text-sm text-gray-400">فروش جاری</div>
                    <div class="text-xl font-bold text-white">${openPipeline.deal_count || 0}</div>
                    <div class="text-xs text-gray-500 mt-1">پتانسیل: ${formatToman(openPipeline.total_potential_amount || 0)} تومان</div>
                </div>
                <div class="glass p-4 border-r-4 border-pink-500">
                    <div class="text-sm text-gray-400">نرخ تبدیل معاملات</div>
                    <div class="text-xl font-bold text-white">${dealConversion.conversion_rate || 0}%</div>
                    <div class="text-xs text-gray-500 mt-1">از ${dealConversion.total_deals || 0} معامله</div>
                    <div class="text-xs text-gray-500">عدم تبدیل: ${dealConversion.non_conversion_rate || 0}%</div>
                </div>
                <div class="glass p-4 border-r-4 border-blue-500">
                    <div class="text-sm text-gray-400">نرخ تبدیل لید جدید</div>
                    <div class="text-xl font-bold text-white">${leadConversion.conversion_rate || 0}%</div>
                    <div class="text-xs text-gray-500 mt-1">از ${leadConversion.total_new_leads || 0} لید جدید</div>
                    <div class="text-xs text-gray-500">عدم تبدیل: ${leadConversion.non_conversion_rate || 0}%</div>
                </div>
            </div>

            <!-- Deals Without Payments Row -->
            ${dealsWithoutPayments.deal_count > 0 ? `
            <div class="glass p-6 mb-6 border-r-4 border-yellow-500">
                <div class="flex items-center justify-between mb-4">
                    <div>
                        <div class="text-lg font-bold text-yellow-400 mb-2">⚠️ معاملات بدون پرداخت</div>
                        <div class="text-sm text-gray-400">معاملاتی که مبلغ دارند اما هنوز پرداختی ثبت نشده</div>
                    </div>
                    <div class="text-right">
                        <div class="text-3xl font-bold text-yellow-400">${dealsWithoutPayments.deal_count || 0}</div>
                        <div class="text-sm text-gray-400">معامله</div>
                    </div>
                </div>
                <div class="mt-4 pt-4 border-t border-gray-700">
                    <div class="text-sm text-gray-400 mb-1">مجموع مبلغ قابل پرداخت:</div>
                    <div class="text-2xl font-bold text-white">${formatToman(dealsWithoutPayments.total_potential_amount || 0)} تومان</div>
                </div>
            </div>
            ` : ''}

            <!-- Product Sales Counts -->
            <div class="glass p-6 mb-6 border-r-4 border-indigo-500">
                <div class="flex items-center justify-between mb-4">
                    <div>
                        <div class="text-lg font-bold text-indigo-400 mb-1">فروش محصولات (بر اساس پرداخت)</div>
                        <div class="text-sm text-gray-400">تعداد معاملاتی که در بازه انتخاب‌شده حداقل یک پرداخت دارند</div>
                    </div>
                </div>
                ${productSales.length > 0 ? `
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                        ${productSales.map(item => `
                            <div class="bg-gray-800/50 p-4 rounded flex items-center justify-between">
                                <div class="text-sm text-gray-300">${item.product_name || '-'}</div>
                                <div class="text-xl font-bold text-white">${item.deal_count || 0}</div>
                            </div>
                        `).join('')}
                    </div>
                ` : `
                    <div class="text-sm text-gray-500">داده‌ای برای این بازه وجود ندارد.</div>
                `}
            </div>

            <!-- Charts Row -->
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                <!-- Daily Sales Chart -->
                <div class="glass p-6">
                    <h3 class="text-lg font-bold mb-4 text-yellow-500">فروش روزانه</h3>
                    <div id="daily-sales-chart" class="h-64"></div>
                    ${this.renderDailySalesChart(kpi.daily_sales || [])}
                </div>

                <!-- Lost Sales Reasons Chart -->
                <div class="glass p-6">
                    <h3 class="text-lg font-bold mb-4 text-yellow-500">دلایل شکست معاملات</h3>
                    <div id="lost-sales-chart" class="h-64"></div>
                    ${this.renderLostSalesChart(lost)}
                </div>
            </div>

            <!-- Lead Conversion Details -->
            <div class="glass p-6 mb-6">
                <h3 class="text-lg font-bold mb-4 text-yellow-500">جزئیات نرخ تبدیل لید جدید</h3>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="bg-gray-800/50 p-4 rounded">
                        <div class="text-sm text-gray-400">لیدهای جدید</div>
                        <div class="text-2xl font-bold text-white">${leadConversion.total_new_leads || 0}</div>
                    </div>
                    <div class="bg-gray-800/50 p-4 rounded">
                        <div class="text-sm text-gray-400">تبدیل شده</div>
                        <div class="text-2xl font-bold text-green-400">${leadConversion.converted_leads || 0}</div>
                    </div>
                    <div class="bg-gray-800/50 p-4 rounded">
                        <div class="text-sm text-gray-400">نرخ عدم تبدیل</div>
                        <div class="text-2xl font-bold text-red-400">${leadConversion.non_conversion_rate || 0}%</div>
                    </div>
                </div>
            </div>

            <!-- Time to Conversion -->
            <div class="glass p-6">
                <h3 class="text-lg font-bold mb-4 text-yellow-500">میانگین زمان تبدیل</h3>
                <div class="text-3xl font-bold text-white mb-2">
                    ${Math.round(timeToConversion.avg_conversion_days || 0)} روز
                </div>
                <div class="text-sm text-gray-400">
                    مبنای محاسبه: از ثبت معامله تا اولین پرداخت ثبت شده
                </div>
            </div>

            <!-- Deal Details Table -->
            <div class="glass p-6 mt-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-bold text-yellow-500">جزئیات معاملات</h3>
                    <button onclick="app.toggleKpiDealDetails()" class="bg-gray-700 hover:bg-gray-600 text-white px-4 py-2 rounded transition">
                        <i class="fa-solid fa-eye ml-2"></i> ${this.state.showKpiDealDetails ? 'مخفی کن' : 'نمایش'}
                    </button>
                </div>
                <div id="kpi-deal-details-table" class="${this.state.showKpiDealDetails ? '' : 'hidden'}">
                    <div class="overflow-x-auto">
                        <table class="w-full text-xs border-collapse">
                            <thead class="sticky top-0 bg-gray-700">
                                <tr>
                                    <th class="border border-gray-600 p-2 text-right">عنوان معامله</th>
                                    <th class="border border-gray-600 p-2 text-right">مسئول</th>
                                    <th class="border border-gray-600 p-2 text-right">وضعیت</th>
                                    <th class="border border-gray-600 p-2 text-right">مبلغ قرارداد</th>
                                    <th class="border border-gray-600 p-2 text-right">پرداخت شده</th>
                                    <th class="border border-gray-600 p-2 text-right">مانده</th>
                                    <th class="border border-gray-600 p-2 text-right">اولین پرداخت</th>
                                </tr>
                            </thead>
                            <tbody id="kpi-deals-table-body" class="bg-gray-900">
                                <tr class="text-center text-gray-500">
                                    <td colspan="8" class="py-8">در حال بارگذاری...</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        `;

        this.populateKpiOwnerFilter();
        this.populateKpiProductFilter();
        this.initKpiDatePickers();
    },

    toggleKpiDealDetails() {
        this.state.showKpiDealDetails = !this.state.showKpiDealDetails;
        if (this.state.showKpiDealDetails) {
            this.loadKpiDealDetails();
        }
    },

    async loadKpiDealDetails() {
        const tbody = document.getElementById('kpi-deals-table-body');
        if (!tbody) return;

        tbody.innerHTML = '<tr class="text-center"><td colspan="8" class="py-8">در حال بارگذاری...</td></tr>';

        const filters = this.state.kpiFilters || {};
        const dateFrom = filters.date_from || this.getFirstDayOfMonth();
        const dateTo = filters.date_to || this.getLastDayOfMonth();
        const ownerId = filters.owner_didar_id || '';
        const productId = filters.product_id || '';

        const res = await this.req('get_kpi_deals', {
            date_from: dateFrom,
            date_to: dateTo,
            owner_didar_id: ownerId,
            product_id: productId
        });

        if (res.status === 'success') {
            this.renderKpiDealDetailsTable(res.deals || []);
        } else {
            tbody.innerHTML = '<tr class="text-center text-red-400"><td colspan="8">خطا در بارگذاری: ' + (res.message || 'نامشخص') + '</td></tr>';
        }
    },

    renderKpiDealDetailsTable(deals) {
        const tbody = document.getElementById('kpi-deals-table-body');
        if (!tbody) return;

        if (!deals || deals.length === 0) {
            tbody.innerHTML = '<tr class="text-center text-gray-500"><td colspan="8">معامله‌ای در این بازه وجود ندارد</td></tr>';
            return;
        }

        const formatToman = (amount) => {
            const toman = parseFloat(amount) || 0;
            return this.formatNumber(toman);
        };

        tbody.innerHTML = deals.map((deal, index) => {
            const outstanding = (deal.payable_amount || 0) - (deal.collected_amount || 0);
            const firstPaymentDate = deal.first_payment_date ? deal.first_payment_date.slice(0, 10) : '-';

            return `
                <tr class="${index % 2 === 0 ? 'bg-gray-800' : 'bg-gray-900'}">
                    <td class="border border-gray-700 p-2 text-right">${deal.title || '-'}</td>
                    <td class="border border-gray-700 p-2 text-right">${deal.owner_name || '-'}</td>
                    <td class="border border-gray-700 p-2 text-right">
                        <span class="px-2 py-1 rounded text-xs ${
                            deal.status === 'Won' ? 'bg-green-600' : 
                            deal.status === 'Lost' ? 'bg-red-600' : 'bg-blue-600'
                        }">
                            ${deal.status === 'Won' ? 'موفق' : deal.status === 'Lost' ? 'ناموفق' : 'جاری'}
                        </span>
                    </td>
                    <td class="border border-gray-700 p-2 text-right">${formatToman(deal.payable_amount)}</td>
                    <td class="border border-gray-700 p-2 text-right text-green-400">${formatToman(deal.collected_amount || 0)}</td>
                    <td class="border border-gray-700 p-2 text-right ${outstanding > 0 ? 'text-red-400' : ''}">${formatToman(outstanding)}</td>
                    <td class="border border-gray-700 p-2 text-right">${firstPaymentDate}</td>
                </tr>
            `;
        }).join('');
    },

    renderDailySalesChart(dailySales) {
        if (!dailySales || dailySales.length === 0) {
            return '<div class="text-center text-gray-500 py-8">داده‌ای برای نمایش وجود ندارد</div>';
        }

        const amounts = dailySales.map(d => (parseFloat(d.collected_amount) || 0));
        const maxAmount = Math.max(...amounts, 0);
        const summary = dailySales.map(d => {
            const amount = (parseFloat(d.collected_amount) || 0);
            return `${d.date}: ${this.formatNumber(amount)} تومان`;
        }).join('<br>');

        return `
            <div class="flex items-end space-x-1 h-48 overflow-x-auto pb-2">
                ${dailySales.map(d => {
                    const amount = (parseFloat(d.collected_amount) || 0);
                    const height = maxAmount > 0 ? (amount / maxAmount * 100) : 0;
                    const displayAmount = this.formatNumber(amount);

                    return `
                        <div class="flex-shrink-0 flex flex-col items-center">
                            <div class="w-8 bg-blue-500 rounded-t hover:bg-blue-400 transition cursor-pointer"
                                 style="height: ${Math.max(height, 6)}%"
                                 title="${d.date}: ${displayAmount} تومان">
                            </div>
                            <div class="text-xs text-gray-500 mt-1 transform -rotate-45 origin-left">${d.date.slice(5)}</div>
                        </div>
                    `;
                }).join('')}
            </div>
            <div class="text-xs text-gray-400 mt-3 leading-6">${summary}</div>
        `;
    },

    renderLostSalesChart(lostSales) {
        if (!lostSales || lostSales.length === 0) {
            return '<div class="text-center text-gray-500 py-8">فروش ناموفقی در این بازه وجود ندارد</div>';
        }

        const total = lostSales.reduce((sum, item) => sum + Number(item.total_lost || 0), 0);

        return `
            <div class="space-y-3">
                ${lostSales.map(item => {
                    const count = item.total_lost || 0;
                    const percent = total > 0 ? (count / total * 100) : 0;
                    const reason = item.failure_reason_text || item.failure_reason_code || 'نامشخص';

                    return `
                        <div>
                            <div class="flex justify-between text-sm mb-1">
                                <span class="text-gray-300">${reason}</span>
                                <span class="text-white">${count} (${percent.toFixed(1)}%)</span>
                            </div>
                            <div class="w-full bg-gray-700 rounded-full h-2">
                                <div class="bg-red-500 h-2 rounded-full" style="width: ${percent}%"></div>
                            </div>
                        </div>
                    `;
                }).join('')}
            </div>
        `;
    },

    sumLostSales(lostSales) {
        if (!lostSales || lostSales.length === 0) return 0;
        return lostSales.reduce((sum, item) => sum + Number(item.total_lost || 0), 0);
    },

    sumLostSalesAmount(lostSales) {
        if (!lostSales || lostSales.length === 0) return 0;
        return lostSales.reduce((sum, item) => sum + (parseFloat(item.total_amount) || 0), 0);
    },

    getKpiOwnerOptions(selectedId) {
        let options = '';
        if (this.state.kpiUsers && this.state.kpiUsers.length > 0) {
            const activeUsers = this.state.kpiUsers.filter(user => Number(user.is_active) === 1);
            activeUsers.forEach(user => {
                const name = user.display_name || user.email || (user.first_name + ' ' + user.last_name);
                const didarId = user.didar_user_id || user.owner_id || '';
                const isSelected = String(didarId) === String(selectedId);
                options += `<option value="${didarId}" ${isSelected ? 'selected' : ''}>${name}</option>`;
            });
        }
        return options;
    },

    getKpiProductOptions(selectedId) {
        let options = '';
        if (this.state.kpiProducts && this.state.kpiProducts.length > 0) {
            this.state.kpiProducts.forEach(product => {
                options += `<option value="${product.id}" ${product.id == selectedId ? 'selected' : ''}>${product.name || 'محصول'}</option>`;
            });
        }
        return options;
    },

    getFirstDayOfMonth() {
        const now = new Date();
        return `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}-01`;
    },

    getLastDayOfMonth() {
        const now = new Date();
        return `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}-${new Date(now.getFullYear(), now.getMonth() + 1, 0).getDate()}`;
    },

    getJalaliMonthRange() {
        const fallback = {
            date_from: this.getFirstDayOfMonth(),
            date_to: this.getLastDayOfMonth()
        };
        if (typeof persianDate === 'undefined') {
            return fallback;
        }
        try {
            const start = new persianDate().startOf('month').toDate();
            const end = new persianDate().endOf('month').toDate();
            const format = (dateObj) => {
                const y = dateObj.getFullYear();
                const m = String(dateObj.getMonth() + 1).padStart(2, '0');
                const d = String(dateObj.getDate()).padStart(2, '0');
                return `${y}-${m}-${d}`;
            };
            return {
                date_from: format(start),
                date_to: format(end)
            };
        } catch (error) {
            console.warn('getJalaliMonthRange failed', error);
            return fallback;
        }
    },

    formatNumber(num) {
        if (num === null || num === undefined || num === '') return '0';
        return new Intl.NumberFormat('fa-IR').format(num);
    },
    
    // ============ SUPPORT MODULE ============
    
    /**
     * Open support modal
     */
    openSupportModal() {
        const contactId = window.currentContactId;
        if (!contactId) {
            Swal.fire('خطا', 'لطفاً ابتدا یک لید را انتخاب کنید', 'error');
            return;
        }
        
        // Reset form
        document.getElementById('support-contact-id').value = contactId;
        document.getElementById('support-id').value = '';
        document.getElementById('support-channel').value = '';
        document.getElementById('support-direction').value = '';
        document.getElementById('support-result-note').value = '';
        
        this.openModal('modalSupport');
    },
    
    /**
     * Save support log
     */
    async saveSupport(e) {
        e.preventDefault();
        
        const form = e.target;
        const data = new FormData(form);
        
        Swal.fire({
            title: 'در حال ثبت...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        try {
            const res = await this.req('create_support', Object.fromEntries(data));
            Swal.close();
            
            if (res.status === 'success') {
                Swal.fire({
                    toast: true,
                    position: 'bottom-end',
                    icon: 'success',
                    title: 'پشتیبانی با موفقیت ثبت شد',
                    showConfirmButton: false,
                    timer: 2000
                });
                
                this.closeModal('modalSupport');
                
                // Reload support logs if in support tab
                const contactId = data.get('contact_id');
                if (contactId) {
                    await this.loadSupportLogs(contactId);
                }
            } else {
                Swal.fire('خطا', res.message || 'خطا در ثبت پشتیبانی', 'error');
            }
        } catch (error) {
            console.error('Error saving support:', error);
            Swal.close();
            Swal.fire('خطا', 'خطا در ثبت پشتیبانی: ' + error.message, 'error');
        }
    },
    
    /**
     * Load support logs for a contact
     */
    async loadSupportLogs(contactId) {
        try {
            const res = await this.req('get_support_logs', { contact_id: contactId });
            
            if (res.status === 'success') {
                this.renderSupportLogs(res.support_logs || []);
            } else {
                console.error('Failed to load support logs:', res.message);
                document.getElementById('prof-support').innerHTML = 
                    '<div class="text-center text-red-500 py-4">خطا در بارگذاری پشتیبانی‌ها</div>';
            }
        } catch (error) {
            console.error('Error loading support logs:', error);
            document.getElementById('prof-support').innerHTML = 
                '<div class="text-center text-red-500 py-4">خطا در بارگذاری پشتیبانی‌ها</div>';
        }
    },
    
    /**
     * Render support logs as cards
     */
    renderSupportLogs(logs) {
        const container = document.getElementById('prof-support');
        
        if (!logs || logs.length === 0) {
            container.innerHTML = '<div class="text-center text-gray-500 py-4">هیچ پشتیبانی ثبت نشده است</div>';
            return;
        }
        
        const isAdmin = window.APP_USER?.role === 'admin';
        
        const channelLabels = {
            'call': 'تماس',
            'chat': 'چت',
            'visit': 'حضوری',
            'email': 'ایمیل'
        };
        
        const directionLabels = {
            'incoming': 'ورودی',
            'outgoing': 'خروجی'
        };
        
        const channelIcons = {
            'call': 'fa-phone',
            'chat': 'fa-comment',
            'visit': 'fa-user',
            'email': 'fa-envelope'
        };
        
        const html = logs.map(log => {
            const channelLabel = channelLabels[log.channel] || log.channel;
            const directionLabel = directionLabels[log.direction] || log.direction;
            const channelIcon = channelIcons[log.channel] || 'fa-circle';
            const createdAt = log.created_at ? toPersianDateTime(log.created_at) : '-';
            const creatorName = log.creator_name || 'نامشخص';
            
            return `
                <div class="glass p-4 rounded-lg border border-gray-700/50">
                    <div class="flex justify-between items-start mb-3">
                        <div class="flex items-center gap-3">
                            <div class="w-10 h-10 rounded-full bg-blue-600/20 flex items-center justify-center">
                                <i class="fas ${channelIcon} text-blue-400"></i>
                            </div>
                            <div>
                                <div class="font-bold text-white">${channelLabel} - ${directionLabel}</div>
                                <div class="text-xs text-gray-400 flex items-center gap-2 mt-1">
                                    <i class="fa-solid fa-user text-[10px]"></i>
                                    <span>${creatorName}</span>
                                    <span class="mx-1">•</span>
                                    <i class="fa-solid fa-clock text-[10px]"></i>
                                    <span>${createdAt}</span>
                                </div>
                            </div>
                        </div>
                        ${isAdmin ? `
                            <button onclick="app.editSupportLog(${log.id})" 
                                    class="text-gray-400 hover:text-yellow-500 transition px-2">
                                <i class="fa-solid fa-edit"></i>
                            </button>
                        ` : ''}
                    </div>
                    <div class="text-sm text-gray-300 whitespace-pre-line bg-gray-900/50 p-3 rounded">
                        ${this.escapeHtml(log.result_note || '')}
                    </div>
                </div>
            `;
        }).join('');
        
        container.innerHTML = html;
    },
    
    /**
     * Edit support log (admin only)
     */
    async editSupportLog(id) {
        // TODO: Implement edit functionality if needed
        Swal.fire('اطلاعات', 'قابلیت ویرایش به زودی اضافه می‌شود', 'info');
    },
    
    /**
     * Escape HTML to prevent XSS
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // ============ END SUPPORT MODULE ============
};

// Initialize on page load
// Note: User initialization is handled in the layout file
// This ensures APP_USER is set before app.init() is called
































