#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
آنالیز فایل اکسل فعالیت‌های دیدار
"""

import pandas as pd
import sys
from pathlib import Path

# تنظیم encoding برای Windows
if sys.platform == 'win32':
    import io
    sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
    sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding='utf-8')

EXCEL_FILE = "activities from didar.xlsx"

print("=" * 60)
print("آنالیز فایل اکسل فعالیت‌های دیدار")
print("=" * 60)

try:
    # خواندن فایل اکسل
    df = pd.read_excel(EXCEL_FILE)
    
    print(f"\n✅ فایل با موفقیت خوانده شد")
    print(f"📊 تعداد ردیف‌ها: {len(df)}")
    print(f"📋 تعداد ستون‌ها: {len(df.columns)}")
    
    print("\n" + "-" * 60)
    print("ستون‌های موجود:")
    print("-" * 60)
    for i, col in enumerate(df.columns, 1):
        print(f"  {i}. {col}")
    
    print("\n" + "-" * 60)
    print("نمونه داده‌ها (5 ردیف اول):")
    print("-" * 60)
    print(df.head().to_string())
    
    print("\n" + "-" * 60)
    print("اطلاعات ستون‌ها:")
    print("-" * 60)
    for col in df.columns:
        non_null = df[col].notna().sum()
        null_count = df[col].isna().sum()
        unique_count = df[col].nunique()
        print(f"\n📌 {col}:")
        print(f"  - غیر خالی: {non_null} ({non_null/len(df)*100:.1f}%)")
        print(f"  - خالی: {null_count} ({null_count/len(df)*100:.1f}%)")
        print(f"  - مقادیر منحصر به فرد: {unique_count}")
        
        # نمایش نمونه مقادیر
        if unique_count <= 20:
            print(f"  - مقادیر: {df[col].unique().tolist()}")
        else:
            print(f"  - نمونه مقادیر: {df[col].dropna().unique()[:10].tolist()}")
    
    # بررسی ستون‌های مهم
    print("\n" + "=" * 60)
    print("بررسی ستون‌های کلیدی:")
    print("=" * 60)
    
    # بررسی ستون‌های مربوط به شناسه تماس/لید
    contact_cols = []
    for col in df.columns:
        col_lower = str(col).lower()
        if 'contact' in col_lower or 'person' in col_lower or 'لید' in str(col) or 'مشتری' in str(col) or 'id' in col_lower:
            contact_cols.append(col)
    
    if contact_cols:
        print(f"\n🔗 ستون‌های مربوط به شناسه تماس/لید: {contact_cols}")
        for col in contact_cols:
            print(f"  - {col}: {df[col].nunique()} مقدار منحصر به فرد")
    
    # بررسی ستون‌های مربوط به کاربر/مسئول
    user_cols = []
    for col in df.columns:
        col_lower = str(col).lower()
        if 'user' in col_lower or 'owner' in col_lower or 'کاربر' in str(col) or 'مسئول' in str(col) or 'creator' in col_lower:
            user_cols.append(col)
    
    if user_cols:
        print(f"\n👤 ستون‌های مربوط به کاربر/مسئول: {user_cols}")
        for col in user_cols:
            print(f"  - {col}: {df[col].nunique()} مقدار منحصر به فرد")
    
    # بررسی ستون‌های مربوط به تاریخ
    date_cols = []
    for col in df.columns:
        col_lower = str(col).lower()
        if 'date' in col_lower or 'time' in col_lower or 'تاریخ' in str(col) or 'زمان' in str(col):
            date_cols.append(col)
    
    if date_cols:
        print(f"\n📅 ستون‌های مربوط به تاریخ: {date_cols}")
        for col in date_cols:
            print(f"  - {col}: {df[col].dtype}")
            if df[col].notna().any():
                print(f"    نمونه: {df[col].dropna().iloc[0] if len(df[col].dropna()) > 0 else 'N/A'}")
    
    # بررسی ستون‌های مربوط به نوع فعالیت
    type_cols = []
    for col in df.columns:
        col_lower = str(col).lower()
        if 'type' in col_lower or 'نوع' in str(col) or 'category' in col_lower or 'دسته' in str(col):
            type_cols.append(col)
    
    if type_cols:
        print(f"\n📝 ستون‌های مربوط به نوع فعالیت: {type_cols}")
        for col in type_cols:
            print(f"  - {col}: {df[col].nunique()} نوع منحصر به فرد")
            if df[col].nunique() <= 20:
                print(f"    انواع: {df[col].unique().tolist()}")
    
    # بررسی ستون‌های مربوط به توضیحات/یادداشت
    note_cols = []
    for col in df.columns:
        col_lower = str(col).lower()
        if 'note' in col_lower or 'description' in col_lower or 'comment' in col_lower or 'توضیح' in str(col) or 'یادداشت' in str(col) or 'متن' in str(col):
            note_cols.append(col)
    
    if note_cols:
        print(f"\n📄 ستون‌های مربوط به توضیحات/یادداشت: {note_cols}")
    
    # بررسی توزیع فعالیت‌ها بر اساس تماس
    if contact_cols:
        print(f"\n" + "=" * 60)
        print("توزیع فعالیت‌ها:")
        print("=" * 60)
        contact_col = contact_cols[0]
        activity_count = df.groupby(contact_col).size()
        print(f"\n📊 آمار فعالیت‌ها بر اساس {contact_col}:")
        print(f"  - تعداد تماس‌های منحصر به فرد: {activity_count.shape[0]}")
        print(f"  - میانگین فعالیت‌ها برای هر تماس: {activity_count.mean():.2f}")
        print(f"  - حداکثر فعالیت‌ها برای یک تماس: {activity_count.max()}")
        print(f"  - حداقل فعالیت‌ها برای یک تماس: {activity_count.min()}")
        print(f"\n  توزیع:")
        print(f"    - 1 فعالیت: {(activity_count == 1).sum()} تماس")
        print(f"    - 2-5 فعالیت: {((activity_count >= 2) & (activity_count <= 5)).sum()} تماس")
        print(f"    - 6-10 فعالیت: {((activity_count >= 6) & (activity_count <= 10)).sum()} تماس")
        print(f"    - بیش از 10 فعالیت: {(activity_count > 10).sum()} تماس")
    
    # بررسی ساختار جدول activities موجود
    print("\n" + "=" * 60)
    print("بررسی ساختار جدول activities در دیتابیس:")
    print("=" * 60)
    
    import sqlite3
    try:
        conn = sqlite3.connect("crm_database_new.db")
        cursor = conn.cursor()
        cursor.execute("PRAGMA table_info(activities)")
        columns = cursor.fetchall()
        print("\n📋 ستون‌های جدول activities:")
        for col in columns:
            print(f"  - {col[1]} ({col[2]})")
        
        cursor.execute("SELECT COUNT(*) FROM activities")
        count = cursor.fetchone()[0]
        print(f"\n📊 تعداد فعالیت‌های موجود در دیتابیس: {count}")
        
        conn.close()
    except Exception as e:
        print(f"\n⚠️ خطا در بررسی دیتابیس: {e}")
    
except FileNotFoundError:
    print(f"\n❌ فایل '{EXCEL_FILE}' پیدا نشد!")
    print("لطفاً مطمئن شوید فایل در همان پوشه قرار دارد.")
except Exception as e:
    print(f"\n❌ خطا در خواندن فایل: {e}")
    import traceback
    traceback.print_exc()

print("\n" + "=" * 60)
print("✅ آنالیز تکمیل شد")
print("=" * 60)




