#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
آنالیز و تطبیق دو فایل اکسل بر اساس شماره تماس
"""

import pandas as pd
import sys
import os
from datetime import datetime

# تنظیم encoding برای Windows
if sys.platform == 'win32':
    import io
    sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
    sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding='utf-8')

def normalize_phone(phone):
    """نرمال‌سازی شماره تماس برای تطبیق بهتر"""
    if pd.isna(phone) or phone == '':
        return ''
    
    # تبدیل به رشته و حذف فاصله‌ها و کاراکترهای خاص
    phone_str = str(phone).strip()
    
    # حذف کاراکترهای غیر عددی (به جز +)
    phone_clean = ''.join(c for c in phone_str if c.isdigit() or c == '+')
    
    # اگر خالی شد، برگرداندن رشته خالی
    if not phone_clean:
        return ''
    
    # حذف پیش‌شماره‌های مختلف و تبدیل به فرمت استاندارد
    if phone_clean.startswith('0098'):
        phone_clean = '0' + phone_clean[4:]
    elif phone_clean.startswith('+98'):
        phone_clean = '0' + phone_clean[3:]
    elif phone_clean.startswith('98') and len(phone_clean) > 10:
        phone_clean = '0' + phone_clean[2:]
    
    # حذف صفرهای ابتدایی اضافی (بیش از یک صفر)
    while len(phone_clean) > 11 and phone_clean.startswith('00'):
        phone_clean = phone_clean[1:]
    
    # اگر شماره با 0 شروع نمی‌شود و 10 رقم دارد، اضافه کردن 0
    if len(phone_clean) == 10 and not phone_clean.startswith('0'):
        phone_clean = '0' + phone_clean
    
    # فقط شماره‌های 11 رقمی که با 0 شروع می‌شوند را قبول می‌کنیم
    if len(phone_clean) == 11 and phone_clean.startswith('0'):
        return phone_clean
    
    # اگر شماره معتبر نیست، برگرداندن همان مقدار نرمال‌سازی شده
    return phone_clean

def analyze_excel_file(file_path, file_name):
    """آنالیز ساختار فایل اکسل"""
    print(f"\n{'='*60}")
    print(f"📊 آنالیز فایل: {file_name}")
    print(f"{'='*60}")
    
    try:
        df = pd.read_excel(file_path)
        
        print(f"\n✅ فایل با موفقیت خوانده شد")
        print(f"📏 تعداد ردیف‌ها: {len(df)}")
        print(f"📏 تعداد ستون‌ها: {len(df.columns)}")
        
        print(f"\n📋 نام ستون‌ها:")
        for i, col in enumerate(df.columns, 1):
            print(f"  {i}. {col}")
        
        print(f"\n📊 نمونه داده‌ها (5 ردیف اول):")
        print(df.head().to_string())
        
        print(f"\n📈 اطلاعات آماری:")
        print(df.info())
        
        return df
        
    except Exception as e:
        print(f"❌ خطا در خواندن فایل: {e}")
        return None

def main():
    print("="*60)
    print("🔍 شروع آنالیز و تطبیق فایل‌های اکسل")
    print("="*60)
    
    # مسیر فایل‌ها
    file1_path = "پایگاه داده اولیه.xlsx"
    file2_path = "لید های با فروش قبلی.xlsx"
    output_path = "پایگاه داده با فروش قبلی.xlsx"
    
    # خواندن و آنالیز فایل اول
    print("\n" + "="*60)
    print("📁 فایل 1: پایگاه داده اولیه")
    print("="*60)
    df_base = analyze_excel_file(file1_path, "پایگاه داده اولیه")
    
    if df_base is None:
        print("❌ خطا: نتوانست فایل پایگاه داده اولیه را بخواند")
        sys.exit(1)
    
    # خواندن و آنالیز فایل دوم
    print("\n" + "="*60)
    print("📁 فایل 2: لید های با فروش قبلی")
    print("="*60)
    df_previous_sales = analyze_excel_file(file2_path, "لید های با فروش قبلی")
    
    if df_previous_sales is None:
        print("❌ خطا: نتوانست فایل لید های با فروش قبلی را بخواند")
        sys.exit(1)
    
    # پیدا کردن ستون شماره تماس در هر دو فایل
    print("\n" + "="*60)
    print("🔍 پیدا کردن ستون‌های شماره تماس")
    print("="*60)
    
    # در فایل دوم، ستون دوم باید شماره تماس باشد
    if len(df_previous_sales.columns) < 2:
        print("❌ خطا: فایل لید های با فروش قبلی باید حداقل 2 ستون داشته باشد")
        sys.exit(1)
    
    phone_col_file2_primary = df_previous_sales.columns[1]  # ستون دوم (ایندکس 1) - اصلی
    phone_col_file2_secondary = df_previous_sales.columns[0]  # ستون اول - برای تطبیق بیشتر
    print(f"✅ ستون شماره تماس اصلی در فایل لید های با فروش قبلی: '{phone_col_file2_primary}'")
    print(f"✅ ستون شماره تماس فرعی در فایل لید های با فروش قبلی: '{phone_col_file2_secondary}'")
    
    # پیدا کردن ستون شماره تماس در فایل اول
    # جستجو برای ستون‌هایی که ممکن است شماره تماس باشند
    possible_phone_cols = [col for col in df_base.columns if any(keyword in str(col).lower() for keyword in ['شماره', 'تماس', 'موبایل', 'تلفن', 'phone', 'mobile'])]
    
    if not possible_phone_cols:
        print("⚠️ هشدار: ستون شماره تماس به صورت خودکار پیدا نشد")
        print("📋 لطفاً نام ستون شماره تماس را از لیست زیر انتخاب کنید:")
        for i, col in enumerate(df_base.columns, 1):
            print(f"  {i}. {col}")
        
        # استفاده از اولین ستون که ممکن است شماره تماس باشد
        phone_col_file1 = df_base.columns[0]
        print(f"\n⚠️ استفاده از ستون اول به عنوان شماره تماس: '{phone_col_file1}'")
    else:
        phone_col_file1 = possible_phone_cols[0]
        print(f"✅ ستون شماره تماس در فایل پایگاه داده اولیه: '{phone_col_file1}'")
    
    # نرمال‌سازی شماره‌های تماس در فایل دوم
    print("\n" + "="*60)
    print("🔄 نرمال‌سازی شماره‌های تماس")
    print("="*60)
    
    # نرمال‌سازی ستون دوم (اصلی)
    df_previous_sales['normalized_phone_primary'] = df_previous_sales[phone_col_file2_primary].apply(normalize_phone)
    # نرمال‌سازی ستون اول (فرعی)
    df_previous_sales['normalized_phone_secondary'] = df_previous_sales[phone_col_file2_secondary].apply(normalize_phone)
    
    # نرمال‌سازی شماره‌های تماس در فایل اول
    df_base['normalized_phone'] = df_base[phone_col_file1].apply(normalize_phone)
    
    # ایجاد مجموعه شماره‌های تماس از فایل دوم (از هر دو ستون)
    previous_sales_phones = set()
    
    # اضافه کردن شماره‌های ستون دوم (اصلی)
    phones_primary = df_previous_sales['normalized_phone_primary'].dropna()
    previous_sales_phones.update({p for p in phones_primary if p != ''})
    
    # اضافه کردن شماره‌های ستون اول (فرعی)
    phones_secondary = df_previous_sales['normalized_phone_secondary'].dropna()
    previous_sales_phones.update({p for p in phones_secondary if p != ''})
    
    print(f"✅ تعداد شماره‌های تماس منحصر به فرد در ستون دوم: {len({p for p in phones_primary if p != ''})}")
    print(f"✅ تعداد شماره‌های تماس منحصر به فرد در ستون اول: {len({p for p in phones_secondary if p != ''})}")
    print(f"✅ تعداد کل شماره‌های تماس منحصر به فرد (از هر دو ستون): {len(previous_sales_phones)}")
    
    # اضافه کردن ستون جدید به فایل اول
    print("\n" + "="*60)
    print("➕ اضافه کردن ستون 'دارای فروش قبلی'")
    print("="*60)
    
    df_result = df_base.copy()
    df_result['دارای فروش قبلی'] = df_result['normalized_phone'].apply(
        lambda x: 'بله' if normalize_phone(x) in previous_sales_phones else 'خیر'
    )
    
    # حذف ستون normalized_phone از خروجی
    df_result = df_result.drop(columns=['normalized_phone'])
    
    # آمار تطبیق
    matched_count = (df_result['دارای فروش قبلی'] == 'بله').sum()
    total_count = len(df_result)
    
    print(f"✅ تعداد لیدهای دارای فروش قبلی: {matched_count} از {total_count}")
    print(f"📊 درصد تطبیق: {(matched_count/total_count*100):.2f}%")
    
    # ذخیره فایل جدید
    print("\n" + "="*60)
    print("💾 ذخیره فایل جدید")
    print("="*60)
    
    try:
        df_result.to_excel(output_path, index=False, engine='openpyxl')
        print(f"✅ فایل جدید با موفقیت ذخیره شد: {output_path}")
        print(f"📏 تعداد ردیف‌ها: {len(df_result)}")
        print(f"📏 تعداد ستون‌ها: {len(df_result.columns)}")
        
        print(f"\n📋 ستون‌های فایل جدید:")
        for i, col in enumerate(df_result.columns, 1):
            print(f"  {i}. {col}")
        
    except Exception as e:
        print(f"❌ خطا در ذخیره فایل: {e}")
        sys.exit(1)
    
    print("\n" + "="*60)
    print("✅ آنالیز و تطبیق با موفقیت انجام شد!")
    print("="*60)

if __name__ == "__main__":
    main()

