<?php
/**
 * اسکریپت دیباگ برای بررسی مشکل لاگین کاربران
 */

define('DB_FILE', __DIR__ . '/crm_database.db');

if (!file_exists(DB_FILE)) {
    die("❌ فایل دیتابیس یافت نشد: " . DB_FILE);
}

$pdo = new PDO('sqlite:' . DB_FILE);
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html dir="rtl" lang="fa">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug Login Users</title>
    <style>
        body {
            font-family: 'Vazirmatn', 'Tahoma', sans-serif;
            background: #0f172a;
            color: #f8fafc;
            padding: 20px;
            max-width: 1200px;
            margin: 0 auto;
        }
        .container {
            background: rgba(30, 41, 59, 0.8);
            backdrop-filter: blur(8px);
            border: 1px solid rgba(255,255,255,0.05);
            border-radius: 12px;
            padding: 30px;
        }
        h1 { color: #eab308; }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 10px;
            text-align: right;
            border: 1px solid #334155;
        }
        th {
            background: #1e293b;
            color: #eab308;
        }
        .highlight { background: #1e293b; }
        .problem { background: rgba(239, 68, 68, 0.2); }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 بررسی کاربران برای لاگین</h1>
        
        <h2>📋 لیست همه کاربران:</h2>
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>display_name</th>
                    <th>username</th>
                    <th>first_name</th>
                    <th>last_name</th>
                    <th>طول display_name</th>
                    <th>طول username</th>
                    <th>فاصله‌های اضافی</th>
                    <th>is_active</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $users = $pdo->query("SELECT id, display_name, username, first_name, last_name, is_active FROM users ORDER BY display_name")->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($users as $user) {
                    $displayName = $user['display_name'] ?? '';
                    $username = $user['username'] ?? '';
                    $displayNameTrimmed = trim($displayName);
                    $usernameTrimmed = trim($username);
                    
                    $hasExtraSpaces = ($displayName !== $displayNameTrimmed) || ($username !== $usernameTrimmed);
                    $isEmpty = empty($displayName) && empty($username);
                    $isProblem = $hasExtraSpaces || $isEmpty;
                    
                    $rowClass = $isProblem ? 'problem' : '';
                    
                    echo "<tr class='{$rowClass}'>";
                    echo "<td>{$user['id']}</td>";
                    echo "<td>" . htmlspecialchars($displayName) . "</td>";
                    echo "<td>" . htmlspecialchars($username) . "</td>";
                    echo "<td>" . htmlspecialchars($user['first_name'] ?? '') . "</td>";
                    echo "<td>" . htmlspecialchars($user['last_name'] ?? '') . "</td>";
                    echo "<td>" . mb_strlen($displayName, 'UTF-8') . "</td>";
                    echo "<td>" . mb_strlen($username, 'UTF-8') . "</td>";
                    echo "<td>" . ($hasExtraSpaces ? '⚠️ بله' : '✅ خیر') . "</td>";
                    echo "<td>" . ($user['is_active'] ? '✅' : '❌') . "</td>";
                    echo "</tr>";
                }
                ?>
            </tbody>
        </table>
        
        <h2>🔍 تست جستجو برای کاربران مشکل‌دار:</h2>
        <pre><?php
        $problemUsers = ['محمدحسین حنفی', 'امیر سهرابی'];
        
        foreach ($problemUsers as $searchName) {
            echo "\n=== جستجو برای: {$searchName} ===\n";
            
            // 1. جستجوی دقیق
            $stmt = $pdo->prepare("SELECT id, display_name, username, first_name, last_name FROM users WHERE display_name = ? AND is_active = 1");
            $stmt->execute([$searchName]);
            $exact = $stmt->fetch(PDO::FETCH_ASSOC);
            echo "1. جستجوی دقیق: " . ($exact ? "✅ پیدا شد (ID: {$exact['id']})" : "❌ پیدا نشد") . "\n";
            
            // 2. جستجوی با TRIM
            $stmt = $pdo->prepare("SELECT id, display_name, username, first_name, last_name FROM users WHERE TRIM(display_name) = ? AND is_active = 1");
            $stmt->execute([trim($searchName)]);
            $trimmed = $stmt->fetch(PDO::FETCH_ASSOC);
            echo "2. جستجوی با TRIM: " . ($trimmed ? "✅ پیدا شد (ID: {$trimmed['id']})" : "❌ پیدا نشد") . "\n";
            
            // 3. جستجوی با LIKE
            $stmt = $pdo->prepare("SELECT id, display_name, username, first_name, last_name FROM users WHERE display_name LIKE ? AND is_active = 1 LIMIT 1");
            $stmt->execute(['%' . trim($searchName) . '%']);
            $like = $stmt->fetch(PDO::FETCH_ASSOC);
            echo "3. جستجوی با LIKE: " . ($like ? "✅ پیدا شد (ID: {$like['id']}, display_name: '{$like['display_name']}')" : "❌ پیدا نشد") . "\n";
            
            // 4. جستجو در first_name + last_name
            $nameParts = explode(' ', trim($searchName), 2);
            if (count($nameParts) >= 2) {
                $stmt = $pdo->prepare("SELECT id, display_name, username, first_name, last_name FROM users WHERE (first_name LIKE ? AND last_name LIKE ?) AND is_active = 1 LIMIT 1");
                $stmt->execute(['%' . $nameParts[0] . '%', '%' . $nameParts[1] . '%']);
                $namePartsResult = $stmt->fetch(PDO::FETCH_ASSOC);
                echo "4. جستجو در first_name + last_name: " . ($namePartsResult ? "✅ پیدا شد (ID: {$namePartsResult['id']})" : "❌ پیدا نشد") . "\n";
            }
            
            // 5. نمایش همه کاربران مشابه
            $stmt = $pdo->prepare("SELECT id, display_name, username, first_name, last_name FROM users WHERE (display_name LIKE ? OR first_name LIKE ? OR last_name LIKE ?) AND is_active = 1");
            $searchTerm = '%' . trim($searchName) . '%';
            $stmt->execute([$searchTerm, $searchTerm, $searchTerm]);
            $similar = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo "5. کاربران مشابه:\n";
            foreach ($similar as $sim) {
                echo "   - ID: {$sim['id']}, display_name: '{$sim['display_name']}', username: '{$sim['username']}', first: '{$sim['first_name']}', last: '{$sim['last_name']}'\n";
            }
        }
        ?></pre>
        
        <h2>💡 پیشنهادات:</h2>
        <ul>
            <li>اگر کاربری با فاصله‌های اضافی پیدا شد، باید display_name را اصلاح کنید</li>
            <li>اگر display_name خالی است، باید از first_name + last_name استفاده شود</li>
            <li>اگر username خالی است، باید از display_name کپی شود</li>
        </ul>
    </div>
</body>
</html>


