<?php
/**
 * ساخت دیتابیس جدید از فایل‌های اکسل
 * 
 * این اسکریپت:
 * 1. بکاپ از دیتابیس فعلی می‌گیرد
 * 2. دیتابیس جدید می‌سازد
 * 3. لیدها، فعالیت‌ها و معاملات را از اکسل import می‌کند
 * 4. Virtual Stages را محاسبه می‌کند
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);
set_time_limit(600); // 10 minutes

date_default_timezone_set('Asia/Tehran');

// ============================================
// تنظیمات
// ============================================

$oldDbFile = __DIR__ . '/crm_database (4).db';
$newDbFile = __DIR__ . '/crm_database_excel_import.db';
$backupDbFile = __DIR__ . '/crm_database_backup_' . date('Y-m-d_His') . '.db';

$excelFiles = [
    'leads' => __DIR__ . '/all leads in didar.xlsx',
    'activities' => __DIR__ . '/activities from didar.xlsx',
    'deals' => __DIR__ . '/deals from didar.xlsx'
];

// Owner Mapping
$allowedOwners = [
    'محمدحسین حنفی' => '495c6315-3db2-416d-836c-24c2e74c350a',
    'خانم شکیبا' => 'd596af82-2c35-4ede-90ff-8bcdd5b373a9',
    'خانم توانا' => '5140cc36-21cd-4e82-85df-c01f63610d1c',
    'خانم شایسته' => '75f9d286-0d05-43e0-a6ed-9fdb8ad89a0f',
    'آقای امید' => '4471c7d2-2a0b-49cd-9565-73509016cae8'
];

$stats = [
    'leads' => ['total' => 0, 'filtered' => 0, 'imported' => 0, 'failed' => 0],
    'activities' => ['total' => 0, 'imported' => 0, 'skipped' => 0],
    'deals' => ['total' => 0, 'imported' => 0, 'skipped' => 0],
    'virtual_stages' => ['created' => 0],
    'errors' => []
];

// ============================================
// توابع کمکی
// ============================================

function logMessage($message) {
    echo "[" . date('Y-m-d H:i:s') . "] " . $message . "\n";
}

function normalizePhone($phone) {
    if (empty($phone)) return '';
    // فقط اعداد
    $phone = preg_replace('/[^0-9]/', '', $phone);
    return $phone;
}

function convertPersianDate($persianDate) {
    if (empty($persianDate)) return null;
    
    // فرمت: 1400/06/15
    $parts = explode('/', trim($persianDate));
    if (count($parts) !== 3) return null;
    
    $year = intval($parts[0]);
    $month = intval($parts[1]);
    $day = intval($parts[2]);
    
    // تبدیل تقریبی شمسی به میلادی (برای دقت بیشتر می‌توان از کتابخانه استفاده کرد)
    // این یک تبدیل ساده است
    $gregorianYear = $year + 621;
    $gregorianMonth = $month;
    $gregorianDay = $day;
    
    // بررسی اعتبار تاریخ
    if (!checkdate($gregorianMonth, $gregorianDay, $gregorianYear)) {
        return null;
    }
    
    return sprintf('%04d-%02d-%02d', $gregorianYear, $gregorianMonth, $gregorianDay);
}

function parseJsonPhones($jsonString) {
    if (empty($jsonString)) return '';
    
    try {
        $data = json_decode($jsonString, true);
        if (is_array($data) && !empty($data)) {
            $firstPhone = $data[0]['شماره'] ?? $data[0]['number'] ?? '';
            return normalizePhone($firstPhone);
        }
    } catch (Exception $e) {
        // Ignore
    }
    
    return '';
}

function mapOwnerNameToId($ownerName, $allowedOwners) {
    $normalized = trim($ownerName);
    
    // جستجوی دقیق
    if (isset($allowedOwners[$normalized])) {
        return $allowedOwners[$normalized];
    }
    
    // جستجوی جزئی
    foreach ($allowedOwners as $name => $id) {
        if (strpos($normalized, $name) !== false || strpos($name, $normalized) !== false) {
            return $id;
        }
    }
    
    return null;
}

function mapDealStatus($status) {
    $status = trim($status);
    if ($status === 'موفق') return 'Won';
    if ($status === 'ناموفق') return 'Lost';
    return 'Pending';
}

function determineVirtualStageFromExcel($contactId, $activities, $deals) {
    // 1. بررسی آخرین فعالیت
    if (!empty($activities)) {
        $lastActivity = $activities[0];
        
        $activityType = $lastActivity['نوع فعالیت'] ?? '';
        $resultNote = $lastActivity['نتیجه فعالیت'] ?? '';
        
        // Mapping از نوع فعالیت
        if (stripos($activityType, 'عدم پاسخ') !== false) {
            return 'contact_no_answer';
        }
        if (stripos($activityType, 'پیگیری') !== false) {
            return 'contact_followup';
        }
        if (stripos($activityType, 'پشتیبانی') !== false) {
            return 'support';
        }
        
        // بررسی result_note
        if (stripos($resultNote, 'لید با خرید قبلی') !== false) {
            return 'with_purchase';
        }
        if (stripos($resultNote, 'لید بدون خرید قبلی') !== false) {
            return 'without_purchase';
        }
    }
    
    // 2. بررسی Deals
    if (!empty($deals)) {
        $deal = $deals[0];
        $status = $deal['وضعیت معامله'] ?? '';
        $pipelineStage = $deal['مرحله کاریز معامله'] ?? '';
        
        if ($status === 'موفق') {
            // بررسی فعالیت‌های مرتبط برای course_delivered
            $hasCourseDelivery = false;
            foreach ($activities as $act) {
                $actType = $act['نوع فعالیت'] ?? '';
                if (stripos($actType, 'دوره') !== false || stripos($actType, 'برگزاری') !== false) {
                    $hasCourseDelivery = true;
                    break;
                }
            }
            
            if ($hasCourseDelivery) {
                return 'course_delivered';
            }
            
            return 'payment';
        }
        
        // Mapping از مرحله کاریز
        if (stripos($pipelineStage, 'لید ورودی') !== false) {
            return 'new';
        }
        if (stripos($pipelineStage, 'پرزنت') !== false || stripos($pipelineStage, 'پیگیری') !== false) {
            return 'contact_followup';
        }
        if (stripos($pipelineStage, 'دریافت وجه') !== false) {
            return 'payment';
        }
        if (stripos($pipelineStage, 'پشتیبانی') !== false) {
            return 'support';
        }
        
        return 'deal_registered';
    }
    
    // 3. پیش‌فرض
    return 'new';
}

// ============================================
// بررسی وجود فایل‌ها
// ============================================

logMessage("بررسی فایل‌ها...");

if (!file_exists($oldDbFile)) {
    die("❌ دیتابیس قدیم یافت نشد: $oldDbFile\n");
}

foreach ($excelFiles as $key => $file) {
    if (!file_exists($file)) {
        die("❌ فایل اکسل یافت نشد: $file\n");
    }
}

logMessage("✅ همه فایل‌ها موجود هستند");

// ============================================
// بکاپ از دیتابیس فعلی
// ============================================

logMessage("📦 ایجاد بکاپ از دیتابیس فعلی...");

try {
    if (file_exists($oldDbFile)) {
        copy($oldDbFile, $backupDbFile);
        logMessage("✅ بکاپ ایجاد شد: $backupDbFile");
    }
} catch (Exception $e) {
    logMessage("⚠️ خطا در ایجاد بکاپ: " . $e->getMessage());
}

// ============================================
// بررسی PhpSpreadsheet
// ============================================

logMessage("📚 بررسی کتابخانه PhpSpreadsheet...");

$usePhpSpreadsheet = false;
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    require_once __DIR__ . '/vendor/autoload.php';
    $usePhpSpreadsheet = true;
    logMessage("✅ PhpSpreadsheet یافت شد");
} else {
    logMessage("⚠️ PhpSpreadsheet یافت نشد - نیاز به نصب: composer require phpoffice/phpspreadsheet");
    logMessage("💡 در حال استفاده از روش جایگزین...");
}

// ============================================
// ایجاد دیتابیس جدید
// ============================================

logMessage("🗄️ ایجاد دیتابیس جدید...");

if (file_exists($newDbFile)) {
    unlink($newDbFile);
}

$newPdo = new PDO('sqlite:' . $newDbFile);
$newPdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$newPdo->exec('PRAGMA journal_mode=WAL;');
$newPdo->exec('PRAGMA busy_timeout=30000;');

// ============================================
// ایجاد جداول (از index.php)
// ============================================

logMessage("📋 ایجاد جداول...");

$tableCommands = [
    "CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        didar_user_id TEXT UNIQUE NOT NULL,
        email TEXT UNIQUE NOT NULL,
        first_name TEXT,
        last_name TEXT,
        display_name TEXT,
        role TEXT CHECK(role IN ('admin', 'agent')) DEFAULT 'agent',
        is_active INTEGER DEFAULT 1,
        api_key TEXT,
        last_sync DATETIME,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS persons (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        didar_contact_id TEXT UNIQUE NOT NULL,
        owner_didar_id TEXT,
        code TEXT,
        first_name TEXT,
        last_name TEXT,
        mobile_phone TEXT,
        secondary_mobile_phone TEXT,
        work_phone TEXT,
        email TEXT,
        contact_type TEXT,
        city TEXT,
        job_title TEXT,
        acquaintance_source TEXT,
        acquaintance_detail TEXT,
        content_topic TEXT,
        national_id TEXT,
        has_previous_purchase INTEGER DEFAULT 0,
        birth_date DATE,
        register_time DATETIME,
        visibility_type TEXT,
        background_info TEXT,
        last_sync DATETIME,
        raw_json TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS deals (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        didar_deal_id TEXT UNIQUE NOT NULL,
        owner_didar_id TEXT,
        contact_didar_id TEXT,
        title TEXT,
        code TEXT,
        status TEXT CHECK(status IN ('Pending', 'Won', 'Lost')),
        pipeline_stage_id TEXT,
        pipeline_stage_title TEXT,
        estimated_price REAL,
        final_price REAL,
        probability INTEGER,
        is_paid INTEGER DEFAULT 0,
        payment_short_link TEXT,
        register_time DATETIME,
        last_update_time DATETIME,
        won_time DATETIME,
        lost_time DATETIME,
        visibility_type TEXT,
        last_sync DATETIME,
        raw_json TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS activities (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        didar_activity_id TEXT UNIQUE NOT NULL,
        activity_type_id TEXT NOT NULL,
        activity_type_title TEXT,
        owner_didar_id TEXT,
        deal_didar_id TEXT,
        contact_didar_id TEXT,
        title TEXT,
        note TEXT,
        result_note TEXT,
        is_done INTEGER DEFAULT 0,
        due_date DATETIME,
        done_date DATETIME,
        duration INTEGER,
        activity_category TEXT,
        direction TEXT CHECK(direction IN ('incoming', 'outgoing')),
        stage TEXT,
        register_date DATETIME,
        last_update_time DATETIME,
        last_sync DATETIME,
        raw_json TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS virtual_stages (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        contact_didar_id TEXT NOT NULL,
        stage_name TEXT CHECK(stage_name IN (
            'inactive_failed', 'inactive_success', 'with_purchase', 'without_purchase',
            'new', 'contact_no_answer', 'contact_followup', 'deal_registered', 'payment', 
            'payment_followup', 'course_delivered', 'support', 'refer_crm'
        )),
        entered_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        last_activity_id TEXT,
        notes TEXT,
        UNIQUE(contact_didar_id, stage_name)
    )",
    "CREATE TABLE IF NOT EXISTS config (
        key TEXT PRIMARY KEY,
        value TEXT,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS sync_log (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        entity_type TEXT CHECK(entity_type IN ('user', 'person', 'deal', 'activity')),
        entity_id TEXT,
        action TEXT CHECK(action IN ('fetch', 'create', 'update')),
        status TEXT CHECK(status IN ('success', 'failed')),
        error_message TEXT,
        http_code INTEGER,
        response_data TEXT,
        synced_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS pipeline_cache (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        pipeline_id TEXT UNIQUE NOT NULL,
        pipeline_title TEXT,
        stages_json TEXT,
        cached_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )",
    "CREATE TABLE IF NOT EXISTS referrals (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        contact_didar_id TEXT NOT NULL,
        referrer_didar_id TEXT,
        note TEXT,
        phones TEXT,
        status TEXT DEFAULT 'pending',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )"
];

foreach ($tableCommands as $cmd) {
    try {
        $newPdo->exec($cmd);
    } catch (PDOException $e) {
        logMessage("⚠️ خطا در ایجاد جدول: " . $e->getMessage());
    }
}

logMessage("✅ جداول ایجاد شدند");

// ============================================
// کپی جداول از دیتابیس قدیم
// ============================================

logMessage("📥 کپی جداول از دیتابیس قدیم...");

try {
    $oldPdo = new PDO('sqlite:' . $oldDbFile);
    $oldPdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // کپی users
    logMessage("  - کپی users...");
    $users = $oldPdo->query("SELECT * FROM users")->fetchAll(PDO::FETCH_ASSOC);
    $stmt = $newPdo->prepare("INSERT OR REPLACE INTO users (didar_user_id, email, first_name, last_name, display_name, role, is_active, api_key, last_sync, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    foreach ($users as $user) {
        $stmt->execute([
            $user['didar_user_id'],
            $user['email'],
            $user['first_name'] ?? null,
            $user['last_name'] ?? null,
            $user['display_name'] ?? null,
            $user['role'] ?? 'agent',
            $user['is_active'] ?? 1,
            $user['api_key'] ?? null,
            $user['last_sync'] ?? null,
            $user['created_at'] ?? date('Y-m-d H:i:s')
        ]);
    }
    logMessage("    ✅ " . count($users) . " کاربر کپی شد");
    
    // کپی config
    logMessage("  - کپی config...");
    $configs = $oldPdo->query("SELECT * FROM config")->fetchAll(PDO::FETCH_ASSOC);
    $stmt = $newPdo->prepare("INSERT OR REPLACE INTO config (key, value, updated_at) VALUES (?, ?, ?)");
    foreach ($configs as $config) {
        $stmt->execute([$config['key'], $config['value'], $config['updated_at'] ?? date('Y-m-d H:i:s')]);
    }
    logMessage("    ✅ " . count($configs) . " تنظیمات کپی شد");
    
    // کپی pipeline_cache
    logMessage("  - کپی pipeline_cache...");
    $pipelines = $oldPdo->query("SELECT * FROM pipeline_cache")->fetchAll(PDO::FETCH_ASSOC);
    $stmt = $newPdo->prepare("INSERT OR REPLACE INTO pipeline_cache (pipeline_id, pipeline_title, stages_json, cached_at) VALUES (?, ?, ?, ?)");
    foreach ($pipelines as $pipeline) {
        $stmt->execute([
            $pipeline['pipeline_id'],
            $pipeline['pipeline_title'] ?? null,
            $pipeline['stages_json'] ?? null,
            $pipeline['cached_at'] ?? date('Y-m-d H:i:s')
        ]);
    }
    logMessage("    ✅ " . count($pipelines) . " pipeline کپی شد");
    
} catch (Exception $e) {
    logMessage("⚠️ خطا در کپی جداول: " . $e->getMessage());
}

// ============================================
// خواندن فایل‌های اکسل
// ============================================

logMessage("📖 خواندن فایل‌های اکسل...");

if (!$usePhpSpreadsheet) {
    die("❌ نیاز به PhpSpreadsheet است. لطفاً اجرا کنید: composer require phpoffice/phpspreadsheet\n");
}

use PhpOffice\PhpSpreadsheet\IOFactory;

// خواندن Leads
logMessage("  - خواندن Leads...");
$spreadsheet = IOFactory::load($excelFiles['leads']);
$sheet = $spreadsheet->getActiveSheet();
$leadsData = [];

// استفاده از toArray برای سرعت بیشتر
$rows = $sheet->toArray();
if (empty($rows)) {
    die("❌ فایل اکسل Leads خالی است!\n");
}

$headers = array_shift($rows); // ردیف اول = هدرها
$headerCount = count($headers);

foreach ($rows as $rowIndex => $row) {
    if (empty($row)) continue;
    
    $rowData = [];
    for ($i = 0; $i < $headerCount && $i < count($row); $i++) {
        if (isset($headers[$i]) && !empty($headers[$i])) {
            $rowData[$headers[$i]] = $row[$i] ?? null;
        }
    }
    
    if (!empty($rowData)) {
        $leadsData[] = $rowData;
    }
    
    if (($rowIndex + 1) % 1000 === 0) {
        logMessage("    خوانده شده: " . ($rowIndex + 1) . " ردیف...");
    }
}

$stats['leads']['total'] = count($leadsData);
logMessage("    ✅ " . count($leadsData) . " لید خوانده شد");

// فیلتر کردن لیدها بر اساس owner
logMessage("  - فیلتر کردن لیدها...");
$filteredLeads = [];
foreach ($leadsData as $lead) {
    $ownerName = $lead['مسئول مشتری'] ?? '';
    if (mapOwnerNameToId($ownerName, $allowedOwners) !== null) {
        $filteredLeads[] = $lead;
    }
}
$stats['leads']['filtered'] = count($filteredLeads);
logMessage("    ✅ " . count($filteredLeads) . " لید فیلتر شد");

// ============================================
// Import Leads
// ============================================

logMessage("💾 Import لیدها...");

$newPdo->beginTransaction();
$importedContactIds = [];

$stmt = $newPdo->prepare("
    INSERT OR REPLACE INTO persons 
    (didar_contact_id, owner_didar_id, code, first_name, last_name, 
     mobile_phone, secondary_mobile_phone, work_phone, email, 
     contact_type, city, job_title, acquaintance_source, acquaintance_detail, 
     content_topic, register_time, visibility_type, has_previous_purchase, 
     last_sync, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
");

$batchCount = 0;
foreach ($filteredLeads as $lead) {
    try {
        $contactId = $lead['کد دیدار مشتری'] ?? '';
        if (empty($contactId)) continue;
        
        $ownerName = $lead['مسئول مشتری'] ?? '';
        $ownerId = mapOwnerNameToId($ownerName, $allowedOwners);
        if (!$ownerId) continue;
        
        $firstName = $lead['نام مشتری'] ?? '';
        $lastName = $lead['نام خانوادگی مشتری'] ?? '';
        $mobilePhone = normalizePhone($lead['تلفن همراه مشتری'] ?? '');
        $secondaryMobile = parseJsonPhones($lead['تلفن های دیگر مشتری'] ?? '');
        $workPhone = normalizePhone($lead['تلفن ثابت مشتری'] ?? '');
        $email = $lead['ایمیل مشتری'] ?? '';
        $city = $lead['شهر_1'] ?? '';
        $jobTitle = $lead['شغل'] ?? '';
        $acquaintanceSource = $lead['شیوه آشنایی'] ?? '';
        $acquaintanceDetail = $lead['جزئیات شیوه آشنایی'] ?? '';
        $contentTopic = $lead['موضوع محتوا'] ?? '';
        $registerTime = convertPersianDate($lead['تاریخ ثبت مشتری'] ?? '');
        $visibilityType = $lead['مجوز مشاهده مشتری'] ?? '';
        $code = $lead['کد دیدار مشتری'] ?? '';
        
        $stmt->execute([
            $contactId,
            $ownerId,
            $code,
            $firstName,
            $lastName,
            $mobilePhone,
            $secondaryMobile,
            $workPhone,
            $email,
            'Lead',
            $city,
            $jobTitle,
            $acquaintanceSource,
            $acquaintanceDetail,
            $contentTopic,
            $registerTime,
            $visibilityType,
            0, // has_previous_purchase
        ]);
        
        $importedContactIds[] = $contactId;
        $stats['leads']['imported']++;
        $batchCount++;
        
        if ($batchCount % 100 === 0) {
            $newPdo->commit();
            $newPdo->beginTransaction();
            logMessage("    Import شده: $batchCount لید...");
        }
    } catch (Exception $e) {
        $stats['leads']['failed']++;
        $stats['errors'][] = "Lead import error: " . $e->getMessage();
    }
}

$newPdo->commit();
logMessage("    ✅ " . $stats['leads']['imported'] . " لید import شد");

// ============================================
// خواندن و Import Activities
// ============================================

logMessage("\n📖 خواندن Activities...");

$spreadsheet = IOFactory::load($excelFiles['activities']);
$sheet = $spreadsheet->getActiveSheet();
$activitiesData = [];

// استفاده از toArray برای سرعت بیشتر
$rows = $sheet->toArray();
if (empty($rows)) {
    logMessage("    ⚠️ فایل اکسل Activities خالی است!");
} else {
    $headers = array_shift($rows); // ردیف اول = هدرها
    $headerCount = count($headers);
    
    foreach ($rows as $rowIndex => $row) {
        if (empty($row)) continue;
        
        $rowData = [];
        for ($i = 0; $i < $headerCount && $i < count($row); $i++) {
            if (isset($headers[$i]) && !empty($headers[$i])) {
                $rowData[$headers[$i]] = $row[$i] ?? null;
            }
        }
        
        if (!empty($rowData)) {
            $activitiesData[] = $rowData;
        }
        
        if (($rowIndex + 1) % 1000 === 0) {
            logMessage("    خوانده شده: " . ($rowIndex + 1) . " ردیف...");
        }
    }
}

$stats['activities']['total'] = count($activitiesData);
logMessage("    ✅ " . count($activitiesData) . " فعالیت خوانده شد");

// گروه‌بندی Activities بر اساس contact_id
$activitiesByContact = [];
foreach ($activitiesData as $activity) {
    $contactId = $activity['کد اشخاص فعالیت'] ?? '';
    if (!empty($contactId) && in_array($contactId, $importedContactIds)) {
        if (!isset($activitiesByContact[$contactId])) {
            $activitiesByContact[$contactId] = [];
        }
        $activitiesByContact[$contactId][] = $activity;
    }
}

logMessage("  - " . count($activitiesByContact) . " contact با فعالیت پیدا شد");

// Import Activities
logMessage("💾 Import Activities...");

$newPdo->beginTransaction();
$activityCounter = 0;

$stmt = $newPdo->prepare("
    INSERT OR REPLACE INTO activities 
    (didar_activity_id, activity_type_id, activity_type_title, owner_didar_id,
     contact_didar_id, title, note, result_note, is_done,
     due_date, done_date, register_date, last_sync, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
");

foreach ($activitiesByContact as $contactId => $activities) {
    foreach ($activities as $activity) {
        try {
            // تولید ID برای activity (اگر در اکسل نیست)
            $activityId = 'excel_' . $contactId . '_' . $activityCounter;
            $activityType = $activity['نوع فعالیت'] ?? '';
            $title = $activity['عنوان فعالیت'] ?? $activityType;
            $note = $activity['متن فعالیت'] ?? '';
            $resultNote = $activity['نتیجه فعالیت'] ?? '';
            $isDone = (stripos($activity['وضعیت اجرای فعالیت'] ?? '', 'انجام شده') !== false) ? 1 : 0;
            $dueDate = convertPersianDate($activity['تاریخ برنامه ریزی شده اجرا فعالیت'] ?? '');
            $doneDate = convertPersianDate($activity['تاریخ انجام شدن فعالیت'] ?? '');
            $registerDate = $dueDate ?? date('Y-m-d H:i:s');
            $ownerName = $activity['مسئول اجرا فعالیت'] ?? '';
            $ownerId = mapOwnerNameToId($ownerName, $allowedOwners);
            
            if (!$ownerId) {
                $ownerId = $allowedOwners['محمدحسین حنفی']; // پیش‌فرض
            }
            
            $stmt->execute([
                $activityId,
                'excel_type', // activity_type_id
                $activityType,
                $ownerId,
                $contactId,
                $title,
                $note,
                $resultNote,
                $isDone,
                $dueDate,
                $doneDate,
                $registerDate
            ]);
            
            $activityCounter++;
            $stats['activities']['imported']++;
            
            if ($activityCounter % 100 === 0) {
                $newPdo->commit();
                $newPdo->beginTransaction();
                logMessage("    Import شده: $activityCounter فعالیت...");
            }
        } catch (Exception $e) {
            $stats['activities']['skipped']++;
            $stats['errors'][] = "Activity import error: " . $e->getMessage();
        }
    }
}

$newPdo->commit();
logMessage("    ✅ " . $stats['activities']['imported'] . " فعالیت import شد");

// ============================================
// خواندن و Import Deals
// ============================================

logMessage("\n📖 خواندن Deals...");

$spreadsheet = IOFactory::load($excelFiles['deals']);
$sheet = $spreadsheet->getActiveSheet();
$dealsData = [];

// استفاده از toArray برای سرعت بیشتر
$rows = $sheet->toArray();
if (empty($rows)) {
    logMessage("    ⚠️ فایل اکسل Deals خالی است!");
} else {
    $headers = array_shift($rows); // ردیف اول = هدرها
    $headerCount = count($headers);
    
    foreach ($rows as $rowIndex => $row) {
        if (empty($row)) continue;
        
        $rowData = [];
        for ($i = 0; $i < $headerCount && $i < count($row); $i++) {
            if (isset($headers[$i]) && !empty($headers[$i])) {
                $rowData[$headers[$i]] = $row[$i] ?? null;
            }
        }
        
        if (!empty($rowData)) {
            $dealsData[] = $rowData;
        }
        
        if (($rowIndex + 1) % 1000 === 0) {
            logMessage("    خوانده شده: " . ($rowIndex + 1) . " ردیف...");
        }
    }
}

$stats['deals']['total'] = count($dealsData);
logMessage("    ✅ " . count($dealsData) . " معامله خوانده شد");

// فیلتر Deals بر اساس imported contact ids
$dealsByContact = [];
foreach ($dealsData as $deal) {
    $contactId = $deal['کد دیدار شخص معامله'] ?? '';
    if (!empty($contactId) && in_array($contactId, $importedContactIds)) {
        if (!isset($dealsByContact[$contactId])) {
            $dealsByContact[$contactId] = [];
        }
        $dealsByContact[$contactId][] = $deal;
    }
}

logMessage("  - " . count($dealsByContact) . " contact با معامله پیدا شد");

// Import Deals
logMessage("💾 Import Deals...");

$newPdo->beginTransaction();
$dealCounter = 0;

$stmt = $newPdo->prepare("
    INSERT OR REPLACE INTO deals 
    (didar_deal_id, owner_didar_id, contact_didar_id, title, code, status,
     pipeline_stage_title, estimated_price, final_price, probability,
     register_time, won_time, visibility_type, last_sync, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
");

foreach ($dealsByContact as $contactId => $deals) {
    foreach ($deals as $deal) {
        try {
            $dealId = $deal['کد دیدار معامله'] ?? '';
            if (empty($dealId)) {
                $dealId = 'excel_' . $contactId . '_' . $dealCounter;
            }
            
            $title = $deal['عنوان معامله'] ?? '';
            $price = floatval($deal['ارزش معامله'] ?? 0);
            $finalPrice = floatval($deal['مبلغ نهایی'] ?? $price);
            $status = mapDealStatus($deal['وضعیت معامله'] ?? '');
            $registerTime = convertPersianDate($deal['تاریخ ایجاد معامله'] ?? '');
            $wonTime = ($status === 'Won') ? convertPersianDate($deal['تاریخ انجام معامله'] ?? '') : null;
            $pipelineStageTitle = $deal['مرحله کاریز معامله'] ?? '';
            $probability = intval($deal['احتمال موفقیت معامله'] ?? 0);
            $ownerName = $deal['مسئول معامله'] ?? '';
            $ownerId = mapOwnerNameToId($ownerName, $allowedOwners);
            $visibilityType = $deal['مجوز مشاهده معامله'] ?? '';
            
            if (!$ownerId) {
                $ownerId = $allowedOwners['محمدحسین حنفی']; // پیش‌فرض
            }
            
            $stmt->execute([
                $dealId,
                $ownerId,
                $contactId,
                $title,
                $dealId, // code
                $status,
                $pipelineStageTitle,
                $price,
                $finalPrice,
                $probability,
                $registerTime,
                $wonTime,
                $visibilityType
            ]);
            
            $dealCounter++;
            $stats['deals']['imported']++;
            
            if ($dealCounter % 100 === 0) {
                $newPdo->commit();
                $newPdo->beginTransaction();
                logMessage("    Import شده: $dealCounter معامله...");
            }
        } catch (Exception $e) {
            $stats['deals']['skipped']++;
            $stats['errors'][] = "Deal import error: " . $e->getMessage();
        }
    }
}

$newPdo->commit();
logMessage("    ✅ " . $stats['deals']['imported'] . " معامله import شد");

// ============================================
// ایجاد Virtual Stages
// ============================================

logMessage("\n🎯 ایجاد Virtual Stages...");

$newPdo->beginTransaction();
$stageCounter = 0;

$stmt = $newPdo->prepare("
    INSERT OR REPLACE INTO virtual_stages (contact_didar_id, stage_name, entered_at)
    VALUES (?, ?, CURRENT_TIMESTAMP)
");

foreach ($importedContactIds as $contactId) {
    try {
        // دریافت Activities مرتبط
        $activitiesStmt = $newPdo->prepare("
            SELECT * FROM activities 
            WHERE contact_didar_id = ? 
            ORDER BY register_date DESC, created_at DESC
        ");
        $activitiesStmt->execute([$contactId]);
        $activities = $activitiesStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // دریافت Deals مرتبط
        $dealsStmt = $newPdo->prepare("
            SELECT * FROM deals 
            WHERE contact_didar_id = ? 
            ORDER BY register_time DESC
        ");
        $dealsStmt->execute([$contactId]);
        $deals = $dealsStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // تبدیل به فرمت مورد نیاز تابع
        $activitiesFormatted = [];
        foreach ($activities as $act) {
            $activitiesFormatted[] = [
                'نوع فعالیت' => $act['activity_type_title'] ?? '',
                'نتیجه فعالیت' => $act['result_note'] ?? ''
            ];
        }
        
        $dealsFormatted = [];
        foreach ($deals as $deal) {
            $dealsFormatted[] = [
                'وضعیت معامله' => $deal['status'] === 'Won' ? 'موفق' : ($deal['status'] === 'Lost' ? 'ناموفق' : 'جاری'),
                'مرحله کاریز معامله' => $deal['pipeline_stage_title'] ?? ''
            ];
        }
        
        // تعیین Virtual Stage
        $virtualStage = determineVirtualStageFromExcel($contactId, $activitiesFormatted, $dealsFormatted);
        
        // ذخیره
        $stmt->execute([$contactId, $virtualStage]);
        
        $stageCounter++;
        $stats['virtual_stages']['created']++;
        
        if ($stageCounter % 100 === 0) {
            $newPdo->commit();
            $newPdo->beginTransaction();
            logMessage("    ایجاد شده: $stageCounter virtual stage...");
        }
    } catch (Exception $e) {
        $stats['errors'][] = "Virtual stage error for $contactId: " . $e->getMessage();
    }
}

$newPdo->commit();
logMessage("    ✅ " . $stats['virtual_stages']['created'] . " virtual stage ایجاد شد");

// ============================================
// گزارش نهایی
// ============================================

logMessage("\n" . str_repeat("=", 60));
logMessage("✅ Import کامل انجام شد!");
logMessage(str_repeat("=", 60));
logMessage("\n📊 آمار نهایی:");
logMessage("  📋 لیدها:");
logMessage("     - کل در اکسل: " . $stats['leads']['total']);
logMessage("     - فیلتر شده: " . $stats['leads']['filtered']);
logMessage("     - Import شده: " . $stats['leads']['imported']);
logMessage("     - خطا: " . $stats['leads']['failed']);
logMessage("  📞 فعالیت‌ها:");
logMessage("     - کل در اکسل: " . $stats['activities']['total']);
logMessage("     - Import شده: " . $stats['activities']['imported']);
logMessage("     - Skip شده: " . $stats['activities']['skipped']);
logMessage("  💼 معاملات:");
logMessage("     - کل در اکسل: " . $stats['deals']['total']);
logMessage("     - Import شده: " . $stats['deals']['imported']);
logMessage("     - Skip شده: " . $stats['deals']['skipped']);
logMessage("  🎯 Virtual Stages:");
logMessage("     - ایجاد شده: " . $stats['virtual_stages']['created']);

if (!empty($stats['errors'])) {
    logMessage("\n⚠️ خطاها (" . count($stats['errors']) . " مورد):");
    foreach (array_slice($stats['errors'], 0, 10) as $error) {
        logMessage("     - " . $error);
    }
    if (count($stats['errors']) > 10) {
        logMessage("     ... و " . (count($stats['errors']) - 10) . " خطای دیگر");
    }
}

logMessage("\n📁 فایل‌ها:");
logMessage("  - دیتابیس جدید: $newDbFile");
logMessage("  - بکاپ دیتابیس: $backupDbFile");
logMessage("\n✅ آماده برای آپلود!");

