#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
اسکریپت کامل کردن Virtual Stages باقی‌مانده
این اسکریپت فقط Virtual Stages را برای لیدهایی که هنوز Virtual Stage ندارند ایجاد می‌کند
"""

import sqlite3
from datetime import datetime

DB_FILE = 'crm_database_excel_import.db'

def log_message(message):
    timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    print(f"[{timestamp}] {message}")

def determine_virtual_stage_from_excel(contact_id, activities, deals):
    """تعیین Virtual Stage بر اساس Activities و Deals"""
    # 1. بررسی آخرین فعالیت
    if activities and len(activities) > 0:
        last_activity = activities[0]
        activity_type = str(last_activity.get('نوع فعالیت', '')).strip()
        result_note = str(last_activity.get('نتیجه فعالیت', '')).strip()
        
        # Mapping از نوع فعالیت
        if 'عدم پاسخ' in activity_type:
            return 'contact_no_answer'
        if 'پیگیری' in activity_type:
            return 'contact_followup'
        if 'پشتیبانی' in activity_type:
            return 'support'
        
        # بررسی result_note
        if 'لید با خرید قبلی' in result_note:
            return 'with_purchase'
        if 'لید بدون خرید قبلی' in result_note:
            return 'without_purchase'
    
    # 2. بررسی Deals
    if deals and len(deals) > 0:
        deal = deals[0]
        status = str(deal.get('وضعیت معامله', '')).strip()
        pipeline_stage = str(deal.get('مرحله کاریز معامله', '')).strip()
        
        if status == 'موفق':
            # بررسی فعالیت‌های مرتبط برای course_delivered
            has_course_delivery = False
            if activities:
                for act in activities:
                    act_type = str(act.get('نوع فعالیت', '')).strip()
                    if 'دوره' in act_type or 'برگزاری' in act_type:
                        has_course_delivery = True
                        break
            
            if has_course_delivery:
                return 'course_delivered'
            
            return 'payment'
        
        # Mapping از مرحله کاریز
        if 'لید ورودی' in pipeline_stage:
            return 'new'
        if 'پرزنت' in pipeline_stage or 'پیگیری' in pipeline_stage:
            return 'contact_followup'
        if 'دریافت وجه' in pipeline_stage:
            return 'payment'
        if 'پشتیبانی' in pipeline_stage:
            return 'support'
        
        return 'deal_registered'
    
    # 3. پیش‌فرض
    return 'new'

# ============================================
# شروع
# ============================================

log_message("🎯 شروع کامل کردن Virtual Stages...")

conn = sqlite3.connect(DB_FILE)
cursor = conn.cursor()

# پیدا کردن لیدهایی که Virtual Stage ندارند
log_message("📋 پیدا کردن لیدهای بدون Virtual Stage...")

cursor.execute("""
    SELECT p.didar_contact_id 
    FROM persons p
    LEFT JOIN virtual_stages vs ON p.didar_contact_id = vs.contact_didar_id
    WHERE vs.contact_didar_id IS NULL
""")

contacts_without_stage = [row[0] for row in cursor.fetchall()]

total = len(contacts_without_stage)
log_message(f"    ✅ {total} لید بدون Virtual Stage پیدا شد")

if total == 0:
    log_message("✅ همه Virtual Stages کامل هستند!")
    conn.close()
    exit(0)

# ایجاد Virtual Stages
log_message("💾 ایجاد Virtual Stages...")

insert_stage_sql = """INSERT OR REPLACE INTO virtual_stages (contact_didar_id, stage_name, entered_at)
    VALUES (?, ?, CURRENT_TIMESTAMP)"""

stage_counter = 0
errors = []

for contact_id in contacts_without_stage:
    try:
        # دریافت Activities مرتبط
        cursor.execute("""
            SELECT activity_type_title, result_note 
            FROM activities 
            WHERE contact_didar_id = ? 
            ORDER BY register_date DESC, created_at DESC
        """, (contact_id,))
        activities_rows = cursor.fetchall()
        activities = [{'نوع فعالیت': row[0] or '', 'نتیجه فعالیت': row[1] or ''} for row in activities_rows]
        
        # دریافت Deals مرتبط
        cursor.execute("""
            SELECT status, pipeline_stage_title 
            FROM deals 
            WHERE contact_didar_id = ? 
            ORDER BY register_time DESC
        """, (contact_id,))
        deals_rows = cursor.fetchall()
        deals = []
        for row in deals_rows:
            status = 'موفق' if row[0] == 'Won' else ('ناموفق' if row[0] == 'Lost' else 'جاری')
            deals.append({'وضعیت معامله': status, 'مرحله کاریز معامله': row[1] or ''})
        
        # تعیین Virtual Stage
        virtual_stage = determine_virtual_stage_from_excel(contact_id, activities, deals)
        
        # ذخیره
        cursor.execute(insert_stage_sql, (contact_id, virtual_stage))
        
        stage_counter += 1
        
        if stage_counter % 100 == 0:
            conn.commit()
            log_message(f"    ایجاد شده: {stage_counter} از {total} ({stage_counter*100/total:.1f}%)...")
    
    except Exception as e:
        errors.append(f"Error for {contact_id}: {e}")

conn.commit()
log_message(f"    ✅ {stage_counter} Virtual Stage ایجاد شد")

# گزارش نهایی
log_message("\n" + "="*60)
log_message("✅ کامل کردن Virtual Stages انجام شد!")
log_message("="*60)
log_message(f"\n📊 آمار:")
log_message(f"  - Virtual Stages ایجاد شده: {stage_counter}")
log_message(f"  - خطاها: {len(errors)}")

if errors:
    log_message(f"\n⚠️ خطاها:")
    for error in errors[:10]:
        log_message(f"     - {error}")
    if len(errors) > 10:
        log_message(f"     ... و {len(errors) - 10} خطای دیگر")

# بررسی نهایی
cursor.execute("SELECT COUNT(*) FROM virtual_stages")
total_stages = cursor.fetchone()[0]

cursor.execute("SELECT COUNT(*) FROM persons")
total_persons = cursor.fetchone()[0]

log_message(f"\n📊 آمار نهایی:")
log_message(f"  - کل لیدها: {total_persons}")
log_message(f"  - Virtual Stages: {total_stages}")
log_message(f"  - پوشش: {total_stages*100/total_persons:.1f}%")

conn.close()
log_message("\n✅ آماده!")

