<?php
// Test phone number normalization

function normalizePhone($phone) {
    // Remove all non-digits
    $cleaned = preg_replace('/\D/', '', $phone);
    
    // If empty, return as is
    if (empty($cleaned)) {
        return $cleaned;
    }
    
    // If 10 digits and starts with 9, add 0 at the beginning (Iranian mobile)
    if (strlen($cleaned) === 10 && substr($cleaned, 0, 1) === '9') {
        return '0' . $cleaned;
    }
    
    // If 11 digits and starts with 0, return as is
    if (strlen($cleaned) === 11 && substr($cleaned, 0, 1) === '0') {
        return $cleaned;
    }
    
    // If 11 digits but doesn't start with 0, might be international format
    // Try to convert: if it's 11 digits starting with 9, add 0
    if (strlen($cleaned) === 11 && substr($cleaned, 0, 1) === '9') {
        return '0' . $cleaned;
    }
    
    // Otherwise return as is
    return $cleaned;
}

// Test cases
$testCases = [
    '9025028611' => '09025028611',  // 10 digits -> 11 digits
    '09025028611' => '09025028611', // Already 11 digits
    '09123456789' => '09123456789', // Already 11 digits
    '9123456789' => '09123456789',  // 10 digits -> 11 digits
    '00989123456789' => '00989123456789', // International format
    '02112345678' => '02112345678', // Landline (starts with 0, not 9)
];

echo "=== Phone Normalization Test ===\n\n";

foreach ($testCases as $input => $expected) {
    $result = normalizePhone($input);
    $match = ($result === $expected) ? '✓' : '✗';
    echo "$match Input: $input\n";
    echo "   Expected: $expected\n";
    echo "   Got:      $result\n";
    if ($result !== $expected) {
        echo "   *** MISMATCH ***\n";
    }
    echo "\n";
}

// Test the actual case
echo "=== Actual Case Test ===\n";
$phoneFromDb = '9025028611';
$phoneFromDidar = '09025028611';

$normalizedDb = normalizePhone($phoneFromDb);
$normalizedDidar = normalizePhone($phoneFromDidar);

echo "Phone from DB: $phoneFromDb -> $normalizedDb\n";
echo "Phone from Didar: $phoneFromDidar -> $normalizedDidar\n";
echo "Match: " . ($normalizedDb === $normalizedDidar ? 'YES ✓' : 'NO ✗') . "\n";
?>

